#include "camerathread.h"

#include <QtDebug>

void CameraThread::onOCRDone(QString st)
{
    ocrActive = false;
    qDebug()<<"Done: " + st;
}
void CameraThread::onOCRError(QString st)
{
    qDebug()<<"Error: " + st;
}

void CameraThread::run()
{
    if (!overlay)
    {
        return;
    }

    //a list containing sharpness levels of the last six frames
    QList<int> sharpnessList;

    //attach the lens to the sensor
    sensor.attach(&lens);

    //make a helper autofocus object
    FCam::AutoFocus autoFocus(&lens);

    //make an asynchronous file writer to save images in the background
    FCam::AsyncFileWriter writer;

    //set initial parameters for the viewfinder
    viewfinder.exposure = 33333;
    viewfinder.gain = 1.0f;
    viewfinder.whiteBalance = 5000;

    //run viewfinder at 25 fps
    viewfinder.frameTime = 40000;

    //enable histograms and sharpness maps
    viewfinder.histogram.enabled = true;
    viewfinder.histogram.region = FCam::Rect(0, 0, 640, 480);
    viewfinder.sharpness.enabled = true;
    viewfinder.sharpness.size = FCam::Size(16, 12);

    //dump image data directly into the frame buffer
    viewfinder.image = overlay->framebuffer();

    //stream the viewfinder
    sensor.stream(viewfinder);

    //set image properties and allocate storage
    shot.image = FCam::Image(640, 480, FCam::UYVY, FCam::Image::AutoAllocate);

    //sharness of the current frame
    int currentFrameSharpness = 0;

    //sharness of the previous frame
    int previousFrameSharpness = 0;

    int nr = 0;

    //state of ocr
    ocrActive = false;

    while (keepGoing) {
        //handle FCam events
        FCam::Event e;
        while (FCam::getNextEvent(&e)) {
            switch(e.type) {
            case FCam::Event::Error:
                printf("FCam Error: %s\n", e.description.c_str());
                if (e.data == FCam::Event::DriverLockedError || e.data == FCam::Event::DriverMissingError) {
                    emit panic(e);
                    sensor.stop();
                    exitLock.lock();
                    printf("Terminating\n");
                    return;
                }
                break;
            case FCam::Event::Warning:
                printf("FCam Warning: %s\n", e.description.c_str());
                break;
            case FCam::Event::N900LensClosed:
                emit lensCoverClosed();
                break;
            case FCam::Event::N900LensOpened:
                emit lensCoverOpened();
                break;
            };
        }

        //drain the frame queue
        do {
            //while OCR is active lets just consume frames
            if (ocrActive)
            {
                frame = sensor.getFrame();
                continue;
            }

            //autofocus routine triggers if sharpness of the current frame is lower than given limit
            if(autoFocus.idle() && currentFrameSharpness < 900)
            {
                autoFocus.setTarget(FCam::Rect(0, 0, 640, 480));
                autoFocus.startSweep();
            }

            //auto-expose and auto-white-balance the frame
            //update the viewfinder
            do {
                frame = sensor.getFrame();
                autoExpose(&viewfinder, frame);
                autoWhiteBalance(&viewfinder, frame);
                sensor.stream(viewfinder);
                autoFocus.update(frame);
            }
            while (!autoFocus.idle()); //keep repeating these routines while the lens is not focused

            //this frame should be the result of the shot we made
            if(frame.shot().id == viewfinder.id)
            {
                shot.exposure     = viewfinder.exposure;
                shot.gain         = viewfinder.gain;
                shot.whiteBalance = viewfinder.whiteBalance;
                shot.frameTime = 0;

                //test if it's safe to access the image data
                if(frame.image().valid())
                {                 
                    image = frame.image();

                    currentFrameSharpness = 0;

                    //evaluate sharpness of the current frame
                    for (size_t y = 20; y < image.height()-20; y += 20) {
                        for (size_t x = 20; x < image.width()-20; x += 20) {
                        currentFrameSharpness += abs(image(x, y)[1] - image(x-1, y)[1]);
                        currentFrameSharpness += abs(image(x, y)[1] - image(x+1, y)[1]);
                        currentFrameSharpness += abs(image(x, y)[1] - image(x, y-1)[1]);
                        currentFrameSharpness += abs(image(x, y)[1] - image(x, y+1)[1]);
                        }
                    }

                    currentFrameSharpness = currentFrameSharpness / 10;

                    qDebug() << QString::number(currentFrameSharpness);

                    //compare sharpness of the current and the previous frame
                    if(abs(currentFrameSharpness-previousFrameSharpness > 20) || currentFrameSharpness < 900)
                    {
                        //if the difference is too high, clear the queue
                        sharpnessList.clear();
                    }

                    //if we have gathered more than six frames into the list
                    if(sharpnessList.length() > 6)
                    {
                        //compare sharpness of the current frame and the first frame in the list
                        //if the difference is too high, remove the first frame in the list
                        if(abs(sharpnessList[0]-currentFrameSharpness > 50))
                        {                           
                            sharpnessList.removeAt(0);
                        }
                        //the current frame met requiments: clear the list and request OCR for the frame
                        else
                        {

                            sharpnessList.clear();
                            nr++;
                            QString path = "/home/user/MyDocs/DCIM/kuvaTEST" + QString::number(nr) + ".jpg";
                            writer.saveJPEG(frame, path.toStdString(), 100);
                            ocrActive = true;
                        }
                    }
                    //append sharpness of the current frame into the list
                    else
                    {
                        sharpnessList.append(currentFrameSharpness);
                    }
                    previousFrameSharpness = currentFrameSharpness;
                }
            }
        } while (sensor.framesPending());
    }
    sensor.stop();
}

bool CameraThread::lensCovered()
{
    FILE * file = fopen("/sys/devices/platform/gpio-switch/cam_shutter/state", "r");
    char state = fgetc(file); // file contains either "open" or "closed"
    fclose(file);
    return state == 'c';
}


