#include "cameraoverlay.h"

#define X_UYVY 0x59565955

CameraOverlay::CameraOverlay(QWidget *par) : QWidget(par)  {
    QWidget::setBackgroundRole(QPalette::Window); 
    QWidget::setAutoFillBackground(true); 
    QPalette overlayPalette = QWidget::palette(); 
    overlayPalette.setColor 
        (QPalette::Window,  
         colorKey()); 
    QWidget::setPalette(overlayPalette); 

    //open the overlay device
    overlay_fd = open("/dev/fb1", O_RDWR);

    if (overlay_fd == -1) {
        perror("open");
    }

    //get the current overlay and plane settings
    if (ioctl(overlay_fd, FBIOGET_VSCREENINFO, &overlay_info)) {
        perror("FBIO_VSCREENINFO");
    }
    if (ioctl(overlay_fd, OMAPFB_QUERY_PLANE, &plane_info)) {
        perror("OMAPFB_QUERY_PLANE");
    }

    //disable the plane so we can allocate memory for it.
    plane_info.enabled = 0;
    plane_info.pos_x = 0;
    plane_info.pos_y = 0; 
    plane_info.out_width = 640;
    plane_info.out_height = 480;
    if (ioctl(overlay_fd, OMAPFB_SETUP_PLANE, &plane_info)) {
        perror("OMAPFB_SETUP_PLANE");
    }

    //allocate the memory
    mem_info.size = 640*480*2;
    mem_info.type = 0;
    if (ioctl(overlay_fd, OMAPFB_SETUP_MEM, &mem_info)) {
        perror("OMAPFB_SETUP_MEM");
    }

    //mmap it into an FCam image
    void *ptr = mmap(NULL, mem_info.size, PROT_WRITE, MAP_SHARED, overlay_fd, 0);

    if (ptr == MAP_FAILED) {
        perror("mmap");
    }
    framebuffer_ = FCam::Image(640, 480, FCam::UYVY, (unsigned char *)ptr);

    //clear the memory in case there was something hanging around from an earlier invocation
    memset(ptr, 128, 640*480*2);

    //set the overlay properties
    overlay_info.xres = 640;
    overlay_info.yres = 480;
    overlay_info.xres_virtual = 640;
    overlay_info.yres_virtual = 480;
    overlay_info.xoffset = 0;
    overlay_info.yoffset = 0;
    overlay_info.nonstd = OMAPFB_COLOR_YUV422;

    if (ioctl(overlay_fd, FBIOPUT_VSCREENINFO, &overlay_info)) {
        perror("FBIOPUT_VSCREENINFO");
    }

    //record the original color key
    if (ioctl(overlay_fd, OMAPFB_GET_COLOR_KEY, &old_color_key)){
        perror("OMAPFB_GET_COLOR_KEY");
    }
    //set up the color key
    struct omapfb_color_key color_key;
    color_key.key_type = OMAPFB_COLOR_KEY_GFX_DST;
    QColor key = colorKey();
    color_key.trans_key = ((key.red() >> 3) << 11) | ((key.green() >> 2) << 5) | ((key.blue() >> 3));
    if (ioctl(overlay_fd, OMAPFB_SET_COLOR_KEY, &color_key)) {
        perror("OMAPFB_SET_COLOR_KEY");
    }

    filterInstalled = false;

    //calculate relative line width
    lineLength = this->geometry().width()*0.15;

    //calculate relative point for the cross' centre
    lineCenterPoint = QPointF((plane_info.out_width-75)/2, (plane_info.out_height-50)/2);
}

void CameraOverlay::paintEvent(QPaintEvent *event)
{
    //set painter properties
    QPainter painter(this);
    QPen whitePen(painter.pen());
    whitePen.setWidth(5);
    whitePen.setColor(Qt::white);
    painter.setPen(whitePen);

    //draw the cross
    painter.drawLine(lineCenterPoint.x()-50, lineCenterPoint.y(), lineCenterPoint.x()+50, lineCenterPoint.y());
    painter.drawLine(lineCenterPoint.x(), lineCenterPoint.y()-50, lineCenterPoint.x(), lineCenterPoint.y()+50);
}

bool CameraOverlay::eventFilter(QObject *, QEvent *event) {
    if (event->type() == QEvent::Move ||
        event->type() == QEvent::Resize ||
        event->type() == QEvent::Show) {
        enable();
    } else if (event->type() == QEvent::Hide) {
        disable();
    }

    //we don't capture this event, it should be propagated as normal
    return false;
}

void CameraOverlay::showEvent(QShowEvent *) {
    enable();
}

void CameraOverlay::hideEvent(QHideEvent *) {
    disable();
}

void CameraOverlay::resizeEvent(QResizeEvent *) {
    enable();
}

void CameraOverlay::moveEvent(QMoveEvent *) {
    enable();
}


CameraOverlay::~CameraOverlay() {
    old_color_key.trans_key = 0x842;
    old_color_key.background = 0x0;
    old_color_key.key_type = OMAPFB_COLOR_KEY_GFX_DST;
    old_color_key.channel_out = OMAPFB_CHANNEL_OUT_LCD;
    
    if (ioctl(overlay_fd, OMAPFB_SET_COLOR_KEY, &old_color_key)) {
        perror("OMAPFB_SET_COLOR_KEY");
    }
    disable();
    ::close(overlay_fd);
}

void CameraOverlay::enable() {

    //shift the plane according to where the widget is, but keep it at 640x480
    QPoint global = mapToGlobal(QPoint(0, 0));

    //round to even X
    global.setX(global.x()/2);
    global.setX(global.x()*2);

    int xoff = global.x() > 0 ? global.x() : 0;
    int yoff = global.y() > 0 ? global.y() : 0;
    int xcrop = global.x() < 0 ? -global.x() : 0;
    int ycrop = global.y() < 0 ? -global.y() : 0;

    if (xcrop >= 640 || ycrop >= 480 || xoff >= 640 || yoff >= 480) {
        disable();
        return;
    }

    //set the size and position on screen
    plane_info.enabled = 1;
    plane_info.pos_x = xoff;
    plane_info.pos_y = yoff;
    plane_info.out_width = 640 - xcrop - xoff;
    plane_info.out_height = 480 - ycrop - yoff;

    if (ioctl(overlay_fd, OMAPFB_SETUP_PLANE, &plane_info)) {
        perror("OMAPFB_SETUP_PLANE");
    }

    //the image is always 640x480
    overlay_info.xres_virtual = 640;
    overlay_info.yres_virtual = 480;
    //set the portion of it that's visible on screen
    overlay_info.xres = plane_info.out_width;
    overlay_info.yres = plane_info.out_height;
    overlay_info.xoffset = xcrop;
    overlay_info.yoffset = ycrop;
    overlay_info.nonstd = OMAPFB_COLOR_YUV422;    
    if (ioctl(overlay_fd, FBIOPUT_VSCREENINFO, &overlay_info)) {
        perror("FBIOPUT_VSCREENINFO");
    }

    if (!filterInstalled) {
        //if anything moves above me, we need to know about it to update the overlay
        for (QObject *obj = parent(); obj; obj = obj->parent()) {
            obj->installEventFilter(this);
        }
        filterInstalled = true;
    }
}

void CameraOverlay::disable() {
    plane_info.enabled = 0;
    if (ioctl(overlay_fd, OMAPFB_SETUP_PLANE, &plane_info)) {
        perror("OMAPFB_SETUP_PLANE");
    }
}

FCam::Image CameraOverlay::framebuffer() {
    return framebuffer_;
}
