#include <QMenu>
#include <QtCore>
#include <QCursor>
#include <QColorDialog>
#include <QPainter>
#include <QPropertyAnimation>
#include "maemo5colordialog.h"
#include "drawpad.h"
#include "ui_drawpad.h"

Drawpad::Drawpad(QWidget *parent) :
    QWidget(parent),
    ui(new Ui::Drawpad)
{
    m_paper = 0;
//    background = 0;
    ui->setupUi(this);
    gridSize = QSize(100,100);

    QGraphicsScene *scene = new QGraphicsScene();
    ui->view->setScene(scene);

    strokeRecorder = new StrokeRecorder();
    scene->addItem(strokeRecorder);
    connect(strokeRecorder,SIGNAL(recorded()),
            this,SLOT(onStrokeRecorded()));

    eraser = new Eraser();
    scene->addItem(eraser);

    addAction(ui->actionFreeDraw);
    addAction(ui->actionMove);
    addAction(ui->actionErase);

    QActionGroup *group = new QActionGroup(this);
    group->addAction(ui->actionFreeDraw);
    group->addAction(ui->actionErase);
    group->addAction(ui->actionMove);
    ui->actionMove->setChecked(true);
    connect(group,SIGNAL(triggered(QAction*)),
            this,SLOT(onActionTriggered(QAction*)));

    addAction(ui->actionBrush);
    addAction(ui->actionColor);

    brushMenu = new QMenu(this);
    brushMenu->addAction(ui->actionBrush1);
    brushMenu->addAction(ui->actionBrush2);
    brushMenu->addAction(ui->actionBrush3);
    brushMenu->addAction(ui->actionBrush4);

    connect(&mapper,SIGNAL(mapped(int)),
            this,SLOT(setBrushType(int)));

    connect(ui->actionBrush1,SIGNAL(triggered()),
            &mapper,SLOT(map()));
    connect(ui->actionBrush2,SIGNAL(triggered()),
            &mapper,SLOT(map()));
    connect(ui->actionBrush3,SIGNAL(triggered()),
            &mapper,SLOT(map()));
    connect(ui->actionBrush4,SIGNAL(triggered()),
            &mapper,SLOT(map()));
    mapper.setMapping(ui->actionBrush1,0);
    mapper.setMapping(ui->actionBrush2,1);
    mapper.setMapping(ui->actionBrush3,2);
    mapper.setMapping(ui->actionBrush4,3);

    m_directionActions << ui->actionUp;
    m_directionActions << ui->actionDown;
    m_directionActions << ui->actionLeft;
    m_directionActions << ui->actionRight;

    m_mode = FreeMoveMode;

    /// @todo Should not set default paper
    setPaper(new SketchPaper());

    setBrushType(1);

    setColor(Qt::black);

    setMode(FreeDrawMode);

}

Drawpad::~Drawpad()
{
    delete ui;
}

void Drawpad::setPaper(Paper *new_paper){
    if (m_paper == new_paper) {
        refresh();
        return;
    }

    QGraphicsScene *s = ui->view->scene();

    Mode oldMode = m_mode;
    if (m_paper){
        setMode(FreeMoveMode); // Disable all binding

        m_paper->deleteLater();
        m_paper = 0;
    }

    m_paper = new_paper;

    if (m_paper){
        connect(strokeRecorder,SIGNAL(recorded()),
                m_paper,SLOT(setDirty()));
        connect(eraser,SIGNAL(erased()),
                m_paper,SLOT(setDirty()));
        connect(eraser,SIGNAL(erased()),
                ui->view->scene(),SLOT(invalidate()));
        s->addItem(m_paper);

        setMode(oldMode); // Restore to original mode

        refresh();
    }
}

void Drawpad::refresh(){
    QGraphicsScene *s = ui->view->scene();
    QRectF rect = m_paper->boundingRect();
//    qDebug() << __func__<< s->sceneRect() << rect;
    s->setSceneRect(rect);

    QColor bg = m_paper->background().color();
    ui->view->scene()->setBackgroundBrush(bg);
    ui->view->setBackgroundBrush(bg);

}

Paper* Drawpad::paper(){
    return m_paper;
}

void Drawpad::setMode(Mode val){
    if (m_mode == val)
        return;

    switch (m_mode) { // Original mode
        case FreeMoveMode:
            ui->view->setDragMode(QGraphicsView::NoDrag);
            break;

        case FreeDrawMode:
//            ui->actionFreeDraw->setChecked(false);

            strokeRecorder->stop();
            break;

        case EraserMode:
//            ui->actionErase->setChecked(false);
            eraser->stop();
            break;
    }

    m_mode = val;

    switch (m_mode) {
        case FreeMoveMode:
            ui->view->setDragMode(QGraphicsView::ScrollHandDrag	);
            ui->actionMove->setChecked(true);
            break;

        case FreeDrawMode:
            ui->actionFreeDraw->setChecked(true);
            strokeRecorder->record(m_paper);
            break;

        case EraserMode:
            ui->actionErase->setChecked(true);
            eraser->start(m_paper);
            break;
    }

}

Drawpad::Mode Drawpad::mode(){
    return m_mode;
}

QGraphicsView *Drawpad::view(){
    return ui->view;
}

QDomDocument Drawpad::toXml(){
    QDomDocument doc;
    QDomElement elem = m_paper->createElement(doc);
    doc.appendChild(elem);
    return doc;
}

void Drawpad::changeEvent(QEvent *e)
{
    QWidget::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void Drawpad::onStrokeRecorded(){
    strokeRecorder->record(m_paper);
}

void Drawpad::on_actionBrush_triggered()
{
    brushMenu->exec(QCursor::pos());
}

void Drawpad::onActionTriggered(QAction *action){
    if (action == ui->actionMove ){
        setMode(FreeMoveMode);
    } else if (action == ui->actionFreeDraw){
        setMode(FreeDrawMode);
    } else {
        setMode(EraserMode);
    }

}

void Drawpad::setBrushType(int type){
    QPen pen = strokeRecorder->pen();
    int width = 1;
    m_brushType = type;
    QIcon icon;
    switch (type){
    case 0:
        icon = QIcon(":/drawpad/brush1.png");
        width = 1;
        break;
    case 1:
        icon = QIcon(":/drawpad/brush2.png");
        width = 3;
        break;
    case 2:
        icon = QIcon(":/drawpad/brush3.png");
        width = 5;
        break;
    case 3:
    default:
        icon = QIcon(":/drawpad/brush4.png");
        width = 7;
        break;
    }

    ui->actionBrush->setIcon(icon);
    pen.setWidth(width);
    strokeRecorder->setPen(pen);
}

int Drawpad::brushType(){
    return m_brushType;
}

void Drawpad::setColor(QColor color){
    m_color = color;

    QPen pen = strokeRecorder->pen();
    pen.setColor(color);
    qDebug() << __func__ << pen;
    strokeRecorder->setPen(pen);

    QPixmap pixmap(48,48);
    QRect rect(0,0,48,48);

    pen = QPen();
    pen.setColor(Qt::white);
    pen.setStyle(Qt::SolidLine);
    pen.setWidth(5);

    QBrush brush;
    brush.setColor(color);
    brush.setStyle(Qt::SolidPattern);

    QPainter painter(&pixmap);
    painter.setPen(pen);
    painter.setBrush(brush);
    painter.drawRect(rect);
    painter.end();

//    pixmap.fill(color);

    QIcon icon(pixmap);
    ui->actionColor->setIcon(icon);

}

QColor Drawpad::color(){
    return m_color;
}

bool Drawpad::save(QString file){
    QGraphicsScene *scene = ui->view->scene();
    QRectF source = m_paper->childrenBoundingRect();

    QRectF target = source;
    target.moveTopLeft(QPointF(0,0));

    qDebug() << target << source;

    QPixmap pixmap(target.width(),target.height());
    pixmap.fill();
    QPainter painter;
    painter.begin(&pixmap);
    scene->render(&painter,target,source,Qt::KeepAspectRatio);
    painter.end();

    return pixmap.save(file);
}

QList<QAction*> Drawpad::directionActions(){
    return m_directionActions;
}

void Drawpad::setGridSize(QSize val){
    qDebug() << __func__ << val;
    gridSize = val;
}

void Drawpad::move(int vmove,int hmove){
    QPointF start = center();

    QRectF rect1 = viewRect(); // Current rect
//    qDebug() << __func__ << rect1;
    rect1.moveTopLeft( QPointF( rect1.x() + hmove * gridSize.width()
                      , rect1.y() + vmove * gridSize.height() ) );

    QPointF target = rect1.center();

//    qDebug() << rect1 << viewportSize << start << target;

    QRectF rect2 = m_paper->rect();
    QRectF rect3 = rect1.united(rect2);

    m_paper->setRect(rect3.toRect());

    refresh();


    QPropertyAnimation *animation = new QPropertyAnimation(this,"center",this);
    animation->setStartValue(start);
    animation->setEndValue(target);
    animation->setDuration(800);
    animation->start();

    connect(animation,SIGNAL(finished()),
            animation,SLOT(deleteLater()));
}

void Drawpad::setCenter(QPointF val){
    ui->view->centerOn(val);
}

QPointF Drawpad::center(){
    QRectF rect = viewRect();
    return rect.center();
}

QRectF Drawpad::viewRect() {
    QSize size = ui->view->size();

    QRect rect(0,0,size.width(),size.height());
    QPolygonF polygon = ui->view->mapToScene(rect);
    return polygon.boundingRect();
}

void Drawpad::on_actionColor_triggered()
{
    Maemo5ColorDialog dialog(this);
//    QColorDialog dialog(this);
    dialog.setCurrentColor(m_color);
//    dialog.setOption(QColorDialog::ShowAlphaChannel,false);
    if (dialog.exec() == QDialog::Accepted ){
        qDebug() << __func__ << dialog.selectedColor();
        setColor(dialog.selectedColor());
        setMode(FreeDrawMode);
    }


}

void Drawpad::on_actionLeft_triggered()
{
    move(0,-1);
}

void Drawpad::on_actionRight_triggered()
{
    move(0,1)    ;
}

void Drawpad::on_actionUp_triggered()
{
    move(-1,0);
}

void Drawpad::on_actionDown_triggered()
{
    move(1,0);
}
