/*
 * Copyright (c) 2009 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Hallyson Melo <hallyson.melo@indt.org.br>
 *          Walter Guerra <walter.guerra@signove.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */
 
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <stdlib.h>
#include <string.h>


#include "nm_vnc_setup.h"
#include "nm_settings.h"
#include "nm_environment_manager.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

typedef struct _NmVncSetupPrivate NmVncSetupPrivate;

struct _NmVncSetupPrivate
{
    GtkWidget *checkbutton;

    GtkWidget *port_label;
    GtkWidget *loop_label;

    GtkWidget *port_entry;
    GtkWidget *loop_checkbox;
    
    gchar *env;
};

#define NM_VNC_SETUP_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_VNC_SETUP_TYPE, NmVncSetupPrivate))

static void nm_vnc_setup_class_init (NmVncSetupClass *);
static void nm_vnc_setup_init       (NmVncSetup *);
static void nm_vnc_setup_dispose    (GObject *);
static void nm_vnc_setup_finalize   (GObject *);

static void nm_vnc_setup_defaults (NmVncSetup *, gboolean);
static void nm_vnc_setup_create_widget (NmVncSetup *);
static void nm_vnc_setup_enable_cb (GtkToggleButton *, gpointer);
static void nm_vnc_setup_enable_options (NmVncSetup *, gboolean);
    
G_DEFINE_TYPE (NmVncSetup, nm_vnc_setup, NM_CONN_WIDGET_TYPE)

static void
nm_vnc_setup_class_init (NmVncSetupClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);
    NmConnWidgetClass *conn_widget_class = NM_CONN_WIDGET_CLASS (klass);

    g_type_class_add_private (klass, sizeof (NmVncSetupPrivate));

    object_class->dispose = nm_vnc_setup_dispose;
    object_class->finalize = nm_vnc_setup_finalize;

    conn_widget_class->apply_changes = nm_vnc_setup_apply_changes;
}

static void
nm_vnc_setup_init (NmVncSetup *self)
{
    nm_vnc_setup_create_widget (self);
}

static void
nm_vnc_setup_dispose (GObject *object)
{
    G_OBJECT_CLASS (nm_vnc_setup_parent_class)->dispose (object);
}

static void
nm_vnc_setup_finalize (GObject *object)
{
    G_OBJECT_CLASS (nm_vnc_setup_parent_class)->finalize (object);
}

GtkWidget*
nm_vnc_setup_new (gchar *env)
{
    GtkWidget *self = g_object_new (NM_VNC_SETUP_TYPE, NULL);
    
    NmVncSetupPrivate *priv;

    priv = NM_VNC_SETUP_GET_PRIVATE (self);
    
    priv->env = g_strdup(env);
    
    g_debug("[%s] - priv->env: %s", __FUNCTION__, priv->env);
    
    return self;
}

void
nm_vnc_setup_enable (NmVncSetup *self, gboolean enable)
{
    NmVncSetupPrivate *priv;

    priv = NM_VNC_SETUP_GET_PRIVATE (self);

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(priv->checkbutton), enable);
}

gboolean
nm_vnc_setup_is_enabled (NmVncSetup *self)
{
    NmVncSetupPrivate *priv;

    priv = NM_VNC_SETUP_GET_PRIVATE (self);
    return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(priv->checkbutton));
}

static void
nm_vnc_setup_create_widget (NmVncSetup *self)
{
    GtkWidget *table;
    NmVncSetupPrivate *priv;

    priv = NM_VNC_SETUP_GET_PRIVATE (self);

    table = gtk_table_new (5, 2, TRUE);

    priv->checkbutton = gtk_check_button_new_with_label (_("enable"));
    g_signal_connect (priv->checkbutton, "toggled", G_CALLBACK(nm_vnc_setup_enable_cb),
                      self);

    /* Port number */
    priv->port_label = gtk_label_new(_("Port Number: "));
    gtk_misc_set_alignment(GTK_MISC(priv->port_label), 1, 0.5);
    priv->port_entry = gtk_entry_new();

    /* loop */
    priv->loop_label = gtk_label_new(_("Loop: "));
    gtk_misc_set_alignment(GTK_MISC(priv->loop_label), 1, 0.5);
    priv->loop_checkbox = gtk_check_button_new ();


    gtk_table_attach_defaults(GTK_TABLE (table), priv->checkbutton, 0, 1, 0, 1);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->port_label, 0, 1, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->port_entry, 1, 2, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->loop_label, 0, 1, 2, 3);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->loop_checkbox, 1, 2, 2, 3);
    
    GtkWidget *scroll;
    scroll = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll), 
                                    GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroll),
                                           table);    
    
    gtk_container_add(GTK_CONTAINER (self), scroll);

    nm_vnc_setup_enable_options(self, FALSE);
}

static void
nm_vnc_setup_defaults (NmVncSetup *self, gboolean pressed)
{
    NmVncSetupPrivate *priv;

    // FIXME: This is not a good solution. It should be done during class init
    NmSettings *settings = nm_settings_new ();
    nm_settings_start (settings);

    priv = NM_VNC_SETUP_GET_PRIVATE (self);

    if (pressed) {

        gtk_entry_set_text (GTK_ENTRY (priv->port_entry),
                             nm_settings_get_value (settings, priv->env,
                                                     NM_VNC_SETTINGS_GROUP,
                                                     NM_VNC_PORT));

                                                     
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priv->loop_checkbox),
                             nm_settings_get_boolean (settings, priv->env,
                                                     NM_VNC_SETTINGS_GROUP,
                                                     NM_VNC_LOOP));


    } else {
        gtk_entry_set_text(GTK_ENTRY (priv->port_entry), "");
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON (priv->loop_checkbox), FALSE );

    }
}

static void
nm_vnc_setup_enable_cb (GtkToggleButton *button, gpointer data)
{
    NmVncSetupPrivate *priv;
    gboolean pressed = gtk_toggle_button_get_active (button);

    priv = NM_VNC_SETUP_GET_PRIVATE (data);
    
    nm_vnc_setup_defaults (data, pressed);

    // FIXME: remove this if the combo box is needed for something else
    nm_vnc_setup_enable_options (data, pressed);
    nm_conn_widget_settings_changed (data);
}

static void
nm_vnc_setup_enable_options (NmVncSetup *self, gboolean enable)
{
    NmVncSetupPrivate *priv;
    priv = NM_VNC_SETUP_GET_PRIVATE (self);

    gtk_widget_set_sensitive (priv->port_label, enable);
    gtk_widget_set_sensitive (priv->port_entry, enable);
    gtk_widget_set_sensitive (priv->loop_label, enable);
    gtk_widget_set_sensitive (priv->loop_checkbox, enable);
}

gboolean
nm_vnc_setup_apply_changes (NmConnWidget *self)
{
    g_debug("[%s]", __FUNCTION__);
    NmVncSetup *vncsetup = NM_VNC_SETUP(self);
    
    NmVncSetupPrivate *priv;
    priv = NM_VNC_SETUP_GET_PRIVATE (vncsetup);
    
    nm_vnc_setup_save_changes(vncsetup);
    
    if ( nm_vnc_setup_is_enabled(vncsetup) ) {
        
        const gchar *port = gtk_entry_get_text(GTK_ENTRY(priv->port_entry));
        int loop = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(priv->loop_checkbox));
        
        gchar command[50];
        
        sprintf(command, "sudo /usr/sbin/nm-vnc-setup start %d %s", loop, port);
        
        g_debug("[%s] - command: %s ", __FUNCTION__, command);
        
        system("sudo /usr/sbin/nm-vnc-setup stop");
        system(command);
        
        /* Export with avahi */
        sprintf(command, "sudo /usr/sbin/nm-avahi start_vnc %s", port); // Set the port
        system(command);
        
    }
    else {
        system("sudo /usr/sbin/nm-vnc-setup stop");
        system("sudo /usr/sbin/nm-avahi stop_vnc");
    }
    
    
    return TRUE;   
}

gboolean 
nm_vnc_setup_activate_environment (gchar *env)
{
    g_debug("[%s]", __FUNCTION__);
    gboolean ret = TRUE;
    
    NmSettings *settings = nm_settings_new ();
    nm_settings_start (settings);
    
    gboolean enabled = nm_settings_get_boolean (settings, env,
                                                     NM_VNC_SETTINGS_GROUP,
                                                     NM_SETTINGS_ENABLE);

    GtkWidget *vnc = nm_vnc_setup_new(env);
    nm_vnc_setup_enable(NM_VNC_SETUP(vnc), enabled);
    ret = nm_vnc_setup_apply_changes(NM_CONN_WIDGET(vnc));

    return ret;
}

void 
nm_vnc_setup_save_changes (NmVncSetup *self) 
{
    NmVncSetupPrivate *priv = NM_VNC_SETUP_GET_PRIVATE (self);

    NmSettings *settings = nm_settings_new();
  
    nm_settings_set_boolean (settings, priv->env, NM_VNC_SETTINGS_GROUP,
                        NM_SETTINGS_ENABLE,  nm_vnc_setup_is_enabled(self));  
 
    if (nm_vnc_setup_is_enabled(self)) {
        nm_settings_set_value (settings, priv->env, NM_VNC_SETTINGS_GROUP,
                            NM_VNC_PORT,
                            gtk_entry_get_text (GTK_ENTRY (priv->port_entry)));
                            
        nm_settings_set_boolean (settings, priv->env, NM_VNC_SETTINGS_GROUP,
                            NM_VNC_LOOP,
                            gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (priv->loop_checkbox)));
       
    }

    nm_settings_save(settings);
}

gboolean
vnc_setup_check_enabled (NmVncSetup *self)
{
    NmVncSetupPrivate *priv = NM_VNC_SETUP_GET_PRIVATE (self);

    if ( system("sudo /usr/sbin/nm-vnc-setup isrunning") ) {

        nm_vnc_setup_enable (self, TRUE);
        nm_vnc_setup_enable_cb (GTK_TOGGLE_BUTTON (priv->checkbutton), self);
        return TRUE;
    } else {

        nm_vnc_setup_enable (self, FALSE);
        nm_vnc_setup_enable_cb (GTK_TOGGLE_BUTTON (priv->checkbutton), self);
        return FALSE;
    }
/*    return TRUE;*/
}

void
nm_vnc_setup_fill_widget(NmVncSetup *self)
{
    NmVncSetupPrivate *priv = NM_VNC_SETUP_GET_PRIVATE (self);
    
    if (strcmp(priv->env, nm_environment_manager_get_active())) {
/*        g_debug("[%s] - %s is not the active environment", __FUNCTION__, priv->env);*/
        NmSettings *settings = nm_settings_new ();
        nm_settings_start (settings);
/*        g_debug("[%s] - vnc enabled: %d", __FUNCTION__, */
/*                    nm_settings_get_boolean (settings, priv->env, NM_USB_SETTINGS_GROUP, NM_SETTINGS_ENABLE));*/
        
        nm_vnc_setup_enable(self, nm_settings_get_boolean (settings, priv->env, NM_VNC_SETTINGS_GROUP, NM_SETTINGS_ENABLE));
        
    }
    else {
/*        g_debug("[%s] - %s is the active environment", __FUNCTION__, priv->env);*/
        vnc_setup_check_enabled (self);
    }
}

