/*
 * Copyright (c) 2009 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Hallyson Melo <hallyson.melo@indt.org.br>
 *          Kyller Gorgonio <kyllercg@gmail.com>
 *          Walter Guerra <walter.guerra@signove.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */
 
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <stdlib.h>
#include <string.h>


#include "nm_unfs_setup.h"
#include "nm_settings.h"
#include "nm_environment_manager.h"
#include "nm_file_dialog.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

typedef struct _NmUnfsSetupPrivate NmUnfsSetupPrivate;

struct _NmUnfsSetupPrivate
{
    GtkWidget *checkbutton;

    GtkWidget *port_label;
    GtkWidget *singleuser_label;
    GtkWidget *exported_paths_label;
    GtkWidget *allowed_ips_label;
/*    GtkWidget *debug_label;*/

    GtkWidget *port_entry;
    GtkWidget *singleuser_checkbox;
    GtkWidget *exported_paths_button;
    GtkWidget *allowed_ips_button;
/*    GtkWidget *debug_checkbox;*/
    
    gchar *env;
};

#define NM_UNFS_SETUP_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_UNFS_SETUP_TYPE, NmUnfsSetupPrivate))

static void nm_unfs_setup_class_init (NmUnfsSetupClass *);
static void nm_unfs_setup_init       (NmUnfsSetup *);
static void nm_unfs_setup_dispose    (GObject *);
static void nm_unfs_setup_finalize   (GObject *);

static void nm_unfs_setup_defaults (NmUnfsSetup *, gboolean);
static void nm_unfs_setup_create_widget (NmUnfsSetup *);
static void nm_unfs_setup_enable_cb (GtkToggleButton *, gpointer);
static void nm_unfs_setup_enable_options (NmUnfsSetup *, gboolean);
static void nm_unfs_setup_manage_exported_paths (gpointer , gpointer );
static void nm_unfs_setup_manage_allowed_ips (gpointer , gpointer );
    
G_DEFINE_TYPE (NmUnfsSetup, nm_unfs_setup, NM_CONN_WIDGET_TYPE)

static void
nm_unfs_setup_class_init (NmUnfsSetupClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);
    NmConnWidgetClass *conn_widget_class = NM_CONN_WIDGET_CLASS (klass);

    g_type_class_add_private (klass, sizeof (NmUnfsSetupPrivate));

    object_class->dispose = nm_unfs_setup_dispose;
    object_class->finalize = nm_unfs_setup_finalize;

    conn_widget_class->apply_changes = nm_unfs_setup_apply_changes;
}

static void
nm_unfs_setup_init (NmUnfsSetup *self)
{
    nm_unfs_setup_create_widget (self);
}

static void
nm_unfs_setup_dispose (GObject *object)
{
    G_OBJECT_CLASS (nm_unfs_setup_parent_class)->dispose (object);
}

static void
nm_unfs_setup_finalize (GObject *object)
{
    G_OBJECT_CLASS (nm_unfs_setup_parent_class)->finalize (object);
}

GtkWidget*
nm_unfs_setup_new (gchar *env)
{
    g_debug("[%s] - env: %s", __FUNCTION__, env);
    
    GtkWidget *self = g_object_new (NM_UNFS_SETUP_TYPE, NULL);
    
    NmUnfsSetupPrivate *priv;

    priv = NM_UNFS_SETUP_GET_PRIVATE (self);
    
    priv->env = g_strdup(env);    
    
    return self;
}

void
nm_unfs_setup_enable (NmUnfsSetup *self, gboolean enable)
{
    NmUnfsSetupPrivate *priv;

    priv = NM_UNFS_SETUP_GET_PRIVATE (self);

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(priv->checkbutton), enable);
}

gboolean
nm_unfs_setup_is_enabled (NmUnfsSetup *self)
{
    NmUnfsSetupPrivate *priv;

    priv = NM_UNFS_SETUP_GET_PRIVATE (self);
    return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(priv->checkbutton));
}

static void
nm_unfs_setup_create_widget (NmUnfsSetup *self)
{
    GtkWidget *table;
    NmUnfsSetupPrivate *priv;

    priv = NM_UNFS_SETUP_GET_PRIVATE (self);

    table = gtk_table_new (5, 2, TRUE);

    priv->checkbutton = gtk_check_button_new_with_label (_("enable"));
    g_signal_connect (priv->checkbutton, "toggled", G_CALLBACK(nm_unfs_setup_enable_cb),
                      self);

    /* Port number */
    priv->port_label = gtk_label_new(_("Port Number: "));
    gtk_misc_set_alignment(GTK_MISC(priv->port_label), 1, 0.5);
    priv->port_entry = gtk_entry_new();

    /* Single User */
    priv->singleuser_label = gtk_label_new(_("Single User: "));
    gtk_misc_set_alignment(GTK_MISC(priv->singleuser_label), 1, 0.5);
    priv->singleuser_checkbox = gtk_check_button_new ();

    /* Exported Paths */
    priv->exported_paths_label = gtk_label_new(_("Exported Paths: "));
    gtk_misc_set_alignment(GTK_MISC(priv->exported_paths_label), 1, 0.5);
    priv->exported_paths_button = gtk_button_new_with_label ("...");
    g_signal_connect (G_OBJECT (priv->exported_paths_button),
                       "clicked",
                       G_CALLBACK (nm_unfs_setup_manage_exported_paths), self);

    /* Allowed IPs */
    priv->allowed_ips_label = gtk_label_new(_("Allowed IPs: "));
    gtk_misc_set_alignment(GTK_MISC(priv->allowed_ips_label), 1, 0.5);
    priv->allowed_ips_button = gtk_button_new_with_label ("...");
    g_signal_connect (G_OBJECT (priv->allowed_ips_button),
                       "clicked",
                       G_CALLBACK (nm_unfs_setup_manage_allowed_ips), self);

    gtk_table_attach_defaults(GTK_TABLE (table), priv->checkbutton, 0, 1, 0, 1);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->exported_paths_label, 0, 1, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->exported_paths_button, 1, 2, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->allowed_ips_label, 0, 1, 2, 3);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->allowed_ips_button, 1, 2, 2, 3);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->port_label, 0, 1, 3, 4);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->port_entry, 1, 2, 3, 4);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->singleuser_label, 0, 1, 4, 5);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->singleuser_checkbox, 1, 2, 4, 5);
    
    GtkWidget *scroll;
    scroll = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll), 
                                    GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroll),
                                           table);    
    
    gtk_container_add(GTK_CONTAINER (self), scroll);

    nm_unfs_setup_enable_options(self, FALSE);
}

static void
nm_unfs_setup_defaults (NmUnfsSetup *self, gboolean pressed)
{
    NmUnfsSetupPrivate *priv;

    // FIXME: This is not a good solution. It should be done during class init
    NmSettings *settings = nm_settings_new ();
    nm_settings_start (settings);

    priv = NM_UNFS_SETUP_GET_PRIVATE (self);

    if (pressed) {

        gtk_entry_set_text (GTK_ENTRY (priv->port_entry),
                             nm_settings_get_value (settings, priv->env,
                                                     NM_UNFS_SETTINGS_GROUP,
                                                     NM_UNFS_PORT));

                                                     
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priv->singleuser_checkbox),
                             nm_settings_get_boolean (settings, priv->env,
                                                     NM_UNFS_SETTINGS_GROUP,
                                                     NM_UNFS_SINGLEUSER));

                                                     
    } else {
        gtk_entry_set_text(GTK_ENTRY (priv->port_entry), "");
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON (priv->singleuser_checkbox), FALSE );
    }
}

static void
nm_unfs_setup_enable_cb (GtkToggleButton *button, gpointer data)
{
    NmUnfsSetupPrivate *priv;
    gboolean pressed = gtk_toggle_button_get_active (button);

    priv = NM_UNFS_SETUP_GET_PRIVATE (data);
    
    nm_unfs_setup_defaults (data, pressed);

    // FIXME: remove this if the combo box is needed for something else
    nm_unfs_setup_enable_options (data, pressed);
    nm_conn_widget_settings_changed (data);
}

static void
nm_unfs_setup_enable_options (NmUnfsSetup *self, gboolean enable)
{
    NmUnfsSetupPrivate *priv;
    priv = NM_UNFS_SETUP_GET_PRIVATE (self);
    
    gtk_widget_set_sensitive (priv->exported_paths_label, enable);
    gtk_widget_set_sensitive (priv->exported_paths_button, enable);
    gtk_widget_set_sensitive (priv->allowed_ips_label, enable);
    gtk_widget_set_sensitive (priv->allowed_ips_button, enable);
    gtk_widget_set_sensitive (priv->port_label, enable);
    gtk_widget_set_sensitive (priv->port_entry, enable);
    gtk_widget_set_sensitive (priv->singleuser_label, enable);
    gtk_widget_set_sensitive (priv->singleuser_checkbox, enable);
}

gboolean
nm_unfs_setup_apply_changes (NmConnWidget *self)
{
    g_debug("[%s]", __FUNCTION__);
    NmUnfsSetup *unfssetup = NM_UNFS_SETUP(self);
    
    NmUnfsSetupPrivate *priv;
    priv = NM_UNFS_SETUP_GET_PRIVATE (unfssetup);
    
    nm_unfs_setup_save_changes(unfssetup);
    
    if ( nm_unfs_setup_is_enabled(unfssetup) ) {

        /* Update the exports */
        GtkWidget *dialog = nm_file_dialog_new(priv->env, NM_UNFS_SETTINGS_GROUP, NM_FILE_DIALOG_FOLDER_DIALOG);
        gsize length = 0;
        gchar **list = nm_file_dialog_get_list(NM_FILE_DIALOG(dialog), &length);
        
        gchar *concat_paths = "";
        int i;
        for (i = 0; i < length; i++) {
            concat_paths = g_strconcat(concat_paths, " ", list[i], NULL);
        }
        
        GtkWidget *dialog2 = nm_file_dialog_new(priv->env, NM_UNFS_SETTINGS_GROUP, NM_FILE_DIALOG_TEXT_DIALOG);
        length = 0;
        gchar **list2 = nm_file_dialog_get_list(NM_FILE_DIALOG(dialog2), &length);
        
        gchar *concat_ips = "127.0.0.1,";
        for (i = 0; i < length; i++) {
            concat_ips = g_strconcat(concat_ips, list2[i], ",", NULL);
        }
        
        g_debug("[%s] - concat_list: %s %s", __FUNCTION__, concat_ips, concat_paths);
        gchar command[50 + strlen(concat_paths) + strlen(concat_ips)];
        
        sprintf(command, "sudo /usr/sbin/nm-unfs-setup gen_exports %s %s", concat_ips, concat_paths);
        system(command);
        
        gtk_widget_destroy(dialog);
        gtk_widget_destroy(dialog2);
        g_free(concat_ips);
        g_free(concat_paths);
        /* END updating the exports */
        
        const gchar *port = gtk_entry_get_text(GTK_ENTRY(priv->port_entry));
        int singleuser = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(priv->singleuser_checkbox));

        sprintf(command, "sudo /usr/sbin/nm-unfs-setup config %d %s", singleuser, port);
        
        g_debug("[%s] - command: %s ", __FUNCTION__, command);
        
        /* Update the config file*/
        system(command);
        
        /* Restart the deamon */ 
        system("sudo /usr/sbin/nm-unfs-setup stop");
        system("sudo /usr/sbin/nm-unfs-setup start");
        
        /* Export with avahi */
        sprintf(command, "sudo /usr/sbin/nm-avahi start_nfs %s", port);
        system(command);
        
    }
    else {
        system("sudo /usr/sbin/nm-unfs-setup gen_exports none");
        system("sudo /usr/sbin/nm-unfs-setup stop");
        system("sudo /usr/sbin/nm-avahi stop_nfs");
    }
    
    
    return TRUE;   
}

gboolean 
nm_unfs_setup_activate_environment (gchar *env)
{
    g_debug("[%s]", __FUNCTION__);
    gboolean ret = TRUE;
    
    NmSettings *settings = nm_settings_new ();
    nm_settings_start (settings);
    
    gboolean enabled = nm_settings_get_boolean (settings, env,
                                                     NM_UNFS_SETTINGS_GROUP,
                                                     NM_SETTINGS_ENABLE);

    GtkWidget *unfs = nm_unfs_setup_new(env);
    nm_unfs_setup_enable(NM_UNFS_SETUP(unfs), enabled);
    ret = nm_unfs_setup_apply_changes(NM_CONN_WIDGET(unfs));

    return ret;
}

void 
nm_unfs_setup_save_changes (NmUnfsSetup *self) 
{
    NmUnfsSetupPrivate *priv = NM_UNFS_SETUP_GET_PRIVATE (self);

    NmSettings *settings = nm_settings_new();
  
    nm_settings_set_boolean (settings, priv->env, NM_UNFS_SETTINGS_GROUP,
                        NM_SETTINGS_ENABLE,  nm_unfs_setup_is_enabled(self));  
 
    if (nm_unfs_setup_is_enabled(self)) {
        nm_settings_set_value (settings, priv->env, NM_UNFS_SETTINGS_GROUP,
                            NM_UNFS_PORT,
                            gtk_entry_get_text (GTK_ENTRY (priv->port_entry)));
                            
        nm_settings_set_boolean (settings, priv->env, NM_UNFS_SETTINGS_GROUP,
                            NM_UNFS_SINGLEUSER,
                            gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (priv->singleuser_checkbox)));

    }

    nm_settings_save(settings);
}

gboolean
unfs_setup_check_enabled (NmUnfsSetup *self)
{
    NmUnfsSetupPrivate *priv = NM_UNFS_SETUP_GET_PRIVATE (self);

    if ( system("sudo /usr/sbin/nm-unfs-setup isrunning") ) {

        nm_unfs_setup_enable (self, TRUE);
        nm_unfs_setup_enable_cb (GTK_TOGGLE_BUTTON (priv->checkbutton), self);
        return TRUE;
    } else {

        nm_unfs_setup_enable (self, FALSE);
        nm_unfs_setup_enable_cb (GTK_TOGGLE_BUTTON (priv->checkbutton), self);
        return FALSE;
    }
/*    return TRUE;*/
}

void
nm_unfs_setup_fill_widget(NmUnfsSetup *self)
{
    NmUnfsSetupPrivate *priv = NM_UNFS_SETUP_GET_PRIVATE (self);
    
    if (strcmp(priv->env, nm_environment_manager_get_active())) {
/*        g_debug("[%s] - %s is not the active environment", __FUNCTION__, priv->env);*/
        NmSettings *settings = nm_settings_new ();
        nm_settings_start (settings);
/*        g_debug("[%s] - unfs enabled: %d", __FUNCTION__, */
/*                    nm_settings_get_boolean (settings, priv->env, NM_USB_SETTINGS_GROUP, NM_SETTINGS_ENABLE));*/
        
        nm_unfs_setup_enable(self, nm_settings_get_boolean (settings, priv->env, NM_UNFS_SETTINGS_GROUP, NM_SETTINGS_ENABLE));
        
    }
    else {
/*        g_debug("[%s] - %s is the active environment", __FUNCTION__, priv->env);*/
        unfs_setup_check_enabled (self);
    }
}

static void
nm_unfs_setup_manage_exported_paths(gpointer instance, gpointer self)
{
    
    NmUnfsSetupPrivate *priv = NM_UNFS_SETUP_GET_PRIVATE (self);
    
    g_debug("[%s] - env: %s", __FUNCTION__, priv->env);
    
    GtkWidget *dialog = nm_file_dialog_new(priv->env, NM_UNFS_SETTINGS_GROUP, NM_FILE_DIALOG_FOLDER_DIALOG);
   
    nm_file_dialog_show_dialog(NM_FILE_DIALOG(dialog));

    gtk_widget_destroy(dialog);

}

static void
nm_unfs_setup_manage_allowed_ips(gpointer instance, gpointer self)
{
    
    NmUnfsSetupPrivate *priv = NM_UNFS_SETUP_GET_PRIVATE (self);
    
    g_debug("[%s] - env: %s", __FUNCTION__, priv->env);
    
    GtkWidget *dialog = nm_file_dialog_new(priv->env, NM_UNFS_SETTINGS_GROUP, NM_FILE_DIALOG_IPS);
   
    nm_file_dialog_show_dialog(NM_FILE_DIALOG(dialog));

    gtk_widget_destroy(dialog);

}

