/*
 * Copyright (c) 2008 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Marcos Morais <morais@dee.ufcg.edu.br>
 *          Kyller Gorgonio <kyllercg@gmail.com>
 *          Andre Rodrigues <andre.rodrigues@signove.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <gconf/gconf-client.h>
#include <hildon/hildon-banner.h>

#include "nm_wlan_setup.h"
#include "nm_wlan_utils.h"
#include "nm_ifaces_utils.h"
#include "nm_settings.h"
#include "nm_environment_manager.h"


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

static void nm_wlan_setup_defaults (NmWlanSetup *, gboolean);
static void nm_wlan_setup_create_widget (NmWlanSetup *);
static void nm_wlan_setup_enable_cb (GtkToggleButton *, gpointer);
static void nm_wlan_setup_enable_options (NmWlanSetup *, gboolean);
static int nm_wlan_setup_enable_dhcpd(NmWlanSetup *, gboolean);

typedef struct _NmWlanSetupPrivate NmWlanSetupPrivate;

struct _NmWlanSetupPrivate
{
    GtkWidget *checkbutton;

    GtkWidget *ip_label;
    GtkWidget *gw_label;
    GtkWidget *mask_label;
    GtkWidget *essid_label;
    GtkWidget *dhcpd_label;
    GtkWidget *wep_label;

    GtkWidget *ip_entry;
    GtkWidget *gw_entry;
    GtkWidget *mask_entry;
    GtkWidget *essid_entry;
    GtkWidget *dhcpd_entry;
    GtkWidget *wep_entry;

    NmWlanUtils *wlan_utils;
    
    gchar *env;
};

#define NM_WLAN_SETUP_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_WLAN_SETUP_TYPE, NmWlanSetupPrivate))

static void nm_wlan_setup_class_init (NmWlanSetupClass *);
static void nm_wlan_setup_init       (NmWlanSetup *);
static void nm_wlan_setup_dispose    (GObject *);
static void nm_wlan_setup_finalize   (GObject *);

G_DEFINE_TYPE (NmWlanSetup, nm_wlan_setup, NM_CONN_WIDGET_TYPE)

static void
nm_wlan_setup_class_init (NmWlanSetupClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);
    NmConnWidgetClass *conn_widget_class = NM_CONN_WIDGET_CLASS (klass);

    g_type_class_add_private (klass, sizeof (NmWlanSetupPrivate));

    object_class->dispose = nm_wlan_setup_dispose;
    object_class->finalize = nm_wlan_setup_finalize;

    conn_widget_class->apply_changes = nm_wlan_setup_apply_changes;
}

static void
nm_wlan_setup_init (NmWlanSetup *self)
{
    NmWlanSetupPrivate *priv;

    priv = NM_WLAN_SETUP_GET_PRIVATE (self);
    priv->wlan_utils = nm_wlan_utils_new ();

    nm_wlan_setup_create_widget (self);
}

static void
nm_wlan_setup_dispose (GObject *object)
{
    G_OBJECT_CLASS (nm_wlan_setup_parent_class)->dispose (object);
}

static void
nm_wlan_setup_finalize (GObject *object)
{
    G_OBJECT_CLASS (nm_wlan_setup_parent_class)->finalize (object);
}

GtkWidget*
nm_wlan_setup_new (gchar *env)
{
    GtkWidget *self = g_object_new (NM_WLAN_SETUP_TYPE, NULL);
    
    NmWlanSetupPrivate *priv;

    priv = NM_WLAN_SETUP_GET_PRIVATE (self);
    
    priv->env = g_strdup(env);
    
    return self;
}

void
nm_wlan_setup_enable (NmWlanSetup *self, gboolean enable)
{
    NmWlanSetupPrivate *priv;

    priv = NM_WLAN_SETUP_GET_PRIVATE (self);

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(priv->checkbutton), enable);
}

gboolean
nm_wlan_setup_is_enabled (NmWlanSetup *self)
{
    NmWlanSetupPrivate *priv;

    priv = NM_WLAN_SETUP_GET_PRIVATE (self);
    return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(priv->checkbutton));
}

static void
nm_wlan_setup_create_widget(NmWlanSetup *self)
{
    GtkWidget *table;
    NmWlanSetupPrivate *priv;

    priv = NM_WLAN_SETUP_GET_PRIVATE (self);

    table = gtk_table_new (7, 2, TRUE);

    priv->checkbutton = gtk_check_button_new_with_label (_("enable"));
    g_signal_connect (priv->checkbutton, "toggled", G_CALLBACK(nm_wlan_setup_enable_cb),
                      self);
    
    /* DHCPD server */
    priv->dhcpd_label = gtk_label_new(_("DHCP Server: "));      
    gtk_misc_set_alignment(GTK_MISC(priv->dhcpd_label), 1, 0.5);
    priv->dhcpd_entry = gtk_check_button_new ();
    
    /* IP address */
    priv->ip_label = gtk_label_new(_("IP address: "));
    gtk_misc_set_alignment(GTK_MISC(priv->ip_label), 1, 0.5);
    priv->ip_entry = gtk_entry_new();

    /* Gateway */
    priv->gw_label = gtk_label_new(_("Gateway: "));
    gtk_misc_set_alignment(GTK_MISC(priv->gw_label), 1, 0.5);
    priv->gw_entry = gtk_entry_new();

    /* Network mask */
    priv->mask_label = gtk_label_new(_("Network mask: "));
    gtk_misc_set_alignment(GTK_MISC(priv->mask_label), 1, 0.5);
    priv->mask_entry = gtk_entry_new();

    /* ESSID */
    priv->essid_label = gtk_label_new(_("ESSID: "));
    gtk_misc_set_alignment(GTK_MISC(priv->essid_label), 1, 0.5);
    priv->essid_entry = gtk_entry_new();
    
    /* WEP KEY */
    priv->wep_label = gtk_label_new(_("WEP Key: "));
    gtk_misc_set_alignment(GTK_MISC(priv->wep_label), 1, 0.5);
    priv->wep_entry = gtk_entry_new();

    gtk_table_attach_defaults(GTK_TABLE(table), priv->checkbutton, 0, 1, 0, 1);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->dhcpd_label, 0, 1, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->dhcpd_entry, 1, 2, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE(table), priv->ip_label, 0, 1, 2, 3);
    gtk_table_attach_defaults(GTK_TABLE(table), priv->ip_entry, 1, 2, 2, 3);
    gtk_table_attach_defaults(GTK_TABLE(table), priv->gw_label, 0, 1, 3, 4);
    gtk_table_attach_defaults(GTK_TABLE(table), priv->gw_entry, 1, 2, 3, 4);
    gtk_table_attach_defaults(GTK_TABLE(table), priv->mask_label, 0, 1, 4, 5);
    gtk_table_attach_defaults(GTK_TABLE(table), priv->mask_entry, 1, 2, 4, 5);
    gtk_table_attach_defaults(GTK_TABLE(table), priv->essid_label, 0, 1, 5, 6);
    gtk_table_attach_defaults(GTK_TABLE(table), priv->essid_entry, 1, 2, 5, 6);
    gtk_table_attach_defaults(GTK_TABLE(table), priv->wep_label, 0, 1, 6, 7);
    gtk_table_attach_defaults(GTK_TABLE(table), priv->wep_entry, 1, 2, 6, 7);

    GtkWidget *scroll;
    scroll = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll), 
                                    GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroll),
                                           table); 


    gtk_container_add(GTK_CONTAINER(self), scroll);

    nm_wlan_setup_enable_options (self, FALSE);
    gtk_widget_show_all(GTK_WIDGET(self));
}

static void
nm_wlan_setup_defaults (NmWlanSetup *self, gboolean pressed)
{
    NmWlanSetupPrivate *priv;
    GConfClient* gcClient = NULL;
    
    NmSettings *settings = nm_settings_new ();
    nm_settings_start (settings);

    gcClient = gconf_client_get_default();
    g_assert(GCONF_IS_CLIENT(gcClient));

    priv = NM_WLAN_SETUP_GET_PRIVATE (self);

    if (pressed) {

        gconf_client_set_string(gcClient, NM_GCONF_KEY_WLAN_TYPE, NM_WLAN_DEFAULT_TYPE , NULL);
        gconf_client_set_string(gcClient, NM_GCONF_KEY_WLAN_IPV4_TYPE, NM_WLAN_DEFAULT_IPV4_TYPE , NULL);
        gconf_client_set_string(gcClient, NM_GCONF_KEY_WLAN_HIDDEN, NM_WLAN_DEFAULT_HIDDEN , NULL);
        gconf_client_set_string(gcClient, NM_GCONF_KEY_WLAN_NAME, NM_WLAN_DEFAULT_NAME , NULL);
        gconf_client_set_string(gcClient, NM_GCONF_KEY_WLAN_SECURITY, NM_WLAN_DEFAULT_SECURITY , NULL);

        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priv->dhcpd_entry),
                             nm_settings_get_boolean (settings, priv->env,
                                                     NM_WLAN_SETTINGS_GROUP,
                                                     NM_SETTINGS_DHCPD));

        
        gtk_entry_set_text (GTK_ENTRY (priv->ip_entry),
                             nm_settings_get_value (settings, priv->env,
                                                     NM_WLAN_SETTINGS_GROUP,
                                                     NM_SETTINGS_IP));        

        gtk_entry_set_text (GTK_ENTRY (priv->gw_entry),
                             nm_settings_get_value (settings, priv->env,
                                                     NM_WLAN_SETTINGS_GROUP,
                                                     NM_SETTINGS_GATEWAY));



        gtk_entry_set_text (GTK_ENTRY (priv->mask_entry),
                             nm_settings_get_value (settings, priv->env,
                                                     NM_WLAN_SETTINGS_GROUP,
                                                     NM_SETTINGS_MASK));

        gtk_entry_set_text (GTK_ENTRY (priv->essid_entry),
                             nm_settings_get_value (settings, priv->env,
                                                     NM_WLAN_SETTINGS_GROUP,
                                                     NM_WLAN_SETTINGS_ESSID ));

        gtk_entry_set_text (GTK_ENTRY (priv->wep_entry),
                             nm_settings_get_value (settings, priv->env,
                                                     NM_WLAN_SETTINGS_GROUP,
                                                     NM_WLAN_SETTINGS_WEPKEY ));

    } else {

        gtk_entry_set_text(GTK_ENTRY (priv->ip_entry), "");
        gtk_entry_set_text(GTK_ENTRY (priv->gw_entry), "");
        gtk_entry_set_text(GTK_ENTRY (priv->mask_entry), "");
        gtk_entry_set_text(GTK_ENTRY (priv->essid_entry), "");
        gtk_entry_set_text(GTK_ENTRY (priv->wep_entry), "");
    }
}

static void
nm_wlan_setup_enable_cb (GtkToggleButton *button, gpointer data)
{
    NmWlanSetupPrivate *priv;
    gboolean pressed = gtk_toggle_button_get_active (button);

    priv = NM_WLAN_SETUP_GET_PRIVATE (data);

    nm_wlan_setup_defaults(data, pressed);

    // FIXME: remove this if the combo box is needed for something else
    nm_wlan_setup_enable_options (data, pressed);

    nm_conn_widget_settings_changed (data);
}

static void
nm_wlan_setup_enable_options (NmWlanSetup *self, gboolean enable)
{
    NmWlanSetupPrivate *priv;
    priv = NM_WLAN_SETUP_GET_PRIVATE (self);

    gtk_widget_set_sensitive (priv->dhcpd_label, enable);
    gtk_widget_set_sensitive (priv->dhcpd_entry, enable);
    gtk_widget_set_sensitive (priv->ip_label, enable);
    gtk_widget_set_sensitive (priv->ip_entry, enable);
    gtk_widget_set_sensitive (priv->gw_label, enable);
    gtk_widget_set_sensitive (priv->gw_entry, enable);
    gtk_widget_set_sensitive (priv->mask_label, enable);
    gtk_widget_set_sensitive (priv->mask_entry, enable);
    gtk_widget_set_sensitive (priv->essid_label, enable);
    gtk_widget_set_sensitive (priv->essid_entry, enable);
    gtk_widget_set_sensitive (priv->wep_label, enable);
    gtk_widget_set_sensitive (priv->wep_entry, enable);
    
}

gboolean
nm_wlan_setup_apply_changes (NmConnWidget *self)
{
    g_debug("[%s] - Applying wlan env", __FUNCTION__);
    
    NmWlanSetup *wlansetup = NM_WLAN_SETUP (self);
    NmWlanSetupPrivate *priv;
    priv = NM_WLAN_SETUP_GET_PRIVATE (wlansetup);

    nm_wlan_setup_save_changes (NM_WLAN_SETUP(self));

    const gchar* value = NULL;
    GSList *essid_list = NULL;
    gint i = 0;
    GConfClient* gcClient = NULL;

    gcClient = gconf_client_get_default ();
    g_assert (GCONF_IS_CLIENT (gcClient));

    gboolean is_active = nm_wlan_utils_is_active ();

    // Disable the dhcp server
    nm_wlan_setup_enable_dhcpd(wlansetup, FALSE);

    if (nm_wlan_setup_is_enabled (wlansetup)) {
        
        if (is_active) {
            nm_wlan_utils_deactivate ();
        }
        

        gconf_client_set_string (gcClient, NM_GCONF_KEY_WLAN_TYPE, NM_WLAN_DEFAULT_TYPE , NULL);
        gconf_client_set_string (gcClient, NM_GCONF_KEY_WLAN_IPV4_TYPE, NM_WLAN_DEFAULT_IPV4_TYPE , NULL);
        gconf_client_set_string (gcClient, NM_GCONF_KEY_WLAN_HIDDEN, NM_WLAN_DEFAULT_HIDDEN , NULL);
        gconf_client_set_string (gcClient, NM_GCONF_KEY_WLAN_NAME, NM_WLAN_DEFAULT_NAME , NULL);
        gconf_client_set_string (gcClient, NM_GCONF_KEY_WLAN_SECURITY, NM_WLAN_DEFAULT_SECURITY , NULL);
    
        
        value = gtk_entry_get_text (GTK_ENTRY (priv->wep_entry));
        int wep_size = strlen(value);
        g_debug("[%s] - wep key: %s , size: %d", __FUNCTION__, value, wep_size);
        if (wep_size != 0 ) {
            if (wep_size == 5 || wep_size == 10 || wep_size == 13) {
                gconf_client_set_string (gcClient, NM_GCONF_KEY_WLAN_SECURITY, "WEP", NULL);
                gconf_client_set_int (gcClient, NM_GCONF_KEY_WLAN_WEPDEFKEY, 1, NULL);
                value = gtk_entry_get_text (GTK_ENTRY (priv->wep_entry));
                gconf_client_set_string (gcClient, NM_GCONF_KEY_WLAN_WEPKEY, value , NULL);
            }
            else {
                hildon_banner_show_information (NULL, NULL, _("The WEP key needs to have 5, 10 or 13 caracters\nIgnoring key"));
            }
        }

        
        value = gtk_entry_get_text (GTK_ENTRY (priv->ip_entry));
        gconf_client_set_string (gcClient, NM_GCONF_KEY_WLAN_IP , value, NULL);

        value = gtk_entry_get_text (GTK_ENTRY (priv->mask_entry));
        gconf_client_set_string (gcClient, NM_GCONF_KEY_WLAN_MASK, value, NULL);

        value = gtk_entry_get_text (GTK_ENTRY (priv->essid_entry));
        gconf_client_set_string (gcClient, NM_GCONF_KEY_WLAN_NAME, value , NULL);

        /* the ESSID value is stored as a list of integers in gconf */
        for (i = 0; value[i] && i < 50; i++)
            essid_list = g_slist_append (essid_list, GINT_TO_POINTER ((gint) (value[i])));

        gconf_client_set_list (gcClient, NM_GCONF_KEY_WLAN_ESSID, GCONF_VALUE_INT, essid_list, NULL);

        if (nm_wlan_utils_activate ()) {

            /* Enable dhcpd if checkbox active */
            if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(priv->dhcpd_entry))) {
                nm_wlan_setup_enable_dhcpd(wlansetup, TRUE);
            }
            

        } else {

            hildon_banner_show_information (NULL, NULL, _("Error activating WLAN Ad-Hoc network"));
            return FALSE;
        }
    } else if (is_active) {

        if (nm_wlan_utils_deactivate ()) {

/*            hildon_banner_show_information(NULL, NULL,*/
/*                                            "WLAN Ad-Hoc network disabled");*/
        } else {

            return FALSE;
        }
    }

    return TRUE;
}

static int
nm_wlan_setup_enable_dhcpd(NmWlanSetup *self, gboolean enable)
{
    NmWlanSetupPrivate *priv = NM_WLAN_SETUP_GET_PRIVATE (self);
    
    system("sudo /usr/sbin/nm-udhcpd stop wlan0");
    
    if (enable == TRUE) {
    	NmSettings *settings = nm_settings_get_context();
        gboolean share_wlan = nm_settings_get_boolean (settings, priv->env, NM_INTERNET_SETTINGS_GROUP,
    											       NM_INTERNET_IN_INTERFACE_WLAN);
        const gchar *ip = gtk_entry_get_text (GTK_ENTRY (priv->ip_entry));
        const gchar *netmask = gtk_entry_get_text (GTK_ENTRY (priv->mask_entry));
        gchar command[150];

        sprintf(command, "sudo /usr/sbin/nm-udhcpd start wlan0 %s %s", ip, netmask);
    	/* if sharing internet through wlan, set dhcp gateway and dns */
        if (share_wlan == TRUE) {
            gchar *dns = nm_settings_get_value (settings, priv->env, NM_INTERNET_SETTINGS_GROUP,
                                                NM_INTERNET_DNS);
        	gchar internet[50];

        	sprintf(internet, " %s %s", ip, dns);
        	strcat(command, internet);
        }
        g_debug("[%s] - command: %s", __FUNCTION__, command);
        system(command);
    }
    return 1;

}

gboolean 
nm_wlan_setup_activate_environment (gchar *env)
{
    g_debug("[%s]", __FUNCTION__);
    gboolean ret = TRUE;
    
    NmSettings *settings = nm_settings_new ();
    nm_settings_start (settings);
    
    gboolean enabled = nm_settings_get_boolean (settings, env,
                                                     NM_WLAN_SETTINGS_GROUP,
                                                     NM_SETTINGS_ENABLE);

    GtkWidget *wlan = nm_wlan_setup_new(env);
    nm_wlan_setup_enable(NM_WLAN_SETUP(wlan), enabled);
    ret = nm_wlan_setup_apply_changes(NM_CONN_WIDGET(wlan));

    
    return ret;    
}

void 
nm_wlan_setup_save_changes (NmWlanSetup *self)
{
    
    NmWlanSetupPrivate *priv = NM_WLAN_SETUP_GET_PRIVATE (self);
    
    NmSettings *settings = nm_settings_new();
    
    nm_settings_set_boolean (settings, priv->env, NM_WLAN_SETTINGS_GROUP,
                        NM_SETTINGS_ENABLE,  nm_wlan_setup_is_enabled(self));
    
    if (nm_wlan_setup_is_enabled(self)) {
        nm_settings_set_boolean (settings, priv->env, NM_WLAN_SETTINGS_GROUP,
                                 NM_SETTINGS_DHCPD, 
                                 gtk_toggle_button_get_active(priv->dhcpd_entry) );
        nm_settings_set_value (settings, priv->env, NM_WLAN_SETTINGS_GROUP,
                            NM_SETTINGS_IP,
                            gtk_entry_get_text (GTK_ENTRY (priv->ip_entry)));
        nm_settings_set_value (settings, priv->env, NM_WLAN_SETTINGS_GROUP,
                            NM_SETTINGS_GATEWAY,
                            gtk_entry_get_text (GTK_ENTRY (priv->gw_entry)));
        nm_settings_set_value (settings, priv->env, NM_WLAN_SETTINGS_GROUP,
                            NM_SETTINGS_MASK,
                            gtk_entry_get_text (GTK_ENTRY (priv->mask_entry)));
        nm_settings_set_value (settings, priv->env, NM_WLAN_SETTINGS_GROUP,
                            NM_WLAN_SETTINGS_ESSID,
                            gtk_entry_get_text (GTK_ENTRY (priv->essid_entry)));
        nm_settings_set_value (settings, priv->env, NM_WLAN_SETTINGS_GROUP,
                            NM_WLAN_SETTINGS_WEPKEY,
                            gtk_entry_get_text (GTK_ENTRY (priv->wep_entry)));
    }
    
    nm_settings_save(settings);
}

gboolean
wlan_setup_check_enabled (NmWlanSetup *self)
{
    NmWlanSetupPrivate *priv = NM_WLAN_SETUP_GET_PRIVATE (self);

    if (nm_wlan_utils_is_active()) {

        nm_wlan_setup_enable (self, TRUE);
        nm_wlan_setup_enable_cb (GTK_TOGGLE_BUTTON (priv->checkbutton), self);
        return TRUE;
    } else {

        nm_wlan_setup_enable (self, FALSE);
        nm_wlan_setup_enable_cb (GTK_TOGGLE_BUTTON (priv->checkbutton), self);
        return FALSE;
    }
}

void
nm_wlan_setup_fill_widget(NmWlanSetup *self)
{
    NmWlanSetupPrivate *priv = NM_WLAN_SETUP_GET_PRIVATE (self);
    
    if (strcmp(priv->env, nm_environment_manager_get_active())) {
        NmSettings *settings = nm_settings_new ();
        nm_settings_start (settings);
        nm_wlan_setup_enable(self, nm_settings_get_boolean (settings, priv->env, NM_WLAN_SETTINGS_GROUP, NM_SETTINGS_ENABLE));
        
    }
    else {
        wlan_setup_check_enabled (self);
    }
}

