/*
 * Copyright (c) 2008 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Hallyson Melo <hallyson.melo@indt.org.br>
 *          Kyller Gorgonio <kyllercg@gmail.com>
 *          Walter Guerra <walter.guerra@signove.com>
 *          Andre Rodrigues <andre.rodrigues@signove.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <stdlib.h>
#include <string.h>
#include <hildon/hildon-banner.h>

#include "nm_usb_setup.h"
#include "nm_usb_utils.h"
#include "nm_ifaces_utils.h"
#include "nm_main_dialog.h"
#include "nm_settings.h"
#include "nm_environment_manager.h"


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

static void nm_usb_setup_defaults (NmUsbSetup *, gboolean);
static void nm_usb_setup_create_widget (NmUsbSetup *);
static void nm_usb_setup_enable_cb (GtkComboBox *, gpointer);
static void nm_usb_setup_enable_options (NmUsbSetup *, gboolean);
static int nm_usb_setup_enable_dhcpd(NmUsbSetup *, NmSettings *, gboolean);

typedef struct _NmUsbSetupPrivate NmUsbSetupPrivate;

struct _NmUsbSetupPrivate
{
    GtkWidget *checkbutton;

    GtkWidget *ip_label;
    GtkWidget *gw_label;
    GtkWidget *mask_label;
    GtkWidget *dhcpd_label;

    GtkWidget *ip_entry;
    GtkWidget *gw_entry;
    GtkWidget *mask_entry;
    GtkWidget *dhcpd_entry;
    
    gchar *env;
};

#define NM_USB_SETUP_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_USB_SETUP_TYPE, NmUsbSetupPrivate))

static void nm_usb_setup_class_init (NmUsbSetupClass *);
static void nm_usb_setup_init       (NmUsbSetup *);
static void nm_usb_setup_dispose    (GObject *);
static void nm_usb_setup_finalize   (GObject *);

G_DEFINE_TYPE (NmUsbSetup, nm_usb_setup, NM_CONN_WIDGET_TYPE)

static void
nm_usb_setup_class_init (NmUsbSetupClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);
    NmConnWidgetClass *conn_widget_class = NM_CONN_WIDGET_CLASS (klass);

    g_type_class_add_private (klass, sizeof (NmUsbSetupPrivate));

    object_class->dispose = nm_usb_setup_dispose;
    object_class->finalize = nm_usb_setup_finalize;

    conn_widget_class->apply_changes = nm_usb_setup_apply_changes;
}

static void
nm_usb_setup_init (NmUsbSetup *self)
{
    nm_usb_setup_create_widget (self);
}

static void
nm_usb_setup_dispose (GObject *object)
{
    G_OBJECT_CLASS (nm_usb_setup_parent_class)->dispose (object);
}

static void
nm_usb_setup_finalize (GObject *object)
{
    G_OBJECT_CLASS (nm_usb_setup_parent_class)->finalize (object);
}

GtkWidget*
nm_usb_setup_new (gchar *env)
{
    GtkWidget *self = g_object_new (NM_USB_SETUP_TYPE, NULL);
    
    NmUsbSetupPrivate *priv;

    priv = NM_USB_SETUP_GET_PRIVATE (self);
    
    priv->env = g_strdup(env);
    
    g_debug("[%s] - priv->env: %s", __FUNCTION__, priv->env);
    
    return self;
}

/* set to network mode */
void
nm_usb_setup_enable (NmUsbSetup *self, gchar *mode)
{
    NmUsbSetupPrivate *priv;

    priv = NM_USB_SETUP_GET_PRIVATE (self);
    if (mode == NULL) {
        mode = g_strdup(NM_USB_NETWORK_LABEL);
    }
    
    g_debug("[%s] - mode: %s", __FUNCTION__, mode);
    
    if (!strcmp(mode, NM_USB_NETWORK_LABEL)) {
        gtk_combo_box_set_active (GTK_COMBO_BOX(priv->checkbutton), NM_USB_NETWORK);
    }
    else if (!strcmp(mode, NM_USB_MASS_STORAGE_LABEL)) {
        gtk_combo_box_set_active (GTK_COMBO_BOX(priv->checkbutton), NM_USB_MASS_STORAGE);;
    }
    else {
        gtk_combo_box_set_active(GTK_COMBO_BOX(priv->checkbutton), NM_USB_HOST_MODE);
    }
    g_debug("[%s]", __FUNCTION__);
}

/* Is in network mode */
gboolean
nm_usb_setup_mode_is_network (NmUsbSetup *self)
{
    NmUsbSetupPrivate *priv;

    priv = NM_USB_SETUP_GET_PRIVATE (self);
    return (gtk_combo_box_get_active (GTK_COMBO_BOX(priv->checkbutton)) == NM_USB_NETWORK);
}

gboolean
nm_usb_setup_mode_is_host (NmUsbSetup *self)
{
    NmUsbSetupPrivate *priv;

    priv = NM_USB_SETUP_GET_PRIVATE (self);
    return (gtk_combo_box_get_active (GTK_COMBO_BOX(priv->checkbutton)) == NM_USB_HOST_MODE);
}

gboolean
nm_usb_setup_mode_is_mass_storage (NmUsbSetup *self)
{
    NmUsbSetupPrivate *priv;

    priv = NM_USB_SETUP_GET_PRIVATE (self);
    return (gtk_combo_box_get_active (GTK_COMBO_BOX(priv->checkbutton)) == NM_USB_MASS_STORAGE);
}

static void
nm_usb_setup_create_widget (NmUsbSetup *self)
{
    GtkWidget *table;
    NmUsbSetupPrivate *priv;

    priv = NM_USB_SETUP_GET_PRIVATE (self);

    table = gtk_table_new (5, 2, TRUE);

    priv->checkbutton = gtk_combo_box_new_text();
    g_signal_connect (priv->checkbutton, "changed", G_CALLBACK(nm_usb_setup_enable_cb),
                      self);
    gtk_combo_box_insert_text (GTK_COMBO_BOX(priv->checkbutton), NM_USB_MASS_STORAGE, NM_USB_MASS_STORAGE_LABEL);
    gtk_combo_box_insert_text (GTK_COMBO_BOX(priv->checkbutton), NM_USB_NETWORK, NM_USB_NETWORK_LABEL);
#ifdef DIABLO
    gtk_combo_box_insert_text (GTK_COMBO_BOX(priv->checkbutton), NM_USB_HOST_MODE, NM_USB_HOST_MODE_LABEL);
#endif
#ifdef FREMANTLE
	gtk_combo_box_insert_text(priv->checkbutton, NM_USB_PC_SUITE, NM_USB_PC_SUITE_LABEL);
#endif	
    /* DHCPD server */
    priv->dhcpd_label = gtk_label_new(_("DHCP Server: "));
    gtk_misc_set_alignment(GTK_MISC(priv->dhcpd_label), 1, 0.5);
    priv->dhcpd_entry = gtk_check_button_new ();

    /* IP address */
    priv->ip_label = gtk_label_new(_("IP address: "));
    gtk_misc_set_alignment(GTK_MISC(priv->ip_label), 1, 0.5);
    priv->ip_entry = gtk_entry_new();

    /* Gateway */
    priv->gw_label = gtk_label_new(_("Gateway: "));
    gtk_misc_set_alignment(GTK_MISC(priv->gw_label), 1, 0.5);
    priv->gw_entry = gtk_entry_new();

    /* Network mask */
    priv->mask_label = gtk_label_new(_("Network mask: "));
    gtk_misc_set_alignment(GTK_MISC(priv->mask_label), 1, 0.5);
    priv->mask_entry = gtk_entry_new();

    gtk_table_attach_defaults(GTK_TABLE (table), priv->checkbutton, 0, 2, 0, 1);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->dhcpd_label, 0, 1, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->dhcpd_entry, 1, 2, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->ip_label, 0, 1, 2, 3);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->ip_entry, 1, 2, 2, 3);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->gw_label, 0, 1, 3, 4);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->gw_entry, 1, 2, 3, 4);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->mask_label, 0, 1, 4, 5);
    gtk_table_attach_defaults(GTK_TABLE( table), priv->mask_entry, 1, 2, 4, 5);
    
    GtkWidget *scroll;
    scroll = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll), 
                                    GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroll),
                                           table);    
    
    gtk_container_add(GTK_CONTAINER (self), scroll);

#ifdef DIABLO
    gtk_widget_set_size_request(self, 366, -1);
#endif
#ifdef FREMANTLE
	gtk_widget_set_size_request(self, 360, -1);
#endif
    gtk_widget_show_all(self);
    gtk_combo_box_set_active(priv->checkbutton, NM_USB_MASS_STORAGE);
}

static void
nm_usb_setup_defaults (NmUsbSetup *self, gboolean pressed)
{
    NmUsbSetupPrivate *priv;

    NmSettings *settings = nm_settings_new ();
    nm_settings_start (settings);

    priv = NM_USB_SETUP_GET_PRIVATE (self);

    if (pressed) {
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priv->dhcpd_entry),
                             nm_settings_get_boolean (settings, priv->env,
                                                     NM_USB_SETTINGS_GROUP,
                                                     NM_SETTINGS_DHCPD));

        gtk_entry_set_text (GTK_ENTRY (priv->ip_entry),
                             nm_settings_get_value (settings, priv->env,
                                                     NM_USB_SETTINGS_GROUP,
                                                     NM_SETTINGS_IP));

                                                     
        gtk_entry_set_text (GTK_ENTRY (priv->gw_entry),
                             nm_settings_get_value (settings, priv->env,
                                                     NM_USB_SETTINGS_GROUP,
                                                     NM_SETTINGS_GATEWAY));

        gtk_entry_set_text (GTK_ENTRY (priv->mask_entry),
                             nm_settings_get_value (settings, priv->env,
                                                     NM_USB_SETTINGS_GROUP,
                                                     NM_SETTINGS_MASK));
    } else {
        gtk_entry_set_text(GTK_ENTRY (priv->ip_entry), "");
        gtk_entry_set_text(GTK_ENTRY (priv->gw_entry), "");
        gtk_entry_set_text(GTK_ENTRY (priv->mask_entry), "");
    }
}

static void
nm_usb_setup_enable_cb (GtkComboBox *button, gpointer data)
{
    g_debug("[%s] - callback called", __FUNCTION__);
    
    NmUsbSetupPrivate *priv;

    priv = NM_USB_SETUP_GET_PRIVATE (data);
    
    gboolean pressed = nm_usb_setup_mode_is_network(data);
    
    nm_usb_setup_defaults (data, pressed);

    nm_usb_setup_enable_options (data, pressed);
    nm_conn_widget_settings_changed (data);
}

static void
nm_usb_setup_enable_options (NmUsbSetup *self, gboolean enable)
{
    NmUsbSetupPrivate *priv;
    priv = NM_USB_SETUP_GET_PRIVATE (self);
    
    g_debug("[%s] - gtk_combo_box_get_active = %d", __FUNCTION__, 
            gtk_combo_box_get_active (priv->checkbutton));
    
    if (gtk_combo_box_get_active (priv->checkbutton) == NM_USB_NETWORK) {
        gtk_widget_show (priv->dhcpd_label);
        gtk_widget_show (priv->dhcpd_entry);
        gtk_widget_show (priv->ip_label);
        gtk_widget_show (priv->ip_entry);
        gtk_widget_show (priv->gw_label);
        gtk_widget_show (priv->gw_entry);
        gtk_widget_show (priv->mask_label);
        gtk_widget_show (priv->mask_entry);
    }
    else {
        gtk_widget_hide (priv->dhcpd_label);
        gtk_widget_hide (priv->dhcpd_entry);
        gtk_widget_hide (priv->ip_label);
        gtk_widget_hide (priv->ip_entry);
        gtk_widget_hide (priv->gw_label);
        gtk_widget_hide (priv->gw_entry);
        gtk_widget_hide (priv->mask_label);
        gtk_widget_hide (priv->mask_entry);        
    }

}

gboolean
nm_usb_setup_apply_changes (NmConnWidget *self)
{
    g_debug("[%s]", __FUNCTION__);
    
    NmUsbSetup *usbsetup = NM_USB_SETUP (self);
    gboolean is_loaded = nm_usb_utils_is_loaded();
    NmSettings *settings = nm_settings_get_context();
    
    NmUsbSetupPrivate *priv;
    priv = NM_USB_SETUP_GET_PRIVATE (usbsetup);
    
    nm_usb_setup_save_changes (usbsetup);
    
    // Disable the dhcp server
    nm_usb_setup_enable_dhcpd(usbsetup, settings, FALSE);

    if (nm_usb_setup_mode_is_network (usbsetup)) {
        if (is_loaded) {
            nm_usb_utils_unload ();
        }

        if (nm_ifaces_utils_save_config_e (NM_IFACES_USB_ST,
            gtk_entry_get_text (GTK_ENTRY (priv->ip_entry)),
            gtk_entry_get_text (GTK_ENTRY (priv->mask_entry)),
            gtk_entry_get_text (GTK_ENTRY (priv->gw_entry))) == FALSE) {

            exit (1);
        } 

        if (nm_usb_utils_load (nm_settings_get_value (settings, priv->env,
                                                NM_USB_SETTINGS_GROUP,
                                                NM_SETTINGS_GATEWAY))) {
            
            g_debug("Usb network loaded");
            
            /* Enable dhcpd if checkbox active */
            if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(priv->dhcpd_entry))) {
                nm_usb_setup_enable_dhcpd(usbsetup, settings, TRUE);
            }
            
            
        } else {

            return FALSE;
        }

    } else if (is_loaded) {

        if (!nm_usb_utils_unload ()) {
            return FALSE;
        }
    }
    
    if (nm_usb_setup_mode_is_host (usbsetup)) {
        nm_usb_utils_put_in_host_mode();
        
    } else if (nm_usb_setup_mode_is_mass_storage (usbsetup)) {
        nm_usb_utils_put_in_otg_mode();
    }

    return TRUE;
}

static int
nm_usb_setup_enable_dhcpd(NmUsbSetup *self, NmSettings *settings, gboolean enable)
{
    NmUsbSetupPrivate *priv = NM_USB_SETUP_GET_PRIVATE (self);
    
    system("sudo /usr/sbin/nm-udhcpd stop usb0");
    
    if (enable == TRUE) {
        const gchar *ip = gtk_entry_get_text (GTK_ENTRY (priv->ip_entry));
        const gchar *netmask = gtk_entry_get_text (GTK_ENTRY (priv->mask_entry));
        gboolean share_usb = nm_settings_get_boolean (settings, priv->env, NM_INTERNET_SETTINGS_GROUP,
													  NM_INTERNET_IN_INTERFACE_USB);
        gchar command[150];

        sprintf(command, "sudo /usr/sbin/nm-udhcpd start usb0 %s %s", ip, netmask);
    	/* if sharing internet through usb, set dhcp gateway and dns */
        if (share_usb == TRUE) {
            gchar *dns = nm_settings_get_value (settings, priv->env, NM_INTERNET_SETTINGS_GROUP,
                                                NM_INTERNET_DNS);
        	gchar internet[50];

        	sprintf(internet, " %s %s", ip, dns);
        	strcat(command, internet);
        }

        system(command);
    }
    return 1;

}

gboolean 
nm_usb_setup_activate_environment (gchar *env)
{
    g_debug("[%s]", __FUNCTION__);
    gboolean ret = TRUE;
    
    NmSettings *settings = nm_settings_new ();
    nm_settings_start (settings);
    
    gchar *mode = nm_settings_get_boolean (settings, env,
                                           NM_USB_SETTINGS_GROUP,
                                           NM_USB_SETTINGS_MODE);

    GtkWidget *usb = nm_usb_setup_new(env);
    nm_usb_setup_enable(NM_USB_SETUP(usb), mode);
    ret = nm_usb_setup_apply_changes(NM_CONN_WIDGET(usb));

    return ret;
}

void 
nm_usb_setup_save_changes (NmUsbSetup *self) 
{
    NmUsbSetupPrivate *priv = NM_USB_SETUP_GET_PRIVATE (self);

    NmSettings *settings = nm_settings_new();
  
    nm_settings_set_boolean (settings, priv->env, NM_USB_SETTINGS_GROUP,
                        NM_SETTINGS_ENABLE,  nm_usb_setup_mode_is_network(self));
    
    nm_settings_set_value (settings, priv->env, NM_USB_SETTINGS_GROUP,
                           NM_USB_SETTINGS_MODE,
                           gtk_combo_box_get_active_text (GTK_COMBO_BOX (priv->checkbutton)));
 
    if (nm_usb_setup_mode_is_network(self)) {
        nm_settings_set_boolean (settings, priv->env, NM_USB_SETTINGS_GROUP,
                                 NM_SETTINGS_DHCPD, 
                                 gtk_toggle_button_get_active(priv->dhcpd_entry) );
        nm_settings_set_value (settings, priv->env, NM_USB_SETTINGS_GROUP,
                               NM_SETTINGS_IP,
                               gtk_entry_get_text (GTK_ENTRY (priv->ip_entry)));
        nm_settings_set_value (settings, priv->env, NM_USB_SETTINGS_GROUP,
                               NM_SETTINGS_GATEWAY,
                               gtk_entry_get_text (GTK_ENTRY (priv->gw_entry)));
        nm_settings_set_value (settings, priv->env, NM_USB_SETTINGS_GROUP,
                              NM_SETTINGS_MASK,
                              gtk_entry_get_text (GTK_ENTRY (priv->mask_entry)));
        
    } 
        
    nm_settings_set_value (settings, priv->env, NM_USB_SETTINGS_GROUP,
                            NM_USB_SETTINGS_MODE,
                            gtk_combo_box_get_active_text (GTK_COMBO_BOX (priv->checkbutton)));

    nm_settings_save(settings);
}

gboolean
usb_setup_check_enabled (NmUsbSetup *self)
{
    NmUsbSetupPrivate *priv = NM_USB_SETUP_GET_PRIVATE (self);

    if (nm_usb_utils_is_loaded ()) {

        nm_usb_setup_enable (self, NM_USB_NETWORK_LABEL);
/*        nm_usb_setup_enable_cb (GTK_COMBO_BOX (priv->checkbutton), self);*/
        return TRUE;
    } else {

        nm_usb_setup_enable (self, NM_USB_MASS_STORAGE_LABEL);
/*        nm_usb_setup_enable_cb (GTK_TOGGLE_BUTTON (priv->checkbutton), self);*/
        return FALSE;
    }
}

void
nm_usb_setup_fill_widget(NmUsbSetup *self)
{
    NmUsbSetupPrivate *priv = NM_USB_SETUP_GET_PRIVATE (self);
    
    if (strcmp(priv->env, nm_environment_manager_get_active())) {
        NmSettings *settings = nm_settings_new ();
        nm_settings_start (settings);
        
        nm_usb_setup_enable(self, nm_settings_get_value (settings, priv->env, NM_USB_SETTINGS_GROUP, NM_USB_SETTINGS_MODE));
        
    }
    else {
        usb_setup_check_enabled (self);
    }
}

