/*
 * Copyright (c) 2009 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Walter Guerra <walter.guerra@signove.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */



#include <gtk/gtk.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <hildon/hildon.h>
#include <string.h>

#include "nm_environment_manager.h"
#include "nm_settings.h"


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

enum
{
   CONNECTION_DEVICE_COLUMN,
   CONNECTION_MAC_COLUMN,
   N_COLUMNS
};

enum
{
   NM_ENVIRONMENT_MANAGER_BUTTON_OK,
   NM_ENVIRONMENT_MANAGER_BUTTON_NEW,
   NM_ENVIRONMENT_MANAGER_BUTTON_DELETE
};


typedef struct _NmEnvironmentManagerPrivate NmEnvironmentManagerPrivate;

struct _NmEnvironmentManagerPrivate
{
    GtkWidget *treeview;
    GtkWidget *ok_button;
    GtkWidget *new_button;
    GtkWidget *delete_button;
    
    gboolean delete_active;
};

#define NM_ENVIRONMENT_MANAGER_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_ENVIRONMENT_MANAGER_TYPE, NmEnvironmentManagerPrivate))

static void nm_environment_manager_class_init (NmEnvironmentManagerClass *klass);
static void nm_environment_manager_init       (NmEnvironmentManager *self);
static void nm_environment_manager_dispose    (GObject *object);
static void nm_environment_manager_finalize   (GObject *object);

G_DEFINE_TYPE (NmEnvironmentManager, nm_environment_manager, GTK_TYPE_DIALOG)

static GtkWidget* nm_environment_manager_create_tree_view (NmEnvironmentManager *);
static void nm_environment_manager_add_device (NmEnvironmentManager *, gchar *);
static void nm_environment_manager_del_env (NmEnvironmentManager *);
static void nm_environment_manager_new_dialog(NmEnvironmentManager *);
static void nm_environment_manager_apply_config(NmEnvironmentManager *);
static void nm_environment_manager_row_deleted_cb(GtkTreeModel  *, GtkTreePath *, gpointer);
static void nm_environment_manager_row_inserted_cb(GtkTreeModel  *, GtkTreePath *, GtkTreeIter *, gpointer);
static gint nm_environment_manager_get_number_of_items(GtkTreeModel *); 

static void
nm_environment_manager_class_init (NmEnvironmentManagerClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    g_type_class_add_private (klass, sizeof (NmEnvironmentManagerPrivate));

    object_class->dispose = nm_environment_manager_dispose;
    object_class->finalize = nm_environment_manager_finalize;
}

static void
nm_environment_manager_init (NmEnvironmentManager *self)
{
    GtkWidget *scroll;
    NmEnvironmentManagerPrivate *priv;
    
    priv = NM_ENVIRONMENT_MANAGER_GET_PRIVATE(self); 
    
    // Create the window
    gtk_window_set_title (GTK_WINDOW (self), "Environment Manager" );
    gtk_window_resize (GTK_WINDOW (self), 100, 300);
    
    priv->treeview = nm_environment_manager_create_tree_view (self);
    
    // Add the items to the treeview
    gchar **list;
    gsize length;
    
    list = nm_environment_manager_get_list(&length);
    
    int i;
    for (i = 0; i < length; i++) {
        if (strcmp(list[i], NM_ENVIRONMENT_MANAGER_NONE)) {
            /* If its not the None env, put it at the list */
            nm_environment_manager_add_device(self, list[i]);
        }
    }


    scroll = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll), 
                                    GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroll),
                                           priv->treeview);
    
    gtk_widget_show_all(scroll);

    // Add the content area
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (self)->vbox),
                    scroll, TRUE, TRUE,
                    6 /* padding */);
                    
    // Add buttons
    priv->ok_button = gtk_dialog_add_button (GTK_DIALOG (self), "Select", NM_ENVIRONMENT_MANAGER_BUTTON_OK);
    priv->new_button = gtk_dialog_add_button (GTK_DIALOG (self), "New", NM_ENVIRONMENT_MANAGER_BUTTON_NEW);
    priv->delete_button = gtk_dialog_add_button (GTK_DIALOG (self), "Delete", NM_ENVIRONMENT_MANAGER_BUTTON_DELETE);

    
    g_signal_connect (G_OBJECT (gtk_tree_view_get_model(GTK_TREE_VIEW(priv->treeview))),
                       "row-deleted",
                       G_CALLBACK (nm_environment_manager_row_deleted_cb), self);

    g_signal_connect (G_OBJECT (gtk_tree_view_get_model(GTK_TREE_VIEW(priv->treeview))),
                       "row-inserted",
                       G_CALLBACK (nm_environment_manager_row_inserted_cb), self);
    
    // Check if there is only one item to disable the delete button
    if (nm_environment_manager_get_number_of_items(gtk_tree_view_get_model(GTK_TREE_VIEW(priv->treeview))) == 1) {
        gtk_widget_set_sensitive(priv->delete_button, FALSE);
        priv->delete_active = FALSE;
    }
    else {
        priv->delete_active = TRUE;
    }
}

static void
nm_environment_manager_dispose (GObject *object)
{

}

static void
nm_environment_manager_finalize (GObject *object)
{
    G_OBJECT_CLASS (nm_environment_manager_parent_class)->finalize (object);
}

GtkWidget*
nm_environment_manager_new ()
{ 
    return g_object_new (NM_ENVIRONMENT_MANAGER_TYPE, NULL);
}

gchar * 
nm_environment_manager_show_dialog(NmEnvironmentManager *self) 
{
    NmEnvironmentManagerPrivate *priv;
    
    priv = NM_ENVIRONMENT_MANAGER_GET_PRIVATE(self); 
    
    gchar *selected_env = NULL;
    while (TRUE) {
        gint ret = gtk_dialog_run( GTK_DIALOG(self) );
        
        g_debug("Env dialog ret: %d", ret);
        
        if (ret == NM_ENVIRONMENT_MANAGER_BUTTON_OK || ret == GTK_RESPONSE_DELETE_EVENT) {
            nm_environment_manager_apply_config(self);
            gtk_widget_hide(GTK_WIDGET(self));
            
            GtkTreeModel *model;
            GtkTreeIter iter;
            GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(priv->treeview));
            int selected = gtk_tree_selection_get_selected (selection, &model, &iter);
            if (selected) {
                gtk_tree_model_get (model, &iter, 
                        CONNECTION_DEVICE_COLUMN, &selected_env,
                           -1 );
                
                g_debug("[%s] - There is a selected item: %s", __FUNCTION__, selected_env);
            }
            
            break;
        }
        else if (ret == NM_ENVIRONMENT_MANAGER_BUTTON_DELETE) {
            // Remove the item from the tree
            nm_environment_manager_del_env(self);
        }
        else if (ret == NM_ENVIRONMENT_MANAGER_BUTTON_NEW) {
            // Add an item to the view
            nm_environment_manager_new_dialog(self);
        }
    }
    
    g_debug("[%s] - return: %s", __FUNCTION__, selected_env);
    
    return selected_env;
}

static GtkWidget*
nm_environment_manager_create_tree_view (NmEnvironmentManager *self)
{
    GtkWidget *treeview;
    GtkTreeStore *store;
    GtkCellRenderer *renderer;
    GtkTreeSelection *selection;
    GtkTreeViewColumn *column;
    NmEnvironmentManagerPrivate *priv;

    priv = NM_ENVIRONMENT_MANAGER_GET_PRIVATE (self);
    store = gtk_tree_store_new (N_COLUMNS,
                                G_TYPE_STRING,
                                G_TYPE_STRING);

    treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));

    /* Creates the render */
    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes ("Device",
                                                       renderer,
                                                       "text", CONNECTION_DEVICE_COLUMN,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

    /* Setup the selection handler */
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
    gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
    
    
/*    GValue value = {0};*/
/*    g_value_init (&value, G_TYPE_BOOLEAN);*/
/*    g_value_set_boolean(&value, TRUE);*/
/*    g_object_set_property (priv->treeview, "can-focus", &value);*/
    
    return treeview;
}

static void
nm_environment_manager_add_device (NmEnvironmentManager *self, gchar *name)
{
    GtkTreeModel *model;
    NmEnvironmentManagerPrivate *priv;
    GtkTreeIter iter;

    priv = NM_ENVIRONMENT_MANAGER_GET_PRIVATE (self);

    model = gtk_tree_view_get_model (GTK_TREE_VIEW (priv->treeview));

    if (model != NULL) {
        gtk_tree_store_append (GTK_TREE_STORE(model), &iter, NULL);
        gtk_tree_store_set (GTK_TREE_STORE(model), &iter,
                            CONNECTION_DEVICE_COLUMN, name,
                            -1);
    

    }
}

static void 
nm_environment_manager_del_env (NmEnvironmentManager *self) 
{
    GtkTreeIter iter;
    GtkTreeModel *model;
    
    NmEnvironmentManagerPrivate *priv;
    
    priv = NM_ENVIRONMENT_MANAGER_GET_PRIVATE (self);
    
    GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(priv->treeview));
    
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
        // TODO Do now allow delete the last env
        gtk_tree_store_remove(GTK_TREE_STORE(model), &iter);
    }
    
    if (iter.user_data != NULL) {
        gtk_tree_selection_select_iter(selection, &iter);
    }
    else {
        gtk_tree_model_get_iter_first(model, &iter);
        gtk_tree_selection_select_iter(selection, &iter);
    }
    
}

static void
nm_environment_manager_apply_config(NmEnvironmentManager *self)
{
    g_debug("%s", __FUNCTION__);
    
    int ret;
    // Get the items from the treeview
    NmSettings *settings = nm_settings_new();
    
    GtkTreeIter iter;
    GtkTreeModel *model;
    
    NmEnvironmentManagerPrivate *priv;
    
    priv = NM_ENVIRONMENT_MANAGER_GET_PRIVATE (self);
    
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(priv->treeview));
    
    ret = gtk_tree_model_get_iter_first(model, &iter);
    
    int i = 0;
    gchar *list[30];
    while (ret == TRUE) {
        gtk_tree_model_get(model, &iter, 
                           CONNECTION_DEVICE_COLUMN, &list[i],
                           -1 );
                           
        g_debug("item[%d]: %s", i, list[i]);
        i++;

        ret = gtk_tree_model_iter_next(model, &iter);
    }
    
    // Save the settings
    nm_settings_set_string_list(settings, NULL, NM_SETTINGS_ENVIRONMENT_GROUP, NM_SETTINGS_ENVIRONMENT_LIST, list, i);
    
    // Apply the config
    nm_settings_save(settings);
    
}

gchar** 
nm_environment_manager_get_list(gsize *length) 
{
    g_debug("[%s]", __FUNCTION__);
    
    NmSettings *settings = nm_settings_new();
    if (!nm_settings_start (settings)) {

        hildon_banner_show_information(NULL, NULL,
                                        "Could not read config file!");
        return NULL;
    }
    gchar **list = nm_settings_get_string_list(settings, NULL, NM_SETTINGS_ENVIRONMENT_GROUP, NM_SETTINGS_ENVIRONMENT_LIST, length);

    if (list == NULL) {
        g_debug("list is null");

        (*length) = 0;

        return NULL;
    }
    g_debug("list is NOT null");
    return list;
}

static void 
nm_environment_manager_new_dialog(NmEnvironmentManager *self) 
{
    NmEnvironmentManagerPrivate *priv;
    
    priv = NM_ENVIRONMENT_MANAGER_GET_PRIVATE (self);
    
    GtkWidget *dialog_new = gtk_dialog_new();
    
    gtk_window_set_title (GTK_WINDOW (dialog_new), "Enter the name of the new environment" );
    
    GtkWidget *entry = gtk_entry_new();
    gtk_entry_set_width_chars(GTK_ENTRY(entry), 10);
    
    gtk_widget_show_all(entry);
    
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog_new)->vbox),
                    entry, TRUE, TRUE,
                    6 /* padding */);
    
    gtk_dialog_add_button (GTK_DIALOG (dialog_new), "OK", 1);
    gtk_dialog_add_button (GTK_DIALOG (dialog_new), "Cancel", 2);
    
    gint ret = gtk_dialog_run( GTK_DIALOG(dialog_new) );
    if (ret == 1) {
         const gchar *new_entry = gtk_entry_get_text (GTK_ENTRY(entry));
         if (strcmp(new_entry, "") && strcmp(new_entry, NM_ENVIRONMENT_MANAGER_NONE)) {
            nm_environment_manager_add_device(self, new_entry);
         }
    }

    gtk_widget_destroy (dialog_new);
    
}

static void 
nm_environment_manager_row_deleted_cb(GtkTreeModel *model, GtkTreePath *path, gpointer self)
{
    NmEnvironmentManagerPrivate *priv = NM_ENVIRONMENT_MANAGER_GET_PRIVATE(self); 
   
    gint size = nm_environment_manager_get_number_of_items(model);

    if (size < 2) {
        gtk_widget_set_sensitive (priv->delete_button, FALSE);
        priv->delete_active = FALSE;
    }

}

static void 
nm_environment_manager_row_inserted_cb(GtkTreeModel  *model, GtkTreePath *path, GtkTreeIter *iter, gpointer self)
{
    NmEnvironmentManagerPrivate *priv = NM_ENVIRONMENT_MANAGER_GET_PRIVATE(self); 
    
    gint size = nm_environment_manager_get_number_of_items(model);
    
    if (priv->delete_active == FALSE && size > 1) {
        gtk_widget_set_sensitive (priv->delete_button, TRUE);
        priv->delete_active = TRUE;
    }
}

static gint
nm_environment_manager_get_number_of_items(GtkTreeModel *model) 
{
    GtkTreeIter iter;
    gboolean valid;
    gint size = 0;
    
    valid = gtk_tree_model_get_iter_first (model, &iter);
    while(valid) {
        size++;
        valid = gtk_tree_model_iter_next (model, &iter);
    }
    
    return size;  
}

gchar* 
nm_environment_manager_get_active()
{
    FILE *tmpfile = fopen(TMPFILE, "r");
    if (tmpfile) {
        /* tmp file exists */
        
        char tmp[50];
        fscanf(tmpfile, "%s", tmp);
        
        fclose(tmpfile);
        
        gchar *active = g_strdup(tmp);
        g_debug("[%s] - Active: %s", __FUNCTION__, active);
        return active;
        
    }
    
    /* File does not exist */
    tmpfile = fopen(TMPFILE, "w");
    if (tmpfile) {
        fprintf(tmpfile, "None");
        
        fclose(tmpfile);
        
        gchar *active = g_strdup(NM_ENVIRONMENT_MANAGER_NONE);
        return active;
    }

    /* File does not exist and can not be created */
    g_debug("[%s] - Active environment temp file can not be created", __FUNCTION__);
    return g_strdup(NM_ENVIRONMENT_MANAGER_NONE); 
}

void 
nm_environment_manager_set_active(gchar *env)
{
    FILE *tmpfile = fopen(TMPFILE, "w");
    if (tmpfile) {
        fprintf(tmpfile, "%s", env);
        
        fclose(tmpfile);
        return;
    }
    
    /* File can not be written */
    g_debug("[%s] - Active environment temp file can not be created", __FUNCTION__);
}

