/*
 * Copyright (c) 2009 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Walter Guerra <walter.guerra@signove.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */
 
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <stdlib.h>
#include <string.h>


#include "nm_samba_setup.h"
#include "nm_settings.h"
#include "nm_environment_manager.h"
#include "nm_file_dialog.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif


typedef struct _NmSambaSetupPrivate NmSambaSetupPrivate;

struct _NmSambaSetupPrivate
{
    GtkWidget *checkbutton;

    GtkWidget *workgroup_label;
    GtkWidget *user_label;
    GtkWidget *passwd_label;
    GtkWidget *read_only_label;
    GtkWidget *exported_paths_label;


    GtkWidget *workgroup_entry;
    GtkWidget *user_entry;
    GtkWidget *passwd_entry;
    GtkWidget *read_only_checkbox;
    GtkWidget *exported_paths_button;
    
    gchar *env;
    gboolean is_installed;
};

#define NM_SAMBA_SETUP_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_SAMBA_SETUP_TYPE, NmSambaSetupPrivate))

static void nm_samba_setup_class_init (NmSambaSetupClass *);
static void nm_samba_setup_init       (NmSambaSetup *);
static void nm_samba_setup_dispose    (GObject *);
static void nm_samba_setup_finalize   (GObject *);

static void nm_samba_setup_defaults (NmSambaSetup *, gboolean);
static void nm_samba_setup_create_widget (NmSambaSetup *);
static void nm_samba_setup_enable_cb (GtkToggleButton *, gpointer);
static void nm_samba_setup_enable_options (NmSambaSetup *, gboolean);
static void nm_samba_setup_manage_exported_paths (gpointer , gpointer );
static gboolean nm_samba_setup_is_installed (NmSambaSetup *);
    
G_DEFINE_TYPE (NmSambaSetup, nm_samba_setup, NM_CONN_WIDGET_TYPE)

static void
nm_samba_setup_class_init (NmSambaSetupClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);
    NmConnWidgetClass *conn_widget_class = NM_CONN_WIDGET_CLASS (klass);

    g_type_class_add_private (klass, sizeof (NmSambaSetupPrivate));

    object_class->dispose = nm_samba_setup_dispose;
    object_class->finalize = nm_samba_setup_finalize;

    conn_widget_class->apply_changes = nm_samba_setup_apply_changes;
}

static void
nm_samba_setup_init (NmSambaSetup *self)
{
    nm_samba_setup_create_widget (self);
}

static void
nm_samba_setup_dispose (GObject *object)
{
    G_OBJECT_CLASS (nm_samba_setup_parent_class)->dispose (object);
}

static void
nm_samba_setup_finalize (GObject *object)
{
    G_OBJECT_CLASS (nm_samba_setup_parent_class)->finalize (object);
}

GtkWidget*
nm_samba_setup_new (gchar *env)
{
    g_debug("[%s] - env: %s", __FUNCTION__, env);
    
    GtkWidget *self = g_object_new (NM_SAMBA_SETUP_TYPE, NULL);
    
    NmSambaSetupPrivate *priv;

    priv = NM_SAMBA_SETUP_GET_PRIVATE (self);
    
    priv->env = g_strdup(env);    
    
    return self;
}

void
nm_samba_setup_enable (NmSambaSetup *self, gboolean enable)
{
    NmSambaSetupPrivate *priv;

    priv = NM_SAMBA_SETUP_GET_PRIVATE (self);

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(priv->checkbutton), enable);
}

gboolean
nm_samba_setup_is_enabled (NmSambaSetup *self)
{
    NmSambaSetupPrivate *priv;

    priv = NM_SAMBA_SETUP_GET_PRIVATE (self);
    return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(priv->checkbutton));
}

static gboolean
nm_samba_setup_is_installed (NmSambaSetup *self)
{
    gint ret = system("sudo /usr/sbin/nm-samba-setup isinstalled");
    
    g_debug("[%s] - Samba is installed: %d", __FUNCTION__, ret);
    
    return ret;
}

static void
nm_samba_setup_create_widget (NmSambaSetup *self)
{
    GtkWidget *table;
    NmSambaSetupPrivate *priv;

    priv = NM_SAMBA_SETUP_GET_PRIVATE (self);

    priv->is_installed = nm_samba_setup_is_installed(self);

    if (!priv->is_installed) {
        GtkWidget *not_installed = gtk_label_new(_("SAMBA package is not installed"));
#ifdef DIABLO
        gtk_widget_set_size_request(not_installed, 359, -1);
#endif
        gtk_container_add(GTK_CONTAINER (self), not_installed);
        return;      
    }

    table = gtk_table_new (6, 2, TRUE);

    priv->checkbutton = gtk_check_button_new_with_label (_("enable"));
    g_signal_connect (priv->checkbutton, "toggled", G_CALLBACK(nm_samba_setup_enable_cb),
                      self);

    /* Workgroup  */
    priv->workgroup_label = gtk_label_new(_("Workgroup: "));
    gtk_misc_set_alignment(GTK_MISC(priv->workgroup_label), 1, 0.5);
    priv->workgroup_entry = gtk_entry_new();

    /* Exported Paths */
    priv->exported_paths_label = gtk_label_new(_("Exported Paths: "));
    gtk_misc_set_alignment(GTK_MISC(priv->exported_paths_label), 1, 0.5);
    priv->exported_paths_button = gtk_button_new_with_label ("...");
    g_signal_connect (G_OBJECT (priv->exported_paths_button),
                       "clicked",
                       G_CALLBACK (nm_samba_setup_manage_exported_paths), self);

    /* User */
    priv->user_label = gtk_label_new(_("User: "));
    gtk_misc_set_alignment(GTK_MISC(priv->user_label), 1, 0.5);
    priv->user_entry = gtk_entry_new ();

    /* Password */
    priv->passwd_label = gtk_label_new(_("Password: "));
    gtk_misc_set_alignment(GTK_MISC(priv->passwd_label), 1, 0.5);
    priv->passwd_entry = gtk_entry_new ();

    /* Read only */
    priv->read_only_label = gtk_label_new(_("Read Only: "));
    gtk_misc_set_alignment(GTK_MISC(priv->passwd_label), 1, 0.5);
    priv->read_only_checkbox = gtk_check_button_new ();

    gtk_table_attach_defaults(GTK_TABLE (table), priv->checkbutton, 0, 1, 0, 1);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->exported_paths_label, 0, 1, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->exported_paths_button, 1, 2, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->workgroup_label, 0, 1, 2, 3);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->workgroup_entry, 1, 2, 2, 3);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->user_label, 0, 1, 3, 4);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->user_entry, 1, 2, 3, 4);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->passwd_label, 0, 1, 4, 5);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->passwd_entry, 1, 2, 4, 5);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->read_only_label, 0, 1, 5, 6);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->read_only_checkbox, 1, 2, 5, 6);
    
    GtkWidget *scroll;
    scroll = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll), 
                                    GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroll),
                                           table);    
    
    gtk_container_add(GTK_CONTAINER (self), scroll);

    nm_samba_setup_enable_options(self, FALSE);
}

static void
nm_samba_setup_defaults (NmSambaSetup *self, gboolean pressed)
{
    NmSambaSetupPrivate *priv;

    // FIXME: This is not a good solution. It should be done during class init
    NmSettings *settings = nm_settings_new ();
    nm_settings_start (settings);

    priv = NM_SAMBA_SETUP_GET_PRIVATE (self);

    if (pressed) {

        gtk_entry_set_text (GTK_ENTRY (priv->workgroup_entry),
                             nm_settings_get_value (settings, priv->env,
                                                     NM_SAMBA_SETTINGS_GROUP,
                                                     NM_SAMBA_WORKGROUP));

        gtk_entry_set_text (GTK_ENTRY (priv->user_entry),
                             nm_settings_get_value (settings, priv->env,
                                                     NM_SAMBA_SETTINGS_GROUP,
                                                     NM_SAMBA_USER));

        gtk_entry_set_text (GTK_ENTRY (priv->passwd_entry),
                             nm_settings_get_value (settings, priv->env,
                                                     NM_SAMBA_SETTINGS_GROUP,
                                                     NM_SAMBA_PASSWD));
                                                     
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priv->read_only_checkbox),
                             nm_settings_get_boolean (settings, priv->env,
                                                     NM_SAMBA_SETTINGS_GROUP,
                                                     NM_SAMBA_READ_ONLY));

                                                     
    } else {
        gtk_entry_set_text(GTK_ENTRY (priv->workgroup_entry), "");
        gtk_entry_set_text(GTK_ENTRY (priv->user_entry), "");
        gtk_entry_set_text(GTK_ENTRY (priv->passwd_entry), "");
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON (priv->read_only_checkbox), FALSE );
    }
}

static void
nm_samba_setup_enable_cb (GtkToggleButton *button, gpointer data)
{
    NmSambaSetupPrivate *priv;
    gboolean pressed = gtk_toggle_button_get_active (button);

    priv = NM_SAMBA_SETUP_GET_PRIVATE (data);
    
    nm_samba_setup_defaults (data, pressed);

    // FIXME: remove this if the combo box is needed for something else
    nm_samba_setup_enable_options (data, pressed);
    nm_conn_widget_settings_changed (data);
}

static void
nm_samba_setup_enable_options (NmSambaSetup *self, gboolean enable)
{
    NmSambaSetupPrivate *priv;
    priv = NM_SAMBA_SETUP_GET_PRIVATE (self);
    
    gtk_widget_set_sensitive (priv->exported_paths_label, enable);
    gtk_widget_set_sensitive (priv->exported_paths_button, enable);
    gtk_widget_set_sensitive (priv->workgroup_label, enable);
    gtk_widget_set_sensitive (priv->workgroup_entry, enable);
    gtk_widget_set_sensitive (priv->user_label, enable);
    gtk_widget_set_sensitive (priv->user_entry, enable);
    gtk_widget_set_sensitive (priv->passwd_label, enable);
    gtk_widget_set_sensitive (priv->passwd_entry, enable);
    gtk_widget_set_sensitive (priv->read_only_label, enable);
    gtk_widget_set_sensitive (priv->read_only_checkbox, enable);
}

gboolean
nm_samba_setup_apply_changes (NmConnWidget *self)
{
    g_debug("[%s]", __FUNCTION__);
    NmSambaSetup *sambasetup = NM_SAMBA_SETUP(self);
    
    NmSambaSetupPrivate *priv;
    priv = NM_SAMBA_SETUP_GET_PRIVATE (sambasetup);
 
    if (!priv->is_installed) {
        return TRUE;
    }
    
    nm_samba_setup_save_changes(sambasetup);
    
    system("sudo /usr/sbin/nm-samba-setup stop");
    system("sudo /usr/sbin/nm-avahi stop_samba");
    
    if ( nm_samba_setup_is_enabled(sambasetup) ) {
    
        const gchar *workgroup = gtk_entry_get_text(GTK_ENTRY(priv->workgroup_entry));
        const gchar *user = gtk_entry_get_text(GTK_ENTRY(priv->user_entry));
        const gchar *passwd = gtk_entry_get_text(GTK_ENTRY(priv->passwd_entry));
        int read_only = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(priv->read_only_checkbox));

        /* Update the config file */
        GtkWidget *dialog = nm_file_dialog_new(priv->env, NM_SAMBA_SETTINGS_GROUP, NM_FILE_DIALOG_FOLDER_DIALOG);
        gsize length = 0;
        gchar **list = nm_file_dialog_get_list(NM_FILE_DIALOG(dialog), &length);
        
        gchar *concat_paths = g_strdup("");
        int i;
        for (i = 0; i < length; i++) {
            concat_paths = g_strconcat(concat_paths, " ", list[i], NULL);
        }
        
        g_debug("[%s] - concat_list: %s", __FUNCTION__, concat_paths);
        gchar command[50 + strlen(concat_paths)];
        
        sprintf(command, "sudo /usr/sbin/nm-samba-setup config %s %d %s %s", workgroup, read_only, user, concat_paths);
        system(command);
        
        /* Set the samba password */
        sprintf(command, "sudo /usr/sbin/nm-samba-setup set_passwd %s %s", user, passwd);
        system(command);
        
        g_debug("smbpasswd line: %s", command);
        
        /* Start the deamon */ 
        system("sudo /usr/sbin/nm-samba-setup start");
        
        /* Export with avahi */
        sprintf(command, "sudo /usr/sbin/nm-avahi start_samba %d", 139); // Set the port
        system(command);
        
        gtk_widget_destroy(dialog);
        g_free(concat_paths);
        
    }
    
    
    return TRUE;   
}

gboolean 
nm_samba_setup_activate_environment (gchar *env)
{
    g_debug("[%s]", __FUNCTION__);
    gboolean ret = TRUE;
    
    NmSettings *settings = nm_settings_new ();
    nm_settings_start (settings);
    
    gboolean enabled = nm_settings_get_boolean (settings, env,
                                                     NM_SAMBA_SETTINGS_GROUP,
                                                     NM_SETTINGS_ENABLE);

    GtkWidget *samba = nm_samba_setup_new(env);
    nm_samba_setup_enable(NM_SAMBA_SETUP(samba), enabled);
    ret = nm_samba_setup_apply_changes(NM_CONN_WIDGET(samba));

    return ret;
}

void 
nm_samba_setup_save_changes (NmSambaSetup *self) 
{
    NmSambaSetupPrivate *priv = NM_SAMBA_SETUP_GET_PRIVATE (self);

    if (!priv->is_installed) {
        return;
    }

    NmSettings *settings = nm_settings_new();
  
    nm_settings_set_boolean (settings, priv->env, NM_SAMBA_SETTINGS_GROUP,
                        NM_SETTINGS_ENABLE,  nm_samba_setup_is_enabled(self));  
 
    if (nm_samba_setup_is_enabled(self)) {
        nm_settings_set_value (settings, priv->env, NM_SAMBA_SETTINGS_GROUP,
                            NM_SAMBA_WORKGROUP,
                            gtk_entry_get_text (GTK_ENTRY (priv->workgroup_entry)));

        nm_settings_set_value (settings, priv->env, NM_SAMBA_SETTINGS_GROUP,
                            NM_SAMBA_USER,
                            gtk_entry_get_text (GTK_ENTRY (priv->user_entry)));
    
        nm_settings_set_value (settings, priv->env, NM_SAMBA_SETTINGS_GROUP,
                            NM_SAMBA_PASSWD,
                            gtk_entry_get_text (GTK_ENTRY (priv->passwd_entry)));
                            
        nm_settings_set_boolean (settings, priv->env, NM_SAMBA_SETTINGS_GROUP,
                            NM_SAMBA_READ_ONLY,
                            gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (priv->read_only_checkbox)));

    }

    nm_settings_save(settings);
}

gboolean
samba_setup_check_enabled (NmSambaSetup *self)
{
    NmSambaSetupPrivate *priv = NM_SAMBA_SETUP_GET_PRIVATE (self);

    if ( system("sudo /usr/sbin/nm-samba-setup isrunning") ) {

        nm_samba_setup_enable (self, TRUE);
        nm_samba_setup_enable_cb (GTK_TOGGLE_BUTTON (priv->checkbutton), self);
        return TRUE;
    } else {

        nm_samba_setup_enable (self, FALSE);
        nm_samba_setup_enable_cb (GTK_TOGGLE_BUTTON (priv->checkbutton), self);
        return FALSE;
    }
/*    return TRUE;*/
}

void
nm_samba_setup_fill_widget(NmSambaSetup *self)
{
    NmSambaSetupPrivate *priv = NM_SAMBA_SETUP_GET_PRIVATE (self);

    if (!priv->is_installed) {
        return;
    }
    
    if (strcmp(priv->env, nm_environment_manager_get_active())) {
/*        g_debug("[%s] - %s is not the active environment", __FUNCTION__, priv->env);*/
        NmSettings *settings = nm_settings_new ();
        nm_settings_start (settings);
/*        g_debug("[%s] - samba enabled: %d", __FUNCTION__, */
/*                    nm_settings_get_boolean (settings, priv->env, NM_USB_SETTINGS_GROUP, NM_SETTINGS_ENABLE));*/
        
        nm_samba_setup_enable(self, nm_settings_get_boolean (settings, priv->env, NM_SAMBA_SETTINGS_GROUP, NM_SETTINGS_ENABLE));
        
    }
    else {
/*        g_debug("[%s] - %s is the active environment", __FUNCTION__, priv->env);*/
        samba_setup_check_enabled (self);
    }
}

static void
nm_samba_setup_manage_exported_paths(gpointer instance, gpointer self)
{
    
    NmSambaSetupPrivate *priv = NM_SAMBA_SETUP_GET_PRIVATE (self);
    
    g_debug("[%s] - env: %s", __FUNCTION__, priv->env);
    
    GtkWidget *dialog = nm_file_dialog_new(priv->env, NM_SAMBA_SETTINGS_GROUP, NM_FILE_DIALOG_FOLDER_DIALOG);
   
    nm_file_dialog_show_dialog(NM_FILE_DIALOG(dialog));

    gtk_widget_destroy(dialog);

}
