/*
 * Copyright (c) 2008 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Hallyson Melo <hallyson.melo@indt.org.br>
 *          Kyller Gorgonio <kyllercg@gmail.com>
 *          Walter Guerra <walter.guerra@signove.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
/*#include <bt-dbus.h>*/

#include <glib-object.h>

#include "nm_marshal.h"
#include "nm_bt_utils.h"

#ifndef NM_BLUETOOTH_INTERFACE
#define NM_BLUETOOTH_INTERFACE "/org/bluez/hci0"
#endif



/* NmBtUtils signals */
enum {
    DEVICE_FOUND,
    SEARCH_COMPLETED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

static DBusGConnection* nm_bt_utils_create_connection (NmBtUtils *self);
static gchar* nm_bt_utils_get_connection_name (NmBtUtils *self);
static gchar* nm_bt_utils_activate_service (NmBtUtils *self);
static gchar* nm_bt_utils_get_existing_connection_name (NmBtUtils *self);

static void nm_bt_utils_discovery_completed (DBusGProxy *, gpointer);
static void nm_bt_utils_device_found (DBusGProxy *object, gchar *address,
                                      gchar *name, gpointer data);
static void nm_bt_utils_disconnect_discovery_signals (NmBtUtils *self);


typedef struct _NmBtUtilsPrivate NmBtUtilsPrivate;

struct _NmBtUtilsPrivate
{
    DBusGConnection *connection;
    DBusGProxy *discovery_proxy;

    gchar *dbus_name;
    gchar *conn_name;

    gchar *bt_addr;
};

#define NM_BT_UTILS_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_BT_UTILS_TYPE, NmBtUtilsPrivate))

static void nm_bt_utils_class_init (NmBtUtilsClass *klass);
static void nm_bt_utils_init       (NmBtUtils *self);
static void nm_bt_utils_dispose    (GObject *object);
static void nm_bt_utils_finalize   (GObject *object);

G_DEFINE_TYPE (NmBtUtils, nm_bt_utils, G_TYPE_OBJECT)

static void
nm_bt_utils_class_init (NmBtUtilsClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    g_type_class_add_private (klass, sizeof (NmBtUtilsPrivate));

    signals[DEVICE_FOUND] = g_signal_new ("device-found",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                    0, NULL, NULL,
                    nm_marshal_VOID__STRING_STRING,
                    G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_STRING, NULL);

    signals[SEARCH_COMPLETED] = g_signal_new ("discovery-completed",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                    0, NULL, NULL,
                    g_cclosure_marshal_VOID__VOID,
                    G_TYPE_NONE, 0, NULL);

    object_class->dispose = nm_bt_utils_dispose;
    object_class->finalize = nm_bt_utils_finalize;
}

static void
nm_bt_utils_init (NmBtUtils *self)
{
    NmBtUtilsPrivate *priv;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    priv->connection = NULL;
}

static void
nm_bt_utils_dispose (GObject *object)
{
    NmBtUtilsPrivate *priv;

    priv = NM_BT_UTILS_GET_PRIVATE (object);

    if (priv->connection) {
        dbus_g_connection_unref (priv->connection);
    }

    G_OBJECT_CLASS (nm_bt_utils_parent_class)->dispose (object);
}

static void
nm_bt_utils_finalize (GObject *object)
{
    G_OBJECT_CLASS (nm_bt_utils_parent_class)->finalize (object);
}

NmBtUtils*
nm_bt_utils_new (void)
{
    return g_object_new (NM_BT_UTILS_TYPE, NULL);
}

static DBusGConnection*
nm_bt_utils_create_connection (NmBtUtils *self)
{
    DBusGConnection *connection;

    connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, NULL);
    if (!connection) {
        g_warning ("D-Bus Connection not created");
    }

    return connection;
}

/************************************************************************
 Tool methods to start d-bus network connection.
************************************************************************/

gboolean
nm_bt_utils_connect (NmBtUtils *self, gchar *bt_addr, const gchar *gw)
{
    DBusGProxy *p_conn = NULL;
    NmBtUtilsPrivate *priv;
    GError *error = NULL;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    g_free (priv->bt_addr);
    priv->bt_addr = g_strdup (bt_addr);

    if (!priv->connection) {

        priv->connection = nm_bt_utils_create_connection (self);
        g_return_val_if_fail (priv->connection != NULL, FALSE);
    }

    /* Call ActivateService method, wait for reply */
    priv->dbus_name = nm_bt_utils_activate_service (self);
    if (!priv->dbus_name) {

        g_warning ("[%s] NM Could not get DBus name", __FUNCTION__);
        return FALSE;
    }

    /* Call FindConnection method, wait for reply */
    priv->conn_name = nm_bt_utils_get_connection_name (self);
    if (!priv->conn_name) {

        g_warning ("[%s] NM Could not get connection name", __FUNCTION__);
        return FALSE;
    }

    /* Create proxy and call Connect method, wait for reply */
    p_conn = dbus_g_proxy_new_for_name (priv->connection,
                                        priv->dbus_name,
                                        priv->conn_name,
                                        DBUS_INTERFACE_BLUEZ_CONN);
    if (!dbus_g_proxy_call (p_conn, BLUEZ_CONNECT, &error, G_TYPE_INVALID,
                            G_TYPE_STRING, &(priv->conn_name), G_TYPE_INVALID)) {

        if (error->domain == DBUS_GERROR &&
            error->code == DBUS_GERROR_REMOTE_EXCEPTION) {

            g_warning ("Caught remote method exception %s: %s\n\t%s: %d\n",
                        dbus_g_error_get_name (error), error->message,
                        __FILE__, __LINE__);
        } else {

            g_warning ("Error: %s\n\t%s: %d\n", error->message, __FILE__,
                        __LINE__);
        }

        g_error_free (error);
        g_object_unref (p_conn);

        return FALSE;
    }

    g_object_unref (p_conn);

    nm_bt_utils_ifdown();
    nm_bt_utils_ifup(gw);

    return TRUE;
}

/**
 * Verifies if the bt network connection is already set.
 */
gboolean
nm_bt_utils_has_network_connection (NmBtUtils *self)
{
    NmBtUtilsPrivate *priv;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    if (!priv->connection) {

        priv->connection = nm_bt_utils_create_connection (self);
        g_return_val_if_fail (priv->connection != NULL, FALSE);
    }

    /* Call ActivateService method, wait for reply */
    priv->dbus_name = nm_bt_utils_activate_service (self);
    if (!priv->dbus_name) {

        g_warning ("[%s] NM Could not get Dbus name", __FUNCTION__);
        return FALSE;
    }

    /* Call FindConnection method, wait for reply */
    priv->conn_name = nm_bt_utils_get_existing_connection_name (self);
    if (!priv->conn_name) {

        g_debug ("[%s] NM did not found any BT connection (Not connected)", __FUNCTION__);
        return FALSE;
    }

    g_debug ("[%s] NM found the BT connection %s", __FUNCTION__, priv->conn_name);

    return TRUE;
}

gchar*
nm_bt_utils_get_network_connection_end_point (NmBtUtils *self)
{
    DBusGProxy *p_conn;
    NmBtUtilsPrivate *priv;
    GError *error = NULL;
    gchar *res = NULL;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    if (!priv->connection) {

        priv->connection = nm_bt_utils_create_connection (self);
        g_return_val_if_fail (priv->connection != NULL, NULL);
    }

    if (!nm_bt_utils_has_network_connection (self)) {
        return NULL;
    }

    p_conn = dbus_g_proxy_new_for_name (priv->connection,
                                        priv->dbus_name,
                                        priv->conn_name,
                                        DBUS_INTERFACE_BLUEZ_CONN);

    /* Call ListConnections method, wait for reply */
    if (!dbus_g_proxy_call (p_conn, BLUEZ_GET_NAME, &error, G_TYPE_INVALID,
                             G_TYPE_STRING, &res, G_TYPE_INVALID)) {

        if (error) {

            g_warning ("Caught remote method exception %s: %s\n\t%s: %d\n",
                        dbus_g_error_get_name (error), error->message,
                        __FILE__, __LINE__);
        } else {

            g_warning ("Error: %s\n\t%s: %d\n", error->message, __FILE__, __LINE__);
        }

        g_error_free (error);
    }

    g_debug ("[%s] Network connection found with %s", __FUNCTION__, res);

    return res;
}

static gchar*
nm_bt_utils_activate_service (NmBtUtils *self)
{
    GError *error = NULL;
    DBusGProxy *p_manager = NULL;
    gchar *dbus_uid = NULL;
    NmBtUtilsPrivate *priv;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    /* Create a proxy object for the "bus driver" (name "org.bluez.Manager") */
    p_manager = dbus_g_proxy_new_for_name (priv->connection,
                                           DBUS_SERVICE_BLUEZ,
                                           DBUS_PATH_BLUEZ,
                                           DBUS_INTERFACE_BLUEZ);

    /* Call ActivateService method, wait for reply */
    if (!dbus_g_proxy_call (p_manager, BLUEZ_ACTIVATE_SERVICE, &error,
                             G_TYPE_STRING, BLUEZ_NETWORK, G_TYPE_INVALID,
                             G_TYPE_STRING, &dbus_uid, G_TYPE_INVALID)) {

        if (error->domain == DBUS_GERROR && error->code == DBUS_GERROR_REMOTE_EXCEPTION) {

            g_warning ("Caught remote method exception %s: %s\n\t%s: %d\n",
                        dbus_g_error_get_name (error), error->message, __FILE__, __LINE__);
        } else {

            g_warning ("Error: %s\n\t%s: %d\n", error->message, __FILE__, __LINE__);
        }

        g_error_free (error);
        g_object_unref (p_manager);

        return NULL;
    }

    g_object_unref (p_manager);

    return dbus_uid;
}

static gchar*
nm_bt_utils_get_existing_connection_name (NmBtUtils *self)
{
    DBusGProxy *p_conn = NULL;
    NmBtUtilsPrivate *priv;
    GError *error = NULL;
    char **res = NULL;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    /* Create a proxy object for "org.bluez.network.Manager" */
    g_debug("[%s] - priv->dbus_name: %s", __FUNCTION__, priv->dbus_name);
    p_conn = dbus_g_proxy_new_for_name (priv->connection,
                                         priv->dbus_name,
                                         DBUS_PATH_BLUEZ_NET,
                                         DBUS_INTERFACE_BLUEZ_NET);

    /* Call ListConnections method, wait for reply */
    if (!dbus_g_proxy_call (p_conn, BLUEZ_LIST_CONNECTIONS, &error,
                             G_TYPE_INVALID, G_TYPE_STRV, &res,
                             G_TYPE_INVALID)) {

        if (error) {

            g_warning ("Caught remote method exception %s: %s\n\t%s: %d\n",
                        dbus_g_error_get_name (error), error->message,
                        __FILE__, __LINE__);
        } else {

            g_warning ("Error: %s\n\t%s: %d\n", error->message, __FILE__,
                        __LINE__);
        }

        g_error_free (error);
    }

    if (res) {

        gchar *conn_name;
        conn_name = g_strdup (res[0]);
        g_strfreev (res);

        g_debug ("[%s] Connection name: %s", __FUNCTION__, conn_name);
        
        return conn_name;
    }

    return NULL;
}

static gchar*
nm_bt_utils_get_connection_name (NmBtUtils *self)
{
    DBusGProxy *p_conn = NULL;
    NmBtUtilsPrivate *priv;
    GError *error = NULL;
    char *res = NULL;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    res = nm_bt_utils_get_existing_connection_name (self);
    if (res) {

        g_debug ("[%s] Existing connection found: %s", __FUNCTION__, res);
        return res;
    }

    /* Create a proxy object for "org.bluez.network.Manager" */
    p_conn = dbus_g_proxy_new_for_name (priv->connection,
                                         priv->dbus_name,
                                         DBUS_PATH_BLUEZ_NET,
                                         DBUS_INTERFACE_BLUEZ_NET);


    if (res == NULL || res == "") {
        error = NULL;

        if (!dbus_g_proxy_call (p_conn, BLUEZ_CREATE_CONNECTION, &error,
                                 G_TYPE_STRING, priv->bt_addr,
                                 G_TYPE_STRING, BLUEZ_HOST_TYPE_NAP,
                                 G_TYPE_INVALID, G_TYPE_STRING, &res,
                                 G_TYPE_INVALID)) {
            
            if (!dbus_g_proxy_call (p_conn, BLUEZ_CREATE_CONNECTION, &error,
                                  G_TYPE_STRING, priv->bt_addr,
                                  G_TYPE_STRING, BLUEZ_HOST_TYPE_GN,
                                  G_TYPE_INVALID, G_TYPE_STRING, &res,
                                  G_TYPE_INVALID)) {
                
                if ((error->domain == DBUS_GERROR) &&
                    (error->code == DBUS_GERROR_REMOTE_EXCEPTION)) {

                    g_warning ("Caught remote method exception %s: %s\n\t%s: %d\n",
                                dbus_g_error_get_name (error), error->message,
                                __FILE__, __LINE__);
                } else {

                  g_warning ("Error: %s\n\t%s: %d\n", error->message, __FILE__,
                              __LINE__);
                }

                g_error_free (error);
                return NULL;
            
            }
        }
    }

    g_object_unref (p_conn);
    
    return res;
}

gboolean
nm_bt_utils_remove_connection (NmBtUtils *self)
{
    DBusGProxy *p_conn = NULL;
    NmBtUtilsPrivate *priv;
    GError *error = NULL;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    if (!priv->connection) {

        priv->connection = nm_bt_utils_create_connection (self);
        g_return_val_if_fail (priv->connection != NULL, FALSE);
    }

    /* Call ActivateService method, wait for reply */
    priv->dbus_name = nm_bt_utils_activate_service (self);
    if (!priv->dbus_name) {

        g_warning ("[%s] NM Could not get Dbus name", __FUNCTION__);
        return FALSE;
    }

    g_debug ("[%s] Dbus name: %s", __FUNCTION__, priv->dbus_name);

    /* Call FindConnection method, wait for reply */
    priv->conn_name = nm_bt_utils_get_existing_connection_name (self);
    if (!priv->conn_name) {

        g_warning ("[%s] NM Could not get connection name", __FUNCTION__);
        return FALSE;
    }
    g_debug ("[%s] Connection name: %s", __FUNCTION__, priv->dbus_name);

    /* Create proxy and call Connect method, wait for reply */
    p_conn = dbus_g_proxy_new_for_name (priv->connection,
                                        priv->dbus_name,
                                        DBUS_PATH_BLUEZ_NET,
                                        DBUS_INTERFACE_BLUEZ_NET);
    if (!dbus_g_proxy_call (p_conn, BLUEZ_REMOVE_CONNECTION, &error,
                             G_TYPE_STRING, priv->conn_name, G_TYPE_INVALID,
                             G_TYPE_INVALID)) {

        if ((error->domain == DBUS_GERROR) &&
            (error->code == DBUS_GERROR_REMOTE_EXCEPTION)) {

            g_warning ("Caught remote method exception %s: %s\n\t%s: %d\n",
                        dbus_g_error_get_name (error), error->message,
                        __FILE__, __LINE__);
        } else {

            g_warning ("Error: %s\n\t%s: %d\n", error->message, __FILE__,
                        __LINE__);
        }

        g_error_free (error);
        g_object_unref (p_conn);
        return FALSE;
    }

    nm_bt_utils_ifdown();
    g_object_unref (p_conn);
    return TRUE;
}


/************************************************************************
 Tool methods to search for bluetooth devices
************************************************************************/

gboolean
nm_bt_utils_start_device_discovery (NmBtUtils *self)
{
    GError *error = NULL;
    NmBtUtilsPrivate *priv;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    if (!priv->connection) {

        priv->connection = nm_bt_utils_create_connection (self);
        g_return_val_if_fail (priv->connection != NULL, FALSE);
    }

    if (!nm_bt_utils_radio_is_enabled(self)) {

        nm_bt_utils_radio_enable (self, TRUE);
    }

    priv->discovery_proxy = dbus_g_proxy_new_for_name (priv->connection,
                                                        DBUS_SERVICE_BLUEZ,
                                                        NM_BLUETOOTH_INTERFACE,
                                                        DBUS_INTERFACE_BLUEZ_ADP);

    dbus_g_proxy_add_signal(priv->discovery_proxy, BLUEZ_DISCOVERY_COMPLETED,
                             G_TYPE_INVALID);
    dbus_g_proxy_connect_signal(priv->discovery_proxy,
                                 BLUEZ_DISCOVERY_COMPLETED,
                                 G_CALLBACK(nm_bt_utils_discovery_completed),
                                 self, NULL);

    dbus_g_object_register_marshaller(nm_marshal_VOID__STRING_STRING,
                                       G_TYPE_NONE, G_TYPE_STRING,
                                       G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_proxy_add_signal(priv->discovery_proxy, BLUEZ_REMOTE_NAME_UPDATED,
                             G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_proxy_connect_signal(priv->discovery_proxy,
                                 BLUEZ_REMOTE_NAME_UPDATED,
                                 G_CALLBACK(nm_bt_utils_device_found),
                                 self, NULL);

    dbus_g_proxy_call(priv->discovery_proxy, BLUEZ_DISCOVER_DEVICES, &error,
                       G_TYPE_INVALID, G_TYPE_INVALID);
    if (error != NULL) {

        g_warning ("Failed to discover devices: %s\n", error->message);

        nm_bt_utils_disconnect_discovery_signals (self);

        g_error_free(error);
        g_object_unref (priv->discovery_proxy);
        priv->discovery_proxy = NULL;

        return FALSE;
    }

    dbus_g_connection_flush (priv->connection);

    return TRUE;
}

gboolean
nm_bt_utils_stop_device_discovery (NmBtUtils *self)
{
    NmBtUtilsPrivate *priv;
    GError *error = NULL;

    priv = NM_BT_UTILS_GET_PRIVATE (self);
    g_return_val_if_fail (priv->connection != NULL, FALSE);

    if (priv->discovery_proxy) {

        nm_bt_utils_disconnect_discovery_signals (self);

        dbus_g_proxy_call (priv->discovery_proxy, BLUEZ_CANCEL_DISCOVERY,
                            &error, G_TYPE_INVALID, G_TYPE_INVALID);

        if (error != NULL) {

            g_warning ("Failed to discover devices: %s\n", error->message);
        }

        nm_bt_utils_disconnect_discovery_signals (self);

        g_error_free(error);
        g_object_unref (priv->discovery_proxy);
        priv->discovery_proxy = NULL;
    }

    dbus_g_connection_flush (priv->connection);

    return TRUE;
}

gboolean
nm_bt_utils_radio_is_enabled (NmBtUtils *self)
{
    NmBtUtilsPrivate *priv;
    DBusGProxy *proxy = NULL;
    GError *error = NULL;
    gboolean res;
    gchar *mode = NULL;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    if (!priv->connection) {

        priv->connection = nm_bt_utils_create_connection (self);
        g_return_val_if_fail (priv->connection != NULL, FALSE);
    }

    proxy = dbus_g_proxy_new_for_name (priv->connection,
                                        DBUS_SERVICE_BLUEZ,
                                        NM_BLUETOOTH_INTERFACE,
                                        DBUS_INTERFACE_BLUEZ_ADP);

    /* Call GetMode method, wait for reply */
    error = NULL;
    res = dbus_g_proxy_call (proxy, BLUEZ_GET_MODE, &error, G_TYPE_INVALID,
                              G_TYPE_STRING, &mode, G_TYPE_INVALID);

    if (res) {

        res = g_ascii_strcasecmp (mode, BLUEZ_RADIO_MODE_OFF);
        g_free (mode);
    } else {

        g_error_free (error);
    }

    g_object_unref (proxy);
    g_debug ("[%s] Radio is enabled? %s", __FUNCTION__, res ? "TRUE" : "FALSE");

    return res;
}

gboolean
nm_bt_utils_radio_enable (NmBtUtils *self, gboolean enable)
{
    NmBtUtilsPrivate *priv;
    DBusGProxy *proxy;
    GError *error = NULL;
    gchar *mode = NULL;
    gboolean res;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    if (!priv->connection) {

        priv->connection = nm_bt_utils_create_connection (self);
        g_return_val_if_fail (priv->connection != NULL, FALSE);
    }

    proxy = dbus_g_proxy_new_for_name (priv->connection,
                                        DBUS_SERVICE_BLUEZ,
                                        NM_BLUETOOTH_INTERFACE,
                                        DBUS_INTERFACE_BLUEZ_ADP);

    mode = (enable ? BLUEZ_RADIO_MODE_DISCOVERABLE : BLUEZ_RADIO_MODE_OFF);

    res = dbus_g_proxy_call (proxy, BLUEZ_SET_MODE, &error, G_TYPE_STRING, mode,
                              G_TYPE_INVALID, G_TYPE_INVALID);

    if (!res) {

        g_warning ("[%s] Failed to enable BT radio", __FUNCTION__);
        g_object_unref (proxy);
        g_error_free (error);
    }

    g_object_unref (proxy);

    return res;
}

gchar *
nm_bt_utils_check_pand (NmBtUtils *self, gchar *addr)
{
    NmBtUtilsPrivate *priv;
    DBusGProxy *proxy = NULL;
    GError *error = NULL;
    GArray *handle_array = NULL;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    if (!priv->connection) {
        priv->connection = nm_bt_utils_create_connection (self);
        
        g_return_val_if_fail (priv->connection != NULL, FALSE);
    }

    proxy = dbus_g_proxy_new_for_name (priv->connection,
                                        DBUS_SERVICE_BLUEZ,
                                        NM_BLUETOOTH_INTERFACE,
                                        DBUS_INTERFACE_BLUEZ_ADP);

    if(!dbus_g_proxy_call (proxy, BLUEZ_GET_REMOTE_SERVICE_HANDLES, &error,
                            G_TYPE_STRING, addr,
                            G_TYPE_STRING, BLUEZ_NETWORK_PAN_SERVICE_UUID,
                            G_TYPE_INVALID,
                            DBUS_TYPE_G_UINT_ARRAY, &handle_array,
                            G_TYPE_INVALID)) {

            g_warning ("[%s] Error. Check if pand is enabled on host!",
                        __FUNCTION__);
            g_error_free (error);
            return NULL;
    }
    
    if (handle_array->data == NULL) {
        if(!dbus_g_proxy_call (proxy, BLUEZ_GET_REMOTE_SERVICE_HANDLES, &error,
                                G_TYPE_STRING, addr,
                                G_TYPE_STRING, BLUEZ_NETWORK_GN_SERVICE_UUID,
                                G_TYPE_INVALID,
                                DBUS_TYPE_G_UINT_ARRAY, &handle_array,
                                G_TYPE_INVALID)) {

            g_warning ("[%s] Error. Check if pand is enabled on host!",
                        __FUNCTION__);
            g_error_free (error);
            return NULL;
        
        }
    }

    g_object_unref (proxy);
    return handle_array->data;
}

static void
nm_bt_utils_discovery_completed(DBusGProxy *object, gpointer user_data)
{
    NmBtUtilsPrivate *priv;

    priv = NM_BT_UTILS_GET_PRIVATE (user_data);
    g_signal_emit (user_data, signals[SEARCH_COMPLETED], 0, G_TYPE_NONE);

}

static void
nm_bt_utils_device_found (DBusGProxy *object, gchar *addr, gchar *name,
                           gpointer user_data)
{
    NmBtUtilsPrivate *priv;

    priv = NM_BT_UTILS_GET_PRIVATE (user_data);
    g_signal_emit (user_data, signals[DEVICE_FOUND], 0, name, addr, NULL);
}

static void
nm_bt_utils_disconnect_discovery_signals (NmBtUtils *self)
{
    NmBtUtilsPrivate *priv;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    g_return_if_fail (priv->discovery_proxy != NULL);

    dbus_g_proxy_disconnect_signal (priv->discovery_proxy,
                                     BLUEZ_DISCOVERY_COMPLETED,
                                     G_CALLBACK (nm_bt_utils_discovery_completed),
                                     self);

    dbus_g_proxy_disconnect_signal (priv->discovery_proxy,
                                     BLUEZ_REMOTE_NAME_UPDATED,
                                     G_CALLBACK (nm_bt_utils_device_found),
                                     self);
}

/************************************************************************
 Tool methods to up and down the interfaces
************************************************************************/

gboolean
nm_bt_utils_ifup (const gchar *gw)
{
    // FIXME: replace bnep0 by the file retrieved from dbus
    gint status = system("sudo /sbin/ifup bnep0");

/*    system (g_strconcat ("sudo /sbin/ip route del default via ", gw,*/
/*                          " dev bnep0", NULL));*/
/*    system (g_strconcat ("sudo /sbin/ip route append default via ", gw,*/
/*                          " dev bnep0", NULL));*/

    return status == 0;
}

gboolean
nm_bt_utils_ifdown (void)
{
    // FIXME: replace bnep0 by the file retrieved from dbus
    gint status = system("sudo /sbin/ifdown bnep0");
    return status == 0;
}

