/*
 * Copyright (C) 2010 Nokia Corporation
 *   @author Mathias Hasselmann <mathias.hasselmann@maemo.org>
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"
#include "appdata.h"
#include "gui.h"
#include "accounts.h"
#include "find_duplicate_contacts/merger.h"

#include <libosso-abook/osso-abook.h>
#include <gtk/gtk.h>
#include <glib.h>
#include <unistd.h>

// Application data.
static AppData appdata;

void osso_abook_merge_with_cb (const char *, gpointer);

static void merge_contacts_cb (GtkWidget *, gpointer);


/* How to extend the menus in the contacts application.
 * For more documentation see:
 * http://maemo.org/api_refs/5.0/5.0-final/libosso-abook/OssoABookMenuExtension.html */

typedef struct {
  OssoABookMenuExtension parent_instance;
} ExampleMenuExtension;

typedef struct {
  OssoABookMenuExtensionClass parent_class;
} ExampleMenuExtensionClass;

G_GNUC_INTERNAL GType
merge_contacts_menu_extension_get_type (void) G_GNUC_CONST;

/* Defines the new plugin object */
OSSO_ABOOK_DEFINE_MENU_PLUGIN
        (ExampleMenuExtension,
         merge_contacts_menu_extension,
         OSSO_ABOOK_TYPE_MENU_EXTENSION,,);

static void
merge_contacts_menu_extension_init (ExampleMenuExtension *ext)
{
}


static void arrow_clicked_cb (HildonEditToolbar *toolbar, gpointer user_data) 
{
   gtk_widget_destroy (appdata.window);
}

static void select_service(GtkWidget *, gint, gpointer);

/**
 * Callback is called after merging contact
 *
 * @uid: new id of contact
 * @user_data: data
 */ 
void osso_abook_merge_with_cb (const char *uid, gpointer user_data) 
{
    // Not correctly merge.
    if (uid == NULL) {
	hildon_banner_show_informationf (NULL, NULL, "Merge fail or is cancelled");
    } else {
	hildon_banner_show_informationf (NULL, NULL, "Merge is complete");
    }
    
}

/**
 * Callback for "merge" button from menu.
 */ 
static void merge_button_cb(GtkWidget *button, gpointer data) 
{
    GList *l = NULL;
    GList *l1 = osso_abook_contact_view_get_selection(appdata.view1);
    l1 = g_list_first(l1);
    while (l1) {
	l = g_list_append(l, l1->data);
	l1 = g_list_next(l1);
    }

    GList *l2 = osso_abook_contact_view_get_selection(appdata.view2);
    l2 = g_list_first(l2);
    while (l2) {
	l = g_list_append(l, l2->data);
	l2 = g_list_next(l2);
    }

    if (g_list_length(l) < 2) {
	hildon_banner_show_informationf (NULL, NULL, "You must choose at least two contacts!");	
    } else {
	osso_abook_merge_contacts_and_save(l, GTK_WINDOW(appdata.window), 
		    (OssoABookMergeWithCb) osso_abook_merge_with_cb, NULL);
    }

}

/**
 * Return account for specified selector.
 *
 */ 
Account *get_account_from_selector(GtkWidget *selector)
{
    GtkTreeIter iter;
    Account *acc;
    GtkTreeModel *selector_model;

    if (hildon_touch_selector_get_selected (HILDON_TOUCH_SELECTOR (selector), 0, &iter)) {
	
	selector_model = hildon_touch_selector_get_model (HILDON_TOUCH_SELECTOR (selector), 0);
	gtk_tree_model_get (selector_model, &iter, 0, &acc, -1);
	return acc;
    }

    return NULL;
}

/**
 * Return true if contact should be visible.
 *
 */ 
gboolean is_visible_contact_for_account(OssoABookContact *contact, Account *acc) 
{
     if (contact) { 
	GList *l = osso_abook_contact_get_attributes (E_CONTACT(contact), acc->vcard_field_name);
	    
	GList *values = NULL;

	l = g_list_first(l);
	if (g_list_length(l) == 0) {
	    return FALSE;
	}
	while (l) {
	    EVCardAttribute *attr = l->data;
	    values = e_vcard_attribute_get_values (attr);
	    values = g_list_first(values);
	    while (values) {
		GList *domains = g_list_first(acc->domains);
		if (domains) {
		    while (domains) { 
			if (g_str_has_suffix(values->data, domains->data)) {
			    return TRUE;
			}
			domains = g_list_next(domains);
		    }
		} else {
		    return TRUE;
		}

		values = g_list_next(values);
	    }
	    l = g_list_next (l);
	}
    
    }
    return FALSE; 
}

/**
 * Retrun true if name of contact has prefix text from live.
 */ 
gboolean is_true_from_live(GtkWidget *live, OssoABookContact *contact) 
{
    gboolean live_result = TRUE;
    const char *text = g_strdown (hildon_live_search_get_text(live));
    if (text) {
	const char *contact_name = g_strdown (g_strdup(osso_abook_contact_get_name(contact)));
	gchar **names = g_strsplit(contact_name, " ", 0);
	gint j = 0;
	while (names[j] != NULL) { 
	    live_result = g_str_has_prefix (names[j], text);
	    if (live_result) {
		return live_result;
	    }
	    j++;
	}


	g_free(contact_name);
	g_strfreev(names);
    } 

    return live_result;
}
/**
 * Function for filter of left list of contacts.
 */ 
gboolean filter_func_left(GtkTreeModel *model,		
		    GtkTreeIter *iter,
		    gpointer data) 
{
    OssoABookContact *contact;
    gtk_tree_model_get(model, iter, 0, &contact,  -1);
 
    gboolean live_result = is_true_from_live(appdata.live_left, contact);
   
    Account *acc = data;
    if (!g_strcmp0(acc->vcard_field_name, "All contacts")) {
	return TRUE && live_result;
    }

   
    return live_result && is_visible_contact_for_account(contact, acc);
}

/**
 * Function for filter of left list of contacts.
 */ 
gboolean filter_func_right(GtkTreeModel *model,		
		    GtkTreeIter *iter,
		    gpointer data) 
{
    OssoABookContact *contact;
    gtk_tree_model_get(model, iter, 0, &contact,  -1);
 
    gboolean live_result = is_true_from_live(appdata.live_right, contact);
   
    Account *acc = data;
    if (!g_strcmp0(acc->vcard_field_name, "All contacts")) {
	return TRUE && live_result;
    }

   
    return live_result && is_visible_contact_for_account(contact, acc);
}

/**
 * Callback is called when user choose service.
 */ 
static void select_service_right(GtkWidget *selector, gint column, gpointer data) 
{
    Account *acc = get_account_from_selector(selector);
    osso_abook_filter_model_set_visible_func(appdata.filter_model_right, 
		    filter_func_right, acc, NULL);   
}


/**
 * Callback is called when user choose service.
 */ 
static void select_service_left(GtkWidget *selector, gint column, gpointer data) 
{

    Account *acc = get_account_from_selector(selector); 
    osso_abook_filter_model_set_visible_func(appdata.filter_model_left, 
		    filter_func_left, acc, NULL);
}

/**
 * Init application data.
 */  
static void init_data(AppData *data) 
{
    Account *acc;
 
    // Accounts 
    data->accounts = get_enable_accounts();

    // Add item with All contacts
    acc = g_new0(Account, 1);
    acc->vcard_field_name = g_strdup("All contacts");
    acc->display_name = g_strdup("All contacts");
    data->accounts = g_list_prepend(data->accounts, acc);

}

/**
 * Deinit data.
 */  
static void de_init_data(AppData *data) 
{
    // Free accounts
    g_list_free(data->accounts);
}

static void destroy_window_cb(GtkWidget *widget, gpointer data) 
{
    de_init_data(&appdata);
}


static void
find_duplicates_cb (GtkWidget *widget,
        gpointer   user_data)
{
    OssoABookRoster *aggregator;
    GtkWidget *window;

    aggregator = osso_abook_aggregator_get_default (NULL);
    window = merger_window_new (OSSO_ABOOK_AGGREGATOR (aggregator));
    gtk_widget_show (window);
}

/**
 * Callback for "Merge Contacts" button from main menu address book.
 *
 */ 
static void
merge_contacts_cb (GtkWidget *widget,
                gpointer   user_data)
{   
    
    GtkWidget *window, *hbox, *toolbar;
    GtkWidget *vbox_left, *vbox_right;
    OssoABookContactModel *model_left, *model_right;
    GtkWidget *view1, *view2;
    OssoABookFilterModel *filter_model_left, *filter_model_right;
    GtkWidget *picker_select_service_right, *selector_right, 
	      *picker_select_service_left, *selector_left; 
    HildonLiveSearch *live_left, *live_right;
    GtkWidget *button, *toolbar_hbox; 
    HildonAppMenu *menu;
    HildonSizeType buttonsize;

    /* Initialize the locale stuff */
    /*setlocale(LC_ALL, "");
    bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
    bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
    textdomain(GETTEXT_PACKAGE);

    gconf_init(0, NULL, NULL);*/

    init_data(&appdata);

    window = hildon_stackable_window_new ();
    appdata.window = window;

    vbox_left = GTK_BOX (gtk_vbox_new (FALSE, 10));
    vbox_right = GTK_BOX (gtk_vbox_new (FALSE, 10));

    hbox = GTK_BOX (gtk_hbox_new (FALSE, 10));
    
    /* ===============
     * LEFT HAND
     * ===============*/
    // Selector of service
    selector_left = create_selector(appdata.accounts, 
		    renderer_account_name, 
		    touch_selector_print_account_func); 
    g_signal_connect(G_OBJECT(selector_left), "changed", 
		    G_CALLBACK(select_service_left), NULL);

    // Create left picker button.
    picker_select_service_left = create_picker_button(selector_left, "Select account", "");
    gtk_box_pack_start (vbox_left, GTK_WIDGET(picker_select_service_left), FALSE, FALSE, 0);
    
    model_left = osso_abook_contact_model_get_default();
    filter_model_left = osso_abook_filter_model_new(model_left);
       
    osso_abook_filter_model_set_visible_func(filter_model_left, filter_func_left, 
    		    g_list_first(appdata.accounts)->data, NULL);
    appdata.filter_model_left = filter_model_left;

    view1 = osso_abook_contact_view_new(HILDON_UI_MODE_EDIT, model_left, filter_model_left);
    osso_abook_contact_view_set_minimum_selection(view1, 1);
    osso_abook_contact_view_set_maximum_selection(view1, 30);
    gtk_box_pack_start (GTK_CONTAINER(vbox_left), GTK_WIDGET(view1), TRUE, TRUE, 0);
    appdata.view1 = view1;

    // Left live search 
    live_left = HILDON_LIVE_SEARCH (hildon_live_search_new ());
    hildon_live_search_set_filter (live_left, filter_model_left); 
    hildon_live_search_widget_hook (live_left, vbox_left, view1);
    gtk_box_pack_start (GTK_BOX (vbox_left), GTK_WIDGET (live_left), FALSE, FALSE, 0);
    appdata.live_left = live_left;
   
    /* ===============
     * RIGHT HAND
     * ===============*/
    // Selector of service
    selector_right = create_selector(appdata.accounts, 
		    renderer_account_name, 
		    touch_selector_print_account_func); 
    g_signal_connect(G_OBJECT(selector_right), "changed", 
		    G_CALLBACK(select_service_right), NULL);
   
    // Create right service picker button.
    picker_select_service_right = create_picker_button(
		    selector_right, "Select account", "");
    gtk_box_pack_start (vbox_right, GTK_WIDGET(picker_select_service_right), FALSE, FALSE, 0);
  
    model_right = osso_abook_contact_model_get_default();
    filter_model_right = osso_abook_filter_model_new(model_right);
    osso_abook_filter_model_set_visible_func(filter_model_right, filter_func_right, 
    		    g_list_first(appdata.accounts)->data, NULL);
    appdata.filter_model_right = filter_model_right;

    view2 = osso_abook_contact_view_new(HILDON_UI_MODE_EDIT, model_right, filter_model_right);
    osso_abook_contact_view_set_minimum_selection(view2, 1);
    osso_abook_contact_view_set_maximum_selection(view2, 30);
    gtk_box_pack_start (GTK_CONTAINER(vbox_right), GTK_WIDGET(view2), TRUE, TRUE, 0);
    appdata.view2 = view2;

    // Right live search 
    live_right = HILDON_LIVE_SEARCH (hildon_live_search_new ());
    hildon_live_search_set_filter (live_right, filter_model_right); 
    hildon_live_search_widget_hook (live_right, vbox_right, view2);
    gtk_box_pack_start (GTK_BOX (vbox_right), GTK_WIDGET (live_right), FALSE, FALSE, 0);
    appdata.live_right = live_right;

   
    gtk_box_pack_start (GTK_CONTAINER(hbox), GTK_WIDGET(vbox_left), TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_CONTAINER(hbox), GTK_WIDGET(vbox_right), TRUE, TRUE, 0);
    gtk_container_add (GTK_CONTAINER(window), GTK_WIDGET(hbox));
   
    // create and attach menu to the window
    /*toolbar = hildon_edit_toolbar_new_with_text ("Merge Contacts", "Merge");
    hildon_window_set_edit_toolbar (HILDON_WINDOW (window), HILDON_EDIT_TOOLBAR (toolbar));
    g_signal_connect (toolbar, "button-clicked", G_CALLBACK (merge_button_cb), NULL);
    g_signal_connect (toolbar, "arrow-clicked", G_CALLBACK (arrow_clicked_cb), NULL);
    gtk_window_fullscreen (GTK_WINDOW (window));*/
    buttonsize = HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH;
    menu = HILDON_APP_MENU (hildon_app_menu_new ());
    GtkAccelGroup *accel;
    accel = gtk_accel_group_new ();

    button = hildon_gtk_button_new (buttonsize);
    gtk_button_set_label (GTK_BUTTON (button), "Merge");
    g_signal_connect_after (button, "clicked", G_CALLBACK (merge_button_cb), NULL);
    hildon_app_menu_append (menu, GTK_BUTTON (button));

    gtk_widget_add_accelerator (button, "activate", accel, GDK_r, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);

    button = hildon_gtk_button_new (buttonsize);
    gtk_button_set_label (GTK_BUTTON (button), "Find duplicate contacts");
    g_signal_connect_after (button, "clicked", G_CALLBACK (find_duplicates_cb), NULL);
    hildon_app_menu_append (menu, GTK_BUTTON (button));

    gtk_widget_show_all (GTK_WIDGET (menu));

    hildon_window_set_app_menu (HILDON_WINDOW (window), menu);

    gtk_window_add_accel_group (GTK_WINDOW (window), accel);
    g_object_unref (accel);

    gtk_widget_show_all (window);
}

/* Entries to show in the menu, the values are:
 * - label: a label for the menu item;
 * - accel_mods:  modifier keys to use for the menu item;
 * - callback: a function to call when the menu item is activated;
 * - name: the name of the menu item.
 */
static OssoABookMenuEntry entries[] = {
  { "Merge contacts", 0, 0, G_CALLBACK (merge_contacts_cb), NULL },
};

static const OssoABookMenuEntry *
get_menu_entries (OssoABookMenuExtension *menu_extension)
{
  return entries;
}

static int
get_n_menu_entries (OssoABookMenuExtension *menu_extension)
{
  return G_N_ELEMENTS (entries);
}

static void
merge_contacts_menu_extension_class_init (ExampleMenuExtensionClass *cls)
{
  OssoABookMenuExtensionClass *menu_extension_class;

  menu_extension_class = OSSO_ABOOK_MENU_EXTENSION_CLASS (cls);
  menu_extension_class->get_n_menu_entries = get_n_menu_entries;
  menu_extension_class->get_menu_entries = get_menu_entries;
  /* Which menus should be extended with our buttons, separated by a ";" */
  menu_extension_class->name =  OSSO_ABOOK_MENU_NAME_MAIN_VIEW
                            ";" OSSO_ABOOK_MENU_NAME_CONTACT_VIEW
                            ";" OSSO_ABOOK_MENU_NAME_MECARD_VIEW
                            ";" OSSO_ABOOK_MENU_NAME_SIM_VIEW;
}

static void
merge_contacts_menu_extension_class_finalize (ExampleMenuExtensionClass *cls)
{
}

