/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include <gtk/gtk.h>

#include "matchcell.h"
#include "debug.h"
#include "merger.h"


#define MATCH_CELL_RENDERER_GET_PRIVATE(obj) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((obj), TYPE_MATCH_CELL_RENDERER, \
                                  MatchCellRendererPrivate))

enum {
    PROP_0,
    PROP_MATCH,
};

struct _MatchCellRendererPrivate
{
    Match *match;
};

G_DEFINE_TYPE (MatchCellRenderer, match_cell_renderer, GTK_TYPE_CELL_RENDERER)

GtkCellRenderer *
match_cell_renderer_new (void)
{
    return g_object_new (TYPE_MATCH_CELL_RENDERER, NULL);
}

static void
match_cell_renderer_get_size (GtkCellRenderer *cell,
                              GtkWidget       *widget,
                              GdkRectangle    *cell_area,
                              gint            *x_offset,
                              gint            *y_offset,
                              gint            *width,
                              gint            *height)
{
    /* This doesn't really matter, the values are ignored */
    if (x_offset)
        *x_offset = 0;
    if (y_offset)
        *y_offset = 0;
    if (width)
        *width = HILDON_ICON_PIXEL_SIZE_FINGER;
    if (height)
        *height = HILDON_ICON_PIXEL_SIZE_FINGER;
}

static void
get_pixbuf_rectangle (GdkPixbuf    *pixbuf,
                      GdkRectangle *rectangle)
{
    g_return_if_fail (pixbuf);
    g_return_if_fail (rectangle);

    rectangle->x = 0;
    rectangle->y = 0;
    rectangle->width = gdk_pixbuf_get_width (pixbuf);
    rectangle->height = gdk_pixbuf_get_height (pixbuf);
}

static void
match_cell_renderer_render (GtkCellRenderer      *cell,
                            GdkWindow            *window,
                            GtkWidget            *widget,
                            GdkRectangle         *background_area,
                            GdkRectangle         *cell_area,
                            GdkRectangle         *expose_area,
                            GtkCellRendererState  flags)
{
    static const gint avatar_distance = 6;

    MatchCellRenderer *self = (MatchCellRenderer *)cell;
    OssoABookContact *c1;
    OssoABookContact *c2;
    PangoLayout *layout_contact1;
    PangoLayout *layout_contact2;
    PangoLayout *layout_description;
    GdkPixbuf *pixbuf1;
    GdkPixbuf *pixbuf2;
    PangoRectangle rectangle_contact1;
    PangoRectangle rectangle_contact2;
    PangoRectangle rectangle_description;
    GdkRectangle rectangle_pixbuf1;
    GdkRectangle rectangle_pixbuf2;
    gint description_max_width;
    gint contact_max_width;
    GtkStyle *style;
    gchar *tmp;
    gchar *markup;
    gchar *description_foreground;
    gint text_height;
    gint text_space;
    GdkColor color;
    gint xpad;
    gint ypad;
    cairo_t *cr;
    GtkStateType state = 0;

    /* Prepare the layouts for the contact names */

    match_get_contacts (self->priv->match, &c1, &c2);

    layout_contact1 = gtk_widget_create_pango_layout (widget, NULL);
    pango_layout_set_text (layout_contact1,
            osso_abook_contact_get_display_name (c1), -1);

    layout_contact2 = gtk_widget_create_pango_layout (widget, NULL);
    pango_layout_set_text (layout_contact2,
            osso_abook_contact_get_display_name (c2), -1);

    /* Prepare the layouts for the description */

    style = gtk_rc_get_style_by_paths (gtk_settings_get_default (), NULL,
                NULL, GTK_TYPE_LABEL);
    if (gtk_style_lookup_color (style, "SecondaryTextColor", &color)) {
        tmp = gdk_color_to_string (&color);
        description_foreground = g_strdup_printf (" foreground=\"%s\"", tmp);
        g_free (tmp);
    } else {
        description_foreground = g_strdup ("");
    }
    layout_description = gtk_widget_create_pango_layout (widget, NULL);
    tmp = g_markup_escape_text (match_get_description (self->priv->match), -1);
    markup = g_strdup_printf ("<span size=\"x-small\"%s>%s</span>",
            description_foreground, tmp);
    pango_layout_set_markup (layout_description, markup, -1);
    g_free (markup);
    g_free (tmp);
    g_free (description_foreground);

    /* Prepare the pixbufs for the avatars */

    pixbuf1 = get_avatar (c1, widget);
    pixbuf2 = get_avatar (c2, widget);

    /* Calculate the positions of the various elements */

    get_pixbuf_rectangle (pixbuf1, &rectangle_pixbuf1);
    get_pixbuf_rectangle (pixbuf2, &rectangle_pixbuf2);

    xpad = GTK_CELL_RENDERER (self)->xpad;
    ypad = GTK_CELL_RENDERER (self)->ypad;

    description_max_width = cell_area->width - rectangle_pixbuf1.width -
        rectangle_pixbuf2.width - xpad * 2 - avatar_distance * 2;
    pango_layout_set_ellipsize (layout_description, PANGO_ELLIPSIZE_MIDDLE);
    pango_layout_set_width (layout_description,
            description_max_width * PANGO_SCALE);

    contact_max_width = (description_max_width - avatar_distance * 2) / 2;
    pango_layout_set_ellipsize (layout_contact1, PANGO_ELLIPSIZE_MIDDLE);
    pango_layout_set_width (layout_contact1, contact_max_width * PANGO_SCALE);
    pango_layout_set_ellipsize (layout_contact2, PANGO_ELLIPSIZE_MIDDLE);
    pango_layout_set_width (layout_contact2, contact_max_width * PANGO_SCALE);

    pango_layout_get_pixel_extents (layout_contact1, NULL, &rectangle_contact1);
    pango_layout_get_pixel_extents (layout_contact2, NULL, &rectangle_contact2);
    pango_layout_get_pixel_extents (layout_description, NULL, &rectangle_description);

    rectangle_pixbuf1.x = xpad + cell_area->x;
    rectangle_pixbuf2.x = cell_area->width - rectangle_pixbuf2.width - xpad;

    rectangle_pixbuf1.y = cell_area->y +
        (cell_area->height - rectangle_pixbuf1.height) / 2;
    rectangle_pixbuf2.y = cell_area->y +
        (cell_area->height - rectangle_pixbuf2.height) / 2;

    rectangle_contact1.x = rectangle_pixbuf1.x + rectangle_pixbuf1.width +
        avatar_distance;
    rectangle_contact2.x = rectangle_pixbuf2.x - rectangle_contact2.width -
        avatar_distance;
    rectangle_description.x = (cell_area->width -
        rectangle_description.width) / 2 + cell_area->x;

    text_height  = rectangle_contact1.height +
                   rectangle_description.height +
                   ypad * 2;
    text_space = (cell_area->height - text_height) / 3;
    rectangle_contact1.y = text_space + cell_area->y + ypad;
    rectangle_contact2.y = rectangle_contact1.y;
    rectangle_description.y = rectangle_contact1.y +
        rectangle_contact1.height + text_space;

    /* Draw pixbufs */

    cr = gdk_cairo_create (window);

    gdk_cairo_set_source_pixbuf (cr, pixbuf1, rectangle_pixbuf1.x,
            rectangle_pixbuf1.y);
    gdk_cairo_rectangle (cr, &rectangle_pixbuf1);
    cairo_fill (cr);

    gdk_cairo_set_source_pixbuf (cr, pixbuf2, rectangle_pixbuf2.x,
            rectangle_pixbuf2.y);
    gdk_cairo_rectangle (cr, &rectangle_pixbuf2);
    cairo_fill (cr);

    /* Draw text */

    gtk_paint_layout (widget->style, window, state, TRUE, expose_area,
            widget, "cellrenderertext", rectangle_contact1.x,
            rectangle_contact1.y, layout_contact1);
    gtk_paint_layout (widget->style, window, state, TRUE, expose_area,
            widget, "cellrenderertext", rectangle_contact2.x,
            rectangle_contact2.y, layout_contact2);
    gtk_paint_layout (widget->style, window, state, TRUE, expose_area,
            widget, "cellrenderertext", rectangle_description.x,
            rectangle_description.y, layout_description);
    
    /* Free stuff */

    cairo_destroy (cr);

    g_object_unref (layout_contact1);
    g_object_unref (layout_contact2);
    g_object_unref (layout_description);
}

static void
match_cell_renderer_get_property (GObject    *object,
                                  guint       param_id,
                                  GValue     *value,
                                  GParamSpec *pspec)
{
    MatchCellRenderer *self = MATCH_CELL_RENDERER (object);

    switch (param_id) {
        case PROP_MATCH:
            g_value_set_boxed (value, self->priv->match);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
match_cell_renderer_set_property (GObject      *object,
                                  guint         param_id,
                                  const GValue *value,
                                  GParamSpec   *pspec)
{
    MatchCellRenderer *self = MATCH_CELL_RENDERER (object);

    switch (param_id) {
        case PROP_MATCH:
            if (self->priv->match)
                match_unref (self->priv->match);
            self->priv->match = g_value_dup_boxed (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
match_cell_renderer_init (MatchCellRenderer *self)
{
    self->priv = MATCH_CELL_RENDERER_GET_PRIVATE (self);
}

static void
match_cell_renderer_finalize (GObject *object)
{
    MatchCellRenderer *self = MATCH_CELL_RENDERER (object);

    if (self->priv->match)
        match_unref (self->priv->match);

    G_OBJECT_CLASS (match_cell_renderer_parent_class)->finalize (object);
}

static void
match_cell_renderer_class_init (MatchCellRendererClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);
    GtkCellRendererClass *cell_class = GTK_CELL_RENDERER_CLASS (class);

    object_class->finalize = match_cell_renderer_finalize;
    object_class->get_property = match_cell_renderer_get_property;
    object_class->set_property = match_cell_renderer_set_property;

    cell_class->get_size = match_cell_renderer_get_size;
    cell_class->render = match_cell_renderer_render;

    g_object_class_install_property (
            object_class,
            PROP_MATCH,
            g_param_spec_boxed (
                "match",
                "Match object",
                "The match to render",
                TYPE_MATCH,
                G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

    g_type_class_add_private (object_class, sizeof (MatchCellRendererPrivate));
}
