/*
 * OpenConnect GUI (a GTK/Hildon GUI client for OpenConnect)
 *
 * Copyright (c) 2009-2010 Pascal Jermini <lorelei@garage.maemo.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to:
 *
 *   Free Software Foundation, Inc.
 *   51 Franklin Street, Fifth Floor,
 *   Boston, MA 02110-1301 USA
 */

#include <hildon/hildon.h>
#include <stdlib.h>

#include "openconnect-gui.h"
#include "about.h"
#include "profiles.h"

gboolean read_output(GIOChannel *channel, GIOCondition condition, gpointer data)
{
	GError *error = NULL;
	GIOStatus status;
	gchar *line;
	gsize len, term;

	OpenConnectGUI *self = (OpenConnectGUI*)data;

	status = g_io_channel_read_line(self->channel, &line, &len, &term, 
			&error);

	if (status != G_IO_STATUS_NORMAL || line == NULL || error != NULL) {
		if (self->channel != NULL) {
			g_io_channel_shutdown(self->channel, TRUE, NULL);
		}

		gtk_widget_set_sensitive(GTK_WIDGET(self->connButton), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(self->disconnButton), 
				FALSE);

		self->channel = NULL;
		return FALSE;
	}
		
	GtkTextIter iter;
	GtkTextMark *insert_mark;

   	gtk_text_buffer_get_end_iter(self->logBuffer, &iter);
	gtk_text_buffer_insert(self->logBuffer,  &iter, line, -1);
	gtk_text_buffer_get_end_iter(self->logBuffer, &iter);
	insert_mark = gtk_text_buffer_get_insert(self->logBuffer);
	gtk_text_buffer_place_cursor(self->logBuffer, &iter);
	gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(self->logWidget), 
			insert_mark, 0, TRUE, 0 ,0);

	g_free(line);

	return TRUE;
}

static void vpnConnect (GtkWidget *widget, gpointer   data)
{
	gint fout, argc;
	gchar *command;
	gchar **argv;
	gchar *profile = NULL, *username, *password, *server;
	OpenConnectGUI *self = (OpenConnectGUI*)data;

	profile = hildon_touch_selector_get_current_text(HILDON_TOUCH_SELECTOR(self->profileSelector));

	if (profile == NULL) {
		GtkWidget* banner;
		banner = hildon_banner_show_information_with_markup (widget, NULL,
			"You must select a profile before connecting!");
		hildon_banner_set_timeout (HILDON_BANNER (banner), 9000);
		return;
	}

	username = g_key_file_get_string(self->keyfile, profile, "username", NULL);
	password = g_key_file_get_string(self->keyfile, profile, "password", NULL);
	server = g_key_file_get_string(self->keyfile, profile, "server", NULL);

	command = g_strdup_printf("/usr/bin/openconnect-wrapper --user=%s --passwd=%s %s", username, password, server);
	g_shell_parse_argv(command, &argc, &argv, NULL);
	g_print("%s\n", command);

	g_spawn_async_with_pipes (NULL, argv, NULL, G_SPAWN_SEARCH_PATH, 
			NULL, NULL, &(self->pid), NULL, &fout, NULL, NULL);

	gtk_widget_set_sensitive(GTK_WIDGET(self->connButton), FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(self->disconnButton), TRUE);

	g_strfreev(argv);
	g_free(command);

	self->channel = g_io_channel_unix_new(fout);
	g_io_add_watch(self->channel, G_IO_IN | G_IO_ERR | G_IO_HUP, 
			read_output, self);
	g_io_channel_unref(self->channel);
}


static void vpnDisconnect (GtkWidget *widget, gpointer   data)
{
	OpenConnectGUI *self = (OpenConnectGUI*)data;
	gtk_widget_set_sensitive(GTK_WIDGET(self->connButton), TRUE);
	gtk_widget_set_sensitive(GTK_WIDGET(self->disconnButton), FALSE);
	kill(self->pid, SIGTERM);
}

static void newProfile_cb (GtkWidget *widget, gpointer   data)
{
	createProfilesWindow(widget, data);
}

static void deleteProfile (GtkWidget *widget, gpointer   data)
{
	GtkWidget *note;
	gint response;
	OpenConnectGUI *self = (OpenConnectGUI*)data;
	gchar *profile;

	profile = hildon_touch_selector_get_current_text(HILDON_TOUCH_SELECTOR(self->profileSelector));

	if (profile == NULL) {
		GtkWidget* banner;
		banner = hildon_banner_show_information_with_markup (widget, NULL,
			"You must select a profile to delete!");
		hildon_banner_set_timeout (HILDON_BANNER (banner), 9000);
	} else {
		gchar *text = g_strdup_printf("Do you want to delete profile \"%s\"?",
							 profile);
		note = hildon_note_new_confirmation (GTK_WINDOW(self->window), text);
		g_free(text);
		response = gtk_dialog_run (GTK_DIALOG (note));
		if (response == GTK_RESPONSE_OK) {
			gchar *sdata;
			gsize confSize;
			g_key_file_remove_group (self->keyfile, profile, NULL);
			/* Error checking! */
			g_file_set_contents (self->confFile, sdata, confSize, NULL);
		}
		gtk_object_destroy (GTK_OBJECT (note));
		load_profiles(self);
	}

	g_free(profile);
}
static void editProfile_cb (GtkWidget *widget, gpointer   data)
{
	OpenConnectGUI *self = (OpenConnectGUI*)data;
	profileWidgets *pwidgets;

	gchar *profile = hildon_touch_selector_get_current_text(HILDON_TOUCH_SELECTOR(self->profileSelector));

	if (profile == NULL) {
		GtkWidget* banner;
		banner = hildon_banner_show_information_with_markup (widget, NULL,
			"You must select a profile to edit!");
		hildon_banner_set_timeout (HILDON_BANNER (banner), 9000);
		return;
	}
	pwidgets = createProfilesWindow(widget, data);
	editProfile(pwidgets, profile);
}

static HildonAppMenu *create_menu(OpenConnectGUI *appData)
{
		GtkWidget *button;
		HildonAppMenu *menu = HILDON_APP_MENU(hildon_app_menu_new());

		button = hildon_gtk_button_new(HILDON_SIZE_AUTO);
		gtk_button_set_label(GTK_BUTTON(button), "New Profile");
		g_signal_connect_after(button, "clicked", 
				G_CALLBACK(newProfile_cb), appData);
		hildon_app_menu_append(menu, GTK_BUTTON(button));

		button = hildon_gtk_button_new(HILDON_SIZE_AUTO);
		gtk_button_set_label(GTK_BUTTON(button), "Delete Profile");
		g_signal_connect_after(button, "clicked", 
				G_CALLBACK(deleteProfile), appData);
		hildon_app_menu_append(menu, GTK_BUTTON(button));

		button = hildon_gtk_button_new(HILDON_SIZE_AUTO);
		gtk_button_set_label(GTK_BUTTON(button), "Edit Profile");
		g_signal_connect_after(button, "clicked", 
				G_CALLBACK(editProfile_cb), appData);
		hildon_app_menu_append(menu, GTK_BUTTON(button));

		button = hildon_gtk_button_new(HILDON_SIZE_AUTO);
		gtk_button_set_label(GTK_BUTTON(button), "About");
		g_signal_connect_after(button, "clicked",
			       	G_CALLBACK(aboutWindow), appData);
		hildon_app_menu_append(menu, GTK_BUTTON(button));

		gtk_widget_show_all(GTK_WIDGET(menu));

		return menu;
}

static GtkWidget *createProfileSelector(OpenConnectGUI *self)
{
	GtkWidget *selector;
	GtkCellRenderer *renderer = NULL;
	HildonTouchSelectorColumn *column = NULL;
	/* Create a touch selector */
	selector = hildon_touch_selector_new();
	/* Create model to store selector's items */
	self->profile_store = gtk_list_store_new (1, G_TYPE_STRING);


	renderer = gtk_cell_renderer_text_new();
	gtk_cell_renderer_set_fixed_size (renderer, -1, 100);
	/* Add the column to the selector */
	column = hildon_touch_selector_append_column (HILDON_TOUCH_SELECTOR (selector), GTK_TREE_MODEL (self->profile_store), renderer, "text", 0, NULL);
	g_object_set (G_OBJECT (column), "text-column", 0, NULL);
	self->profileButton = hildon_picker_button_new(HILDON_SIZE_AUTO, 
					HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_button_set_title(HILDON_BUTTON(self->profileButton), "Select a VPN profile");
	self->profileSelector = selector;
	hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(self->profileButton),
					HILDON_TOUCH_SELECTOR(self->profileSelector));

	return self->profileButton;
}

GtkWidget *createLogView(OpenConnectGUI *self)
{
	GtkWidget *scrolledWindow;
	self->logWidget = hildon_text_view_new();
	PangoFontDescription *font;

	scrolledWindow = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledWindow), 
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	gtk_text_view_set_editable (GTK_TEXT_VIEW (self->logWidget), FALSE);
	gtk_text_view_set_left_margin (GTK_TEXT_VIEW (self->logWidget), 10);
	gtk_text_view_set_right_margin (GTK_TEXT_VIEW (self->logWidget), 10);

	font = pango_font_description_from_string ("Monospace Regular 12");
	gtk_widget_modify_font (self->logWidget, font);


	gtk_container_add(GTK_CONTAINER(scrolledWindow), self->logWidget);
	gtk_widget_show(self->logWidget);

	self->logBuffer = hildon_text_view_get_buffer(HILDON_TEXT_VIEW (self->logWidget));

	return scrolledWindow;
}


int main (int argc, char **argv)
{
	OpenConnectGUI *self;
	HildonProgram *program;
	HildonAppMenu *menu;
	GtkWidget *bbox;
	GtkWidget *mainBox;
	GtkWidget *log, *pickButton;


	self = (OpenConnectGUI*) (malloc(sizeof(OpenConnectGUI)));
	self->channel = NULL;
	self->confFile = g_strdup_printf("%s/.openconnect.conf", g_get_home_dir());

	hildon_gtk_init (&argc, &argv);

	program = hildon_program_get_instance ();

	self->window = hildon_stackable_window_new ();
	gtk_window_set_title ( GTK_WINDOW (self->window), "OpenConnect GUI");

	menu = create_menu(self);

	hildon_program_add_window (program, HILDON_WINDOW (self->window));
	hildon_window_set_app_menu(HILDON_WINDOW(self->window), menu);

	g_signal_connect (G_OBJECT (self->window), "delete_event",
				   	G_CALLBACK (gtk_main_quit), NULL);

	self->connButton = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT, 
					HILDON_BUTTON_ARRANGEMENT_HORIZONTAL, "Connect", NULL);
	g_signal_connect (G_OBJECT(self->connButton), "clicked", G_CALLBACK (vpnConnect),
				   	self);

	self->disconnButton = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT, 
					HILDON_BUTTON_ARRANGEMENT_HORIZONTAL, 
					"Disconnect", NULL);
	g_signal_connect (G_OBJECT(self->disconnButton), "clicked", 
					G_CALLBACK (vpnDisconnect), self);

	pickButton = createProfileSelector(self);
	log = createLogView(self);

	bbox = gtk_hbox_new(FALSE, 0);
	mainBox = gtk_vbox_new(FALSE, 0);

	gtk_box_pack_start(GTK_BOX(bbox), self->connButton, TRUE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(bbox), self->disconnButton, TRUE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(mainBox), pickButton, TRUE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(mainBox), log, TRUE, TRUE, 0);

	gtk_box_pack_start(GTK_BOX(mainBox), bbox, TRUE, FALSE, 0);
	
	gtk_container_add(GTK_CONTAINER(self->window), mainBox);
	gtk_widget_set_sensitive(GTK_WIDGET(self->disconnButton), FALSE);
	gtk_widget_show_all (GTK_WIDGET (self->window));

	/* Load profiles */
	load_profiles(self);

	gtk_main ();

	g_free(self->confFile);
	free(self);
	return 0;
}

