/*
 * OpenConnect GUI (a GTK/Hildon GUI client for OpenConnect)
 *
 * Copyright (c) 2009-2010 Pascal Jermini <lorelei@garage.maemo.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to:
 *
 *   Free Software Foundation, Inc.
 *   51 Franklin Street, Fifth Floor,
 *   Boston, MA 02110-1301 USA
 */

#include <stdlib.h>

#include "profiles.h"

void saveProfile_cb (GtkWidget *widget, gpointer data)
{
	const gchar *profileName, *server, *username, *password, *free_option;
	profileWidgets *self = (profileWidgets*)data;
	gchar *sdata;
	gsize confSize;

	profileName = hildon_entry_get_text(HILDON_ENTRY(self->profileName));
	server = hildon_entry_get_text(HILDON_ENTRY(self->server));
	username = hildon_entry_get_text(HILDON_ENTRY(self->username));
	password = hildon_entry_get_text(HILDON_ENTRY(self->password));
	free_option = hildon_entry_get_text(HILDON_ENTRY(self->free_option));

	if ((g_strcmp0(profileName, "") == 0) ||
					(g_strcmp0(server, "") == 0) ||
					(g_strcmp0(username, "") == 0) ||
					(g_strcmp0(password, "") == 0)) {
		GtkWidget* banner;
		banner = hildon_banner_show_information_with_markup (widget, NULL,
		"Please fill all fields in this form!");
		hildon_banner_set_timeout (HILDON_BANNER (banner), 9000);
		return;
	}

	if (self->editingProfile) {
		if (g_strcmp0(profileName, self->oldName) != 0) {
			/* Remove old group in keyfile */
			g_key_file_remove_group (self->appData->keyfile, self->oldName, NULL);
		}
	}

	g_key_file_set_string(self->appData->keyfile, profileName, "server", server);
	g_key_file_set_string(self->appData->keyfile, profileName, "username", username);
	g_key_file_set_string(self->appData->keyfile, profileName, "password", password);
	g_key_file_set_string(self->appData->keyfile, profileName, "free_option", free_option);

	sdata = g_key_file_to_data (self->appData->keyfile, &confSize, NULL);
	/* XXX error checking! */
	g_file_set_contents (self->appData->confFile, sdata, confSize, NULL);

	g_free(sdata);
	load_profiles(self->appData);
	
	gtk_widget_destroy(self->win);

}

profileWidgets *createProfilesWindow (GtkWidget *widget, gpointer data)
{
	profileWidgets *self;
	OpenConnectGUI *rootData = (OpenConnectGUI*)data;

	GtkWidget *vbox, *profNameBox, *serverBox, *usernameBox, *passwordBox, *free_optionBox;
	GtkWidget *buttonBox, *saveButton;
	GtkWidget *profileNameLabel, *serverLabel, *usernameLabel, 
		  *passwordLabel, *free_optionLabel;
	self = (profileWidgets*)malloc(sizeof(profileWidgets));
	self->appData = rootData;
	self->editingProfile = FALSE;
	
	/* Create the main window */
	self->win = hildon_stackable_window_new();
	gtk_window_set_title(GTK_WINDOW(self->win), 
			"OpenConnect GUI - Profiles");

	profileNameLabel = gtk_label_new("Profile Name");
	gtk_misc_set_alignment(GTK_MISC(profileNameLabel), 1.0, 0.5);
	self->profileName = hildon_entry_new(HILDON_SIZE_HALFSCREEN_WIDTH);
	hildon_gtk_entry_set_input_mode(GTK_ENTRY(self->profileName), HILDON_GTK_INPUT_MODE_FULL | !HILDON_GTK_INPUT_MODE_AUTOCAP);
	serverLabel = gtk_label_new("VPN server");
	gtk_misc_set_alignment(GTK_MISC(serverLabel), 1.0, 0.5);
	self->server = hildon_entry_new(HILDON_SIZE_HALFSCREEN_WIDTH);
	hildon_gtk_entry_set_input_mode(GTK_ENTRY(self->server), HILDON_GTK_INPUT_MODE_FULL | !HILDON_GTK_INPUT_MODE_AUTOCAP);
	usernameLabel = gtk_label_new("Username");
	gtk_misc_set_alignment(GTK_MISC(usernameLabel), 1.0, 0.5);
	self->username = hildon_entry_new(HILDON_SIZE_HALFSCREEN_WIDTH);
	hildon_gtk_entry_set_input_mode(GTK_ENTRY(self->username), HILDON_GTK_INPUT_MODE_FULL | !HILDON_GTK_INPUT_MODE_AUTOCAP);
	passwordLabel = gtk_label_new("Password");
	gtk_misc_set_alignment(GTK_MISC(passwordLabel), 1.0, 0.5);
	self->password = hildon_entry_new(HILDON_SIZE_HALFSCREEN_WIDTH);
	hildon_gtk_entry_set_input_mode(GTK_ENTRY(self->password), HILDON_GTK_INPUT_MODE_FULL | !HILDON_GTK_INPUT_MODE_AUTOCAP);
	gtk_entry_set_visibility(GTK_ENTRY(self->password), FALSE);
	free_optionLabel = gtk_label_new("Free option");
	gtk_misc_set_alignment(GTK_MISC(free_optionLabel), 1.0, 0.5);
	self->free_option = hildon_entry_new(HILDON_SIZE_HALFSCREEN_WIDTH);
	hildon_gtk_entry_set_input_mode(GTK_ENTRY(self->free_option), HILDON_GTK_INPUT_MODE_FULL | !HILDON_GTK_INPUT_MODE_AUTOCAP);
	
	vbox = gtk_vbox_new (FALSE, 0);
	profNameBox = gtk_hbox_new (FALSE, 0);
	serverBox = gtk_hbox_new (FALSE, 0);
	usernameBox = gtk_hbox_new (FALSE, 0);
	passwordBox = gtk_hbox_new (FALSE, 0);
    free_optionBox =  gtk_hbox_new (FALSE, 0);
	buttonBox = gtk_hbox_new (FALSE, 0);

	gtk_box_pack_start (GTK_BOX(profNameBox), profileNameLabel, TRUE, 
			TRUE, 0); 
	gtk_box_pack_start (GTK_BOX(profNameBox), self->profileName, TRUE, 
			FALSE, 0); 
	gtk_box_pack_start (GTK_BOX(serverBox), serverLabel, TRUE, 
			TRUE, 0); 
	gtk_box_pack_start (GTK_BOX(serverBox), self->server, TRUE, 
			FALSE, 0); 
	gtk_box_pack_start (GTK_BOX(usernameBox), usernameLabel, TRUE, 
			TRUE, 0); 
	gtk_box_pack_start (GTK_BOX(usernameBox), self->username, TRUE, 
			FALSE, 0); 
	gtk_box_pack_start (GTK_BOX(passwordBox), passwordLabel, TRUE, 
			TRUE, 0); 
	gtk_box_pack_start (GTK_BOX(passwordBox), self->password, TRUE, 
			FALSE, 0); 
	gtk_box_pack_start (GTK_BOX(free_optionBox), free_optionLabel, TRUE, 
			TRUE, 0); 
	gtk_box_pack_start (GTK_BOX(free_optionBox), self->free_option, TRUE, 
			FALSE, 0); 

	saveButton = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT, 
			HILDON_BUTTON_ARRANGEMENT_HORIZONTAL, "Save", 
			NULL);
	g_signal_connect (G_OBJECT(saveButton), "clicked", 
			G_CALLBACK (saveProfile_cb), self);


	gtk_box_pack_start (GTK_BOX(buttonBox), saveButton, TRUE, 
			TRUE, 0); 

	gtk_box_pack_start(GTK_BOX(vbox), profNameBox, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), serverBox, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), usernameBox, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), passwordBox, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), free_optionBox, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), buttonBox, FALSE, FALSE, 0);
	
	gtk_container_add(GTK_CONTAINER(self->win), vbox);

	/* This call show the window and also add the window to the stack */
	gtk_widget_show_all (self->win);

	return self;

}

void editProfile (gpointer data, gchar* profile)
{
   gchar *username, *password, *server, *free_option;
	profileWidgets *self = (profileWidgets*)data;
	self->editingProfile = TRUE;
	self->oldName = g_strdup(profile);

	username = g_key_file_get_string(self->appData->keyfile, profile, "username", NULL);
	password = g_key_file_get_string(self->appData->keyfile, profile, "password", NULL);
	server = g_key_file_get_string(self->appData->keyfile, profile, "server", NULL);
	free_option = g_key_file_get_string(self->appData->keyfile, profile, "free_option", NULL);

	hildon_entry_set_text(HILDON_ENTRY(self->profileName), profile);
	hildon_entry_set_text(HILDON_ENTRY(self->server), server);
	hildon_entry_set_text(HILDON_ENTRY(self->username), username);
	hildon_entry_set_text(HILDON_ENTRY(self->password), password);
	hildon_entry_set_text(HILDON_ENTRY(self->free_option), free_option);

}

void load_profiles(OpenConnectGUI *self)
{
	GKeyFileFlags flags = G_KEY_FILE_KEEP_COMMENTS;
	GError *error = NULL;
	gint i = 0;


	if (self->keyfile == NULL) {
		self->keyfile = g_key_file_new();
		if (!g_key_file_load_from_file(self->keyfile, self->confFile, flags, &error)) {
			/* Could not load conf. file...skip loading configuration */
			return;
		}
	}

	self->profiles = g_key_file_get_groups(self->keyfile, NULL);

	gtk_list_store_clear(self->profile_store);
	for (i = 0; self->profiles[i] != NULL; ++i) {
		GtkTreeIter iter;
		/* Populate model */
		gtk_list_store_append (self->profile_store, &iter);
		gtk_list_store_set (self->profile_store, &iter, 0, self->profiles[i], -1);
	}
	hildon_touch_selector_set_model(HILDON_TOUCH_SELECTOR(self->profileSelector), 0, GTK_TREE_MODEL(self->profile_store));
	hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(self->profileSelector), 0, 0);
}

/* vim: set ts=4: */
