#!/usr/bin/env python

from __future__ import with_statement
from __future__ import division

import os
import sys
import logging
import pygame
import pygame.locals

from util import algorithms
import constants
import config
import basicSprite
import greenManSprite
import purpleManSprite
import basicMonster
import item_cache
import helpers


_moduleLogger = logging.getLogger(__name__)


# Warn if soundrr or fonts are disabled
if not pygame.font:
	_moduleLogger.info('Warning, fonts disabled')
if not pygame.mixer:
	AUDIO_BROKEN = True
	_moduleLogger.info('Warning, sound disabled')
else:
	AUDIO_BROKEN = False


FRAMES_PER_SECOND = 30


class LevelView(object):

	_NEXT_LEVEL = "next"
	_REPEAT_LEVEL = "repeat"
	_QUIT = "quit"

	def __init__(self, dataPath, width=800, height=480):
		self._dataPath = dataPath
		self._cache = item_cache.ItemCache()

		try:
			pygame.mixer.pre_init(44100, -16, 1, 256)
		except pygame.error:
			config.gameMusic = False
			config.gameSound = False
			_moduleLogger.exception("Disabling sound support")
		pygame.init()
		pygame.mouse.set_visible(False)
		try:
			pygame.mixer.init()
		except pygame.error:
			config.gameMusic = False
			config.gameSound = False
			_moduleLogger.exception("Disabling sound support")
		config.gameSound &= not AUDIO_BROKEN
		config.gameMusic &= not AUDIO_BROKEN

		self.level1 = None
		self.width = width
		self.height = height
		self.greenMan = None
		self.greenMan_sprites = None
		self.purpleMan_sprites = None
		self.purpleMan = None
		self._hudNeedsUpdate = True

		self._clock = pygame.time.Clock()
		self.screen = pygame.display.set_mode((self.width, self.height), pygame.locals.FULLSCREEN)
		self.background = pygame.Surface(self.screen.get_size()).convert()

	def run(self, levels):
		try:
			config.players = levels[0].parent.parent.playerCount
			config.greenScore = 0
			config.purpleScore = 0
			config.greenHits = 0
			config.purpleHits = 0
			config.greenScore = 0
			config.purpleScore = 0
			config.deathCount = 3

			config.levelselect = 1
			nextLevel = levels.pop(0)
			while True:
				nextLevel.load()
				transition = self.play_level(nextLevel)
				if transition == self._NEXT_LEVEL:
					nextLevel.complete()
					try:
						nextLevel = levels.pop(0)
					except IndexError:
						self.ShowScores()
						break
					config.levelselect += 1
					nextLevel.mark_available()
				elif transition == self._REPEAT_LEVEL:
					pass
				elif transition == self._QUIT:
					break
		finally:
			pygame.quit()
		self._cache.clear_cache()

	def play_level(self, level):
		self.dead = 0

		self._load_cache(level)
		self._load_map(level)
		pygame.display.flip()

		self._hudNeedsUpdate = True
		while True:
			tick_time = self._clock.tick(FRAMES_PER_SECOND)
			next = self._process_iteration()
			if next is not None:
				return next

	def _process_iteration(self):
		if sum(player.pellets for player in self._players()) == self.total:
			if self.greenMan is not None:
				config.greenScore = config.greenScore+self.greenMan.pellets
			if self.purpleMan is not None:
				config.purpleScore = config.purpleScore+self.purpleMan.pellets
			return self._NEXT_LEVEL

		for sprite in self._player_sprites():
			sprite.clear(self.screen, self.background)
		self.monster_sprites.clear(self.screen, self.background)

		for event in pygame.event.get():
			if event.type == pygame.QUIT:
				sys.exit()
			elif event.type == pygame.locals.KEYDOWN:
				if event.key in (
					pygame.locals.K_RIGHT,
					pygame.locals.K_LEFT,
					pygame.locals.K_UP,
					pygame.locals.K_DOWN,
				):
					self.greenMan.MoveKeyDown(event.key)
				elif event.key in (
					pygame.locals.K_f,
					pygame.locals.K_s,
					pygame.locals.K_e,
					pygame.locals.K_c,
				):
					if config.players == 2:
						self.purpleMan.MoveKeyDown(event.key)
				elif event.key == pygame.locals.K_p:
					screenshotPath = os.path.expanduser("~/MyDocs/omnom")
					try:
						os.makedirs(screenshotPath)
					except OSError, e:
						if e.errno != 17:
							raise
					pygame.image.save(self.screen, os.path.join(screenshotPath, "screenshot.png"))
				elif event.key == pygame.locals.K_r:
					return self._REPEAT_LEVEL
				elif event.key == pygame.locals.K_n:
					return self._NEXT_LEVEL
				elif event.key == pygame.locals.K_BACKSPACE:
					return self._QUIT
			elif event.type == pygame.locals.KEYUP:
				if event.key in (
					pygame.locals.K_RIGHT,
					pygame.locals.K_LEFT,
					pygame.locals.K_UP,
					pygame.locals.K_DOWN,
				):
					self.greenMan.MoveKeyUp(event.key)
				elif event.key in (
					pygame.locals.K_f,
					pygame.locals.K_s,
					pygame.locals.K_e,
					pygame.locals.K_c,
				):
					if config.players == 2:
						self.purpleMan.MoveKeyUp(event.key)
			elif event.type == greenManSprite.SUPER_STATE_OVER:
				for player in self._players():
					player.superState = False
				pygame.time.set_timer(greenManSprite.SUPER_STATE_OVER, 0)
				for monster in self.monster_sprites.sprites():
					monster.SetScared(False)
			elif event.type == greenManSprite.SUPER_STATE_START:
				for monster in self.monster_sprites.sprites():
					monster.SetScared(True)
			elif event.type == greenManSprite.PLAYER_EATEN:
				self._hudNeedsUpdate = True
				if all(player.lives < 1 for player in self._players()):
					if self.greenMan is not None:
						self.greenMan.lives = config.lives
					if self.purpleMan is not None:
						self.purpleMan.lives = config.lives
					config.deathCount = config.deathCount-1
					if config.deathCount == 0:
						config.deathCount = 3
						self.dead = True
						self.ShowScores()
						return self._QUIT
					else:
						return self._REPEAT_LEVEL

		for sprite in self._player_sprites():
			sprite.update(self.block_sprites, self.pellet_sprites, self.super_pellet_sprites, self.monster_sprites)
		self.monster_sprites.update(self.block_sprites)

		self.screen.blit(self.background, (0, 0))

		textpos = None
		if pygame.font and self._hudNeedsUpdate:
			font = pygame.font.Font(None, 36)
			if config.players == 2:
				text = font.render("Lives %s		  Level %s" % (config.deathCount, config.levelselect), 1, (255, 255, 255))
			else:
				text = font.render("Lives %s	   Level %s" % (config.deathCount, config.levelselect), 1, (255, 255, 255))
			textpos = text.get_rect(centerx=self.background.get_width()/2)
			self.screen.blit(text, textpos)
			self._hudNeedsUpdate = False

		reclist = [textpos] if textpos is not None else []
		reclist += self.pellet_sprites.draw(self.screen)
		reclist += self.super_pellet_sprites.draw(self.screen)
		for sprite in self._player_sprites():
			reclist += sprite.draw(self.screen)
		reclist += self.monster_sprites.draw(self.screen)
		pygame.display.update(reclist)
		return None

	def _players(self):
		return (
			player
			for player in (self.greenMan, self.purpleMan)
			if player is not None
		)

	def _player_sprites(self):
		return (
			player
			for player in (self.greenMan_sprites, self.purpleMan_sprites)
			if player is not None
		)

	def ShowScores(self):
		screen = pygame.display.set_mode((800, 480), pygame.locals.FULLSCREEN)
		background = pygame.Surface(screen.get_size())
		if config.players == 2:
			background = pygame.image.load(self.level1.data.find_path("2score.png"))
		else:
			background = pygame.image.load(self.level1.data.find_path("1score.png"))

		font = pygame.font.Font(None, 36)
		if config.players == 2:
			text = font.render("Green								Purple", 1, (255, 255, 255))
			textpos = text.get_rect()
			textpos.centerx = background.get_rect().centerx
			textpos.centery = background.get_rect().centery-100
			background.blit(text, textpos)
			screen.blit(background, (0, 0))
			pygame.display.flip()
			text = font.render("Pellets", 1, (255, 255, 255))
			textpos = text.get_rect()
			textpos.centerx = background.get_rect().centerx
			textpos.centery = background.get_rect().centery-40
			background.blit(text, textpos)
			screen.blit(background, (0, 0))
			pygame.display.flip()
			text = font.render("Hits", 1, (255, 255, 255))
			textpos = text.get_rect()
			textpos.centerx = background.get_rect().centerx
			textpos.centery = background.get_rect().centery
			background.blit(text, textpos)
			screen.blit(background, (0, 0))
			pygame.display.flip()
			text = font.render("Score", 1, (255, 255, 255))
			textpos = text.get_rect()
			textpos.centerx = background.get_rect().centerx
			textpos.centery = background.get_rect().centery+40
			background.blit(text, textpos)
			screen.blit(background, (0, 0))
			pygame.display.flip()
		if config.players == 2:
			text = font.render("%s" % (config.greenScore), 1, (255, 255, 255))
			textpos = text.get_rect()
			textpos.centerx = background.get_rect().centerx-150
			textpos.centery = background.get_rect().centery-40
			background.blit(text, textpos)
			screen.blit(background, (0, 0))
			pygame.display.flip()
			text = font.render("%s" % (config.purpleScore), 1, (255, 255, 255))
			textpos = text.get_rect()
			textpos.centerx = background.get_rect().centerx+150
			textpos.centery = background.get_rect().centery-40
			background.blit(text, textpos)
			screen.blit(background, (0, 0))
			pygame.display.flip()
		else:
			text = font.render("Pellets: %s" % (config.greenScore), 1, (255, 255, 255))
			textpos = text.get_rect()
			textpos.centerx = background.get_rect().centerx
			textpos.centery = background.get_rect().centery-40
			background.blit(text, textpos)
			screen.blit(background, (0, 0))
			pygame.display.flip()

		pygame.time.wait(1000)
		config.greenHitsTotal = config.greenHitsTotal+config.greenHits
		config.purpleHitsTotal = config.purpleHitsTotal+config.purpleHits
		if config.players == 2:
			text = font.render("%s" % (config.greenHitsTotal), 1, (255, 255, 255))
			textpos = text.get_rect()
			textpos.centerx = background.get_rect().centerx-150
			textpos.centery = background.get_rect().centery
			background.blit(text, textpos)
			screen.blit(background, (0, 0))
			pygame.display.flip()
			text = font.render("%s" % (config.purpleHitsTotal), 1, (255, 255, 255))
			textpos = text.get_rect()
			textpos.centerx = background.get_rect().centerx+150
			textpos.centery = background.get_rect().centery
			background.blit(text, textpos)
			screen.blit(background, (0, 0))
			pygame.display.flip()
		else:
			text = font.render("Hits: %s" % (config.greenHitsTotal), 1, (255, 255, 255))
			textpos = text.get_rect()
			textpos.centerx = background.get_rect().centerx
			textpos.centery = background.get_rect().centery
			background.blit(text, textpos)
			screen.blit(background, (0, 0))
			pygame.display.flip()

		pygame.time.wait(1000)
		greenDeduction = config.greenHitsTotal*10
		purpleDeduction = config.purpleHitsTotal*10
		config.greenPoints = (config.greenScore+greenDeduction)*10
		config.purplePoints = (config.purpleScore+purpleDeduction)*10
		if config.players == 2:
			text = font.render("%s" % (config.greenPoints), 1, (255, 255, 255))
			textpos = text.get_rect()
			textpos.centerx = background.get_rect().centerx-150
			textpos.centery = background.get_rect().centery+40
			background.blit(text, textpos)
			screen.blit(background, (0, 0))
			pygame.display.flip()
			text = font.render("%s" % (config.purplePoints), 1, (255, 255, 255))
			textpos = text.get_rect()
			textpos.centerx = background.get_rect().centerx+150
			textpos.centery = background.get_rect().centery+40
			background.blit(text, textpos)
			screen.blit(background, (0, 0))
			pygame.display.flip()
		else:
			text = font.render("Score: %s" % (config.greenPoints), 1, (255, 255, 255))
			textpos = text.get_rect()
			textpos.centerx = background.get_rect().centerx
			textpos.centery = background.get_rect().centery+40
			background.blit(text, textpos)
			screen.blit(background, (0, 0))
			pygame.display.flip()

		if self.dead == 1:
			pygame.time.wait(5000)
			text = font.render("G A M E   O V E R", 1, (255, 255, 255))
			textpos = text.get_rect()
			textpos.centerx = background.get_rect().centerx
			textpos.centery = background.get_rect().centery+80
			background.blit(text, textpos)
			screen.blit(background, (0, 0))
			pygame.display.flip()
		else:
			text = font.render("CONGRATULATIONS!", 1, (255, 255, 255))
			textpos = text.get_rect()
			textpos.centerx = background.get_rect().centerx
			textpos.centery = background.get_rect().centery+80
			background.blit(text, textpos)
			screen.blit(background, (0, 0))
			pygame.display.flip()
		pygame.time.wait(5000)
		self.dead = 0

	def _load_cache(self, level):
		actionSoundPaths = (
			level.data.get_action_sound(action)
			for action in ("bite", "super", "kill")
		)
		actionSoundPaths = [path for path in actionSoundPaths if path and os.path.exists(path)]
		if config.gameSound and not all(
			self._cache.is_present("sound", path)
			for path in actionSoundPaths
		):
			self._cache.clear_category("sound")
			for path in actionSoundPaths:
				self._cache.add_item("sound", path, pygame.mixer.Sound(path))

		imagePaths = [
			level.data.get_cell_image(symbol, state)
			for (symbol, state) in algorithms.product(
				("#", "G", "P", ".", "M", "*"),
				("", "chomped", "evil", "scared"),
			)
		]
		imagePaths = [path for path in imagePaths if path and os.path.exists(path)]
		if not all(
			self._cache.is_present("image", path)
			for path in imagePaths
		):
			self._cache.clear_category("image")
			for path in imagePaths:
				img, _ = helpers.load_image(path, -1)
				self._cache.add_item("image", path, img)

	def _load_map(self, level):
		config.areMonstersEvil = False
		self.total = 0

		self.level1 = level

		self.pellet_sprites = pygame.sprite.RenderUpdates()
		self.super_pellet_sprites = pygame.sprite.RenderUpdates()
		self.block_sprites = pygame.sprite.RenderUpdates()
		self.monster_sprites = pygame.sprite.RenderUpdates()

		blockSize = level.data.get_block_size()
		x_offset = blockSize // 2
		y_offset = blockSize // 2

		for y, row in enumerate(level.data.get_walls()):
			yCenterPoint = y*blockSize + y_offset
			for x, symbol in enumerate(row):
				if symbol == " ":
					continue
				xCenterPoint = x*blockSize + x_offset
				centerPoint = (xCenterPoint, yCenterPoint)
				path = level.data.get_cell_image(symbol, "")
				img = self._cache.get_item("image", path)
				block = basicSprite.Sprite(centerPoint, img)
				self.block_sprites.add(block)

		for y, row in enumerate(level.data.get_items()):
			yCenterPoint = y*blockSize + y_offset
			for x, symbol in enumerate(row):
				if symbol == " ":
					continue
				xCenterPoint = x*blockSize + x_offset
				centerPoint = (xCenterPoint, yCenterPoint)
				path = level.data.get_cell_image(symbol, "")
				img = self._cache.get_item("image", path)
				if symbol == ".":
					pellet = basicSprite.Sprite(centerPoint, img)
					self.pellet_sprites.add(pellet)
					self.total = self.total+1
				elif symbol == "*":
					super_pellet = basicSprite.Sprite(centerPoint, img)
					self.super_pellet_sprites.add(super_pellet)
				else:
					raise RuntimeError("Unknown map symbol %r" % symbol)

		for y, row in enumerate(level.data.get_enemies()):
			yCenterPoint = y*blockSize + y_offset
			for x, symbol in enumerate(row):
				if symbol == " ":
					continue
				xCenterPoint = x*blockSize + x_offset
				centerPoint = (xCenterPoint, yCenterPoint)
				monster = basicMonster.Monster(centerPoint, level, self._cache)
				self.monster_sprites.add(monster)

		for y, row in enumerate(level.data.get_players()):
			yCenterPoint = y*blockSize + y_offset
			for x, symbol in enumerate(row):
				if symbol == " ":
					continue
				xCenterPoint = x*blockSize + x_offset
				centerPoint = (xCenterPoint, yCenterPoint)
				if symbol == "G":
					self.greenMan = greenManSprite.greenMan(centerPoint, level, self._cache)
				elif symbol == "P":
					if config.players == 2:
						self.purpleMan = purpleManSprite.purpleMan(centerPoint, level, self._cache)
				else:
					raise RuntimeError("Unknown map symbol %r" % symbol)

		self.greenMan_sprites = pygame.sprite.RenderUpdates(self.greenMan)
		if config.players == 2:
			self.purpleMan_sprites = pygame.sprite.RenderUpdates(self.purpleMan)

		self.background.fill(level.data.get_background())
		self.screen.fill(level.data.get_background())
		self.block_sprites.draw(self.screen)
		self.block_sprites.draw(self.background)


def run(campaignIndex, levelIndex):
	import game_state
	dataPath = os.path.join(os.path.dirname(__file__), "data")
	gameState = game_state.GameState(dataPath)
	gameState.playerCount = 1
	campaigns = gameState.supportedCampaigns
	campaign = campaigns[campaignIndex]
	levels = campaign.levels
	levelsToPlay = levels[levelIndex:]

	levelView = LevelView(dataPath)
	levelView.run(levelsToPlay)


if __name__ == "__main__":
	try:
		os.makedirs(constants._data_path_)
	except OSError, e:
		if e.errno != 17:
			raise

	logFormat = '(%(relativeCreated)5d) %(levelname)-5s %(threadName)s.%(name)s.%(funcName)s: %(message)s'
	logging.basicConfig(level=logging.DEBUG, format=logFormat)

	run(0, 0)
