#include "mainwindow.h"
#include "videodelegatewidget.h"
#include "videoplayer.h"
#include "optionfilters.h"
#include "settingdialog.h"
#include "logindialog.h"
#include "playlistdlg.h"
#include "newplaylistdialog.h"
#include <QStringListModel>
#include <QMaemo5InformationBox>
#include <downloadmanager.h>
#include <QDir>

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent)
{
    setAttribute(Qt::WA_Maemo5StackedWindow);
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
    ytfields = "link(@rel),entry(link(@rel,@href),media:group(media:credit/text(),media:title/text(),media:thumbnail[@yt:name='%1'](@url),yt:duration,yt:uploaded,yt:videoid),gd:rating(@average),yt:statistics(@viewCount),yt:rating)";
    listWidget = new QListWidget(this);
    listWidget->setContextMenuPolicy(Qt::ActionsContextMenu);
    listWidget->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    listWidget->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    listWidget->setFlow(QListView::LeftToRight);
    listWidget->installEventFilter(this);
    hScrollBar = listWidget->horizontalScrollBar();
    vScrollBar = listWidget->verticalScrollBar();
    connect(listWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(itemClicked(QListWidgetItem*)));
    connect(hScrollBar, SIGNAL(valueChanged(int)), this, SLOT(onScrollBarValueChanged(int)));
    connect(vScrollBar, SIGNAL(valueChanged(int)), this, SLOT(onScrollBarValueChanged(int)));
    net = new NetworkAccess(this);
    connect(net, SIGNAL(videoData(const QMap<QString, QString>&)), this, SLOT(addVideo(const QMap<QString, QString>&)));
    connect(net, SIGNAL(dataLoaded(bool)), this, SLOT(dataLoaded(bool)));
    connect(net, SIGNAL(videoThumbnail(int, const QPixmap&)), this, SLOT(addThumbnail(int, const QPixmap&)));
    connect(net, SIGNAL(suggestion(const QStringList&)), this, SLOT(updateSuggestion(const QStringList&)));
    connect(net, SIGNAL(authDone(bool)), this, SLOT(loginResult(bool)));
    connect(net, SIGNAL(postMessage(const QString&)), this, SLOT(replyMessage(const QString&)));
    connect(net, SIGNAL(errorMessage(const QString&)), this, SLOT(errorMessage(const QString&)));
    readSettings();
    setupMenus();
    setupUi();
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
    orientationChanged();
    setActions("logout");
    if (autoLogin && !acct.isEmpty() && !acctpsswd.isEmpty()){
        QTimer::singleShot(0, this, SLOT(login_out()));
    }
    else if (!currentUrl.toString().isEmpty() && !currentUrl.toString().contains("default")){
        QTimer::singleShot(0, this, SLOT(getVideoInfo()));
    }
    else {
        QTimer::singleShot(0, this, SLOT(getStandardFeed()));
    }
}

MainWindow::~MainWindow()
{
    delete net;
}

void MainWindow::readSettings()
{
    acct = maskIt(settings.value("acct", "").toString(), false);
    acctpsswd = maskIt(settings.value("acctpsswd", "").toString(), false);
    autoLogin = settings.value("autologin", false).toBool();
    regionID = settings.value("regionID", "").toString();
    categories = settings.value("categories", "").toString();
    standardFeed = settings.value("standardFeed", "top_rated").toString();
    timeLine = settings.value("timeLine", "this_week").toString();
    nextPage = settings.value("nextPage", false).toBool();
    currentUrl = QUrl(settings.value("currentUrl", "").toString());
    recentSearch = settings.value("recentSearch", QStringList()).toStringList();
    safeSearch = settings.value("safeSearch", "none").toString();
    hqthumb = settings.value("hqthumb", true).toBool();
    net->setUsePreferredServer(settings.value("preferserver", false).toBool());
    QStringList bl;
    bl << "This week" << "Top rated" << "All categories";
    buttonListValue = settings.value("buttonListValue", bl).toStringList();
    downloadedVideo = settings.value("downloadedVideo", QStringList()).toStringList();
    start_index = 1;
    isBusy = false;
    isAuthenicated = false;
    currentChannelUser = "default";
    rescanDownloaded = true;
    clearListWidget = true;
}

void MainWindow::closeEvent(QCloseEvent *event)
{
    settings.setValue("acct", maskIt(acct, true));
    settings.setValue("acctpsswd", maskIt(acctpsswd, true));
    settings.setValue("autologin", autoLogin);
    settings.setValue("regionID", regionID);
    settings.setValue("categories", categories);
    settings.setValue("standardFeed", standardFeed);
    settings.setValue("timeLine", timeLine);
    settings.setValue("nextPage", nextPage);
    settings.setValue("currentUrl", currentUrl.toString());
    settings.setValue("recentSearch", recentSearch);
    settings.setValue("safeSearch", safeSearch);
    settings.setValue("hqthumb", hqthumb);
    settings.setValue("downloadedVideo", downloadedVideo);
    buttonListValue.clear();
    buttonListValue << timeLineButton->valueText() << standardFeedButton->valueText() << categoriesButton->valueText();
    settings.setValue("buttonListValue", buttonListValue);
    event->accept();
}

void MainWindow::setupMenus()
{
    filterGroup = new QActionGroup(this);
    filterGroup->setExclusive(true);
    browse = new QAction(tr("Browse"), filterGroup);
    browse->setCheckable(true);
    browse->setChecked(true);
    channel = new QAction(tr("Your channel"), filterGroup);
    channel->setCheckable(true);
    channel->setChecked(false);
    downloaded = new QAction(tr("Downloaded"), filterGroup);
    downloaded->setCheckable(true);
    downloaded->setChecked(false);
    connect(filterGroup, SIGNAL(triggered(QAction*)), this, SLOT(widgetSwitch(QAction*)));
    menuBar()->addActions(filterGroup->actions());
    login_outAction = new QAction("Sign in", this);
    connect(login_outAction, SIGNAL(triggered()), this, SLOT(login_out()));
    downloadQueue = new QAction(tr("Download queue (%1)").arg(queueVideoList.size()), this);
    downloadQueue->setVisible(!queueVideoList.isEmpty());
    connect(downloadQueue, SIGNAL(triggered()), this, SLOT(startDownloadQueue()));
    menuBar()->addAction(login_outAction);
    menuBar()->addAction(tr("Settings"), this, SLOT(editSettings()));
    menuBar()->addAction(tr("Recently watched"), this, SLOT(recentWatched()));
    menuBar()->addAction(downloadQueue);
    menuBar()->addAction(tr("About"), this, SLOT(aboutDlg()));

    addToPlaylist = new QAction(tr("Add to playlist"), listWidget);
    connect(addToPlaylist, SIGNAL(triggered()), this, SLOT(on_addToPlaylist()));
    removeFromPlaylist = new QAction(tr("Remove from playlist"), listWidget);
    connect(removeFromPlaylist, SIGNAL(triggered()), this, SLOT(on_removeFromPlaylist()));

    addToFavorite = new QAction(tr("Add to favorite"), listWidget);
    connect(addToFavorite, SIGNAL(triggered()), this, SLOT(on_addToFavorite()));
    removeFromFavorite = new QAction(tr("Remove from favorite"), listWidget);
    connect(removeFromFavorite, SIGNAL(triggered()), this, SLOT(on_removeFromFavorite()));

    subscribe = new QAction(tr("Subscribe this user"), listWidget);
    connect(subscribe, SIGNAL(triggered()), this, SLOT(on_subscribe()));

    addToQueue = new QAction(tr("Add to download queue"), listWidget);
    connect(addToQueue, SIGNAL(triggered()), this, SLOT(on_addToQueue()));

    downloadAction = new QAction(tr("Download this video"), listWidget);
    connect(downloadAction, SIGNAL(triggered()), this, SLOT(downloadVideoNow()));

    delUpload = new QAction(tr("Delete this video"), listWidget);
    connect(delUpload, SIGNAL(triggered()), this, SLOT(delUploadVideo()));

    delDownloaded = new QAction(tr("Delete this video"), listWidget);
    connect(delDownloaded, SIGNAL(triggered()), this, SLOT(delDownloadedVideo()));

    userVideos = new QAction(tr("User's videos"), listWidget);
    connect(userVideos, SIGNAL(triggered()), this, SLOT(onUserVideos()));

    listWidget->addAction(addToPlaylist);
    listWidget->addAction(addToFavorite);
    listWidget->addAction(addToQueue);
    listWidget->addAction(downloadAction);
    listWidget->addAction(subscribe);
    listWidget->addAction(removeFromPlaylist);
    listWidget->addAction(removeFromFavorite);
    listWidget->addAction(delUpload);
    listWidget->addAction(delDownloaded);
    listWidget->addAction(userVideos);
}

void MainWindow::setActions(const QString &type)
{
    addToPlaylist->setVisible(false);
    addToFavorite->setVisible(false);
    addToQueue->setVisible(true);
    downloadAction->setVisible(true);
    subscribe->setVisible(false);
    removeFromPlaylist->setVisible(false);
    removeFromFavorite->setVisible(false);
    delUpload->setVisible(false);
    delDownloaded->setVisible(false);
    userVideos->setVisible(true);
    if (type == "playlist"){
        addToPlaylist->setVisible(isAuthenicated);
        addToFavorite->setVisible(isAuthenicated);
        subscribe->setVisible(isAuthenicated);
        removeFromPlaylist->setVisible((currentChannelUser == "default"));
    }
    else if(type == "upload"){
        addToPlaylist->setVisible(isAuthenicated);
        addToFavorite->setVisible(isAuthenicated);
        subscribe->setVisible((currentChannelUser != "default" && isAuthenicated));
        delUpload->setVisible((currentChannelUser == "default"));
        userVideos->setVisible(false);
    }
    else if(type == "recommendation"){
        addToPlaylist->setVisible(true);
        addToFavorite->setVisible(true);
        subscribe->setVisible(true);
    }
    else if(type == "favorite"){
        addToPlaylist->setVisible(isAuthenicated);
        addToFavorite->setVisible((currentChannelUser != "default"));
        subscribe->setVisible(true);
        removeFromFavorite->setVisible((currentChannelUser == "default"));
    }
    else if(type == "downloaded"){
        addToPlaylist->setVisible(isAuthenicated);
        addToFavorite->setVisible(isAuthenicated);
        delDownloaded->setVisible(true);
        subscribe->setVisible(isAuthenicated);
        addToQueue->setVisible(false);
        downloadAction->setVisible(false);
    }
    else if(type == "browse"){
        addToPlaylist->setVisible(isAuthenicated);
        addToFavorite->setVisible(isAuthenicated);
        subscribe->setVisible(isAuthenicated);
    }
}

void MainWindow::setupUi()
{
    /* YouTube Logo */
    ytLabel = new QLabel(this);
    ytLabel->installEventFilter(this);
    ytLabel->setAlignment(Qt::AlignCenter);
    ytLabel->setPixmap(QPixmap(":/images/yt_logo.png"));
    ytLabel->setStyleSheet("color: grey; font-size: 50px; margin-top: -80px; background-color: rgb(0, 0, 0, 150)");
    /* Standard Feeds */
    buttonGroup = new QButtonGroup(this);
    buttonGroup->setExclusive(true);
    connect(buttonGroup, SIGNAL(buttonClicked(QAbstractButton*)), this, SLOT(feedsButtonActivated(QAbstractButton*)));
    timeLineButton = new QMaemo5ValueButton(QIcon::fromTheme("control_date_time"),"", this);
    timeLineButton->setCheckable(true);
    timeLineButton->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    timeLineButton->setValueText(buttonListValue.at(0));
    standardFeedButton = new QMaemo5ValueButton(QIcon::fromTheme("general_rss"),"", this);
    standardFeedButton->setCheckable(true);
    standardFeedButton->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    standardFeedButton->setValueText(buttonListValue.at(1));
    categoriesButton = new QMaemo5ValueButton(QIcon(":/images/general_applications.png"),"", this);
    categoriesButton->setCheckable(true);
    categoriesButton->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    categoriesButton->setValueText(buttonListValue.at(2));
    buttonGroup->addButton(timeLineButton);
    buttonGroup->addButton(standardFeedButton);
    buttonGroup->addButton(categoriesButton);
    /* Channel */
    buttonGroup1 = new QButtonGroup(this);
    buttonGroup1->setExclusive(true);
    connect(buttonGroup1, SIGNAL(buttonClicked(QAbstractButton*)), this, SLOT(channelButtonActivated(QAbstractButton*)));
    uploadButton = new QMaemo5ValueButton(QIcon::fromTheme("keyboard_shift"),"", this);
    uploadButton->setValueText("Uploads");
    uploadButton->setCheckable(true);
    uploadButton->setVisible(false);
    uploadButton->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    favoritesButton = new QMaemo5ValueButton(QIcon::fromTheme("imageviewer_favourite"),"", this);
    favoritesButton->setValueText("Favorites");
    favoritesButton->setCheckable(true);
    favoritesButton->setVisible(false);
    favoritesButton->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    playlistsButton = new QMaemo5ValueButton(QIcon::fromTheme("notes_bullets"),"", this);
    playlistsButton->setValueText("Playlists");
    playlistsButton->setCheckable(true);
    playlistsButton->setVisible(false);
    playlistsButton->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    subscribedButton = new QMaemo5ValueButton(QIcon::fromTheme("rss_reader_goto"),"", this);
    subscribedButton->setValueText("Subscriptions");
    subscribedButton->setCheckable(true);
    subscribedButton->setVisible(false);
    subscribedButton->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    buttonGroup1->addButton(uploadButton);
    buttonGroup1->addButton(favoritesButton);
    buttonGroup1->addButton(playlistsButton);
    buttonGroup1->addButton(subscribedButton);
    previousAction = browse;
    previousButton = subscribedButton;

    /* Mainwindow Layout */
    QWidget *widget = new QWidget(this);
    QVBoxLayout *mainLayout = new QVBoxLayout(widget);
    mainLayout->setMargin(0);
    QHBoxLayout *topHLayout = new QHBoxLayout;
    topHLayout->setContentsMargins(0,5,0,0);
    topHLayout->addWidget(timeLineButton);
    topHLayout->addWidget(standardFeedButton);
    topHLayout->addWidget(categoriesButton);
    topHLayout->addWidget(uploadButton);
    topHLayout->addWidget(favoritesButton);
    topHLayout->addWidget(playlistsButton);
    topHLayout->addWidget(subscribedButton);
    mainWidget = new QStackedWidget(this);
    mainWidget->addWidget(listWidget);
    mainWidget->addWidget(ytLabel);
    mainWidget->setCurrentWidget(ytLabel);
    mainLayout->addLayout(topHLayout);
    mainLayout->addWidget(mainWidget);
    setCentralWidget(widget);
    setWindowTitle("NVideo");
    toolBar = new QToolBar(this);
    toolBar->setAllowedAreas(Qt::BottomToolBarArea);
    this->addToolBar(Qt::ToolBarArea(Qt::BottomToolBarArea), toolBar);
    lineEdit = new QLineEdit(toolBar);
    lineEdit->installEventFilter(this);
    timer = new QTimer(this);
    timer->setSingleShot(true);
    timer->setInterval(500);
    connect(timer, SIGNAL(timeout()), SLOT(autoSuggest()));
    connect(lineEdit, SIGNAL(textEdited(const QString&)), timer, SLOT(start()));
    toolBar->addWidget(lineEdit);
    closeToolBarAction = new QAction(QIcon::fromTheme("general_close"),"", toolBar);
    connect(closeToolBarAction, SIGNAL(triggered()), this, SLOT(closeToolBar()));
    toolBar->addAction(closeToolBarAction);
    toolBar->setVisible(false);
    QStringListModel *model = new QStringListModel(recentSearch);
    c = new QCompleter(model, this);
    connect(c, SIGNAL(activated(const QString&)), this, SLOT(searchVideo(const QString&)));
    c->setCaseSensitivity(Qt::CaseInsensitive);
    c->popup()->installEventFilter(this);
    c->popup()->setContentsMargins(20, 0, 20, 0);
    lineEdit->setCompleter(c);
}

void MainWindow::recentWatched()
{
    QStringList recentWVideos = settings.value("recentlyWatched", QStringList()).toStringList();
    if(!recentWVideos.isEmpty()){
        setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
        start_index = 1;
        clearListWidget = true;
        net->resetIndex();
        net->batchRequest(recentWVideos);
        setWindowTitle("Recently watched");
        setActions("browse");
    }
    else {
        QMaemo5InformationBox::information(this, tr("You haven't watched any video yet"));
    }
}

QStringList MainWindow::scanDownloadFolder()
{
    QStringList videoIdList;
    videoIdList = QStringList();
    QDir dlDir(settings.value("downloadfolder", "/home/user/MyDocs/.videos").toString());
    QStringList filenames = dlDir.entryList(QDir::Files | QDir::NoSymLinks, QDir::Time);
    for (int i = 0; i < filenames.size() && i < 50; ++i){
        if(filenames.at(i).endsWith(".mp4", Qt::CaseSensitive)){
            QString filename = filenames.at(i);
            int sidx = filename.lastIndexOf("(") + 1;
            int eidx = filename.lastIndexOf(")");
            if(sidx > 0 && eidx > sidx){
                filename = filename.mid(sidx, eidx - sidx);
                videoIdList.append(filename);
            }
        }
    }
    rescanDownloaded = false;
    return videoIdList;
}

void MainWindow::startDownloadQueue()
{
    if(queueVideoList.isEmpty()){
        return;
    }
    int reply = QMessageBox::question(this, "Download Queue", tr("%1 videos in download queue, do you want to start download this queue?\n(* Press reset button to clear the download queue)").arg(queueVideoList.size()), QMessageBox::Yes | QMessageBox::No | QMessageBox::Reset, QMessageBox::No);
    if (reply == QMessageBox::Yes){
        DownloadManager *dload = new DownloadManager(this);
        dload->setDownloadFolder(settings.value("downloadfolder", "/home/user/MyDocs/.videos").toString());
        connect(dload, SIGNAL(finished()), dload, SLOT(deleteLater()));
        connect(dload, SIGNAL(dlMessage(const QString&)), this, SLOT(replyMessage(const QString&)));
        connect(dload, SIGNAL(downloadStatus(qint64, qint64, const QString&)), this, SLOT(downloadProgress(qint64, qint64, const QString&)));
        dload->append(queueVideoList);
        downloadQueue->setVisible(false);
        progress = new QProgressDialog(this);
        progress->setWindowTitle("Download progress");
        progress->setLabelText("Downloading...");
        connect(progress, SIGNAL(canceled()), dload, SLOT(abort()));
        connect(dload, SIGNAL(finished()), progress, SLOT(deleteLater()));
        progress->exec();
        queueVideoList.clear();
        rescanDownloaded = true;
    }
    else if (reply == QMessageBox::Reset) {
        queueVideoList.clear();
        downloadQueue->setVisible(false);
    }
}

void MainWindow::downloadVideoNow()
{
    VideoDelegateWidget *itemWidget = qobject_cast<VideoDelegateWidget *>(listWidget->itemWidget(listWidget->currentItem()));
    DownloadManager *dload = new DownloadManager(this);
    dload->setDownloadFolder(settings.value("downloadfolder", "/home/user/MyDocs/.videos").toString());
    connect(dload, SIGNAL(finished()), dload, SLOT(deleteLater()));
    connect(dload, SIGNAL(dlMessage(const QString&)), this, SLOT(replyMessage(const QString&)));
    connect(dload, SIGNAL(downloadStatus(qint64, qint64, const QString&)), this, SLOT(downloadProgress(qint64, qint64, const QString&)));
    dload->append(itemWidget->videodata);
    progress = new QProgressDialog(this);
    progress->setWindowTitle("Download progress");
    progress->setLabelText("Downloading...");
    connect(progress, SIGNAL(canceled()), dload, SLOT(abort()));
    connect(dload, SIGNAL(finished()), progress, SLOT(deleteLater()));
    progress->exec();
    rescanDownloaded = true;
}

void MainWindow::downloadProgress(qint64 cur, qint64 total, const QString &speed)
{
    progress->setMaximum(total);
    progress->setValue(cur);
    progress->setLabelText(speed);
}

void MainWindow::downloadVideo(const QMap<QString, QString> &video)
{
    queueVideoList.append(video);
    downloadQueue->setVisible(true);
    downloadQueue->setText(QString("Download queue (%1)").arg(queueVideoList.size()));
}

void MainWindow::on_addToQueue()
{
    VideoDelegateWidget *itemWidget = qobject_cast<VideoDelegateWidget *>(listWidget->itemWidget(listWidget->currentItem()));
    queueVideoList.append(itemWidget->videodata);
    downloadQueue->setVisible(true);
    downloadQueue->setText(QString("Download queue (%1)").arg(queueVideoList.size()));
}

void MainWindow::widgetSwitch(QAction *action)
{
    currentChannelUser = "default";
    if (action == browse){
        showStandardFeed();
    }
    else if (action == channel){
        if (!isAuthenicated){
            login_out();
        }
        else {
            showChannel();
        }
    }
    else if(action == downloaded){
        if(downloadedVideo.isEmpty() || rescanDownloaded){
            downloadedVideo = scanDownloadFolder();
        }
        if(!downloadedVideo.isEmpty()){
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
            start_index = 1;
            clearListWidget = true;
            net->resetIndex();
            net->batchRequest(downloadedVideo);
            setWindowTitle("Downloaded videos");
            setActions("downloaded");
        }
        else {
            filterGroup->setExclusive(false);
            action->setChecked(false);
            filterGroup->setExclusive(true);
            previousAction->setChecked(true);
        }
        QMaemo5InformationBox::information(this, tr("%1 video found").arg(downloadedVideo.size()));
    }
    previousAction = action;
    buttonGroup1->setExclusive(false);
    previousButton->setChecked(false);
    buttonGroup1->setExclusive(true);
}

void MainWindow::showStandardFeed()
{
    getStandardFeed();
    uploadButton->hide();
    favoritesButton->hide();
    playlistsButton->hide();
    subscribedButton->hide();
    timeLineButton->show();
    standardFeedButton->show();
    categoriesButton->show();
    if(isAuthenicated){
        setActions("recommendation");
    }
    setWindowTitle("Browse");
}

void MainWindow::showChannel()
{
    showRecommendation();
    timeLineButton->hide();
    standardFeedButton->hide();
    categoriesButton->hide();
    uploadButton->show();
    favoritesButton->show();
    playlistsButton->show();
    subscribedButton->show();
    setWindowTitle("Recommended");
    channel->setChecked(true);
}

void MainWindow::login_out()
{
    if (isAuthenicated){
        QMaemo5InformationBox::information(this, tr("You have signed out account \"%1\"").arg(acct));
        acct.clear();
        acctpsswd.clear();
        isAuthenicated = false;
        autoLogin = false;
        net->sign_out();
        filterGroup->setExclusive(false);
        channel->setChecked(false);
        filterGroup->setExclusive(true);
        browse->setChecked(true);
        showStandardFeed();
        setWindowTitle("Browse");
        login_outAction->setText("Sign in");
        mainWidget->removeWidget(playlistListWidget);
        mainWidget->removeWidget(subscribedListWidget);
        setActions("logout");
    }
    else {
        if (acct != "" && acctpsswd != ""){
            net->requestAuthenticated(acct, acctpsswd);
        }
        else {
            LoginDialog signIn;
            connect(&signIn, SIGNAL(acctData(const QString&, const QString&, bool)), this, SLOT(processLogin(const QString&, const QString&, bool)));
            if(signIn.exec()){
                disconnect(&signIn, SIGNAL(acctData(const QString&, const QString&, bool)), this, SLOT(processLogin(const QString&, const QString&, bool)));
            }
            else{
                filterGroup->setExclusive(false);
                channel->setChecked(false);
                filterGroup->setExclusive(true);
                previousAction->setChecked(true);
            }
        }
    }
}

void MainWindow::processLogin(const QString &account, const QString &password, bool b)
{
    acct = account;
    acctpsswd = password;
    autoLogin = b;
    login_out();
}

void MainWindow::loginResult(bool b)
{
    if (b){
        isAuthenicated = true;
        showChannel();
        login_outAction->setText("Sign out");
    }
    else {        
        QMaemo5InformationBox::information(this, tr("Attempt to sign in account \"%1\" failed!").arg(acct));
        acct.clear();
        acctpsswd.clear();
        autoLogin = false;
        filterGroup->setExclusive(false);
        channel->setChecked(false);
        filterGroup->setExclusive(true);
        previousAction->setChecked(true);
    }
}

void MainWindow::editSettings()
{
    SettingDialog settingDialog;
    if(settingDialog.exec()){
        QMaemo5InformationBox::information(this, tr("Your settings has been saved."));
        regionID = settings.value("regionID", "").toString();
        safeSearch = settings.value("safeSearch", "none").toString();
        hqthumb = settings.value("hqthumb", true).toBool();
        net->setUsePreferredServer(settings.value("preferserver", false).toBool());
        rescanDownloaded = true;
    }
    QStringList rSearch = settings.value("recentSearch", QStringList()).toStringList();
    if(rSearch.isEmpty()){
        recentSearch = recentSearch.mid(0, 1);
    }
}


void MainWindow::searchVideo(const QString &searchTerm)
{
    closeToolBar();
    start_index = 1;
    clearListWidget = true;
    net->resetIndex();
    getSearchVideo(searchTerm);
    setActions("browse");
}

void MainWindow::getSearchVideo(const QString &searchTerm)
{
    QString query = searchTerm.trimmed();
    query.replace(" ", "+");
    QUrl url(QString("https://gdata.youtube.com/feeds/api/videos"));
    url.addQueryItem("q", query);
    url.addQueryItem("safeSearch", safeSearch);
    if (categories != "") {
        url.addQueryItem("category", categories);
    }
    url.addQueryItem("orderby", "relevance");
    currentUrl = url;
    getVideoInfo();
    recentSearch.removeAll(searchTerm);
    if (recentSearch.size() > 30) {
        recentSearch.removeAt(30);
    }
    recentSearch.prepend(searchTerm);
}


void MainWindow::getStandardFeed()
{
    QUrl url(QString("https://gdata.youtube.com/feeds/api/standardfeeds/%1%2%3").arg(regionID).arg(standardFeed).arg(categories));
    QStringList timeableList;
    timeableList << "top_rated" << "top_favorites" << "most_viewed" << "most_discussed" << "most_responded";
    if(timeableList.contains(standardFeed)){
        url.addQueryItem("time", timeLine);
        timeLineButton->show();
    }
    else {
        timeLineButton->hide();
    }
    currentUrl = url;
    start_index = 1;
    clearListWidget = true;
    net->resetIndex();
    getVideoInfo();
}

void MainWindow::feedsButtonActivated(QAbstractButton *button)
{
    if (previousButton == button && mainWidget->currentWidget() == filterListWidget){
        mainWidget->setCurrentWidget(listWidget);
        buttonGroup->setExclusive(false);
        button->setChecked(false);
        buttonGroup->setExclusive(true);
        listWidget->setFocus();
    }
    else {
        if(mainWidget->indexOf(filterListWidget) < 0){
            filterListWidget = new QListWidget(this);
            filterListWidget->installEventFilter(this);
            mainWidget->addWidget(filterListWidget);
            connect(filterListWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(filterItemClicked(QListWidgetItem*)));
        }
        mainWidget->setCurrentWidget(filterListWidget);
        filterListWidget->clear();
        if (button == timeLineButton){
            filterListWidget->addItems(OptionFilters::getTimeList());
        }
        else if (button == standardFeedButton){
            filterListWidget->addItems(OptionFilters::getFeedList());
        }
        else {
            filterListWidget->addItems(OptionFilters::getCategoriesList());
        }
        filterListWidget->setFocus();
    }
    previousButton = button;
    filterGroup->setExclusive(false);
    previousAction->setChecked(false);
    browse->setChecked(true);
    filterGroup->setExclusive(true);
    previousAction = browse;
    setWindowTitle("Browse");
    setActions("browse");
}

void MainWindow::filterItemClicked(QListWidgetItem *item)
{
    QString name  = item->text();
    if (timeLineButton->isChecked()){
        timeLineButton->setValueText(name);
        timeLine = OptionFilters::getTimeValue(name);
    }
    else if (standardFeedButton->isChecked()){
        standardFeedButton->setValueText(name);
        standardFeed = OptionFilters::getFeedValue(name);
    }
    else {
        categoriesButton->setValueText(name.replace("&", "&&"));
        categories = OptionFilters::getCategoriesValue(name.replace("&&", "&"));
    }
    QTimer::singleShot(0, this, SLOT(getStandardFeed()));
    buttonGroup->setExclusive(false);
    previousButton->setChecked(false);
    buttonGroup->setExclusive(true);
}

void MainWindow::getVideoInfo()
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    QUrl url = currentUrl;
    url.addQueryItem("start-index", QString("%1").arg(start_index));
    url.addQueryItem("max-results", "10");
    url.addQueryItem("fields", ytfields.arg(hqthumb?"hqdefault":"default"));
    net->getUrl(url);
}

void MainWindow::addVideo(const QMap<QString, QString> &video)
{
    if (clearListWidget){
        mainWidget->setCurrentWidget(listWidget);
        listWidget->clear();
        clearListWidget = false;
    }
    VideoDelegateWidget *itemWidget = new VideoDelegateWidget(video, listWidget);
    QListWidgetItem *item = new QListWidgetItem(listWidget);
    if (isPortrait){
        item->setSizeHint(QSize(480, 143));
    }
    else {
        item->setSizeHint(QSize(340, 340));
    }
    listWidget->addItem(item);
    listWidget->setItemWidget(item, itemWidget);
    ++start_index;
}

void MainWindow::addThumbnail(int i, const QPixmap &thumbnail)
{
    QListWidgetItem *item = listWidget->item(i);
    VideoDelegateWidget *itemWidget = qobject_cast<VideoDelegateWidget *>(listWidget->itemWidget(item));
    itemWidget->thumb->setPixmap(thumbnail);
}

void MainWindow::dataLoaded(bool next)
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
    nextPage = next;
    listWidget->setFocus();
}

void MainWindow::itemClicked(QListWidgetItem *item)
{
    if(!isBusy){
        isBusy = true;
        setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
        setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
        VideoDelegateWidget *itemWidget = qobject_cast<VideoDelegateWidget *>(listWidget->itemWidget(item));
        VideoPlayer *video = new VideoPlayer(isAuthenicated, this);
        video->loadVideo(itemWidget->videodata);
        connect(video, SIGNAL(quit()), this, SLOT(restoreOrientation()));
        connect(video, SIGNAL(playerReady()), this, SLOT(playerReady()));
        connect(video, SIGNAL(addFavorite(const QString&)), net, SLOT(toFavorite(const QString&)));
        connect(video, SIGNAL(likeThisVideo(const QString&, bool)), net, SLOT(rateVideo(const QString&, bool)));
        connect(video, SIGNAL(subscribeVideo(const QString&)), net, SLOT(subscribeRequest(const QString&)));
        connect(video, SIGNAL(addComment(const QString&, const QString&)), net, SLOT(addComments(const QString&, const QString&)));
        disconnect(net, SIGNAL(postMessage(const QString&)), this, SLOT(replyMessage(const QString&)));
        connect(net, SIGNAL(postMessage(const QString&)), video, SLOT(popupMessage(const QString&)));
        connect(video, SIGNAL(addToDownloadQueue(const QMap<QString,QString>&)), this, SLOT(downloadVideo(const QMap<QString,QString>&)));
    }
}

void MainWindow::playerReady()
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
    listWidget->setCurrentRow(-1);
    isBusy = false;
}

void MainWindow::restoreOrientation()
{
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
    connect(net, SIGNAL(postMessage(const QString&)), this, SLOT(replyMessage(const QString&)));
}

void MainWindow::channelButtonActivated(QAbstractButton *button)
{
    if (button == uploadButton){
        showUploads(currentChannelUser);
    }
    else if ( button == favoritesButton){
        showFavorites(currentChannelUser);
    }
    else if (button == playlistsButton){
        showPlaylist();
    }
    else {
        showSubscription();
    }
    setWindowTitle((currentChannelUser=="default")?"Your Channel":currentChannelUser);
    previousButton = button;
}
/* Downloaded Video Actions */

void MainWindow::delDownloadedVideo()
{
    VideoDelegateWidget *itemWidget = qobject_cast<VideoDelegateWidget *>(listWidget->itemWidget(listWidget->currentItem()));
    int reply = QMessageBox::question(this, "Warning!", tr("delete \"%1\" from phone memory?").arg(itemWidget->videodata.value("title")), QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
    if (reply == QMessageBox::Yes){

        QString filename = itemWidget->videodata.value("title");
        filename.replace(QRegExp("[\"<>:/|?*\\\\]"), "_").trimmed();
        filename.prepend(QString("%1/").arg(settings.value("downloadfolder", "/home/user/MyDocs/.videos").toString()));
        filename = QString("%1(%2).mp4").arg(filename).arg(itemWidget->videodata.value("videoid"));
        if (QFile::remove(filename)){
            QMaemo5InformationBox::information(this, tr("Successful deleted selected video"));
            downloadedVideo.removeAll(itemWidget->videodata.value("videoid"));
            listWidget->setRowHidden(listWidget->currentRow(), true);
            listWidget->setCurrentRow(-1);
        }
        else {
            QMaemo5InformationBox::information(this, tr("Fail to delete selected video"));
        }
    }
}

/* Recommendation */
void MainWindow::showRecommendation()
{
    start_index = 1;
    clearListWidget = true;
    net->resetIndex();
    QUrl url(QString("https://gdata.youtube.com/feeds/api/users/default/recommendations"));
    currentUrl = url;
    getVideoInfo();
    setActions("recommendation");
}

void MainWindow::onUserVideos()
{
    VideoDelegateWidget *itemWidget = qobject_cast<VideoDelegateWidget *>(listWidget->itemWidget(listWidget->currentItem()));
    QString username = itemWidget->videodata.value("credit");
    currentChannelUser = username;
    setWindowTitle(currentChannelUser);
    showUploads(username);
    buttonGroup1->setExclusive(false);
    playlistsButton->setChecked(false);
    subscribedButton->setChecked(false);
    uploadButton->setChecked(true);
    buttonGroup1->setExclusive(true);
    filterGroup->setExclusive(false);
    browse->setChecked(false);
    channel->setChecked(false);
    downloaded->setChecked(false);
    filterGroup->setExclusive(true);
    timeLineButton->hide();
    standardFeedButton->hide();
    categoriesButton->hide();
    uploadButton->show();
    favoritesButton->show();
    playlistsButton->show();
    subscribedButton->hide();
}

/* User's uploads */
void MainWindow::showUploads(const QString &username)
{
    start_index = 1;
    clearListWidget = true;
    net->resetIndex();
    QUrl url(QString("https://gdata.youtube.com/feeds/api/users/%1/uploads").arg(username));
    currentUrl = url;
    getVideoInfo();
    setActions("upload");
}

void MainWindow::delUploadVideo()
{
    VideoDelegateWidget *itemWidget = qobject_cast<VideoDelegateWidget *>(listWidget->itemWidget(listWidget->currentItem()));
    int reply = QMessageBox::question(this, "Warning!", tr("delete \"%1\"?").arg(itemWidget->videodata.value("title")), QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
    if (reply == QMessageBox::Yes){
        setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
        connect(net, SIGNAL(playlistsData(const QList< QMap<QString, QString> >&)), this, SLOT(addPlaylist(const QList< QMap<QString, QString> >&)));
        net->delUrl(QUrl(itemWidget->videodata.value("edit")), false, false);
        listWidget->setRowHidden(listWidget->currentRow(), true);
        listWidget->setCurrentRow(-1);
    }
}

/* Favorites */
void MainWindow::showFavorites(const QString &username)
{
    start_index = 1;
    clearListWidget = true;
    net->resetIndex();
    QUrl url(QString("https://gdata.youtube.com/feeds/api/users/%1/favorites").arg(username));
    currentUrl = url;
    getVideoInfo();
    setActions("favorite");
    setWindowTitle((username == "default")?"Your channel":username);
}

void MainWindow::on_addToFavorite()
{
    VideoDelegateWidget *itemWidget = qobject_cast<VideoDelegateWidget *>(listWidget->itemWidget(listWidget->currentItem()));
    net->toFavorite(itemWidget->videodata.value("videoid"));
}

void MainWindow::on_removeFromFavorite()
{
    VideoDelegateWidget *itemWidget = qobject_cast<VideoDelegateWidget *>(listWidget->itemWidget(listWidget->currentItem()));
    net->delUrl(QUrl(itemWidget->videodata.value("edit")), false, false);
    listWidget->setRowHidden(listWidget->currentRow(), true);
    listWidget->setCurrentRow(-1);
}

/* Subscriptions */
void MainWindow::showSubscription()
{
    if(mainWidget->indexOf(subscribedListWidget) < 0){
        subscribedListWidget = new QListWidget(this);
        subscribedListWidget->installEventFilter(this);
        subscribedListWidget->setContextMenuPolicy(Qt::ActionsContextMenu);
        subscribedListWidget->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
        QAction *del = new QAction(tr("unsubscribe"), subscribedListWidget);
        connect(del, SIGNAL(triggered()), this, SLOT(delSubscription()));
        subscribedListWidget->addAction(del);
        mainWidget->addWidget(subscribedListWidget);
        connect(subscribedListWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(subscribedItemClicked(QListWidgetItem*)));
        updateSubscription("default");
    }
    mainWidget->setCurrentWidget(subscribedListWidget);
    subscribedListWidget->setFocus();
}

void MainWindow::updateSubscription(const QString &username)
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    QUrl url(QString("https://gdata.youtube.com/feeds/api/users/%1/subscriptions?fields=entry(title,link[@rel='edit'](@href),yt:countHint,yt:username)").arg(username));
    connect(net, SIGNAL(subscriptionsData(const QList< QMap<QString, QString> >&)), this, SLOT(addSubscription(const QList< QMap<QString, QString> >&)));
    net->getUrl(url);
}

void MainWindow::addSubscription(const QList< QMap<QString, QString> > &subscriptions)
{
    subscribedListWidget->clear();
    subscribedUrls.clear();
    for (int i =0; i < subscriptions.size(); ++i){
        QMap<QString, QString> subscription = subscriptions.at(i);
        QString user = subscription.value("username");
        subscribedListWidget->addItem(QString("%1. %2").arg(i + 1).arg(user));
        subscribedUrls.append(user);
        subscribedUrls.append(subscription.value("edit"));
    }
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
    disconnect(net, SIGNAL(subscriptionsData(const QList< QMap<QString, QString> >&)), this, SLOT(addSubscription(const QList< QMap<QString, QString> >&)));
    subscribedListWidget->setFocus();
}

void MainWindow::subscribedItemClicked(QListWidgetItem *item)
{
    start_index = 1;
    clearListWidget = true;
    net->resetIndex();
    QUrl url(QString("https://gdata.youtube.com/feeds/api/users/%1/uploads").arg(subscribedUrls.at(subscribedListWidget->row(item) * 2)));
    currentUrl = url;
    getVideoInfo();
    setActions("upload");
    currentChannelUser = subscribedUrls.at(subscribedListWidget->row(item) * 2);
    setWindowTitle(currentChannelUser);
    buttonGroup1->setExclusive(false);
    playlistsButton->setChecked(false);
    subscribedButton->setChecked(false);
    uploadButton->setChecked(true);
    buttonGroup1->setExclusive(true);
    filterGroup->setExclusive(false);
    browse->setChecked(false);
    channel->setChecked(false);
    downloaded->setChecked(false);
    filterGroup->setExclusive(true);
}

void MainWindow::on_subscribe()
{
    VideoDelegateWidget *itemWidget = qobject_cast<VideoDelegateWidget *>(listWidget->itemWidget(listWidget->currentItem()));
    net->subscribeRequest(itemWidget->videodata.value("credit"));
}

void MainWindow::delSubscription()
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    connect(net, SIGNAL(subscriptionsData(const QList< QMap<QString, QString> >&)), this, SLOT(addSubscription(const QList< QMap<QString, QString> >&)));
    net->delUrl(QUrl(subscribedUrls.at(subscribedListWidget->currentRow() * 2 + 1)), false, true);
}
/* Playlist */
void MainWindow::playListPlay()
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
    orientationChanged();
    VideoPlayer *video = new VideoPlayer(isAuthenicated, this);
    video->loadVideoList(playlistUrls.at(playlistListWidget->currentRow() * 2), "50", true);
    connect(video, SIGNAL(quit()), this, SLOT(restoreOrientation()));
    connect(video, SIGNAL(playerReady()), this, SLOT(playerReady()));
    connect(video, SIGNAL(addFavorite(const QString&)), net, SLOT(toFavorite(const QString&)));
    connect(video, SIGNAL(likeThisVideo(const QString&, bool)), net, SLOT(rateVideo(const QString&, bool)));
    connect(video, SIGNAL(subscribeVideo(const QString&)), net, SLOT(subscribeRequest(QString)));
    disconnect(net, SIGNAL(postMessage(const QString&)), this, SLOT(replyMessage(const QString&)));
    connect(net, SIGNAL(postMessage(const QString&)), video, SLOT(popupMessage(const QString&)));
}

void MainWindow::showPlaylist()
{
    if(mainWidget->indexOf(playlistListWidget) < 0){
        playlistListWidget = new QListWidget(this);
        playlistListWidget->installEventFilter(this);
        playlistListWidget->setContextMenuPolicy(Qt::ActionsContextMenu);
        playlistListWidget->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
        QAction *playThisList = new QAction(tr("Play"), playlistListWidget);
        connect(playThisList, SIGNAL(triggered()), this, SLOT(playListPlay()));
        playlistListWidget->addAction(playThisList);
        QAction *del = new QAction(tr("Delete"), playlistListWidget);
        connect(del, SIGNAL(triggered()), this, SLOT(delPlaylist()));
        playlistListWidget->addAction(del);
        del->setVisible((currentChannelUser == "default"));
        mainWidget->addWidget(playlistListWidget);
        connect(playlistListWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(playlistItemClicked(QListWidgetItem*)));
        updatePlaylist(currentChannelUser);
        previousChannelUser == currentChannelUser;
    }
    else if (previousChannelUser != currentChannelUser){
        updatePlaylist(currentChannelUser);
        previousChannelUser = currentChannelUser;
        QAction *del = qobject_cast<QAction*>(playlistListWidget->actions().at(1));
        del->setVisible((currentChannelUser == "default"));
    }
    mainWidget->setCurrentWidget(playlistListWidget);
    playlistListWidget->setFocus();
}

void MainWindow::updatePlaylist(const QString &username)
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    QUrl url(QString("https://gdata.youtube.com/feeds/api/users/%1/playlists?fields=entry(title,content(@src),link[@rel='edit'](@href),yt:countHint,yt:playlistId)").arg(username));
    connect(net, SIGNAL(playlistsData(const QList< QMap<QString, QString> >&)), this, SLOT(addPlaylist(const QList< QMap<QString, QString> >&)));
    net->getUrl(url);
}

void MainWindow::addPlaylist(const QList< QMap<QString, QString> > &playlists)
{
    playlistListWidget->clear();
    playlistUrls.clear();
    for (int i =0; i < playlists.size(); ++i){
        QMap<QString, QString> playlist = playlists.at(i);
        playlistListWidget->addItem(QString("%1. %2 (%3 videos)").arg(i + 1).arg(playlist.value("title")).arg(playlist.value("countHint")));
        playlistUrls.append(playlist.value("content"));
        playlistUrls.append(playlist.value("edit"));
    }
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
    disconnect(net, SIGNAL(playlistsData(const QList< QMap<QString, QString> >&)), this, SLOT(addPlaylist(const QList< QMap<QString, QString> >&)));
    playlistListWidget->setFocus();
}

void MainWindow::playlistItemClicked(QListWidgetItem *item)
{
    start_index = 1;
    clearListWidget = true;
    net->resetIndex();
    QUrl url(playlistUrls.at(playlistListWidget->row(item) * 2));
    currentUrl = url;
    getVideoInfo();
    setActions("playlist");
}

void MainWindow::on_addToPlaylist()
{
    VideoDelegateWidget *itemWidget = qobject_cast<VideoDelegateWidget *>(listWidget->itemWidget(listWidget->currentItem()));
    PlayListDlg playlistdlg(itemWidget->videodata.value("videoid"), this);
    connect(&playlistdlg, SIGNAL(playlistID(const QString&, const QString&)), net, SLOT(toPlaylist(const QString&, const QString&)));
    connect(&playlistdlg, SIGNAL(newPlaylistInfo(const QString&)), this, SLOT(createNewPlaylist(const QString&)));
    QUrl url("https://gdata.youtube.com/feeds/api/users/default/playlists?fields=entry(title,content(@src),link[@rel='edit'](@href),yt:countHint,yt:playlistId)");
    connect(net, SIGNAL(playlistsData(const QList< QMap<QString, QString> >&)), &playlistdlg, SLOT(showPlaylists(const QList< QMap<QString, QString> >&)));
    net->getUrl(url);
    if(playlistdlg.exec() && mainWidget->indexOf(playlistListWidget) >= 0){
        updatePlaylist("default");
    }
    disconnect(&playlistdlg, SIGNAL(playlistID(const QString&, const QString&)), net, SLOT(toPlaylist(const QString&, const QString&)));
    disconnect(&playlistdlg, SIGNAL(newPlaylistInfo(const QString&)), this, SLOT(createNewPlaylist(const QString&)));
    disconnect(net, SIGNAL(playlistsData(const QList< QMap<QString, QString> >&)), &playlistdlg, SLOT(showPlaylists(const QList< QMap<QString, QString> >&)));
}

void MainWindow::on_removeFromPlaylist()
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    VideoDelegateWidget *itemWidget = qobject_cast<VideoDelegateWidget *>(listWidget->itemWidget(listWidget->currentItem()));
    connect(net, SIGNAL(playlistsData(const QList< QMap<QString, QString> >&)), this, SLOT(addPlaylist(const QList< QMap<QString, QString> >&)));
    net->delUrl(QUrl(itemWidget->videodata.value("edit")), true, false);
    listWidget->setRowHidden(listWidget->currentRow(), true);
    listWidget->setCurrentRow(-1);
}

void MainWindow::createNewPlaylist(const QString &videoid)
{
    NewPlayListDialog dlg(videoid, this);
    connect(&dlg, SIGNAL(playlistInfo(const QString&, const QString&, const QString&, bool, bool)), net, SLOT(createNewPlaylist(const QString&, const QString&, const QString&, bool, bool)));
    if(dlg.exec() && mainWidget->indexOf(playlistListWidget) >= 0){
        updatePlaylist("default");
    }
    disconnect(&dlg, SIGNAL(playlistInfo(const QString&, const QString&, const QString&, bool, bool)), net, SLOT(createNewPlaylist(const QString&, const QString&, const QString&, bool, bool)));
}

void MainWindow::delPlaylist()
{
    int reply = QMessageBox::question(this, "Warning!", tr("delete %1?").arg(playlistListWidget->currentItem()->text()), QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
    if (reply == QMessageBox::Yes){
        setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
        connect(net, SIGNAL(playlistsData(const QList< QMap<QString, QString> >&)), this, SLOT(addPlaylist(const QList< QMap<QString, QString> >&)));
        net->delUrl(QUrl(playlistUrls.at(playlistListWidget->currentRow() * 2 + 1)), true, false);
    }
}

/* Others */
bool MainWindow::eventFilter(QObject *obj, QEvent *ev)
{
    if (ev->type() == QEvent::KeyPress) {

        bool consumed = false;
        QKeyEvent *keyEvent = static_cast<QKeyEvent*>(ev);
        int key = keyEvent->key();
        switch (key) {
        case Qt::Key_Enter:
            if (lineEdit->text() != "") {
                searchVideo(lineEdit->text());
            }
            else {
                closeToolBar();
            }
            consumed = true;

        case Qt::Key_Backspace:
            lineEdit->backspace();
            if (lineEdit->text() == "") {
                closeToolBar();
            }
            consumed = true;

        case Qt::Key_Up:
        case Qt::Key_Down:
        case Qt::Key_Left:
        case Qt::Key_Right:
        case Qt::Key_Control:
        case Qt::Key_Return:
            break;

        default:
            toolBar->setVisible(true);
            lineEdit->setFocus();
            lineEdit->insert(keyEvent->text().toLower());
            consumed = true;
            break;
        }
        return consumed;
    }
    else {
        return QObject::eventFilter(obj, ev);
    }
}

void MainWindow::closeToolBar()
{
    if(c->popup()) {
        c->popup()->hide();
    }
    timer->stop();
    lineEdit->clear();
    toolBar->setVisible(false);
    listWidget->setFocus();
}

void MainWindow::onScrollBarValueChanged(int value)
{
    //qDebug() << value * 2.35;
    bool fetchMore = false;
    if (nextPage && value > 700){
        if(isPortrait){
            if(value == vScrollBar->maximum() && start_index < 100) {
                fetchMore = true;
            }
        }
        else {
            if(value == hScrollBar->maximum() && start_index < 100) {
                fetchMore = true;
            }
        }
    }
    if (fetchMore){
        nextPage = false;
        getVideoInfo();
    }
}

QString MainWindow::maskIt(const QString &str, bool e)
{
    QByteArray ba(str.toAscii());
    if (e) {
        return QString(ba.toHex());
    }
    else {
        return QString(QByteArray::fromHex(ba));
    }
}
/* Google Suggestion */
void MainWindow::updateSuggestion(const QStringList &list)
{
    QStringList newList(list);
    newList.append(recentSearch);
    QStringListModel *model = qobject_cast<QStringListModel*>(c->model());
    model->setStringList(newList);
}

void MainWindow::autoSuggest()
{
    QString txt = lineEdit->text();
    if (txt.size() > 1){
        QUrl url = QString("http://google.com/complete/search?output=toolbar&q=%1").arg(txt);
        net->getUrl(url);
    }
}
/* Screen rotation */
void MainWindow::orientationChanged()
{
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    if (screenGeometry.width() > screenGeometry.height()){
        isPortrait = false;
        listWidget->setFlow(QListView::LeftToRight);
        for (int i = 0; i < listWidget->count(); ++i){
            listWidget->item(i)->setSizeHint(QSize(340, 340));
        }
    }
    else {
        isPortrait = true;
        listWidget->setFlow(QListView::TopToBottom);
        for (int i = 0; i < listWidget->count(); ++i){
            listWidget->item(i)->setSizeHint(QSize(480, 143));
        }
        closeToolBar();
    }
}

/* Info Message */
void MainWindow::replyMessage(const QString &message)
{
    QMaemo5InformationBox::information(this, message);
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
}

/* Error Message */
void MainWindow::errorMessage(const QString &message)
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
    ytLabel->setText(message);
    mainWidget->setCurrentWidget(ytLabel);
    ytLabel->setFocus();
}

/* About */
void MainWindow::aboutDlg() const
{
    QMessageBox msgBox;
    msgBox.setAttribute(Qt::WA_Maemo5AutoOrientation, true);
    msgBox.setWindowTitle(tr("About"));
    msgBox.setText( QString(tr( "<img src=':/images/yt_logo.png'/><br><br><font size=2 color='gray'>Version: 0.1.1<br>"
                                 "Description: YouTube video browser, searcher, and player.</font>")));
    msgBox.exec();
}

