#include "downloadmanager.h"
#include <QTimer>
#include <QRegExp>
#include <QStringList>

DownloadManager::DownloadManager(QObject *parent) :
    QObject(parent), usePreferredServer(false), dlPath("/home/user/MyDocs/.videos/")
{
    connect(this, SIGNAL(videoUrl(const QString&)), this, SLOT(startDownload(const QString&)));
}

void DownloadManager::abort()
{
    if(currentDownload.isEmpty()){
        return;
    }
    dlReply->abort();
    file.close();
    file.remove();
    dlReply->deleteLater();
    emit dlMessage(QString("\"%1\" download was canceled").arg(downloadQueue.at(0).value("title")));
}

void DownloadManager::append(const QList<QMap<QString, QString> > &videos)
{
    downloadQueue.append(videos);
    if(currentDownload.isEmpty()){
        QTimer::singleShot(0, this, SLOT(startNextDownload()));
    }
}

void DownloadManager::append(const QMap<QString, QString> &video)
{
    if(!downloadQueue.contains(video)){
        downloadQueue.append(video);
    }
    if(currentDownload.isEmpty()){
        QTimer::singleShot(0, this, SLOT(startNextDownload()));
    }
}

void DownloadManager::startNextDownload()
{
    if(downloadQueue.isEmpty()){
        emit finished();
        return;
    }
    downloadRequest(downloadQueue.at(0).value("videoid"));
}

void DownloadManager::downloadRequest(const QString &videoId)
{
    QString basename = downloadQueue.at(0).value("title");
    basename.replace(QRegExp("[\"<>:/|?*\\\\]"), "_").trimmed();
    basename.prepend(dlPath);
    QString filename = QString("%1(%2).mp4").arg(basename).arg(downloadQueue.at(0).value("videoid"));
    if (QFile::exists(filename)){
        emit dlMessage("This video was already in download folder");
        downloadQueue.removeFirst();
        QTimer::singleShot(0, this, SLOT(startNextDownload()));
        return;
    }
    file.setFileName(filename.replace(".mp4", ".partial"));
    if (file.exists()){
        if (!file.open(QIODevice::Append)){
            downloadQueue.removeFirst();
            QTimer::singleShot(0, this, SLOT(startNextDownload()));
            emit dlMessage("Error: download resume failed");
            return;
        }
        emit dlMessage("Resume downloading..");
    }
    else if (!file.open(QIODevice::WriteOnly)){
        downloadQueue.removeFirst();
        QTimer::singleShot(0, this, SLOT(startNextDownload()));
        emit dlMessage("Error: unable to create file");
        return;
    }
    getVideoUrl(videoId);
}

void DownloadManager::startDownload(const QString &dlUrl)
{
    if (!dlUrl.isEmpty()){
        QNetworkRequest request;
        request.setUrl(QUrl(QUrl::fromEncoded(dlUrl.toUtf8())));
        if (file.size() > 0){
            request.setRawHeader("Range", "bytes=" + QByteArray::number(file.size()) + "-");
        }
        dlReply = nam.get(request);
        currentDownload.append(dlReply);
        connect(dlReply, SIGNAL(downloadProgress(qint64, qint64)), this, SLOT(onDownloadProgress(qint64,qint64)));
        connect(dlReply, SIGNAL(finished()), this, SLOT(downloadFinished()));
        connect(dlReply, SIGNAL(readyRead()), this, SLOT(onReadyRead()));
        dlTimer.start();
    }
    else {
        emit dlMessage("Error: unable to obtain download link");
        file.close();
        downloadQueue.removeFirst();
        QTimer::singleShot(0, this, SLOT(startNextDownload()));
    }
}

void DownloadManager::onDownloadProgress(qint64 bytesReceived, qint64 bytesTotal) {
    double speed = bytesReceived * 1000.0 / dlTimer.elapsed();
    QString unit;
    if (speed < 1024) {
        unit = "bytes/sec";
    } else if (speed < 1024*1024) {
        speed /= 1024;
        unit = "kB/s";
    } else {
        speed /= 1024*1024;
        unit = "MB/s";
    }
    int n = downloadQueue.size();
    if (n > 1) {
        bytesTotal += 10;
    }
    emit downloadStatus(bytesReceived, bytesTotal, QString::fromLatin1("%1 left to download: [ %2 %3 ]" ).arg(n).arg(speed, 3, 'f', 1).arg(unit));
}

void DownloadManager::downloadFinished()
{
    QString redirectString = dlReply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl().toString();
    if(!redirectString.isEmpty()){
        dlReply->deleteLater();
        currentDownload.removeAll(dlReply);
        startDownload(redirectString);
        return;
    }
    file.close();
    if (dlReply->error() != QNetworkReply::NoError || file.size() == 0){
        emit dlMessage(QString("Download error: %1").arg(dlReply->error()));
        file.remove();
    }
    else {
        QString filename = file.fileName().left(file.fileName().lastIndexOf("."));
        int i = 1;
        while(!file.rename(QString("%1.mp4").arg(filename)) && i < 5){
            filename += QString::number(i);
            ++i;
        }
        emit dlMessage(QString("%1 has been downloaded").arg(downloadQueue.at(0).value("title")));
    }
    dlReply->deleteLater();
    downloadQueue.removeFirst();
    currentDownload.removeAll(dlReply);
    QTimer::singleShot(0, this, SLOT(startNextDownload()));
}

void DownloadManager::onReadyRead()
{
    file.write(dlReply->readAll());
}

void DownloadManager::getVideoUrl(const QString &videoId)
{
    if (!videoId.isEmpty()){
        QNetworkAccessManager *manager = new QNetworkAccessManager(this);
        QUrl url = QUrl(QString("http://www.youtube.com/get_video_info?video_id=%1&el=vevo&ps=default&gl=US&hl=en").arg(videoId));
        QNetworkRequest request;
        request.setUrl(url);
        connect(manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(processVideoUrl(QNetworkReply*)));
        manager->get(request);
    }
}

void DownloadManager::processVideoUrl(QNetworkReply *reply)
{
    QNetworkAccessManager *manager = qobject_cast<QNetworkAccessManager*>(sender());
    if (reply->error() != QNetworkReply::NoError){
        emit dlMessage(QString("Unable to get video url error: %1").arg(reply->error()));
    }
    else {
        QString mp4url = "";
        QString string = reply->readAll();
        if (string.contains("url_encoded_fmt_stream_map=")){
            QString str(QByteArray::fromPercentEncoding(string.split("url_encoded_fmt_stream_map=").at(1).split("&").at(0).toAscii()));
            QStringList strlist = str.split(",");
            if (strlist.size() > 2) {
                if(strlist.at(strlist.size()-2).endsWith("itag=18")){
                    QString url(QByteArray::fromPercentEncoding(strlist.at(strlist.size()-2).split("&").at(0).split("=").at(1).toAscii()));
                    if (url.startsWith("http://o-o.preferred.") && !usePreferredServer){
                        url.remove("o-o.preferred.").remove(7, url.indexOf(".") - 6);
                    }
                    mp4url = url;
                }
            }
        }
        emit videoUrl(mp4url);
    }
    reply->deleteLater();
    manager->deleteLater();
}

void DownloadManager::setUsePreferredServer(bool b)
{
    usePreferredServer = b;
}

void DownloadManager::setDownloadFolder(const QString &downloadPath)
{
    dlPath = QString("%1/").arg(downloadPath);
}
