#include <QtGui>
#include "videoplayer.h"
#include <QMaemo5InformationBox>
#include <QDBusConnection>
#include <QDBusInterface>

VideoPlayer::VideoPlayer(bool login, QWidget *parent) :
    QMainWindow(parent)
{
    isLogin = login;
    setAttribute(Qt::WA_Maemo5StackedWindow);
    setAttribute(Qt::WA_DeleteOnClose);
    net = new NetworkAccess(this);
    connect(net, SIGNAL(videoUrl(const QString&)), this, SLOT(gotVideoUrl(const QString&)));
    connect(net, SIGNAL(videoData(const QMap<QString, QString>&)), this, SLOT(addVideo(const QMap<QString, QString>&)));
    connect(net, SIGNAL(videoThumbnail(int, const QPixmap&)), this, SLOT(addThumbnail(int, const QPixmap&)));
    connect(net, SIGNAL(dataLoaded(bool)), this, SLOT(dataLoaded(bool)));
    connect(net, SIGNAL(postMessage(const QString&)), this, SLOT(popupMessage(const QString&)));

    QWidget *widget = new QWidget(this);
    setCentralWidget(widget);
    QGridLayout *mainLayout = new QGridLayout(widget);
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->setSpacing(0);
    mainLayout->setColumnStretch(0,2);
    mainLayout->setRowStretch(1,2);

    ytLabel = new QLabel(this);
    ytLabel->setAlignment(Qt::AlignCenter);
    ytLabel->setPixmap(QPixmap(":/images/yt_logo.png"));

    titleLabel = new QLabel(this);
    titleLabel->setStyleSheet("color: rgb(255, 255, 255, 150); margin-left: 5px; font: 22px");
    listWidgetLabel = new QLabel(tr("Related videos"), this);
    listWidgetLabel->setStyleSheet("color: rgb(255, 255, 255, 150); font: 22px");
    videoInfoLabel = new QLabel(this);
    videoInfoLabel->setStyleSheet("color: rgb(255, 255, 255, 150); margin-left: 5px; font: 18px");
    mainLayout->addWidget(titleLabel, 0, 0);
    mainLayout->addWidget(listWidgetLabel, 0, 1);
    mainLayout->addWidget(videoInfoLabel, 2, 0);

    mediaObject = new Phonon::MediaObject(this);
    videoWidget = new Phonon::VideoWidget(this);
    videoWidget->setVisible(false);
    Phonon::AudioOutput *audioOutput = new Phonon::AudioOutput(Phonon::VideoCategory, this);
    Phonon::createPath(mediaObject, videoWidget);
    Phonon::createPath(mediaObject, audioOutput);

    commentEdit = new QTextEdit(this);
    commentEdit->setVisible(false);

    QStackedLayout *stackedLayout = new QStackedLayout;
    stackedLayout->setStackingMode(QStackedLayout::StackAll);
    stackedLayout->addWidget(ytLabel);
    stackedLayout->addWidget(videoWidget);
    stackedLayout->addWidget(commentEdit);
    stackedLayout->setCurrentWidget(ytLabel);

    mainLayout->addLayout(stackedLayout, 1, 0);

    listWidget = new QListWidget(this);
    listWidget->setContextMenuPolicy(Qt::ActionsContextMenu);
    listWidget->setContentsMargins(0, 0, 0, 0);
    listWidget->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    listWidget->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    listWidget->setIconSize(QSize(120, 90));
    listWidget->setWordWrap(true);
    listWidget->setFixedWidth(220);
    connect(listWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(itemClicked(QListWidgetItem*)));
    mainLayout->addWidget(listWidget, 1, 1, 4, 1, Qt::AlignRight);

    mediaObject->setPrefinishMark(5000);
    mediaObject->setTickInterval(1000);
    mediaObject->setTransitionTime(-2000);
    connect(mediaObject, SIGNAL(prefinishMarkReached(qint32)), this, SLOT(onPrefinishMarkReached(qint32)));
    connect(mediaObject, SIGNAL(currentSourceChanged(const Phonon::MediaSource&)), this, SLOT(onCurrentSourceChanged(const Phonon::MediaSource&)));
    connect(mediaObject, SIGNAL(totalTimeChanged(qint64)), this, SLOT(totalTimeChanged(qint64)));
    connect(mediaObject, SIGNAL(stateChanged(Phonon::State, Phonon::State)), this, SLOT(stateChanged(Phonon::State, Phonon::State)));
    connect(mediaObject, SIGNAL(finished()), this, SLOT(done()));
    connect(mediaObject, SIGNAL(tick(qint64)), this, SLOT(tick(qint64)));

    addToQueue = new QAction(tr("Add to queue"), listWidget);
    connect(addToQueue, SIGNAL(triggered()), this, SLOT(onAddToQueue()));
    listWidget->addAction(addToQueue);
    optionAction = new QAction(QIcon(":/images/menu.png"), "", this);
    playAction = new QAction(QIcon(playButtonIcon), "", this);
    playAction->setVisible(false);
    pauseAction = new QAction(QIcon(pauseButtonIcon), "", this);
    pauseAction->setVisible(false);
    stopAction = new QAction(QIcon(stopButtonIcon), "", this);
    nextAction = new QAction(QIcon(nextButtonIcon), "", this);
    nextAction->setVisible(false);
    prevAction = new QAction(QIcon(prevButtonIcon), "", this);
    prevAction->setVisible(false);
    connect(playAction, SIGNAL(triggered()), mediaObject, SLOT(play()));
    connect(pauseAction, SIGNAL(triggered()), mediaObject, SLOT(pause()));
    connect(stopAction, SIGNAL(triggered()), this, SLOT(stop()));
    connect(nextAction, SIGNAL(triggered()), this, SLOT(onNextAction()));
    connect(prevAction, SIGNAL(triggered()), this, SLOT(onPrevAction()));
    connect(optionAction, SIGNAL(triggered()), this, SLOT(toolbarSwitch()));

    optionBar = new QToolBar(this);
    optionBar->addAction(optionAction);
    optionBar->setIconSize(QSize(48, 48));
    QAction *favAction = new QAction(QIcon::fromTheme("imageviewer_favourite"), "", optionBar);
    connect(favAction, SIGNAL(triggered()), this, SLOT(onFavAction()));
    QAction *subscribeAction = new QAction(QIcon::fromTheme("rss_reader_goto"), "", optionBar);
    connect(subscribeAction, SIGNAL(triggered()), this, SLOT(onSubscribeAction()));
    QAction *downloadAction = new QAction(QIcon(":/images/downloadicon.png"), "", this);
    connect(downloadAction, SIGNAL(triggered()), this, SLOT(onDownloadAction()));
    QAction *likeAction = new QAction(QIcon(":/images/likeicon.png"), "", optionBar);
    connect(likeAction, SIGNAL(triggered()), this, SLOT(onLikeAction()));
    QAction *dislikeAction = new QAction(QIcon(":/images/dislikeicon.png"), "", optionBar);
    connect(dislikeAction, SIGNAL(triggered()), this, SLOT(onDislikeAction()));
    QAction *commentAction = new QAction(QIcon(":/images/comment.png"), "", optionBar);
    commentAction->setCheckable(true);
    connect(commentAction, SIGNAL(toggled(bool)), this, SLOT(onCommentAction(bool)));
    optionBar->addAction(downloadAction);
    optionBar->addAction(favAction);
    optionBar->addAction(subscribeAction);
    optionBar->addAction(likeAction);
    optionBar->addAction(dislikeAction);
    optionBar->addAction(commentAction);
    mainLayout->addWidget(optionBar, 4, 0);
    optionBar->setVisible(false);

    bar = new QToolBar(this);
    if (isLogin){
        bar->addAction(optionAction);
    }
    else {
        bar->addAction(downloadAction);
    }
    bar->addAction(stopAction);
    bar->addAction(prevAction);
    bar->addAction(pauseAction);
    bar->addAction(playAction);
    bar->addAction(nextAction);
    Phonon::SeekSlider *seekSlider = new Phonon::SeekSlider(mediaObject, bar);
    bar->addWidget(seekSlider);
    timeLcd = new QLabel(bar);
    bar->addWidget(timeLcd);
    mainLayout->addWidget(bar, 3, 0);

    timeLcd->setStyleSheet("font-size: 18px; background-color: rgb(0, 0, 0, 255)");
    timeLcd->setText("00:00");
    isShown = false;
    timer = new QTimer(this);
    timer->setSingleShot(true);
    timer->setInterval(20000);
    connect(timer, SIGNAL(timeout()), bar, SLOT(hide()));
    connect(timer, SIGNAL(timeout()), optionBar, SLOT(hide()));
    connect(timer, SIGNAL(timeout()), listWidget, SLOT(hide()));
    connect(timer, SIGNAL(timeout()), titleLabel, SLOT(hide()));
    connect(timer, SIGNAL(timeout()), listWidgetLabel, SLOT(hide()));
    connect(timer, SIGNAL(timeout()), videoInfoLabel, SLOT(hide()));
    connect(timer, SIGNAL(timeout()), this, SLOT(setFocus()));
    connect(listWidget->verticalScrollBar(), SIGNAL(valueChanged(int)), this, SLOT(onScrollBarValueChanged(int)));
    currentVidx = 0;
    start_index = 1;
    isPlaylist = false;
    nextPage = false;
    isCommentShown = false;
    showFullScreen();
    recentWVideos = settings.value("recentlyWatched", QStringList()).toStringList();
    keepScreenOn();
}

void VideoPlayer::toolbarSwitch()
{
    timer->start();
    if(bar->isVisible()){
        bar->setVisible(false);
        optionBar->setVisible(true);
    }
    else {
        optionBar->setVisible(false);
        bar->setVisible(true);
    }
}

void VideoPlayer::onCommentAction(bool b)
{
    videoWidget->setVisible(!b);
    ytLabel->setVisible(!b);
    commentEdit->setVisible(b);
    if(b){
        timer->stop();
        commentEdit->setFocus();
        titleLabel->setText("Your comment:");
        isCommentShown = true;
    }
    else{
        if(commentEdit->toPlainText() != ""){
            emit addComment(currentVideo.value("comments"), commentEdit->toPlainText());
            commentEdit->clear();
        }
        timer->start();
        QFontMetrics fm(titleLabel->font());
        titleLabel->setText(QString("%1").arg(fm.elidedText(currentVideo.value("title"), Qt::ElideRight, 460)));
        isCommentShown = false;
    }
}

void VideoPlayer::onSubscribeAction()
{
    emit subscribeVideo(currentVideo.value("credit"));
}

void VideoPlayer::onFavAction()
{
    emit addFavorite(currentVideo.value("videoid"));
}

void VideoPlayer::onDownloadAction()
{
    emit addToDownloadQueue(currentVideo);
    QMaemo5InformationBox::information(this, tr("\"%1\" has been added to download queue").arg(currentVideo.value("title")));
}

void VideoPlayer::onLikeAction()
{
    emit likeThisVideo(currentVideo.value("videoid"), true);
}

void VideoPlayer::onDislikeAction()
{
    emit likeThisVideo(currentVideo.value("videoid"), false);
}

void VideoPlayer::onNextAction()
{
    if(isPlaylist && currentVidx < videos.size() - 1){
        ++currentVidx;
        loadVideo(videos.at(currentVidx));
    }
    else if (playQueue.indexOf(currentVideo) < playQueue.size() - 1){
        loadVideo(playQueue.at(playQueue.indexOf(currentVideo) + 1));
    }
}
void VideoPlayer::onPrevAction()
{
    if(isPlaylist && currentVidx > 0){
        --currentVidx;
        loadVideo(videos.at(currentVidx));
    }
    else if(playQueue.indexOf(currentVideo) > 0){
        loadVideo(playQueue.at(playQueue.indexOf(currentVideo) - 1));
    }
}

void VideoPlayer::onAddToQueue()
{
    playQueue.append(videos.at(listWidget->currentRow()));
    nextAction->setVisible(true);
    QMaemo5InformationBox::information(this, tr("\"%1\" has been added to current play queue").arg(videos.at(listWidget->currentRow()).value("title")));
}

void VideoPlayer::onPrefinishMarkReached(qint32 mark)
{
    Q_UNUSED(mark);
    ++currentVidx;
    if(isPlaylist && currentVidx < videos.size()){
        QMap<QString, QString> video = videos.at(currentVidx);
        loadVideo(video);
    }
    else if (playQueue.indexOf(currentVideo) < playQueue.size() - 1){
        loadVideo(playQueue.at(playQueue.indexOf(currentVideo) + 1));
    }
}

void VideoPlayer::onCurrentSourceChanged(const Phonon::MediaSource&)
{
    QFontMetrics fm(titleLabel->font());
    titleLabel->setText(QString("%1").arg(fm.elidedText(currentVideo.value("title"), Qt::ElideRight, 460)));
    videoInfoLabel->setText(QString("By <font color=\"green\">%1</font> - %2 views | %3 likes | %4 dislikes").arg(currentVideo.value("credit")).arg(currentVideo.value("viewCount")).arg(currentVideo.contains("numLikes")?currentVideo.value("numLikes"):"0").arg(currentVideo.contains("numDislikes")?currentVideo.value("numDislikes"):"0"));
    if(isPlaylist){
        listWidget->setCurrentRow(currentVidx);
    }
}

void VideoPlayer::gotVideoUrl(const QString &url)
{
    if (url.isEmpty()){
        QMessageBox::warning(this, tr("Video restriction"), "This video is only available in YouTube website.");
        emit playerReady();
        ++currentVidx;
        if(isPlaylist && currentVidx < videos.size()){
            QMap<QString, QString> video = videos.at(currentVidx);
            loadVideo(video);
        }
        else if(mediaObject->state() != Phonon::PlayingState){
            stop();
        }
    }
    else {
        mediaObject->setCurrentSource(Phonon::MediaSource(QUrl::fromEncoded(url.toUtf8())));
        QTimer::singleShot(0, this, SLOT(play()));
    }
}

void VideoPlayer::loadVideo(const QMap<QString, QString> &video)
{
    currentVideo = video;
    QString basename = video.value("title");
    basename.replace(QRegExp("[\"<>:/|?*\\\\]"), "_").trimmed();
    basename.prepend(QString("%1/").arg(settings.value("downloadfolder", "/home/user/MyDocs/.videos").toString()));
    QString filename = QString("%1(%2).mp4").arg(basename).arg(video.value("videoid"));
    if (QFile::exists(filename)){
        mediaObject->setCurrentSource(Phonon::MediaSource(filename));
        QTimer::singleShot(0, this, SLOT(play()));
        QAction *dl = qobject_cast<QAction*>(optionBar->actions().at(1));
        dl->setVisible(false);
    }
    else {
        net->getVideoUrl(video.value("videoid"));
        QAction *dl = qobject_cast<QAction*>(optionBar->actions().at(1));
        dl->setVisible(true);
    }
    recentWVideos.removeAll(video.value("videoid"));
    if(recentWVideos.size() == 20){
        recentWVideos.removeAt(20);
    }
    recentWVideos.prepend(video.value("videoid"));
}

void VideoPlayer::play()
{
    if(mediaObject->state() != Phonon::PlayingState){
        mediaObject->play();
    }
    if (!isPlaylist) {
        listWidget->clear();
        videos.clear();
        start_index = 1;
        net->resetIndex();
        if(playQueue.indexOf(currentVideo) < 0){
            playQueue.append(currentVideo);
        }
        loadVideoList(currentVideo.value("related"), "10", isPlaylist);
    }
}

void VideoPlayer::loadVideoList(const QString &playlistUrl, const QString &max, bool playlistMode)
{
    isPlaylist = playlistMode;
    QUrl url(playlistUrl);
    url.addQueryItem("start-index", QString("%1").arg(start_index));
    url.addQueryItem("max-results", max);
    url.addQueryItem("fields", "link(@rel),entry(id,link(@rel,@href),summary,gd:comments,media:group(media:credit/text(),media:description/text(),media:title/text(),media:thumbnail[@yt:name='default'](@url),yt:duration,yt:uploaded,yt:videoid),gd:rating(@average),yt:statistics(@viewCount),yt:rating)");
    net->getUrl(url);
}

void VideoPlayer::addVideo(const QMap<QString, QString> &video)
{
    QFont font;
    font.setPointSize(12);
    QListWidgetItem *item = new QListWidgetItem(video.value("title"), listWidget);
    item->setSizeHint(QSize(120, 90));
    item->setFont(font);
    item->setTextAlignment(Qt::AlignLeft|Qt::AlignTop);
    listWidget->addItem(item);
    videos.append(video);
    ++start_index;
    if (isPlaylist && listWidget->count() == 1) {
        loadVideo(video);
        listWidgetLabel->setText(tr("Playlist"));
        listWidget->removeAction(addToQueue);
    }
}

void VideoPlayer::addThumbnail(int i, const QPixmap &thumbnail)
{
    listWidget->item(i)->setIcon(QIcon(thumbnail));
}

void VideoPlayer::itemClicked(QListWidgetItem *item)
{
    int i = listWidget->row(item);
    loadVideo(videos.at(i));
    if(isPlaylist){
        currentVidx = i;
    }
}

void VideoPlayer::stop()
{
    mediaObject->stop();
    settings.setValue("recentlyWatched", recentWVideos);
    emit quit();
    QTimer::singleShot(0, this, SLOT(close()));
}

void VideoPlayer::mouseReleaseEvent(QMouseEvent*)
{
    widgetsSwitch();
}

void VideoPlayer::keyPressEvent(QKeyEvent *event)
{
    if (!titleLabel->isVisible()){
        widgetsSwitch();
    }
    QWidget::keyPressEvent(event);
}

void VideoPlayer::done()
{
    if (!titleLabel->isVisible()){
        widgetsSwitch();
    }
    timer->stop();
}

void VideoPlayer::totalTimeChanged(qint64 time)
{
    videoDuration = time;
}

void VideoPlayer::widgetsSwitch()
{
    if(isShown){
        bool b = titleLabel->isVisible();
        videoWidget->setVisible(b);
        titleLabel->setVisible(!b);
        listWidgetLabel->setVisible(!b);
        listWidget->setVisible(!b);
        videoInfoLabel->setVisible(!b);
        optionBar->setVisible(false);
        bar->setVisible(!b);
        videoWidget->setVisible(true);
        timer->start();
    }
}

void VideoPlayer::tick(qint64 time)
{
    qint64 timeLeft = videoDuration - time;
    QTime displayTime((timeLeft / 3600000) % 24, (timeLeft / 60000) % 60, (timeLeft / 1000) % 60);
    if (timeLeft >= 3600000){
        timeLcd->setText(displayTime.toString("-h:mm:ss"));
    }
    else {
        timeLcd->setText(displayTime.toString("-mm:ss"));
    }
}

void VideoPlayer::onScrollBarValueChanged(int value)
{
    if(!isCommentShown){
       timer->start();
    }
    bool fetchMore = false;
    if (nextPage && value > 400){
        if(value == listWidget->verticalScrollBar()->maximum() && start_index < 100) {
            fetchMore = true;
        }
    }
    if (fetchMore && !isPlaylist){
        nextPage = false;
        loadVideoList(currentVideo.value("related"), "10", isPlaylist);
    }
}

void VideoPlayer::dataLoaded(bool b)
{
    nextPage = b;
    nextAction->setVisible(((currentVidx < videos.size() - 1 && isPlaylist) || (playQueue.indexOf(currentVideo) != -1 && playQueue.indexOf(currentVideo) < playQueue.size() - 1)));
}

void VideoPlayer::stateChanged(Phonon::State newState, Phonon::State oldState)
{
    Q_UNUSED(oldState);
    switch (newState) {
        case Phonon::ErrorState:
            if (mediaObject->errorType() == Phonon::FatalError) {
                QMessageBox::warning(this, tr("Fatal Error"),
                mediaObject->errorString());
                emit playerReady();
                stop();
            } else {
                QMessageBox::warning(this, tr("Error"),
                mediaObject->errorString());
                emit playerReady();
                stop();
            }
            break;
        case Phonon::PlayingState:
                videoWidget->setVisible(true);
                playAction->setVisible(false);
                pauseAction->setVisible(true);
                stopAction->setVisible(true);
                nextAction->setVisible(((currentVidx < videos.size() - 1 && isPlaylist) || (playQueue.indexOf(currentVideo) != -1 && playQueue.indexOf(currentVideo) < playQueue.size() - 1)));
                prevAction->setVisible(((isPlaylist && currentVidx > 0) || playQueue.indexOf(currentVideo) > 0));
                if (!isShown){
                    isShown = true;
                    if(!isCommentShown){
                        timer->start();
                    }
                    emit playerReady();
                }
                break;
        case Phonon::StoppedState:
                playAction->setVisible(true);
                pauseAction->setVisible(false);
                timeLcd->setText("00:00");
                break;
        case Phonon::PausedState:
                pauseAction->setVisible(false);
                stopAction->setVisible(true);
                playAction->setVisible(true);
                break;
        case Phonon::LoadingState:
                videoWidget->setVisible(false);
                break;
        default:
                break;
    }
}

void VideoPlayer::popupMessage(const QString &message)
{
    QMaemo5InformationBox::information(this, message);
}

void VideoPlayer::keepScreenOn()
{
    QDBusInterface iface("com.nokia.mce", "/com/nokia/mce/request", "com.nokia.mce.request", QDBusConnection::systemBus(), this);
    iface.call("req_display_blanking_pause");
    QTimer::singleShot(9000, this, SLOT(keepScreenOn()));
}
