import os, time, threading
import gobject
import pygst
pygst.require("0.10")
import gst

import logging
log = logging.getLogger("nqaap.SimpleGStreamer")


class SimpleGStreamer (threading.Thread):
    def __init__(self, on_playing_done = None, run_thread = False):
        #Fields
        self.has_file = False
        self.playing = False
        self.__elapsed = 0

        #Event callbacks
        self.on_playing_done = on_playing_done

        #Set up GStreamer
        self.player = gst.element_factory_make("playbin2", "player")
        fakesink = gst.element_factory_make("fakesink", "fakesink")
        self.player.set_property("video-sink", fakesink)
        bus = self.player.get_bus()
        bus.add_signal_watch()
        bus.connect("message", self.on_message)
        
        #Constants
        self.time_format = gst.Format(gst.FORMAT_TIME)
        self.seek_flag = gst.SEEK_FLAG_FLUSH
        
        # thread setup
        if run_thread:
            threading.Thread.__init__ ( self )
            self.setDaemon(True)
            self.start()
            log.debug("Thread started")

    def on_message(self, bus, message):
        t = message.type
        if t == gst.MESSAGE_EOS:        # End-Of-Stream
            self.player.set_state(gst.STATE_NULL)
            self.playing = False
            if ( self.on_playing_done != None): # event callback
                self.on_playing_done(self)
        elif t == gst.MESSAGE_ERROR:
            self.player.set_state(gst.STATE_NULL)
            err, debug = message.parse_error()
            #print "Error: %s" % err, debug
            log.error("Error: %s, (%s)" % (err, debug))
            self.playing = False

        
    def set_file(self, file):
        log.info("set file: %s",file)
        if os.path.isfile(file):
            if (self.playing):
                self.stop()

            file = os.path.abspath(file) # ensure absolute path
            log.debug("set file (absolute path): %s "%file)
            self.player.set_property("uri", "file://" + file)
            self.has_file = True
        else:
            log.error( "File: %s not found" % file)

    def play(self):
        log.info("Started playing")
        self.player.set_state(gst.STATE_PLAYING)
        self.playing = True

    def run(self):

        gobject.threads_init()
        loop = gobject.MainLoop() 
        loop.run()
        
        while True:
            if (self.playing):
                try:
                    self.__elapsed = self.player.query_position(self.time_format, None)[0]
                except:
                    pass
            time.sleep(1)
            
    def stop(self):
        self.player.set_state(gst.STATE_NULL)
        self.playing = False
        log.info("Stopped playing") 
                
    def elapsed(self):
        try:
            self.__elapsed = self.player.query_position(self.time_format, None)[0]
        except:
            pass
        return self.__elapsed

    def duration(self):
        try:
            return self.player.query_duration(self.time_format, None)[0]
        except:
            return 0

    def seek_time(self,ns):
        log.debug("Seeking to: %s", ns)
        self.player.seek_simple(self.time_format, self.seek_flag, ns)

    def __seek_percent(self,percent):
        format = gst.Format(gst.FORMAT_PERCENT)
        self.player.seek_simple(format, self.seek_flag, percent)
        
