/** Code is released under GPLv2
 *
 * See debian/copyright for copyright info
 */

/** Compile binary only with:
 * gcc -Wall `pkg-config --cflags --libs hildon-notify glib-2.0 mafw mafw-shared` nowplaying.c -o nowplayingd
 */

#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <libintl.h>
#include <locale.h>
#include <hildon/hildon-notification.h>
#include <libmafw/mafw.h>
#include <libmafw-shared/mafw-shared.h>

#define WANTED_RENDERER "Mafw-Gst-Renderer"

static GMainLoop *loop = NULL;
static MafwRegistry *registry = NULL;

static void sig_handler( int sig G_GNUC_UNUSED )
{
	if ( g_main_loop_is_running( loop ) ) {
		g_main_loop_quit( loop );
	}
}

static gboolean close_notification( gpointer n )
{
	notify_notification_close( NOTIFY_NOTIFICATION( n ), NULL );

	return FALSE;
}

static void metadata_callback(
	const MafwRenderer *self G_GNUC_UNUSED,
	const gchar *object_id G_GNUC_UNUSED,
	GHashTable *metadata,
	gconstpointer user_data G_GNUC_UNUSED,
	const GError *error G_GNUC_UNUSED )
{
	const gchar *artist = NULL;
	const gchar *title = NULL;
	HildonNotification *n = NULL;

	artist = g_value_get_string( mafw_metadata_first( metadata, MAFW_METADATA_KEY_ARTIST ) );
	title = g_value_get_string( mafw_metadata_first( metadata, MAFW_METADATA_KEY_TITLE ) );

	if ( !artist || *artist == '\0' ) /* || !strcmp (artist, "")) */ {
		artist = dgettext( "mediaplayer", "mp_li_unknown_artist" );
	}

	if ( !title || *title == '\0' ) /* || !strcmp (title, "")) */ {
		title = dgettext( "mediaplayer", "mp_li_unknown_song" );
	}

	n = hildon_notification_new( artist, title, "tasklaunch_media_player", "media" );
	hildon_notification_add_dbus_action( n, "default",
		"tasklaunch_media_player",
		"com.nokia.mediaplayer",
		"/com/nokia/osso/mediaplayer",
		"com.nokia.mediaplayer",
		"open_mp_now_playing",
		G_TYPE_NONE, NULL,
		-1 );
		
	notify_notification_show( NOTIFY_NOTIFICATION( n ), NULL );
	g_timeout_add_seconds( 5, close_notification, n );
}

static void state_changed_cb(
	MafwRenderer *renderer,
	const MafwPlayState state,
	gconstpointer user_data G_GNUC_UNUSED)
{
	switch (state) {
		case Playing:
			mafw_renderer_get_current_metadata( renderer,
				(MafwRendererMetadataResultCB) metadata_callback,
				NULL );
			break;

		case Stopped:
		default:
			break;
  }
}

static void renderer_added_cb(
	const MafwRegistry *registry2 G_GNUC_UNUSED,
	GObject *renderer,
	gconstpointer user_data G_GNUC_UNUSED)
{
	if ( MAFW_IS_RENDERER( renderer ) ) {
		const gchar *name = mafw_extension_get_name( MAFW_EXTENSION( renderer ) );

		if ( !strcmp( name, WANTED_RENDERER ) ) {
			g_signal_connect( renderer, "state-changed", G_CALLBACK (state_changed_cb), NULL );
		}
	}
}

static void mafw_init( void )
{
    GList *renderers = NULL;

	g_signal_connect( registry, "renderer-added", G_CALLBACK( renderer_added_cb ), NULL );
	renderers = mafw_registry_get_renderers( registry );
	while ( renderers ) {
		renderer_added_cb( registry, G_OBJECT( renderers->data ), NULL );
		renderers = g_list_next( renderers );
	}
}

int main (void)
{
	setlocale( LC_ALL, "" );

	g_type_init();
	notify_init("nowplayingd");

	loop = g_main_loop_new( NULL, FALSE );
	g_assert( loop );

	registry = MAFW_REGISTRY( mafw_registry_get_instance() );
	g_assert( registry );

	signal( SIGINT, sig_handler );
	signal( SIGQUIT, sig_handler );
	signal( SIGTERM, sig_handler );

	mafw_shared_init( registry, NULL );

	mafw_init();

	g_main_loop_run( loop );

	g_main_loop_unref( loop );

	mafw_shared_deinit();

	return EXIT_SUCCESS;
}

