/**************************************************************************
*   Copyright (C) 2010 by Kevron Rees                                     *
*                                                                         *
*                                                                         *
*   This file is free software: you can redistribute it and/or modify     *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation, either version 3 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   libobd is distributed in the hope that it will be useful,             *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   If not, see <http://www.gnu.org/licenses/>.                           *
***************************************************************************/

#include "genericobd2.h"
#include "bluetoothmanager.h"
#include "bluetoothadapter.h"
#include "bluetoothdevice.h"

#include <QObject>
#include <QtPlugin>
#include <QProcess>

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <termios.h>
#include <fcntl.h>


Q_EXPORT_PLUGIN2(genericObd2, GenericObd2)

GenericObd2::GenericObd2(QObject* parent)
	: IPluginProvider (parent)
{
	obd = new ObdThread(this);
	QObject::connect(obd,SIGNAL(pidReceived(QString,QString,int,double)),this,SLOT(obdPidReceived(QString,QString,int,double)),Qt::QueuedConnection);
	//QObject::connect(obd,SIGNAL(supportedPids(QList<int>)),this,SIGNAL(supportedPids(QList<int>)));
	QObject::connect(obd,SIGNAL(troubleCodes(QList<QString>)),this, SIGNAL(troubleCodes(QList<QString>)),Qt::QueuedConnection);
	QObject::connect(obd,SIGNAL(singleShotResponse(QByteArray)), this, SIGNAL(singleShotResponse(QByteArray)),Qt::QueuedConnection);
	QObject::connect(obd,SIGNAL(connected(QString)),this,SIGNAL(connected(QString)),Qt::QueuedConnection);
	QObject::connect(obd,SIGNAL(disconnected()),this,SIGNAL(disconnected()),Qt::QueuedConnection);
	QObject::connect(obd,SIGNAL(consoleMessage(QString)),this,SIGNAL(consoleMessage(QString)),Qt::QueuedConnection);

//	obd->start();
}

void GenericObd2::setPort(QString port)
{
	obd->setPort(port);
}

void GenericObd2::setBaud(int baud)
{
	obd->setBaud(baud);
}

void GenericObd2::addRequest(QByteArray request)
{
	if(request.size() < 4)
		return;
	
	///separate pid from mode
	int mode = request.left(2).toInt(false,16);
	int pid = request.right(2).toInt(false,16);
	
	//qDebug()<<"GenericObd2::addRequest - request: "<<request<<" mode: "<<mode<<" pid: "<<pid;

	obd->addRequest(mode, pid, 1, 0);
	
}

void GenericObd2::singleShotRequest(QByteArray request)
{
	obd->singleShotRequest(request);
}

void GenericObd2::blindSingleShotRequest(QByteArray request)
{
	obd->blindSingleShotRequest(request);
}

void GenericObd2::getSupportedPids()
{
	obd->getSupportedPids();
}

void GenericObd2::connect()
{
	obd->connect();
}

void GenericObd2::disconnect()
{
	obd->disconnect();
}

void GenericObd2::requestTroubleCodes()
{
	obd->requestTroubleCodes();
}

void GenericObd2::obdPidReceived(QString pid,QString val,int set,double time)
{
	//qDebug()<<"GenericObd2::pidReceived(): "<<pid<<" val: "<<val;
	emit pidReceived(QByteArray(pid.toStdString().c_str()),val,set,time);
}

void GenericObd2::configure(QDomNode innerxml)
{
	QDomNode n = innerxml.firstChild();
	while(!n.isNull())
	{
		QDomElement e = n.toElement(); // try to convert the node to an element.
		if(!e.isNull())
		{
			if(e.tagName() == "device")
			{
				QString device = e.text();

				if(device.contains(":"))
					device = connectBtDevice(device);

				setPort(device);
			}
			else if(e.tagName() == "baud")
			{
				setBaud(e.text().toInt());
			}
		}
		n = n.nextSibling();
	}
	obd->start();
	obd->connect();
}

QString GenericObd2::connectBtDevice(QString hwaddy)
{
	OrgBluezManagerInterface manager(
			"org.bluez",
			"/", QDBusConnection::systemBus());

	QDBusObjectPath adapterpath = manager.DefaultAdapter();

	if(adapterpath.path() == "")
		return "";

	OrgBluezAdapterInterface adapter(
			"org.bluez",
			adapterpath.path(),
			QDBusConnection::systemBus());

	QList<QDBusObjectPath> devicepathlist = adapter.ListDevices();

	foreach(QDBusObjectPath path, devicepathlist)
	{
		OrgBluezDeviceInterface device("org.bluez",
		path.path(),
		QDBusConnection::systemBus());

		QVariantMap props = device.GetProperties();

		QString addy = props["Address"].toString();

		if(addy == hwaddy)
		{
			QDBusInterface interface("org.bluez",path.path(),"org.bluez.Serial",QDBusConnection::systemBus());

			QDBusReply<QString> reply = interface.call("Connect","spp");

			return reply.value();
		}

	}

	return "";
}
