/*
 * nobdy
 * Copyright (c) 2010, Kevron Rees.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "runner.h"
#include "pluginloader.h"

#include <QtDebug>

Runner::Runner(QString config, QObject *parent) :
    QObject(parent), requestInProgress(false)
{
	PluginLoader loader(config);
	provider = loader.provider();
	provider->setParent(this);

	///TODO: don't hard code this:
	//provider->setPort("/dev/pts/4");
	//provider->setBaud(9600);

	connect(provider,SIGNAL(connected(QString)),this,SLOT(connected(QString)));
	connect(provider,SIGNAL(pidReceived(QByteArray,QString,int,double)),this,SLOT(pidReceived(QByteArray,QString,int,double)));
	connect(provider,SIGNAL(supportedPids(QList<int>)),this,SLOT(supportedPids(QList<int>)));
	connect(provider,SIGNAL(troubleCodes(QList<QString>)),this,SLOT(troubleCodes(QList<QString>)));
	connect(provider,SIGNAL(consoleMessage(QString)),this,SLOT(consoleOutput(QString)));
	connect(provider,SIGNAL(singleShotResponse(QByteArray)),this,SLOT(singleShotResponse(QByteArray)));

	///load up subscribers:
	subscribers = loader.subscribers();
	foreach(IPluginSubscriber* subscriber, subscribers)
	{
		subscriber->setParent(this);

		foreach(QByteArray pid, subscriber->requests())
		{
			pidSubscriberMap[pid.toUpper()].append(subscriber);
			qDebug()<<"adding subscriber: "<<subscriber->name()<<" to pid: "<<pid;
		}
		connect(subscriber,SIGNAL(singleShotRequest(QByteArray,bool)),this, SLOT(handleSingleRequest(QByteArray,bool)));
		connect(subscriber,SIGNAL(addRequest(QByteArray)), this, SLOT(subscriberAddRequest(QByteArray)));
	}

	///add requests on the provider:
	QList<QByteArray> requests = pidSubscriberMap.keys();

	foreach(QByteArray req, requests)
	{
		provider->addRequest(req);
	}
}

Runner::~Runner()
{
	if(provider)
		delete provider;
	foreach(IPluginSubscriber* subscriber, subscribers)
	{
		if(subscriber)
			delete subscriber;
	}
}

void Runner::pidReceived(QByteArray pid,QString val,int set,double time)
{
	SubscriberList list = pidSubscriberMap[pid];

	//qDebug()<<"Runner::pidReceived(): "<<QString(pid);

	if(!list.size())
	{
		qDebug()<<"Runner: no subscribers for pid: "<<pid;
	}

	///Notify the subscribers
	foreach(IPluginSubscriber* subscriber, list)
	{
		subscriber->pidReceived(pid,val,set,time);
	}
}

void Runner::supportedPids(QList<int> list)
{
	///Notify the subscribers
	foreach(IPluginSubscriber* subscriber, subscribers)
	{
		subscriber->supportedPids(list);
	}
}

void Runner::troubleCodes(QList<QString> codes)
{
	///Notify the subscribers
	foreach(IPluginSubscriber* subscriber, subscribers)
	{
		subscriber->troubleCodes(codes);
	}
}

void Runner::subscriberAddRequest(QByteArray pid)
{
	consoleOutput("Runner: subscriber is adding a request..." + QString(pid));
	IPluginSubscriber* subscriber = qobject_cast<IPluginSubscriber*>(sender());

	if(!subscriber)
	{
		qDebug("faled to cast subscriber");
		return;
	}

	QByteArray key = QByteArray(pid.constData());

	qDebug()<<"key: "<<key;

	if(pidSubscriberMap[key].indexOf(subscriber) < 0)
	{
		printf("%s\n",key.constData());
		pidSubscriberMap[key].append(subscriber);
	}
	else qDebug("subscriber trying to resubscribe to the same pid.  how silly...");

	if(pidSubscriberMap[key].size() == 1)
	{
		///if this is the first subscriber to request this pid, add the pid to the provider loop
		provider->addRequest(key);
	}
	else
	{
		qDebug("pid already requested.  we won't re-request it.  we'll just set the subscriber to get it when it comes");
	}
}

void Runner::connected(QString version)
{
	qDebug()<<"connected to provider "<<provider->name()<<" version: "<<version;
}

void Runner::consoleOutput(QString message)
{
	qDebug()<<"message: "<<message;
}

void Runner::handleSingleRequest(QByteArray req, bool hasReply)
{
	if(!hasReply)
	{
		provider->blindSingleShotRequest(req);
		return;
	}

	if(!singleRequestQueue.size())
	{
		QTimer::singleShot(10,this,SLOT(processRequests()));
	}

	IPluginSubscriber* subscriber = qobject_cast<IPluginSubscriber*>(sender());
	singleRequestQueue.append(QPair<QByteArray,IPluginSubscriber*>(req, subscriber));


}

void Runner::singleShotResponse(QByteArray response)
{
	///this should never happen:
	if(!singleRequestQueue.size())
	{
		qDebug("response without a subscriber in queue.  this should never happen.");
		return;
	}


	singleRequestQueue[0].second->singleRequestReceived(singleRequestQueue[0].first, response);
	singleRequestQueue.removeAt(0);

	requestInProgress=false;
}

void Runner::processRequests()
{
	if(!requestInProgress && singleRequestQueue.size())
	{
		provider->singleShotRequest(singleRequestQueue[0].first);
		requestInProgress=true;
	}

	if(singleRequestQueue.size())
	{
		QTimer::singleShot(10, this, SLOT(processRequests()));
	}
}
