/*
 * nobdy logger plugin
 * Copyright (c) 2010, Kevron Rees.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "logger.h"
#include "codes.h"

#include <iostream>
#include <QtPlugin>
#include <QtDebug>
#include <QFile>
#include <QTimer>
#include <QTime>
#include <QSocketNotifier>

using namespace std;

Q_EXPORT_PLUGIN2(logger, Logger)

Logger::Logger(QObject *parent) :
    IPluginSubscriber(parent),obddb(NULL),gps(NULL),tripzero(NULL)
{
	gpsd = gps_open("127.0.0.1","2947");
	if(!gpsd) return;

	gps_stream(gpsd, WATCH_ENABLE|WATCH_NEWSTYLE, NULL);

	QSocketNotifier *gpsSocket = new QSocketNotifier(gpsd->gps_fd, QSocketNotifier::Read, this);
	connect(gpsSocket,SIGNAL(activated(int)),this,SLOT(pollGps(int)));
}

Logger::~Logger()
{
	if(tripzero) delete tripzero;
	if(obddb) delete obddb;
	if(gps) delete gps;
	if(gpsd)gps_close(gpsd);
}

QList<QByteArray> Logger::requests()
{
	m_requests<<ObdIICodes::EngineLoad()<</*ObdIICodes::IntakeAirTemp()<<*/ObdIICodes::Velocity()<<ObdIICodes::EngineRPM()
		<<ObdIICodes::MAFAirFlowRate()<<ObdIICodes::ThrottlePosition()<<ObdIICodes::EngineCoolantTemp();
	return m_requests;
}

void Logger::configure(QDomNode innerxml)
{
	QString logfile;

	QDomNode n = innerxml.firstChild();
	while(!n.isNull())
	{
		QDomElement e = n.toElement(); // try to convert the node to an element.
		if(!e.isNull())
		{
			if(e.tagName() == "logfile")
				logfile=e.text();
		}
		n = n.nextSibling();
	}
	init(logfile);
}

void Logger::pollGps(int)
{
	if(!gpsd || !gps_waiting(gpsd) )
	{
		qDebug("gspd either not running not valid or has no data.");
		return;
	}

	gps_poll(gpsd);
}

void Logger::pidReceived(QByteArray pid,QString val, int, double)
{
	QDateTime time = QDateTime::currentDateTime();

	double realtime = (double)time.toTime_t() + (double)time.time().msec()/1000.0f;

	results[pid.toUpper()] = val.toAscii();

	if(!hasfullresultset() || !obddb) return;

	if(!tripzero)
	{
		tripzero = new trip(dbfile);
		if(reloadTables)
		{
			tripzero->createTable();
		}

	}

	if(!isTransacting)
	{
		isTransacting = true;
		database.transaction();
		QTimer::singleShot(10000, this, SLOT(commit()));
	}

	obddb->create(results[ObdIICodes::EngineLoad()].toDouble(), results[ObdIICodes::EngineCoolantTemp()].toDouble(), results[ObdIICodes::EngineRPM()].toDouble(),
		     results[ObdIICodes::Velocity()].toDouble(),results[ObdIICodes::IntakeAirTemp()].toDouble(),results[ObdIICodes::MAFAirFlowRate()].toDouble(),
		     results[ObdIICodes::ThrottlePosition()].toDouble(),realtime);

	results.clear();

	if(!gpsd || gpsd->fix.mode < 2)
	{
		//qDebug()<<"gpsd not valid or 3D fix not available. mode: "<<gpsd->fix.mode;
		return;
	}

	double lat, lon;
	lat = gpsd->fix.latitude;
	lon = gpsd->fix.longitude;

	qDebug()<<"Lat: "<<lat<<"| Lon: "<<lon;

	gps->create(lat, lon, realtime);


}

void Logger::supportedPids(QList<int> list)
{
	Q_UNUSED(list);
}

void Logger::troubleCodes(QList<QString> codes)
{
	foreach(QString code, codes)
	{
		qDebug()<<"trouble code: "<<code;
	}
}

void Logger::commit()
{
	qDebug("committing");
	database.commit();
	isTransacting = false;
}

void Logger::init(QString log)
{
	dbfile = log;
	reloadTables = false;
	isTransacting = false;
	database = QSqlDatabase::database();
	if(!QFile::exists(log))
	{
		reloadTables = true;
	}

	obddb = new obd(log);
	gps = new Gps(log);

	if(reloadTables)
	{
		obddb->createTable();
		gps->createTable();
	}
}

bool Logger::hasfullresultset()
{
	bool hasfull=true;
	foreach(QByteArray pid, m_requests)
	{
		hasfull &= results.contains(pid.toUpper());
	}

	return hasfull;
}
