/*
 * This code was created by Jeff Molofee '99
 * (ported to Linux/SDL by Ti Leggett '01)
 * (ported to Maemo/SDL_gles by Till Harbaum '10)
 *
 * If you've found this code useful, please let me know.
 *
 * Visit Jeff at http://nehe.gamedev.net/
 *
 * or for port-specific comments, questions, bugreports etc.
 * email to leggett@eecs.tulane.edu or till@harbaum.org
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <SDL.h>
#include <SDL_image.h>
#include <SDL_gles.h>
#include <GLES/gl.h>
#include "glu.h"

/* screen width, height, and bit depth */
#define SCREEN_WIDTH  800
#define SCREEN_HEIGHT 480
#define SCREEN_BPP     16

/* Set up some booleans */
#define TRUE  1
#define FALSE 0

/* This is our SDL surface */
SDL_Surface *surface;
SDL_GLES_Context *context;

GLfloat xrot;            /* X Rotation */
GLfloat yrot;            /* Y Rotation */
GLfloat zrot;            /* Z Rotation */

float points[45][45][3]; /* The Points On The Grid Of Our "Wave" */
int wiggle_count = 0;    /* Counter Used To Control How Fast Flag Waves */
GLfloat hold;            /* Temporarily Holds A Floating Point Value */

GLuint texture[1];       /* Storage For One Texture */

/* function to release/destroy our resources and restoring the old desktop */
void Quit( int returnCode )
{
    SDL_GLES_DeleteContext(context);  

    /* clean up the window */
    SDL_Quit( );

    /* and exit appropriately */
    exit( returnCode );
}

/* function to load in bitmap as a GL texture */
int LoadGLTextures( )
{
    /* Status indicator */
    int Status = FALSE;

    /* Create storage space for the texture */
    SDL_Surface *TextureImage[1]; 

    /* Load The Bitmap, Check For Errors, If Bitmap's Not Found Quit */
    if ( ( TextureImage[0] = IMG_Load( "/opt/nehegles/tim.png" ) ) )
        {

	    /* Set the status to true */
	    Status = TRUE;

	    /* Create The Texture */
	    glGenTextures( 1, &texture[0] );

	    /* Typical Texture Generation Using Data From The Bitmap */
	    glBindTexture( GL_TEXTURE_2D, texture[0] );

	    /* Generate The Texture */
	    glTexImage2D( GL_TEXTURE_2D, 0, GL_RGB, TextureImage[0]->w,
			  TextureImage[0]->h, 0, GL_RGB,
			  GL_UNSIGNED_BYTE, TextureImage[0]->pixels );

	    /* Linear Filtering */
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
        }

    /* Free up any memory we may have used */
    if ( TextureImage[0] )
	    SDL_FreeSurface( TextureImage[0] );

    return Status;
}

/* to resize the window and reset our view */
int resizeWindow( int width, int height )
{
    /* Height / width ration */
    GLfloat ratio;

    /* Protect against a divide by zero */
    if ( height == 0 )
	height = 1;

    ratio = ( GLfloat )width / ( GLfloat )height;

    /* Setup our viewport. */
    glViewport( 0, 0, ( GLint )width, ( GLint )height );

    /* change to the projection matrix and set our viewing volume. */
    glMatrixMode( GL_PROJECTION );
    glLoadIdentity( );

    /* Set our perspective */
    gluPerspective( 45.0f, ratio, 0.1f, 100.0f );

    /* Make sure we're chaning the model view and not the projection */
    glMatrixMode( GL_MODELVIEW );

    /* Reset The View */
    glLoadIdentity( );

    return( TRUE );
}

/* general OpenGL initialization function */
int initGL( GLvoid )
{
    int x, y; /* Loop variables */

    SDL_GLES_Init(SDL_GLES_VERSION_1_1);

    /* specify size of depth buffer (NEW) */
    SDL_GLES_SetAttribute(SDL_GLES_DEPTH_SIZE, 24);

    context = SDL_GLES_CreateContext();
    SDL_GLES_MakeCurrent(context);

    /* Load in the texture */
    if ( !LoadGLTextures( ) )
	return FALSE;

    /* Enable Texture Mapping ( NEW ) */
    glEnable( GL_TEXTURE_2D );

    /* Enable smooth shading */
    glShadeModel( GL_SMOOTH );

    /* Set the background black */
    glClearColor( 0.0f, 0.0f, 0.0f, 0.5f );

    /* Depth buffer setup */
    glClearDepthf( 1.0f );

    /* Enables Depth Testing */
    glEnable( GL_DEPTH_TEST );

    /* The Type Of Depth Test To Do */
    glDepthFunc( GL_LEQUAL );

    /* Really Nice Perspective Calculations */
    glHint( GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST );

#if 0
    /* Fill the back with the texture. The front will only be wireline */
    glPolygonMode( GL_BACK, GL_FILL );
    glPolygonMode( GL_FRONT, GL_LINE );
#endif

    /* Loop Through The X Plane */
    for ( x = 0; x < 45; x++ )
        {
	    /* Loop Through The Y Plane */
	    for ( y = 0; y < 45; y++ )
                {
		    /* Apply The Wave To Our Mesh */
		    points[x][y][0] = ( float )( ( x / 5.0f ) - 4.5f );
		    points[x][y][1] = ( float )( ( y / 5.0f ) - 4.5f );
		    points[x][y][2] = ( float )( sin( ( ( ( x / 5.0f ) * 40.0f ) / 360.0f ) * 3.141592654 * 2.0f ) );
                }
        }

    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);

    return( TRUE );
}

/* Here goes our drawing code */
int drawGLScene( GLvoid )
{
    /* These are to calculate our fps */
    static GLint T0     = 0;
    static GLint Frames = 0;

    int x, y;                   /* Loop Variables */
 
    /* Clear The Screen And Depth Buffer */
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    /* Reset The Current Matrix */
    glLoadIdentity( );

    /* Translate 17 Units Into The Screen */
    glTranslatef( 0.0f, 0.0f, -12.0f );

    glRotatef( xrot, 1.0f, 0.0f, 0.0f ); /* Rotate On The X Axis */
    glRotatef( yrot, 0.0f, 1.0f, 0.0f ); /* Rotate On The Y Axis */
    glRotatef( zrot, 0.0f, 0.0f, 1.0f ); /* Rotate On The Z Axis */

    glBindTexture( GL_TEXTURE_2D, texture[0] ); /* Select Our Texture */

    /* Floating Point For Vertices and Coords */
    GLfloat quadVertices[4][3];
    GLfloat texCoords[4][2];

    /* Start Drawing Our Quads */
    /* Loop Through The X Plane 0-44 (45 Points) */
    for( x = 0; x < 44; x++ )
	{
	    /* Loop Through The Y Plane 0-44 (45 Points) */
	    for( y = 0; y < 44; y++ )
		{
		    /* Create A Floating Point X Value (top and bottom left) */
		    texCoords[0][0] = texCoords[2][0] = ( float )x / 44.0f;
		    /* Create A Floating Point Y Value (top left and right) */
		    texCoords[0][1] = texCoords[1][1] = ( float )y / 44.0f;
		    /* Create A Floating Point X Value+0.0227f (top and bottom right) */
		    texCoords[1][0] = texCoords[3][0] = ( float )( x + 1 ) / 44.0f;
		    /* Create A Floating Point Y Value+0.0227f (bottom left and right) */
		    texCoords[2][1] = texCoords[3][1] = ( float )( y + 1 ) / 44.0f;

		    /* Top Left Corner */
		    quadVertices[0][0] = points[x][y][0];
		    quadVertices[0][1] = points[x][y][1];
		    quadVertices[0][2] = points[x][y][2];

		    /* Top Right Corner */
		    quadVertices[1][0] = points[x+1][y][0];
		    quadVertices[1][1] = points[x+1][y][1];
		    quadVertices[1][2] = points[x+1][y][2];

		    /* Bottom Left Corner */
		    quadVertices[2][0] = points[x][y+1][0];
		    quadVertices[2][1] = points[x][y+1][1];
		    quadVertices[2][2] = points[x][y+1][2];

		    /* Bottom Right Corner */
		    quadVertices[3][0] = points[x+1][y+1][0];
		    quadVertices[3][1] = points[x+1][y+1][1];
		    quadVertices[3][2] = points[x+1][y+1][2];

		    glVertexPointer(3, GL_FLOAT, 0, quadVertices);
		    glTexCoordPointer(2, GL_FLOAT, 0, texCoords);
		    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
                 }
        }
 
    /* Used To Slow Down The Wave (Every 2nd Frame Only) */
    if( wiggle_count == 2 )
        {
	    /* Loop Through The Y Plane */
	    for( y = 0; y < 45; y++ )
		{
		    /* Store Current Value One Left Side Of Wave */
		    hold = points[0][y][2];
		    /* Loop Through The X Plane */
		    for( x = 0; x < 44; x++)
			{
                            /* Current Wave Value Equals Value To The Right */
			    points[x][y][2] = points[x + 1][y][2];
                        }
		    /* Last Value Becomes The Far Left Stored Value */
		    points[44][y][2] = hold;
                }
	    wiggle_count = 0; /* Set Counter Back To Zero */
        }
    wiggle_count++; /* Increase The Counter */

    xrot += 0.3f; /* Increase The X Rotation Variable */
    yrot += 0.2f; /* Increase The Y Rotation Variable */
    zrot += 0.4f; /* Increase The Z Rotation Variable */

    /* Draw it to the screen */
    SDL_GLES_SwapBuffers( );

    /* Gather our frames per second */
    Frames++;
    {
	GLint t = SDL_GetTicks();
	if (t - T0 >= 5000) {
	    GLfloat seconds = (t - T0) / 1000.0;
	    GLfloat fps = Frames / seconds;
	    printf("%d frames in %g seconds = %g FPS\n", Frames, seconds, fps);
	    T0 = t;
	    Frames = 0;
	}
    }

    xrot += 0.3f; /* X Axis Rotation */
    yrot += 0.2f; /* Y Axis Rotation */
    zrot += 0.4f; /* Z Axis Rotation */

    return( TRUE );
}

int main( int argc, char **argv )
{
    /* main loop variable */
    int done = FALSE;
    /* used to collect events */
    SDL_Event event;
    /* whether or not the window is active */
    int isActive = TRUE;

    /* initialize SDL */
    if ( SDL_Init( SDL_INIT_VIDEO ) < 0 )
	{
	    fprintf( stderr, "Video initialization failed: %s\n",
		     SDL_GetError( ) );
	    Quit( 1 );
	}

    surface = SDL_SetVideoMode(0, 0, 16, SDL_SWSURFACE | SDL_FULLSCREEN);
    SDL_ShowCursor(0);

    SDL_WM_SetCaption("NeHe OpenGL lesson 11", "NeHe11");

    /* initialize OpenGL */
    initGL( );

    /* resize the initial window */
    resizeWindow( SCREEN_WIDTH, SCREEN_HEIGHT );

    /* wait for events */
    while ( !done )
	{
	    /* handle the events in the queue */

	    while ( SDL_PollEvent( &event ) )
		{
		    switch( event.type )
			{
			case SDL_ACTIVEEVENT:
			    /* Something's happend with our focus
			     * If we lost focus or we are iconified, we
			     * shouldn't draw the screen
			     */
			    if ( event.active.gain == 0 )
				isActive = FALSE;
			    else
				isActive = TRUE;
			    break;			    
                        case SDL_MOUSEBUTTONDOWN:
			case SDL_QUIT:
			    /* handle quit requests */
			    done = TRUE;
			    break;
			default:
			    break;
			}
		}

	    /* draw the scene */
	    if ( isActive )
		drawGLScene( );
	}

    /* clean ourselves up and exit */
    Quit( 0 );

    /* Should never get here */
    return( 0 );
}
