#include <iostream>
#include <math.h>
#include <QtGui>

#include "Player.h"

Engine::Engine(Player* parent)
{
    this->parent = parent;

    QDBusConnection::systemBus().connect("", "", "org.bluez.AudioSink", "Connected",
                                         this, SLOT(onHeadsetConnected()));
    QDBusConnection::systemBus().connect("", "", "org.bluez.AudioSink", "Disconnected",
                                         this, SLOT(onHeadsetDisconnected()));
    QDBusConnection::systemBus().connect("", "", "org.bluez.Headset", "Connected",
                                         this, SLOT(onHeadsetConnected()));
    QDBusConnection::systemBus().connect("", "", "org.bluez.Headset", "Disconnected",
                                         this, SLOT(onHeadsetDisconnected()));

    QDBusConnection::systemBus().connect("", "/org/freedesktop/Hal/devices/platform_headphone",
                                         "org.freedesktop.Hal.Device", "PropertyModified",
                                         this, SLOT(jackChanged()));

    QDBusConnection::systemBus().connect("", "", "org.freedesktop.Hal.Device", "Condition",
                                         this, SLOT(onHeadsetButtonPressed(QDBusMessage)));


    headsetIsConnected = false;
    jackChanged();


    // Phonon init
    audioOutput = new Phonon::AudioOutput(Phonon::MusicCategory);
    mediaObject = new Phonon::MediaObject();

    seekSlider = new Phonon::SeekSlider();
    seekSliderSimple = new Phonon::SeekSlider();
    seekSliderSimple->setStyleSheet("QSlider::groove {background: rgb(28, 28, 28);"
                                    "height: 56px; border-radius: 5px;}"
                                    "QSlider::handle {background: white;"
                                    "width: 24px; border-radius: 5px;}");
    Phonon::createPath(mediaObject,audioOutput);

    seekSlider->setIconVisible(false);
    seekLayout = new QVBoxLayout;
    timeLayout = new QHBoxLayout;
    currentTime = new QLabel("0:00");
    totalTime = new QLabel(" /  0:00");
    //seekLayout->addWidget(seekSlider);
    timeLayout->addWidget(currentTime);
    timeLayout->addWidget(totalTime);
    timeLayout->setAlignment(Qt::AlignRight|Qt::AlignVCenter);
    seekLayout->addLayout(timeLayout);
    seekLayout->addWidget(seekSlider);

    seekSimpleLayout = new QHBoxLayout;
    seekSimpleLayout->addWidget(seekSliderSimple);

    useSysVol = parent->config->useSysVol;
    // Use system volume
    if (useSysVol)
    {
        sysVolSet = false;
        setSysVol = "dbus-send --type=method_call "
                "--dest=com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer "
                "/com/nokia/mafw/renderer/gstrenderer "
                "com.nokia.mafw.extension.set_extension_property "
                "string:volume variant:uint32:";
        interface = new QDBusInterface("com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer",
                                       "/com/nokia/mafw/renderer/gstrenderer",
                                       "com.nokia.mafw.extension",
                                       QDBusConnection::sessionBus(), this);
        sysVolRequest = interface->call("get_extension_property", "volume");
        sysVol = sysVolRequest.arguments().at(1).value<QDBusVariant>().variant().toInt();
        if (sysVol > 21)
        {
            sysVol = (sysVol - 21) / 79;
            sysVol = int(log(pow(sysVol, 0.69) + 1) / log(2) * 100 + 0.5);
        }

        QDBusConnection::sessionBus().connect("com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer",
                                              "/com/nokia/mafw/renderer/gstrenderer",
                                              "com.nokia.mafw.extension",
                                              "property_changed",
                                              this, SLOT(onPropertyChanged(const QDBusMessage &)));
    }

    volLayout = new QVBoxLayout;
    volumeButton = new QPushButton(QIcon("/opt/necroplayer2/icons/32x32/volume.png"),"");
    volumeButton->setCheckable(true);
    volumeButton->setStyleSheet("background-color: transparent;");
    volLayout->addWidget(volumeButton);
    if (useSysVol)
    {
        sysVolSlider = new QSlider();
        //sysVolSlider->setInvertedAppearance(true);
        sysVolSlider->setMaximum(100);
        sysVolSlider->setSliderPosition(sysVol);
        sysVolSlider->setSizePolicy(QSizePolicy::Maximum,QSizePolicy::Expanding);
        sysVolSlider->setOrientation(Qt::Vertical);
        volLayout->addWidget(sysVolSlider);
    }
    else
    {
        volumeSlider = new Phonon::VolumeSlider();
        volumeSlider->setAudioOutput(audioOutput);
        volumeSlider->setMuteVisible(false);
        volumeSlider->setSizePolicy(QSizePolicy::Maximum,QSizePolicy::Expanding);
        volumeSlider->setOrientation(Qt::Vertical);
        volLayout->addWidget(volumeSlider);
    }
    volLayout->setSpacing(0);

    connect(volumeButton, SIGNAL(clicked()), SLOT(volumeButtonClicked()));
    if (useSysVol)
    {
        connect(sysVolSlider, SIGNAL(sliderMoved(int)), this, SLOT(changeSysVol()));
        connect(sysVolSlider, SIGNAL(sliderReleased()), this, SLOT(changeSysVol()));
    }

    // meta tag resolver
    mediaObjectForMetaInfos = new Phonon::MediaObject();
    connect(mediaObjectForMetaInfos, SIGNAL(stateChanged(Phonon::State, Phonon::State)),
            this, SLOT(metaInfosUpdated(Phonon::State, Phonon::State)));

    // media
    connect(mediaObject, SIGNAL(stateChanged(Phonon::State, Phonon::State)),
            SLOT(mediaObjectStateChanged(Phonon::State)));
    connect(mediaObject, SIGNAL(finished()), SLOT(playEnded()));
    connect(mediaObject, SIGNAL(tick(qint64)), SLOT(refreshPlayedTime()));

    listMedias.clear();
    
    // default value
    currentPlayedMedia = -1;
    lastUpdated = -1;
    
    init_position = 0;
    initial = false;
    initialLoadingState = true;
    stopAfterLoading = false;
    metaInfosUpdating = false;
}

Engine::~Engine()
{   
    //qDebug()<<"Engine destructor";
    if (volumeButton)
        delete volumeButton;
    //if (sysVolSlider)
    //    delete sysVolSlider;
    //if (volumeSlider)
    //    delete volumeSlider;
    if (volLayout)
        delete volLayout;
    if (audioOutput)
        delete audioOutput; audioOutput = NULL;
    if (mediaObject)
        delete mediaObject; mediaObject = NULL;
    if (seekSlider)
        delete seekSlider; seekSlider = NULL;
    if (currentTime)
        delete currentTime; currentTime = NULL;
    if (totalTime)
        delete totalTime; totalTime = NULL;
    if (mediaObjectForMetaInfos)
        delete mediaObjectForMetaInfos; mediaObjectForMetaInfos = NULL;
    if (timeLayout)
        delete timeLayout;
    if (seekLayout)
        delete seekLayout; seekLayout = NULL;

}

void
Engine::stop()
{
    if (mediaObject)
    {
        if (mediaObject->state() == Phonon::LoadingState && !initialLoadingState)
        {
            stopAfterLoading = true;
            return;
        }
        if (mediaObject->state() == Phonon::StoppedState ||
                mediaObject->state() == Phonon::ErrorState ||
                initialLoadingState)
        {
            if (parent->ui->getPlaylist().getBoldRow() > -1)
            {
                parent->ui->getPlaylist().setBold(parent->ui->getPlaylist().getBoldRow(), false);
                parent->ui->getPlaylist().unsetBoldRow();
            }
        }

        mediaObject->stop();
        currentPlayedMedia = parent->ui->getPlaylist().getBoldRow();
        if (currentPlayedMedia == -1)
        {
            parent->ui->setInfos("\n", "");
            totalTime->setText(" /  0:00");
        }

        currentPlayedMedia = -1;
        parent->ui->getControls().setPlayIcon(Phonon::PlayingState);
        parent->ui->getSimpleView().setPlayIcon(Phonon::PlayingState);
        currentTime->setText("0:00");
        if (parent->config->autoaddons)
        {
            if (mediaObject->state() != Phonon::StoppedState &&
                    mediaObject->state() != Phonon::LoadingState &&
                    mediaObject->state() != Phonon::ErrorState)
                parent->ui->hideSongAddOns();
        }
//        parent->ui->setWindowTitle("NecroPlayer2");
    }
}

/**
 * Return the row played, or -1 if an error occured
 */
int
Engine::play(int row, short clickAction)
{
    if (row > 0) initial = false;

    if (currentPlayedMedia != -1 && row == -1)
    {
        switch (mediaObject->state())
        {
        case Phonon::PausedState:
            row = currentPlayedMedia;
            if (mediaObject)
            {
                mediaObject->play();
                parent->ui->getControls().setPlayIcon(Phonon::PausedState);
                parent->ui->getSimpleView().setPlayIcon(Phonon::PausedState);
            }
            break;
        case Phonon::PlayingState:
            mediaObject->pause();
            parent->ui->getControls().setPlayIcon(Phonon::PlayingState);
            parent->ui->getSimpleView().setPlayIcon(Phonon::PlayingState);
            row = currentPlayedMedia;
            break;
        default:
            break;
        }
    }
    else
    {
        if (row < listMedias.count())
        {
            currentPlayedMedia = parent->ui->getPlaylist().getBoldRow();
            if (currentPlayedMedia > -1 && currentPlayedMedia != row)
            {
                parent->ui->getPlaylist().setBold(currentPlayedMedia, false);
            }

            mediaObject->setCurrentSource(listSources[row]);
            mediaObject->setTickInterval(1000);
            mediaObject->setTransitionTime(1000);
            currentTime->setText("0:00");
            seekSlider->setMediaObject(mediaObject);
            seekSliderSimple->setMediaObject(mediaObject);

            if (currentPlayedMedia == -1 or clickAction == 2)
                parent->ui->getPlaylist().setBold(row, true);
            currentPlayedMedia = row;

            mediaObject->play();
            parent->ui->getControls().setPlayIcon(Phonon::PausedState);
            parent->ui->getSimpleView().setPlayIcon(Phonon::PausedState);
            if (parent->config->autoaddons)
            {
                if (!clickAction && parent->ui->getSongAddons().isHidden())
                {
                    parent->ui->showSongAddOns();
                }
            }
        }
        else
        {
            row = -1;
        }
    }

    return row;
}

void Engine::mediaObjectStateChanged(Phonon::State state)
{
    if (initialLoadingState)
    {
        initialLoadingState = false;
    }
    if (stopAfterLoading)
    {
        stopAfterLoading = false;
        this->stop();
    }

    if (state == Phonon::ErrorState)
    {
        parent->next();
    }

    if (state == Phonon::PlayingState && (!parent->config->useTags || parent->config->updateOnPlay))
    {
        if (mediaObject->currentSource().type() != Phonon::MediaSource::Invalid)
        {
            QMultiMap<QString, QString> metaData = mediaObject->metaData();
            Media media;

            if (parent->config->useTags)
            {
                media.title = metaData.value("TITLE");
                media.artist = metaData.value("ARTIST");
                media.album = metaData.value("ALBUM");
            }
            if (media.title == "")
            {
                int nn = listSources.indexOf(mediaObject->currentSource());
                if (nn < 0) return;
                QFileInfo fi(listUris[nn]);
                media.title = fi.fileName();
            }

            media.totalTime = Engine::msToString(mediaObject->totalTime());

            parent->ui->changeItem(media.title, media.artist, media.totalTime, currentPlayedMedia, 1);
            listMedias.replace(currentPlayedMedia, media);
            currentInfosUpdated(currentPlayedMedia);
        }
    }
}

void Engine::addMedia(QStringList files)
{
    if (files.isEmpty())
        return;
    
    //int index = listSources.size();
    QFileInfo fi;
    Media tmp_media;

    foreach (QString string, files) 
    {
        Phonon::MediaSource source(string);
        listSources.append(source);
        listUris.append(string);
        fi.setFile(string);
        parent->ui->addItem(fi.fileName());
        tmp_media.title = fi.fileName();
        tmp_media.artist = "";
        tmp_media.album = "";
        listMedias.append(tmp_media);
    }
}


void Engine::startMetaUpdater(bool reload)
{
    if (!listSources.isEmpty())
    {
        if (reload)
        {
            mediaObjectForMetaInfos->setCurrentSource(listSources.at(0));
        }
        else
        {
            int index_cur = listSources.indexOf(mediaObjectForMetaInfos->currentSource());
            if (index_cur >= lastUpdated || index_cur == -1)
            {
                mediaObjectForMetaInfos->setCurrentSource(listSources.at(lastUpdated + 1));
            }
        }
    }
}

void
Engine::dbusVolumeButtonClicked()
{
    if (volumeButton->isChecked())
    {
        volumeButton->setChecked(false);
    }
    else
    {
        volumeButton->setChecked(true);
    }
    volumeButtonClicked();
}

void
Engine::volumeButtonClicked()
{
    if (volumeButton->isChecked())
    {
        volumeButton->setIcon(QIcon("/opt/necroplayer2/icons/32x32/muted.png"));
        audioOutput->setMuted(true);
    }
    else
    {
        volumeButton->setIcon(QIcon("/opt/necroplayer2/icons/32x32/volume.png"));
        audioOutput->setMuted(false);
    }
}

void
Engine::onPropertyChanged(const QDBusMessage &msg)
{
    if (msg.arguments().at(0).toString() == "volume")
    {
        if (!sysVolSlider->isSliderDown())
        {
            if (!sysVolSet)
            {
                sysVol = msg.arguments().at(1).value<QDBusVariant>().variant().toInt();
                if (sysVol > 21)
                {
                    sysVol = (sysVol - 21) / 79;
                    sysVol = int(log(pow(sysVol, 0.69) + 1) / log(2) * 100 + 0.5);
                }
                else
                {
                    sysVol = int(0.314 * sysVol + 0.5);
                }
                sysVolSlider-> setSliderPosition(sysVol);
            }
            else
            {
                sysVolSet = false;
            }

        }
    }
}

void
Engine::changeSysVol()
{
    sysVol = sysVolSlider->value();
    if (sysVol > 6)
    {
        sysVol = int(pow(pow(2, sysVol / 100) - 1, 1.45) * 79 + 21.5);
    }
    else
    {
        sysVol = int(sysVol / 0.314 + 0.5);
    }
    sysVolSet = true;
    system((setSysVol + QString::number(sysVol)).toUtf8());
}

void
Engine::refreshPlayedTime()
{
    currentTime->setText(Engine::msToString(mediaObject->currentTime()));
    //  totalTime->setText(" / " + Engine::msToString(mediaObject->totalTime()));
    
}

Media
Engine::getInfos(unsigned int row)
{
    Media media;
    if (row < (unsigned int)listMedias.count())
    {
        media = listMedias.at(row);
    }
    return media;
}

QString
Engine::getUri(unsigned int row)
{
    QString ret = "";
    if (row < (unsigned int)listUris.count())
    {
        ret = listUris.at(row);
    }
    return ret;
}

void
Engine::setMetaInfosUpdating()
{
    metaInfosUpdating = true;
}

void
Engine::metaInfosUpdated(Phonon::State newState, Phonon::State oldState)
{
    metaInfosUpdating = true;
    if (parent->config->useTags && !parent->config->updateOnPlay)
    {
        if (oldState == Phonon::ErrorState)
        {
            QMetaObject::invokeMethod(this, "startMetaUpdater", Qt::QueuedConnection);
            return;
        }

    //    if (newState != Phonon::StoppedState && newState != Phonon::PausedState && newState != Phonon::ErrorState)
        if (newState == Phonon::LoadingState)
        {
            return;
        }

    //    if (mediaObjectForMetaInfos->currentSource().type() == Phonon::MediaSource::Invalid)
    //    {
    //        return;
    //    }

    //    if (newState == Phonon::ErrorState)
    //    {
    //        mediaObjectForMetaInfos->setCurrentSource(mediaObjectForMetaInfos->currentSource());
    //    }

        Media media;

        if(newState != Phonon::ErrorState &&
                parent->config->useTags && !parent->config->updateOnPlay)
        {
            QMultiMap<QString, QString> metaData = mediaObjectForMetaInfos->metaData();

            media.title = metaData.value("TITLE");
            if (media.title == "")
            {
                int nn=listSources.indexOf(mediaObjectForMetaInfos->currentSource());
                if (nn < 0) return;
                QFileInfo fi(listUris[nn]);
                media.title = fi.fileName();
            }
            media.artist = metaData.value("ARTIST");
            media.album = metaData.value("ALBUM");
            media.totalTime = Engine::msToString(mediaObjectForMetaInfos->totalTime());
        }
        else
        {
            int nn = listSources.indexOf(mediaObjectForMetaInfos->currentSource());
            if (nn < 0) return;
            QFileInfo fi(listUris[nn]);
            media.title = fi.fileName();
            media.totalTime = Engine::msToString(mediaObjectForMetaInfos->totalTime());
            media.album = "";
            media.artist = "";
        }

        int curr_index = listSources.indexOf(mediaObjectForMetaInfos->currentSource());

        if (curr_index >= 0 && curr_index < listSources.size())
        {
            listMedias.replace(curr_index,media);

            if (newState == Phonon::ErrorState)
            {
                parent->ui->changeItem(media.title, media.artist, media.totalTime, curr_index, -1);
            }
            else
            {
                if (curr_index == parent->ui->getPlaylist().getBoldRow())
                    parent->ui->changeItem(media.title, media.artist, media.totalTime, curr_index, 1);
                else
                    parent->ui->changeItem(media.title, media.artist, media.totalTime, curr_index);
            }

            lastUpdated = curr_index;
            int index = curr_index + 1;
            if (index < listSources.size())
            {
                mediaObjectForMetaInfos->setCurrentSource(listSources.at(index));
            }
            else
            {
                metaInfosUpdating = false;
            }
        }
        else
        {
            metaInfosUpdating = false;
        }

        if (curr_index == parent->config->lastPlayed && !parent->getBoolRestoredPlaylist())
        {
            parent->restoreLastPlayed();
        }
    }
    else if (!parent->getBoolRestoredPlaylist())
    {
        parent->restoreLastPlayed();
        metaInfosUpdating = false;
    }
    else
    {
        metaInfosUpdating = false;
    }
}

void Engine::currentInfosUpdated(int row)
{
//    Phonon::State newState = mediaObject->state();
//    if (newState != Phonon::ErrorState)
//    {
        currentPlayedMedia = row;
        if (!listMedias[row].totalTime.isEmpty())
        {
            totalTime->setText(" /  " + listMedias[row].totalTime);
        }
        else
        {
            totalTime->setText("");
        }

        Media media = getInfos(row);

        QString s;
        QString t;
        t = "";

        if (media.artist.isEmpty())
        {
            s = media.title + "\n";
            //parent->ui->setWindowTitle(media.title);
        }
        else
        {
            s = media.artist + "\n" + media.title;
            //parent->ui->setWindowTitle(media.artist+" - "+ media.title);

            if (!media.album.isEmpty())
            {
                QFileInfo checkFile(QDesktopServices::storageLocation(QDesktopServices::HomeLocation) +
                                    "/.covers/" + media.artist.toLower() + " - " + media.album.toLower() + ".jpg");
                if (checkFile.exists())
                {
                    t = QDesktopServices::storageLocation(QDesktopServices::HomeLocation) +
                            "/.covers/" + media.artist.toLower() + " - " + media.album.toLower() + ".jpg";
                    QByteArray imageFormat = QImageReader::imageFormat(t);
                    if (imageFormat.isEmpty())
                    {
                        t = "";
                    }
                }
            }
            if (listUris[row].contains("http://"))
            {
                t = "stream";
            }
        }
        parent->ui->setInfos(s, t);
//    }
//    else
//    {
//        parent->next(false);
//    }
}

void
Engine::playEnded()
{
    parent->next();
}

int
Engine::delMedia(int row)
{
    int ret = 0;
    if (row < listMedias.count())
    {
        listMedias.remove(row);
        listSources.remove(row);
        listUris.remove(row);
        ret = 1;
        if(lastUpdated>listMedias.count()-1)
            lastUpdated=listMedias.count()-1;
    }
    return ret;
}

int Engine::moveMedia(int row, bool moveUp)
{
    int moved=0;
    Media tmpMedia;
    Phonon::MediaSource tmpSource;
    QString tmpUri;

    if((row>0)&&(moveUp))
    {
        tmpMedia=listMedias.at(row);
        tmpSource=listSources.at(row);
        tmpUri=listUris.at(row);

        listMedias.replace(row,listMedias.at(row-1));
        listSources.replace(row,listSources.at(row-1));
        listUris.replace(row,listUris.at(row-1));

        listMedias.replace(row-1,tmpMedia);
        listSources.replace(row-1,tmpSource);
        listUris.replace(row-1,tmpUri);
        moved=1;
    }

    if((row<listMedias.count()-1)&&(!moveUp))
    {
        tmpMedia=listMedias.at(row);
        tmpSource=listSources.at(row);
        tmpUri=listUris.at(row);

        listMedias.replace(row,listMedias.at(row+1));
        listSources.replace(row,listSources.at(row+1));
        listUris.replace(row,listUris.at(row+1));

        listMedias.replace(row+1,tmpMedia);
        listSources.replace(row+1,tmpSource);
        listUris.replace(row+1,tmpUri);
        moved=-1;
    }

    return moved;
}

void
Engine::clearPlayList()
{
    parent->ui->clearPlaylist();
    listMedias.clear();
    listSources.clear();
    listUris.clear();
    lastUpdated=-1;
    if (getReadingState() == Phonon::StoppedState ||
            getReadingState() == Phonon::LoadingState)
    {
        parent->ui->setInfos("\n", "");
        getTotalTime()->setText(" /  0:00");
    }
}

QString
Engine::msToString(qint64 ms)
{
    QString ret = "";
    qint64 tmp = ms / 1000;
    int seconds = 0;
    int minutes = 0;
    int hours = 0;
    int tmpMins = tmp/60;
    if (tmpMins > 59)
    {
        hours = tmpMins/60;
        minutes = tmpMins%60;       
    }
    else
    {
        minutes = tmpMins;
    }
    seconds = tmp%60;

    if (hours > 0)
    { // with hours
        /**
         * We test if minutes/10 == 0 
         * to not have a display like this -> 1:5:12
         * but a display like this -> 1:05:12
         */
        ret += QString::number(hours) + ":";
        if (minutes/10 == 0)
        {
            ret += "0";
        }
        ret += QString::number(minutes) + ":";
        if (seconds/10 == 0)
        {
            ret += "0";
        }
        ret += QString::number(seconds);
    }
    else
    { // without hours
        ret += QString::number(minutes) + ":";
        if (seconds/10 == 0)
        {
            ret += "0";
        }
        ret += QString::number(seconds);
    }
    return ret;
}

void Engine::setInitialPosition(Phonon::State state)
{
    // qDebug()<<"State: "<<state;



    if(initial && state == Phonon::PlayingState)
    {
        initial=false;

        mediaObject->seek(init_position);
    }
    if(initial && state == Phonon::PausedState)
    {
        mediaObject->seek(init_position);
    }



}

void Engine::restoreState(int row, qint64 time)
{
    //qDebug()<<row<<" :::: "<<time;
    if(row < listMedias.count())
    {
        //qDebug()<<listSources.size();
        mediaObject->setCurrentSource(listSources[row]);
        mediaObject->setTickInterval(1000);
        mediaObject->setTransitionTime(1000);
        currentTime->setText("0:00");
        seekSlider->setMediaObject(mediaObject);
        seekSliderSimple->setMediaObject(mediaObject);

        currentPlayedMedia=row;

        parent->ui->getPlaylist().setBold(row,true);

        mediaObject->pause();
        parent->ui->getControls().setPlayIcon(Phonon::PlayingState);
        parent->ui->getSimpleView().setPlayIcon(Phonon::PlayingState);

        initial=true;
        init_position=time;



    }
}

int Engine::getNUris()
{
    return listUris.size();

}

void Engine::jackChanged()
{
    QDBusInterface interface("org.freedesktop.Hal",
                                 "/org/freedesktop/Hal/devices/platform_headphone",
                                 "org.freedesktop.Hal.Device", QDBusConnection::systemBus(), this);

    if (!interface.isValid()) return;

    if (QDBusReply<bool>(interface.call("GetProperty", "button.state.value")))
    {
        QTimer::singleShot(1000, this, SLOT(jackConnected()));
    }
    else if (isHeadset() && headsetIsConnected)
    {
        onHeadsetDisconnected();
        headsetIsConnected = false;
    }
}

void Engine::jackConnected()
{
    if (isHeadset() && !headsetIsConnected)
    {
        onHeadsetConnected();
        headsetIsConnected = true;
    }
}

bool Engine::isHeadset()
{
    QDBusInterface interface("org.freedesktop.Hal",
                                 "/org/freedesktop/Hal/devices/platform_soc_audio_logicaldev_input",
                                 "org.freedesktop.Hal.Device", QDBusConnection::systemBus(), this);

    if (!interface.isValid()) return false;

    return QDBusReply<QStringList>
            (interface.call("GetProperty", "input.jack.type")).value().contains("headphone");
}

void Engine::onHeadsetConnected()
{
    if (!parent->config->resumeOnPlugged)
    {
        return;
    }

    if (mediaObject->state() == Phonon::PausedState)
    {
        QTimer::singleShot(2000, this, SLOT(resumeOnHeadsetConnected()));
    }
}

void Engine::resumeOnHeadsetConnected()
{
    mediaObject->play();
    parent->ui->getControls().setPlayIcon(Phonon::PausedState);
    parent->ui->getSimpleView().setPlayIcon(Phonon::PausedState);
}

void Engine::onHeadsetDisconnected()
{
    if (!parent->config->pauseOnUnplugged)
    {
        return;
    }

    if (mediaObject->state() == Phonon::PlayingState ||
            (mediaObject->state() == Phonon::LoadingState && !initialLoadingState))
    {
        mediaObject->pause();
        parent->ui->getControls().setPlayIcon(Phonon::PlayingState);
        parent->ui->getSimpleView().setPlayIcon(Phonon::PlayingState);
    }
}

void Engine::onHeadsetButtonPressed(QDBusMessage msg)
{
    if (msg.arguments()[0] == "ButtonPressed")
    {
        if (msg.arguments()[1] == "play-cd" || msg.arguments()[1] == "pause-cd")
            parent->play(-1, 0);
        else if (msg.arguments()[1] == "stop-cd")
            parent->stop();
        else if (msg.arguments()[1] == "next-song")
            parent->next(true);
        else if (msg.arguments()[1] == "previous-song")
            parent->prev();
        else if (msg.arguments()[1] == "fast-forward")
            mediaObject->seek(mediaObject->currentTime() + 3000);
        else if (msg.arguments()[1] == "rewind")
            mediaObject->seek(mediaObject->currentTime() - 3000);
        else if (msg.arguments()[1] == "phone")
            phoneButtonPressed();
        else if (msg.arguments()[1] == "jack_insert" &&
                 msg.path() == "/org/freedesktop/Hal/devices/platform_soc_audio_logicaldev_input")
            jackChanged();
    }
}

void Engine::phoneButtonPressed()
{
    if (parent->config->phoneButtonFn == 0)
        parent->play(-1, 0);
    else if (parent->config->phoneButtonFn == 1)
        parent->stop();
    else if (parent->config->phoneButtonFn == 2)
        parent->next(true);
    else if (parent->config->phoneButtonFn == 3)
        parent->prev();
}
