#include <iostream>
#include "Player.h"
#include "UI.h"
#include "Engine.h"

Player::Player(int & argc, char** argv) : QApplication(argc, argv)
{
    QString lang = qgetenv("LANG").constData();
    QTranslator translator;
    translator.load("necroplayer2_" + lang, "/opt/necroplayer2/translations");
    installTranslator(&translator);

    setApplicationName("necroplayer2");
    timerPreviousOrNextClicking = new QTimer();
    timerStopClicking = new QTimer();
    previousOrNextClicking = false;
    playIfNoMoreClickActive = false;
    stopClicking = false;
    musicPath = "";
    config = new Config();
    config->read();
    engine = new Engine(this);
    ui = new UI(this, config->musicFolders[config->musicIndex],
                config->musicDescriptions[config->musicIndex]);
    if (config->lastPlayed > -1)
    {
        ui->getControls().blockButtons();
        ui->getSimpleView().blockButtons();
        engine->setMetaInfosUpdating();
        unblockButtons();
    }
    restoreConf();
    restorePlaylist();
    if (argc > 1)
    {
        QStringList files;
        for (int i = 1; i < argc; i++)
            files.append(argv[i]);
        addMedia(files);
    }
    
    restoredPlaylist = false;
    engine->startMetaUpdater(true);

    connect(this, SIGNAL(aboutToQuit()), SLOT(onQuit()));
    connect(timerPreviousOrNextClicking, SIGNAL(timeout()), SLOT(previousOrNextTimeOut()));
    connect(timerStopClicking, SIGNAL(timeout()), SLOT(stopTimeOut()));

    timerPreviousOrNextClicking->setSingleShot(true);
    timerStopClicking->setSingleShot(true);

    dbus = new DBusService(this);

//    if (config->defaultPlayer)
//    {
//        dbusMP = new DBusServiceMP(this);
//    }
}

Player::~Player()
{
    if (engine)
        delete engine;
    if (ui)
        delete ui;
}

void Player::unblockButtons()
{
    if (!engine->getMetaInfosUpdating())
    {
        ui->getControls().unblockButtons();
        ui->getSimpleView().unblockButtons();
    }
    else
    {
        QTimer::singleShot(200, this, SLOT(unblockButtons()));
    }
}

void Player::restoreLastPlayed()
{
    restoredPlaylist = true;
    if(config->lastPlayed >= 0)
    {
        ui->setBold(config->lastPlayed, true);
        if(config->savePosition)
        {
            if(config->position > 0)
                engine->restoreState(config->lastPlayed,config->position);
        }
    }
}

void Player::addMedia(QStringList ss)
{
    engine->addMedia(ss);
}

void Player::onQuit()
{
    saveConf();
    if (config->savePlaylistOnQuit)
        savePlaylist();
}

void Player::restorePlaylist(QString filename)
{
    QStringList files;
    if(filename=="")
        filename=config->configDir.path() + "/necroplayer2_list";
    QFile file(filename);
    if (file.open(QIODevice::ReadOnly | QIODevice::Text))
    {
        QTextStream in(&file);
        QString line = in.readLine();
        while (!line.isNull())
        {
            //ignore ext. m3u information
            if(!line.startsWith("#"))
            {
                files.append(line);
            }
            line = in.readLine();
        }
    }
    engine->addMedia(files);
}

void Player::openPlaylist_pls(QString filename)
{
    QStringList files;
    QFile file(filename);
    if (file.open(QIODevice::ReadOnly | QIODevice::Text))
    {
        QTextStream in(&file);
        QString line = in.readLine();
        while (!line.isNull())
        {
            if(line.startsWith("File"))
            {
                line.remove(QRegExp("^File\\d+\\="));
                files.append(line);
            }
            line = in.readLine();
        }
    }
    engine->addMedia(files);
}


void Player::savePlaylist(QString filename)
{
    if(filename == "")
        filename=config->configDir.path() + "/necroplayer2_list";
    
    QFile file(filename);
    int count=engine->getNUris();
    
    QTextStream out(&file);
    if (file.open(QIODevice::WriteOnly | QIODevice::Text))
    {
        for (int i = 0; i < count; i++)
        {
            out << engine->getUri(i) << '\n';
        }
    }
    file.close();

}


void Player::restoreConf()
{
    ui->getControls2().setRandom(config->random);
    ui->getControls2().setRepeat(config->repeat);
    useAppVol = false;
    if (!config->useSysVol)
    {
        engine->setVolume(config->volume);
        useAppVol = true;
    }
}

void Player::play(int row, short clickAction)
{
    if (!stopClicking || clickAction == 2)
    {
        switch (engine->getReadingState())
        {
        case Phonon::PlayingState:
            engine->play(row, clickAction);
            break;
        case Phonon::PausedState:
            engine->play(row, clickAction);
            break;
        default:
            if (row == -1)
            {
                row = ui->getPlaylist().getBoldRow();
                if (row == -1) { row = 0; }
            }
            if (row > -1 && row < ui->playlistCount())
            {
                engine->play(row, clickAction);
            }
            break;
        }
    }
}

void
Player::stop()
{
    engine->stop();
    stopClicking = true;
    timerStopClicking->stop();
    timerStopClicking->start(500);
}

void
Player::stopTimeOut()
{
    stopClicking = false;
}


void
Player::previousOrNextTimeOut()
{
    previousOrNextClicking = false;
}

void
Player::playIfNoMoreClick()
{
    playIfNoMoreClickActive = true;
    if (!previousOrNextClicking)
    {
        play(ui->getPlaylist().getBoldRow(), 1);
        playIfNoMoreClickActive = false;
    }
    else
    {
        QTimer::singleShot(500, this, SLOT(playIfNoMoreClick()));
    }
}

void
Player::next(bool click)
{
    previousOrNextClicking = true;

    int n = ui->getPlaylist().getBoldRow();
    int n_next = n;
    int nPlaylistCount = ui->playlistCount();
    if (config->random)
    {
        if ( nPlaylistCount > 1)
        {
            while (n_next == n )
            {
                n_next = qrand()%nPlaylistCount;
                //qDebug()<<qrand();
                //qDebug()<<qrand()%nPlaylistCount;
            }
        }
        else
        { // <=1
            n_next = 0;
        }     
    }
    else
    {
        n_next++;
        if (n_next >= nPlaylistCount)
        {
            if (config->repeat)
            {
                n_next = 0;
            }
            else
            {
                n_next = nPlaylistCount - 1;
                if (!click)
                {
                    engine->stop();
                }
                return;
            }
        }
    }
    if (nPlaylistCount > 0) // if not empty playlist
    {
        if (n >= 0)
        {
            ui->setBold(n, false);
        }
        ui->setBold(n_next, true);
        timerPreviousOrNextClicking->stop();
        timerPreviousOrNextClicking->start(500);
        if (!click)
        {
            play(n_next, 0);
        }
        else if (engine->getReadingState() != Phonon::StoppedState &&
                 engine->getReadingState() != Phonon::ErrorState &&
                 !engine->getInitialLoadingState() &&
                 !playIfNoMoreClickActive)
        {
            playIfNoMoreClickActive = true;
            QTimer::singleShot(500, this, SLOT(playIfNoMoreClick()));
        }
    }
    else
    {
        engine->stop();
    }
}

void
Player::prev()
{
    previousOrNextClicking = true;

    if (config->random)
    {
        next(true);
    }
    else
    {
        int n = ui->getPlaylist().getBoldRow();
        int n_prev = n;
        int nPlaylistCount=ui->playlistCount();
        n_prev--;
        if (n_prev < 0)
        {
            if (config->repeat)
            {
                n_prev = nPlaylistCount - 1;
            }
            else
            {
                n_prev = 0;
                return;
            }
        }

        if (nPlaylistCount > 0) //not empty playlist
        {
            if (n >= 0)
            {
                ui->setBold(n, false);
            }
            ui->setBold(n_prev, true);
            timerPreviousOrNextClicking->stop();
            timerPreviousOrNextClicking->start(500);
            if (engine->getReadingState() != Phonon::StoppedState &&
                     engine->getReadingState() != Phonon::ErrorState &&
                    !engine->getInitialLoadingState() &&
                     !playIfNoMoreClickActive)
            {
                playIfNoMoreClickActive = true;
                QTimer::singleShot(500, this, SLOT(playIfNoMoreClick()));
            }
        }
        else
        {
            engine->stop();
        }
    }
}

void
Player::saveConf()
{
    if (useAppVol)
    {
        config->volume = engine->getVolume();
    }
    config->lastPlayed = ui->getPlaylist().getBoldRow();
    if(config->savePosition)
    {
        config->position=engine->getCurrentTime();
    }
    config->save();
}

void
Player::showApp()
{
    setActiveWindow(ui);
    ui->raise();
    ui->activateWindow();
}

void
Player::playResource(QString uri)
{
    QFileInfo checkFile(uri);
    if (checkFile.exists())
    {
        QFileInfo fi(uri);
        if (fi.isFile())
        {
            if (ui->getLibrary().fileIsGood(uri))
            {
                ui->getLibrary().setPreviousPlaylistCount();
                engine->setMetaInfosUpdating();
                QStringList files;
                files.append(uri);
                ui->playlistAdd(files);
                ui->getLibrary().addAndPlay();
            }
        }
    }
}
