#include <iostream>
#include <QtGui>

#include "UPlay.h"

Engine::Engine(UPlay* parent)
{
	this->parent = parent;
	
	// Phonon init
	audioOutput = new Phonon::AudioOutput(Phonon::MusicCategory);
	mediaObject = new Phonon::MediaObject();	

	seekSlider = new Phonon::SeekSlider();
	Phonon::createPath(mediaObject,audioOutput);

        
	seekSlider->setIconVisible(false);
	seekLayout = new QVBoxLayout;
	timeLayout = new QHBoxLayout;
	currentTime = new QLabel("0:00");
	totalTime = new QLabel(" / 0:00");
	//seekLayout->addWidget(seekSlider);
	timeLayout->addWidget(currentTime);
	timeLayout->addWidget(totalTime);
	timeLayout->setAlignment(Qt::AlignRight|Qt::AlignVCenter);
	seekLayout->addLayout(timeLayout);
	seekLayout->addWidget(seekSlider);

	volLayout=new QVBoxLayout;

	volumeSlider = new Phonon::VolumeSlider();
	volumeSlider->setAudioOutput(audioOutput);
	volumeSlider->setSizePolicy(QSizePolicy::Maximum,QSizePolicy::Expanding);
	volumeSlider->setOrientation(Qt::Vertical);
	//volumeSlider->setMuteVisible(false);
	volLayout->addWidget(volumeSlider);
	volLayout->setSpacing(0);

	
	// meta tag resolver
	mediaObjectForMetaInfos = new Phonon::MediaObject();
	connect(mediaObjectForMetaInfos, SIGNAL(stateChanged(Phonon::State, Phonon::State)),
				this, SLOT(metaInfosUpdated(Phonon::State, Phonon::State)));

	// media
	connect(mediaObject,SIGNAL(finished()),SLOT(playEnded()));
	connect(mediaObject,SIGNAL(tick(qint64)),SLOT(refreshPlayedTime()));

	listMedias.clear();
	
	// default value
	currentPlayedMedia = -1;
}

Engine::~Engine()
{	
	//qDebug()<<"Engine destructor";
	if (volumeSlider)
		delete volumeSlider;
	if (volLayout)
		delete volLayout;
	if (audioOutput)
		delete audioOutput; audioOutput = NULL;
	if (mediaObject)
		delete mediaObject; mediaObject = NULL;
	if (seekSlider)
		delete seekSlider; seekSlider = NULL;
	if (currentTime)
		delete currentTime; currentTime = NULL;
	if (totalTime)
		delete totalTime; totalTime = NULL;
	if (mediaObjectForMetaInfos)
		delete mediaObjectForMetaInfos; mediaObjectForMetaInfos = NULL;
	if (timeLayout)
		delete timeLayout;
	if (seekLayout)
		delete seekLayout; seekLayout = NULL;

}

void
Engine::stop()
{
	// stop playing
	if (mediaObject)
	{
		mediaObject->stop();

		parent->ui->getPlaylist().setBold(currentPlayedMedia,false);
		currentPlayedMedia = -1;
		parent->ui->getControls().setPlayIcon(Phonon::PlayingState);
		parent->ui->setInfos("\n","00:00");
		currentTime->setText("0:00");
		totalTime->setText(" / 0:00");
		parent->ui->setWindowTitle("NecroPlayer");
	}
}

/**
 * Return the row played, or -1 if an error occured
 */
int
Engine::play(int row)
{
	if (currentPlayedMedia != -1 && row == -1)
	{
		switch (mediaObject->state())
		{	
			case Phonon::PausedState:
				row = currentPlayedMedia;
				if (mediaObject)
				{
					mediaObject->play();
					parent->ui->getControls().setPlayIcon(Phonon::PausedState);
				}
				break;
			case Phonon::PlayingState:
				mediaObject->pause();
				parent->ui->getControls().setPlayIcon(Phonon::PlayingState);
				break;
			default:
				break;
		}
	}
	else
	{
		if (row == -1)
		{
			// TODO
			row = 0;
		}
		if (row <= listMedias.count())
		{
			// Get infos
			Media media = getInfos(row);

			QString s;
			QFileInfo fi(listUris[row]);
			// If the file doesn't have infos, use filename instead
			if (media.title.isEmpty() || media.artist.isEmpty())
			{
				s = fi.fileName() + "\n";
				parent->ui->setWindowTitle(fi.fileName());
			}
			else
			{
				s = media.artist + "\n" + media.title;	
				parent->ui->setWindowTitle(media.artist+" - "+ media.title);
				
			}
			
			// Set the label, display a tray message
			parent->ui->setInfos(s,media.totalTime);

			// preparing reading
			mediaObject->setCurrentSource(listSources[row]);
			mediaObject->setTickInterval(1000);
			mediaObject->setTransitionTime(1000);
			currentTime->setText("0:00");
			seekSlider->setMediaObject(mediaObject);
			totalTime->setText(" / " + media.totalTime);

			if (currentPlayedMedia > -1)
			{
				parent->ui->getPlaylist().setBold(currentPlayedMedia,false);
			}
			currentPlayedMedia = row;
			parent->ui->getPlaylist().setBold(row,true);

			// go
			mediaObject->play();
			parent->ui->getControls().setPlayIcon(Phonon::PausedState);

		}
		else
		{
			row = -1;
		}

	}

	return row;
}

void Engine::addMedia(QStringList files)
{
	if (files.isEmpty())
         return;
	
	int index = listSources.size();
	foreach (QString string, files) 
	{
	  Phonon::MediaSource source(string);

	  listSources.append(source);
	  listUris.append(string);  
	}
	
	if (!listSources.isEmpty())
	{
	  int index_cur = listSources.indexOf(mediaObjectForMetaInfos->currentSource());
	 // qDebug()<<"index_cur: "<<index_cur;
	 // qDebug()<<"index: "<<index;
	  if (index_cur==index-1) //metaInfos not busy
	    mediaObjectForMetaInfos->setCurrentSource(listSources.at(index));
	}
}

void
Engine::refreshPlayedTime()
{
	currentTime->setText(Engine::msToString(mediaObject->currentTime()));
}

Media
Engine::getInfos(unsigned int row)
{
	Media media;
	if (row < (unsigned int)listMedias.count())
	{
		media = listMedias.at(row);
	}
	return media;
}

QString
Engine::getUri(unsigned int row)
{
	QString ret = "";
	if (row < (unsigned int)listUris.count())
	{
		ret = listUris.at(row);
	}
	return ret;
}

void
Engine::metaInfosUpdated(Phonon::State newState, Phonon::State oldState)
{
	// When a file was added, and when
	// his metadata were resolved, we can add it to the displayed playlist
	
	 if (newState != Phonon::StoppedState && newState != Phonon::PausedState) //
	  return;
	 
	 if (mediaObjectForMetaInfos->currentSource().type() == Phonon::MediaSource::Invalid)
          return;
	
	 //mediaObjectForMetaInfos
	  QMap<QString, QString> metaData = mediaObjectForMetaInfos->metaData();

	Media media;
		// get infos
	media.title = metaData.value("TITLE");
	if (media.title == "")
	  media.title = mediaObjectForMetaInfos->currentSource().fileName();
	media.artist = metaData.value("ARTIST");
	media.album = metaData.value("ALBUM");
	media.totalTime = Engine::msToString(mediaObjectForMetaInfos->totalTime());
	
	QString result;
	if(media.artist == "")
	  result = media.title;
	else
	  result = media.artist + " - " + media.title;
	
	parent->ui->addItem(result);
	listMedias.append(media);
	
	int index = listSources.indexOf(mediaObjectForMetaInfos->currentSource()) + 1;
	if (listSources.size() > index) 
	{
	  mediaObjectForMetaInfos->setCurrentSource(listSources.at(index));
	}

}


void
Engine::playEnded()
{
	parent->next();
}

int
Engine::delMedia(int row)
{
	int ret = 0;
	if (row < listMedias.count())
	{
		listMedias.remove(row);
		listSources.remove(row);
		listUris.remove(row);
		ret = 1;
	}
	return ret;
}

void
Engine::clearPlayList()
{
	if (parent && parent->ui)
		parent->ui->clearPlaylist();
	listMedias.clear();
	listSources.clear();
	listUris.clear();
}

QString
Engine::msToString(qint64 ms)
{
	QString ret = "";
	qint64 tmp = ms / 1000;
	int seconds = 0;
	int minutes = 0;
	int hours = 0;
	int tmpMins = tmp/60;
	if (tmpMins > 59)
	{
		hours = tmpMins/60;
		minutes = tmpMins%60;		
	}
	else
	{
		minutes = tmpMins;
	}
	seconds = tmp%60;

	if (hours > 0)
	{ // with hours
		/**
		 * We test if minutes/10 == 0 
		 * to not have a display like this -> 1:5:12
		 * but a display like this -> 1:05:12
		 */
		ret += QString::number(hours) + ":";
		if (minutes/10 == 0)
		{
			ret += "0";
		}
		ret += QString::number(minutes) + ":";
		if (seconds/10 == 0)
		{
			ret += "0";
		}
		ret += QString::number(seconds);
	}
	else
	{ // without hours
		ret += QString::number(minutes) + ":";
		if (seconds/10 == 0)
		{
			ret += "0";
		}
		ret += QString::number(seconds);
	}
	return ret;
}

