/*
 *  Nameday desktop applet (widget) for Maemo.
 *  Copyright (C) 2009 Roman Moravcik
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H 
#include <config.h> 
#endif

#include <dirent.h>
#include <string.h>

#include <gtk/gtk.h>

#include <libxml/xmlmemory.h> 
#include <libxml/parser.h> 
#include <libxml/tree.h>

#include "nameday_dbparser.h"

#define DATABASE_DIR DATADIR"/nameday/db/"

void
nameday_dbparser_open (NamedayDesktopApplet *priv)
{
	xmlDocPtr db = NULL;

	g_return_if_fail (priv);

	if (priv->db_filename == NULL) {
		g_debug ("nameday_dbparser_open: File name not specified.\n");
		return;
	}

	/* open xml document */
	db = xmlParseFile (priv->db_filename);

	if (db == NULL) {
		g_debug ("nameday_dbparser_open: Could not parse file %s\n", priv->db_filename);
	}

	priv->db = db;
}

void nameday_dbparser_close (NamedayDesktopApplet *priv)
{
	g_return_if_fail (priv);

	if (priv->db) {
		/* free the document */
		xmlFreeDoc (priv->db);

		/*
		 * Free the global variables that may
		 * have been allocated by the parser.
		 */
		xmlCleanupParser ();

		/* file was closed successfully */
	}

	priv->db = NULL;
}

gchar *
nameday_dbparser_get_name_from_date (NamedayDesktopApplet *priv,
				     const gchar search_day,
				     const gchar search_month)
{
	xmlNode *root = NULL, *entry = NULL;
	gchar *name = NULL;

	g_return_val_if_fail (priv, NULL);
	g_return_val_if_fail (priv->db, NULL);

	/* Get the root element node */
	root = xmlDocGetRootElement (priv->db);

	if(!root || !root->name || xmlStrcmp (root->name, (const xmlChar *) "NamedayCalendar"))
	{
		g_debug ("nameday_dbparser_get_name_from_date: Unable to parse nameday database.");
		return NULL;
	}

	for (entry = root->children; entry != NULL; entry = entry->next)
	{
		if (entry->type == XML_ELEMENT_NODE  &&
		    !xmlStrcmp (entry->name, (const xmlChar *) "NamedayEntry" ))
		{
			xmlChar *tmp = NULL;

			tmp = xmlGetProp (entry, (const xmlChar *) "month");
			if (atoi ((const char *) tmp) == search_month)
			{
				tmp = xmlGetProp (entry, (const xmlChar *) "day");
				if (atoi ((const char *) tmp) == search_day)
				{
					tmp = xmlGetProp (entry, (const xmlChar *) "name");
					name = g_strdup_printf ("%s", tmp);
				}
			}
		xmlFree (tmp);
		}
	}

	return name;
}

GtkListStore *
nameday_dbparser_liststore_create (NamedayDesktopApplet *priv)
{
	GtkListStore *list_store;
	xmlNode *root = NULL, *entry = NULL;
	GtkTreeIter iter;

	g_return_val_if_fail (priv, NULL);
	g_return_val_if_fail (priv->db, NULL);

	/* Get the root element node */
	root = xmlDocGetRootElement (priv->db);

	if(!root || !root->name || xmlStrcmp (root->name, (const xmlChar *) "NamedayCalendar"))
	{
		g_debug ("nameday_dbparser_liststore_create: Unable to parse nameday database.");
		return NULL;
	}

	list_store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);

	for (entry = root->children; entry != NULL; entry = entry->next)
	{
		if (entry->type == XML_ELEMENT_NODE  &&
		    !xmlStrcmp (entry->name, (const xmlChar *) "NamedayEntry" ))
		{
		    	xmlChar *day = NULL, *month = NULL, *name = NULL;
			gchar *date = NULL;

			day = xmlGetProp (entry, (const xmlChar *) "day");
			month = xmlGetProp (entry, (const xmlChar *) "month");
			name = xmlGetProp (entry, (const xmlChar *) "name");

			date = g_strdup_printf ("%s.%s.", day, month);

			gtk_list_store_append (list_store, &iter);
			gtk_list_store_set (list_store,
					    &iter,
					    COLUMN_DATE, date,
					    COLUMN_NAME, name,
					    -1);

			g_free (date);
			xmlFree (day);
			xmlFree (month);
			xmlFree (name);
		}
	}
	return list_store;
}

GtkListStore *
nameday_dbparser_calendar_list_create (NamedayDesktopApplet *priv)
{
	GtkListStore *calendars_list;
	DIR *dir = NULL;
	struct dirent *dit;
	GtkTreeIter iter;

	g_return_val_if_fail (priv, NULL);

	calendars_list = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);

	dir = opendir (DATABASE_DIR);

	if (dir == NULL) {
		g_debug ("nameday_dbparser_calendar_list_create: Unable to open directory %s\n",
			 DATABASE_DIR);
	}

	while ((dit = readdir (dir)) != NULL) {
		gchar *db_filename = NULL;
		xmlDocPtr db = NULL;
		xmlNode *root = NULL;
		xmlChar *calendar_name = NULL;

		if (!strcmp (dit->d_name, ".") || !strcmp (dit->d_name, ".."))
			continue;

		db_filename = g_strdup_printf ("%s%s", DATABASE_DIR, dit->d_name);

		/* open xml document */
		db = xmlParseFile (db_filename);

		if (db == NULL) {
			g_debug ("nameday_dbparser_calendar_list_create: Could not parse file %s\n",
			         db_filename);
		}

		/* Get the root element node */
		root = xmlDocGetRootElement (db);


		if(!root || !root->name || xmlStrcmp (root->name, (const xmlChar *) "NamedayCalendar"))
		{
			g_debug ("nameday_dbparser_calendar_list_create: Unable to parse nameday database.");
			return NULL;
		}

		calendar_name = xmlGetProp (root, (const xmlChar *) "name");

		gtk_list_store_append (calendars_list, &iter); 
		gtk_list_store_set (calendars_list,
		                    &iter,
		                    COLUMN_DB_NAME, calendar_name,
		                    COLUMN_DB_FILENAME, db_filename,
		                    -1);

		if (calendar_name)
			xmlFree (calendar_name);

		/* free the document */
		xmlFreeDoc (db);

		/*
		 * Free the global variables that may
		 * have been allocated by the parser.
		 */
		xmlCleanupParser ();

		if (db_filename)
			g_free (db_filename);
	}

	if (dir)
		closedir (dir);

	return calendars_list;
}

void
nameday_dbparser_init (NamedayDesktopApplet *priv)
{

	g_return_if_fail (priv);

	/* Open db file */
	nameday_dbparser_open (priv);
}

void
nameday_dbparser_finalize (NamedayDesktopApplet *priv)
{
	g_return_if_fail (priv);

	/* Close db file */
	nameday_dbparser_close (priv);
}
