/* FMRX Enabler for N900 hardware
 * Copyright (C) 2009 Martin Grimme  <martin.grimme@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "n900-fmrx-enabler.h"
#include "n900-fmrx-enablerGlue.h"


static GMainLoop *loop;

/* Generate the GObject boilerplate */
G_DEFINE_TYPE(FMRXEnabler, fmrx_enabler, G_TYPE_OBJECT)

static void daemonize();
static gboolean hci_dev_up(int sock, gboolean *had_bluetooth);
static gboolean hci_dev_down(int sock);
static gboolean set_i2c_enabled(int sock, gboolean value);
static gboolean watchdog_handler(gpointer data);


/* GObject class initializer.
 */
static void
fmrx_enabler_class_init(FMRXEnablerClass *clss) {
    
    dbus_g_object_type_install_info(FMRX_ENABLER_TYPE,
                                    &dbus_glib_fmrx_enabler_object_info);

}

/* GObject object constructor.
 */
static void
fmrx_enabler_init(FMRXEnabler *self) {

    DBusGConnection *conn;
    GError *error = NULL;
    DBusGProxy *driver_proxy;
    guint32 request_name_ret;
    
    self->hci_socket = 0;
    self->driver_loaded = FALSE;
    self->had_bluetooth = FALSE;
    self->last_request = 0;

    /* set up D-Bus service */
    conn = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
    if (conn == NULL) {
        g_print("could not connect to D-Bus: %s", error->message);
        g_error_free(error);
        return;
    }
    
    dbus_g_connection_register_g_object(conn, FMRX_OBJECT_PATH, G_OBJECT(self));
    driver_proxy = dbus_g_proxy_new_for_name(conn,
                                             DBUS_SERVICE_DBUS,
                                             DBUS_PATH_DBUS,
                                             DBUS_INTERFACE_DBUS);

    if (! org_freedesktop_DBus_request_name(driver_proxy, FMRX_SERVICE_NAME,
                                            0, &request_name_ret, &error)) {
        g_print("Failed to request name: %s", error->message);
        g_error_free(error);
        return;
    }
  
    if (request_name_ret != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
        g_error("Got result code %u from requesting name", request_name_ret);
        return;
    }

}


/* Turns this process into a daemon by using the double-(pitch)-fork. :)
 */
static void
daemonize() {

  int devnull;

  if (fork()) exit(0);
  chdir("/");
  setsid();
  umask(0);
  if (fork()) exit(0);

  devnull = open("/dev/null", O_RDWR);
  close(STDIN_FILENO); dup2(STDIN_FILENO, devnull);
  close(STDOUT_FILENO); dup2(STDOUT_FILENO, devnull);
  close(STDERR_FILENO); dup2(STDERR_FILENO, devnull);
  close(devnull);

}


/* Powers up the Bluetooth device.
 */
static gboolean
hci_dev_up(int sock, gboolean *had_bluetooth) {

    if (ioctl(sock, HCIDEVUP, 0) < 0) {
        if (errno == EALREADY) {
            *had_bluetooth = TRUE;
            return TRUE;
        } else {
            *had_bluetooth = FALSE;
            return FALSE;
        }
    } else {
        *had_bluetooth = FALSE;
        return TRUE;
    }

}


/* Powers down the Bluetooth device.
 */
static gboolean
hci_dev_down(int sock) {

    if (ioctl(sock, HCIDEVDOWN, 0) < 0 && errno != EALREADY) {
        return FALSE;
    } else {
        return TRUE;
    }

}


/* Enabled or disables I2C communication on the Bluetooth chip.
 */
static gboolean
set_i2c_enabled(int sock, gboolean value) {

    int rv, opt;
    struct sockaddr_hci addr;

    opt = 0;
    rv = setsockopt(sock, SOL_HCI, HCI_DATA_DIR, &opt, sizeof(opt));
    if (rv < 0) {
        return FALSE;
    }
    addr.hci_family = AF_BLUETOOTH;
    addr.hci_dev = 0;
    rv = bind(sock, (struct sockaddr *) &addr, sizeof(addr));
    if (rv < 0) {
        return FALSE;
    }

    write(sock, value ? I2C_ENABLE : I2C_DISABLE, sizeof(I2C_ENABLE));

    return TRUE;

}


/* Watching for automatically powering down the radio and I2C when not in use
 * for a while.
 */
static gboolean
watchdog_handler(gpointer data) {

    int status;
    GError *error;
    FMRXEnabler *self = (FMRXEnabler*) data;

    g_print("watchdog checking\n");
    if (difftime(time(NULL), self->last_request) > UNLOAD_TIMEOUT) {
    
        self->driver_loaded = FALSE;
        set_i2c_enabled(self->hci_socket, FALSE);
        if (! self->had_bluetooth) {
            hci_dev_down(self->hci_socket);
        }
        close(self->hci_socket);
        
        g_spawn_command_line_sync("/sbin/rmmod radio-bcm2048",
                                  NULL, NULL, &status, &error);
        g_print("FM Tuner unloaded\n");
        return FALSE;
        
    } else {
    
        return TRUE;
    
    }

}


/* D-Bus method: request
 */
gboolean
dbus_fmrx_enabler_request(FMRXEnabler *self,
                          int *result,
                          char **device,
                          GError **error) {

    int status;

    g_print("received client request\n");    

    /* prolong usage time */
    self->last_request = time(NULL);

    if (self->driver_loaded) {
    
        g_print("prolonged usage time\n");
        *result = RESULT_OK;
        
    } else {
        /* attempt to unload module first */
        g_spawn_command_line_sync("/sbin/rmmod radio-bcm2048",
                            NULL, NULL, &status, &error);

        /* open HCI socket */
        g_print("opening HCI socket... ");
        self->hci_socket = socket(AF_BLUETOOTH, SOCK_RAW, BTPROTO_HCI);
        if (self->hci_socket < 0) {
            g_print("failed\n");
            *result = RESULT_BLUETOOTH_FAILURE;
            return TRUE;
        }
        g_print("success\n");
        
        /* power up Bluetooth */
        g_print("powering up Bluetooth device... ");
        if (hci_dev_up(self->hci_socket, &(self->had_bluetooth)) == FALSE) {
            g_print("failed\n");
            *result = RESULT_BLUETOOTH_FAILURE;
            close(self->hci_socket);
            return TRUE;
        }
        g_print("success\n");
        
        /* power up I2C bus */
        g_print("enabling I2C communication... ");
        if (set_i2c_enabled(self->hci_socket, TRUE) == FALSE) {
            g_print("failed\n");
            *result = RESULT_I2C_FAILURE;
            if (! self->had_bluetooth) {
                hci_dev_down(self->hci_socket);
            }
            close(self->hci_socket);
            return TRUE;
        }
        g_print("success\n");

        /* wait for device */
        usleep(250000);

        /* load driver */
        g_print("loading kernel module... ");
        g_spawn_command_line_sync("/sbin/insmod "
                                  RADIO_BCM2048_MODULE
                                  " radio_nr=1",
                                  NULL, NULL, &status, error);
        if (status != 0) {
            g_print("failed\n");
            *result = RESULT_DRIVER_FAILURE;
            set_i2c_enabled(self->hci_socket, FALSE);
            if (! self->had_bluetooth) {
                hci_dev_down(self->hci_socket);
            }
            return TRUE;
        }
        g_print("success\n");
        
        /* wait for device */
        usleep(250000);
        
        /* check for device */
        g_print("checking for device file... ");
        if (!g_file_test("/dev/radio1", G_FILE_TEST_EXISTS)) {
            g_print("failed\n");
            *result = RESULT_DRIVER_FAILURE;
            set_i2c_enabled(self->hci_socket, FALSE);
            if (! self->had_bluetooth) {
                hci_dev_down(self->hci_socket);
            }
            return TRUE;
        }
        g_print("success\n");
       
        /* initialize watchdog */
        g_print("initializing powersave watchdog... ");
        self->driver_loaded = TRUE;
        g_timeout_add(WATCHDOG_TIMEOUT * 1000,
                      watchdog_handler,
                      (gpointer) self);
        g_print("success\n");
        
    }
    
    g_print("FM tuner activated\n");
    *device = g_strdup(FMRADIO_DEVICE);
    
    return TRUE;

}


int
main(int argc, char **argv) {

    GObject *obj;

    if (argc == 1) {
        daemonize();
    }

    g_thread_init(NULL);
    dbus_g_thread_init();

    g_type_init();

    loop = g_main_loop_new(NULL, FALSE);
    obj = g_object_new(FMRX_ENABLER_TYPE, NULL);

    g_main_loop_run(loop);

    return 0;
    
}

