/*
 * db_msa_logic.h - interface fo msa db logic.
 * This file is part of DB-Maemo.
 *
 * Copyright (C) 2009 - Ilia L. Burlak
 *
 * DB-Maemo is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DB-Maemo is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with DB-Maemo; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA  02110-1301  USA
 */
#include "db_interface_function.h"

/** response **/
const char* db_response = "\
<TransitData>\
<TargetID>ui</TargetID>\
<SourceID>db</SourceID>\
<Content>\
<Response class=\"class\" function = \"update\">\
</Response>\
</Content>\
</TransitData>";

/** message **/
const char* db_message = "\
<TransitData>\
<TargetID>ui</TargetID>\
<SourceID>db</SourceID>\
<Content>\
<Response class=\"systemMessages\" function = \"update\">\
<Params>\
<string name=\"moduleName\">database</string>\
<string name=\"code\"/>\
<string name=\"text\"/>\
</Params>\
</Response>\
</Content>\
</TransitData>";

static xmlDocPtr db_generate_response(xmlDocPtr response, xmlNodePtr params,
                                                        char* class_name, char* funciotn_name);
static xmlDocPtr db_generate_message(xmlDocPtr response, char* function_name,
                                                  char* text, char* code);

static char* db_get_request_id(char* class,db_type req_id, xmlDocPtr request);
static gint db_sort_asc(gpointer a, gpointer b);
static gint db_sort_desc(gpointer a, gpointer b);
static gint db_friends_find(gpointer a, gpointer b);
static gint db_sort_time(gpointer a, gpointer b);
static gint db_friends_online(gpointer _data, gpointer _filter);

/**
 * @brief get profile form services 
 * @ui_request = request form ui
 * @return db_request 
**/
xmlDocPtr DB_SEND_REQUEST(xmlDocPtr ui_request)
{
	g_debug("db_send_request: START");
	//xmlDocDump(stdout, ui_request);

	xmlDocPtr db_request;
	xmlDocPtr db_settings;	
	xmlNodePtr node_req_root;
	xmlNodePtr node_req;
	xmlNodePtr node_settings_root;
	xmlNodePtr node_settings;

	xmlXPathObject* Obj;
	xmlChar* xml_conf;
	GConfClient* client;

	db_request = xmlCopyDoc(ui_request, 1);	
//	GList* node_list;
	
	g_debug("\ndb_send_request: db request");
	//xmlDocDump(stdout, db_request);

//	msa_driver_info* info;
	node_req_root = xmlDocGetRootElement(db_request);

	
	/** set new value to source_id node = DB_ID **/
	Obj = db_xpath(SOURCE_TAG_XPATH, db_request);
	if (Obj->nodesetval->nodeNr != 0) {
		node_req = Obj->nodesetval->nodeTab[0];
		xmlNodeSetContent(node_req, BAD_CAST DB_ID);
	}

	/** get data from gconf **/
    gchar *g_conf_path = g_strconcat(DB_PATH_GCONF, DB_LIST_DRIVERS, NULL);
	client = gconf_client_get_default ();
	xml_conf = (xmlChar*)gconf_client_get_string (client, g_conf_path, NULL);
    g_free(g_conf_path);

	db_settings = xmlParseDoc(BAD_CAST xml_conf);
	xmlFree(xml_conf);
	node_settings_root = xmlDocGetRootElement(db_settings);

    /** if drivers list is null then ERROR! **/
    if (node_settings_root == NULL || node_settings_root->children == NULL) {

            db_request = xmlParseDoc(BAD_CAST db_message); 
			db_request = db_generate_message(db_request, INFO_MESSAGE, 
	                                         DB_INIT_ERROR_TEXT, 
                                             DB_INIT_ERROR_CODE);
	        xmlFree(ui_request);	        
            xmlXPathFreeObject(Obj);
            return db_request;  
    }

	/** rem all old targets **/
	rem_targets(db_request);

	/** set new target **/
	for (node_settings = node_settings_root->children; node_settings != NULL; 
										node_settings = node_settings->next) {

		/** if driver on and conf, add new target id **/
		xmlChar* prop_val = xmlGetProp(node_settings,BAD_CAST STATUS_PROP);
		if ( (atoi((char*)prop_val) == 1) ) {			
		    xmlChar *target_id = xmlNodeGetContent(node_settings);
			add_target(db_request, target_id);
			xmlFree(target_id);
		}
		xmlFree(prop_val);

	}	


	xmlFree(ui_request);

	g_debug("\ndb_send_request: REQUEST TO DRIVERS");

// FIXME
//    g_printf("xml request:");	
//    xmlDocDump(stdout, db_request);
//    g_printf("drivers:");
//    xmlDocDump(stdout, db_settings);

    xmlXPathFreeObject(Obj);
    xmlFreeDoc(db_settings);
    //xmlFreeDoc(ui_request);
	return db_request;
}


/**
 * @brief save response from drivers in db
 * @drv_response - response for db form drivers  
 * @return db_response - response for ui from db, or NULL if error
**/
xmlDocPtr DB_SAVE_PROFILE(xmlDocPtr drv_response) {

	g_debug("db_update_profile_set: START");

	//xmlDocPtr answer;
	xmlDocPtr db_resp = NULL;
	
	xmlXPathObject* Obj = db_xpath(RESPONSE_TAG_XPATH, drv_response);	
	xmlNodePtr node_set = Obj->nodesetval->nodeTab[0];
	gchar* resp_class = (gchar*)xmlGetProp(node_set,BAD_CAST "class");
        xmlXPathFreeObject(Obj);

	if(strcmp(resp_class, CLASS_PROFILE) == 0) {

		/** try to save in db **/
		/** get profile id **/
		char* id = db_get_request_id(CLASS_PROFILE, TYPE_RESPONSE, drv_response);

		if (id == NULL) {
		
			g_debug("db: save owner contatct");

			if (create_owner_metacontact(drv_response) == 0) {
				db_resp = xmlParseDoc(BAD_CAST db_message); 
				db_resp = db_generate_message(db_resp, INFO_MESSAGE, 
			                                      DB_SAVE_MESSAGE_PROFILE_TEXT, 
                                                              DB_SAVE_MESSAGE_PROFILE_CODE);
			} else {	
				db_resp = xmlParseDoc(BAD_CAST db_message); 
				db_resp = db_generate_message(db_resp, ERROR_MESSAGE,
						              DB_SAVE_ERROR_PROFILE_TEXT, 
                                                              DB_SAVE_ERROR_PROFILE_CODE);
			}

		} else {
			
			g_debug("db: save contact id = %s", id);

			if (create_or_update_profile(drv_response, id) == 0) {
				db_resp = xmlParseDoc(BAD_CAST db_message); 
				db_resp = db_generate_message(db_resp, INFO_MESSAGE, id, 
							      DB_SAVE_MESSAGE_FRIENDS_PROFILE_CODE);
				
				
			} else {	
				db_resp = xmlParseDoc(BAD_CAST db_message); 
				db_resp = db_generate_message(db_resp, ERROR_MESSAGE, 
							      DB_SAVE_ERROR_FRIENDS_PROFILE_TEXT, 
							      DB_SAVE_ERROR_FRIENDS_PROFILE_CODE);
			}
			
			xmlXPathObject* Obj = db_xpath(RESPONSE_PARAMS_TAG_XPATH, db_resp);
			if (Obj->nodesetval->nodeNr != 0) {
				xmlNodePtr node_set = Obj->nodesetval->nodeTab[0];
				xmlSetProp(node_set, BAD_CAST "id", BAD_CAST id);			
			}
                        xmlXPathFreeObject(Obj);

		}		

	} else {

		rem_targets(db_resp);
		add_target(db_resp, (xmlChar*)UI_ID);

	}

    xmlFreeDoc(drv_response);
    g_free(resp_class);

    return db_resp;
}

/**
 * @brief get response from db to ui
 * @drv_response - response for bd form ui  
 * @return db_response - response for ui from db, or NULL if error
**/
xmlDocPtr DB_GET_PROFILE(xmlDocPtr request)
{
	g_debug("db_get_profile: START");

	xmlDocPtr struct_response = xmlParseDoc(BAD_CAST db_response);
	//xmlDocDump(stdout, struct_response);
	xmlDocPtr db_resp;
	GSList* response_list;

	/** get profile id **/
	//gchar* class_name;
	gchar* func_name = NULL;
    gchar* text_error = NULL;
    gchar* code_error = NULL;

	/** get profile id **/
	char* id = db_get_request_id(CLASS_PROFILE, TYPE_REQUEST, request);

	if (id == NULL) {
		response_list = db_get_owner_metacontact("0");
		func_name = GET_PROFILE;
        text_error = DB_GET_ERROR_PROFILE_TEXT;
        code_error = DB_GET_ERROR_PROFILE_CODE;
	} else {
		response_list = db_get_profile(id);	
        text_error = DB_SAVE_ERROR_FRIENDS_PROFILE_TEXT;
        code_error = DB_SAVE_ERROR_FRIENDS_PROFILE_CODE;		
		func_name = GET_FRIENDS_PROFILE;	
	}	
	g_free(id);

	if (response_list != NULL) {	
		xmlNodePtr node_resp = xmlDocGetRootElement((xmlDocPtr)response_list->data);
		db_resp = db_generate_response(struct_response, node_resp,
					       CLASS_PROFILE, func_name);	

	
	} else {
		db_resp = xmlParseDoc(BAD_CAST db_message); 
		db_resp = db_generate_message(db_resp,ERROR_MESSAGE, 
					                          text_error, 
					                          code_error);
	}
	

    xmlFreeDoc(request);    
	g_debug("db_get_profile: END");	
	return db_resp;
}



/**
 * @brief save list friends from drivers in db
 * @drv_response - response for db form drivers  
 * @return db_response - response for ui from db, or NULL if error
**/
xmlDocPtr DB_SAVE_FRIENDS(xmlDocPtr drv_response)
{

	g_debug("DB_SAVE_FRIENDS: START");

	xmlDocPtr db_resp = NULL;
	xmlXPathObject* Obj = db_xpath(RESPONSE_TAG_XPATH, drv_response);	
	xmlNodePtr node_set = Obj->nodesetval->nodeTab[0];
	gchar* resp_class = (gchar*)xmlGetProp(node_set, BAD_CAST "class");
	gchar* friends_id = NULL;
	gchar* resp_xpath = NULL;
	gint i;
    gboolean save_error = FALSE;

	/** if class FRIENDS, else error **/
	if (strcmp(resp_class, CLASS_FRIENDS) == 0) {

		resp_xpath = g_strconcat(RESPONSE_PARAMS_TAG_XPATH, "/array/struct", NULL);
		xmlXPathObject* Obj = db_xpath(resp_xpath, drv_response);	

		for (i=0; i<Obj->nodesetval->nodeNr; i++) {
		    node_set = Obj->nodesetval->nodeTab[i];
            
            if (node_set == NULL) {
                save_error = TRUE;
                continue;            
            }

            db_resp = xmlNewDoc(BAD_CAST XML_VERSION);
		    friends_id = (gchar*)xmlGetProp(node_set, BAD_CAST "id");
		    xmlUnlinkNode(node_set);
		    xmlDocSetRootElement(db_resp, node_set);
			
		    if (create_or_update_contact(db_resp, friends_id) != 0) {
                save_error = TRUE;
			}
            g_free(friends_id);
            xmlFreeDoc(db_resp);	
		} 	
		
		if (i == 0 || save_error == TRUE) {

			db_resp = xmlParseDoc(BAD_CAST db_message); 
			db_resp = db_generate_message(db_resp, ERROR_MESSAGE, 
					  DB_SAVE_ERROR_FRIENDS_TEXT, 
					  DB_SAVE_ERROR_FRIENDS_CODE);									
		} else { 

			db_resp = xmlParseDoc(BAD_CAST db_message); 
			db_resp = db_generate_message(db_resp, INFO_MESSAGE, 
				                          DB_SAVE_MESSAGE_FRIENDS_TEXT, 
                                          DB_SAVE_MESSAGE_FRIENDS_CODE);							        
        }
	} else {

		rem_targets(db_resp);
		add_target(db_resp, (xmlChar*)UI_ID);

	}

    xmlXPathFreeObject(Obj);
    xmlFreeDoc(drv_response);
    
    g_free(resp_class);	
    g_free(resp_xpath);

    g_debug("DB_SAVE_FRIENDS: END");

    return db_resp;
}


// NEW
/**
 * @brief get response friends list from db to ui
 * @drv_response - request to db from ui  
 * @return db_response - response for ui from db, or NULL if error
**/
xmlDocPtr DB_GET_FRIENDS(xmlDocPtr request)
{
	g_debug("db_get_friends: START");

	xmlDocPtr struct_response = xmlParseDoc(BAD_CAST db_response);
	xmlDocPtr db_resp;

    xmlDocPtr fr_resp = xmlParseDoc(BAD_CAST "<Params><array name=\"contacts\"/></Params>");
    xmlNodePtr fr_resp_root = xmlDocGetRootElement(fr_resp);
    xmlNodePtr fr_resp_node = fr_resp_root->children;

    gint fr_from = 0;
    gint fr_to = 0;

    gchar* fr_from_str = NULL;
    gchar* fr_to_str = NULL;            

    xmlDocPtr friends = NULL;
    
    gboolean is_online_req = FALSE;
    gboolean is_filter_req = FALSE;        

    gint sort_order = ORDER_NO;

    /** get parametrs in Params node**/
    xmlXPathObject* Obj = db_xpath(REQUEST_PARAMS_TAG_XPATH, request);
    if (Obj->nodesetval->nodeNr != 0) {
        xmlNodePtr node_set = node_set = Obj->nodesetval->nodeTab[0];

        /** get From **/
        fr_from_str = (gchar*) xmlGetProp(node_set, BAD_CAST "From");
        if (fr_from_str != NULL) {
                fr_from = atoi((char*)fr_from_str);
        } else {
                fr_from_str = g_strdup("0");
                fr_from = 0;
        }


        /** get TO **/
        fr_to_str = (gchar*) xmlGetProp(node_set, BAD_CAST "To");
        if (fr_to_str != NULL) {
                fr_to = atoi(fr_to_str);
        } else {
                fr_to_str = g_strdup("10");
                fr_to = 10;
        }

        /** type of sort list if NULL then no sort **/
        xmlChar* str = xmlGetProp(node_set, BAD_CAST "Sort");
        if (str != NULL) {
            if (strcmp((char*)str,(char*)"asc") == 0) {
                   sort_order = ORDER_ASC;
            } 
            
            if (strcmp((char*)str,(char*)"desc") == 0) {
                   sort_order = ORDER_DESC;                                    
            }
        }
        xmlFree(str);


        /** filter if null or empty then no filter **/
        str = xmlGetProp(node_set, BAD_CAST "Filter");
        if (str != NULL && strlen((char*)str) > 0) {
            friends = search_contacts((const gchar*) str);  
            is_filter_req = TRUE;
            xmlSetProp(fr_resp_node, BAD_CAST "Filter", BAD_CAST str);
        } // if (str) > 0
        xmlFree(str);

        /** online, 1 - only online users, 0 - online + offline **/
        str = xmlGetProp(node_set, BAD_CAST "Online");
        if (str != NULL && strcmp(str,"0") != 0) {
     
            friends = get_online_contacts(sort_order);
            is_online_req = TRUE;
            xmlSetProp(fr_resp_node, BAD_CAST "Online", "1");			
	    }

    }        

    gint friends_count = 0;
    
    if (is_online_req == FALSE && is_filter_req == FALSE) {
        friends = get_contacts_filter(fr_from_str, fr_to_str, sort_order, &friends_count);
    }

    /** error **/
    if (friends == NULL) {
        db_resp = xmlParseDoc(BAD_CAST db_message);
        db_resp = db_generate_message(db_resp,  ERROR_MESSAGE,
                                                DB_GET_ERROR_FRIENDS_TEXT,
                                                DB_GET_ERROR_FRIENDS_CODE);
        return db_resp;
    }
    
    xmlSetProp(fr_resp_node, BAD_CAST "friendsQuantity", 
            BAD_CAST g_strdup_printf("%d", friends_count));       

    xmlNodePtr friends_root = xmlDocGetRootElement(friends);

    xmlNodePtr friends_nodes = friends_root->children;
    friends_root->children = NULL;
    
    while (friends_nodes != NULL) {
        xmlNodePtr tmpNode = friends_nodes->next;
        xmlUnlinkNode(friends_nodes);            
        xmlAddChild(fr_resp_node, friends_nodes);
        friends_nodes = tmpNode;
    }

    xmlFreeDoc(friends);

        
    db_resp = db_generate_response(struct_response, fr_resp_root,
                                                   CLASS_FRIENDS, GET_FRIENDS);
    xmlXPathFreeObject(Obj);


    g_debug("db_get_profile: END");
    return db_resp;
}



/**
 * @brief get response friends list from db to ui
 * @drv_response - request to db from ui  
 * @return db_response - response for ui from db, or NULL if error
**/
xmlDocPtr DB_GET_FRIENDS_OLD(xmlDocPtr request)
{
	g_debug("db_get_friends: START");

	xmlDocPtr struct_response = xmlParseDoc(BAD_CAST db_response);
	xmlDocPtr db_resp;
	GSList* response_list = NULL;

	response_list = get_contacts(FALSE);

        if (response_list != NULL) {

            g_debug("db_get_profile response list isn't null!");

            gint i;
            gint fr_from = 0;
            gint fr_to = 0;
            xmlChar* str = NULL;

            xmlNodePtr node_set = NULL;
            xmlDocPtr fr_resp = xmlParseDoc(BAD_CAST "<Params><array name=\"contacts\"/></Params>");
            xmlNodePtr fr_resp_root = xmlDocGetRootElement(fr_resp);

            xmlNodePtr fr_resp_node = fr_resp_root->children;

            guint fr_count = g_slist_length(response_list);
            gchar* fr_count_text = g_strdup_printf("%d", fr_count);
            
            xmlSetProp(fr_resp_node, BAD_CAST "friendsQuantity", BAD_CAST fr_count_text);
            g_free(fr_count_text);

            /** get parametrs in Params node**/
            xmlXPathObject* Obj = db_xpath(REQUEST_PARAMS_TAG_XPATH, request);
            if (Obj->nodesetval->nodeNr != 0) {
                    node_set = Obj->nodesetval->nodeTab[0];

                    /** get From **/
                    str = xmlGetProp(node_set,BAD_CAST "From");
                    if (str != NULL) {
                            fr_from = atoi((char*)str);
                    } else {
                            fr_from = 0;
                    }
                	xmlFree(str);

                    /** get TO **/
                    str = xmlGetProp(node_set,BAD_CAST "To");
                    if (str != NULL) {
                            fr_to = atoi((char*)str);
                    } else {
                            fr_to = 10;
                    }
                    xmlFree(str);

                    /** type of sort list if NULL then no sort **/
                    str = xmlGetProp(node_set,BAD_CAST "Sort");
                    if (str != NULL) {

                            if (strcmp((char*)str,(char*)"asc") == 0) {
                                    response_list = g_slist_sort(response_list, (GCompareFunc)db_sort_asc);
                            }
                            if (strcmp((char*)str,(char*)"desc") == 0) {
                                    response_list = g_slist_sort(response_list, (GCompareFunc)db_sort_desc);
                            }
                    }
                    xmlFree(str);

					xmlDocPtr cur_data = NULL;
					GSList* list_element = NULL;

                    /** filter if null or empty then no filter **/
                    str = xmlGetProp(node_set,BAD_CAST "Filter");
                    if (str != NULL && strlen((char*)str) > 0) {


                            fr_to = 1;
                            fr_from = 0;

                            list_element = response_list;
                            response_list = NULL;

                            while (list_element != NULL) {
				                    cur_data = (xmlDocPtr)list_element->data;
                                    list_element = g_slist_remove(list_element, cur_data);
                                    if (db_friends_find(cur_data, (gpointer)str) == 0) {
                                            response_list = g_slist_append(response_list, cur_data);
                                            fr_to++;
                                    }else{
                                	xmlFreeDoc(cur_data);
                                    }
                            }

                            xmlSetProp(fr_resp_node, BAD_CAST "friendsQuantity", BAD_CAST g_strdup_printf("%d", fr_to - 1));
                            xmlSetProp(fr_resp_node, BAD_CAST "Filter", BAD_CAST str);
                    } // if (str) > 0
                    xmlFree(str);

                    /** online, 1 - only online users, 0 - online + offline **/
                    str = xmlGetProp(node_set,BAD_CAST "Online");
                    if (str != NULL && strcmp(str,"0") != 0) {
							
							//GSList* find_element;
                            GSList* list_element;

                            fr_to = 1;
                            fr_from = 0;

                            list_element = response_list;
                            response_list = NULL;

							while (list_element != NULL) {
								cur_data = (xmlDocPtr)list_element->data;
								list_element = g_slist_remove(list_element, cur_data);
							    if (db_friends_online(cur_data, str) == 0) {
								    response_list = g_slist_append(response_list, cur_data);
								    fr_to++;
							    }else{
							        xmlFreeDoc(cur_data);
							    }
							}

                            xmlSetProp(fr_resp_node, BAD_CAST "friendsQuantity", BAD_CAST g_strdup_printf("%d", fr_to - 1));
                            xmlSetProp(fr_resp_node, BAD_CAST "Online", "1");			
					}
                    

        }// if req ok

        /** error **/
        if (response_list == NULL || (fr_to - 1) == 0) {
            db_resp = xmlParseDoc(BAD_CAST db_message);
            db_resp = db_generate_message(db_resp,  ERROR_MESSAGE,
                                                    DB_ERROR_SEARCH_FRIENDS_TEXT,
                                                    DB_ERROR_SEARCH_FRIENDS_CODE);
            return db_resp;
        }

        GSList* list = g_slist_nth(response_list, fr_from);

        for (i=fr_from; i<fr_to && list != NULL; i++) {
            node_set = xmlDocGetRootElement((xmlDocPtr)list->data);
            //xmlUnlinkNode(node_set);
            xmlAddChild(fr_resp_node, xmlCopyNode(node_set, 1));
            //xmlAddChild(fr_resp_node, xmlCopyDoc((xmlDocPtr)list->data, 1));

            if (list->next == NULL) {
                    break;
            }

            //xmlDocSetRootElement((xmlDocPtr)list->data, NULL);
            list = list->next;
        }
        
        /* clean memory */
        while(response_list != NULL)
        {
    	    xmlDocPtr cur_node = (xmlDocPtr)response_list->data;
    	    response_list = g_slist_remove(response_list, cur_node);
    	    xmlFreeDoc(cur_node);
        }

        db_resp = db_generate_response(struct_response, fr_resp_root,
                                                                   CLASS_FRIENDS, GET_FRIENDS);
        xmlXPathFreeObject(Obj);


    } else {
            db_resp = xmlParseDoc(BAD_CAST db_message);
            db_resp = db_generate_message(db_resp, ERROR_MESSAGE,
                                                              DB_GET_ERROR_FRIENDS_TEXT,
                                                              DB_GET_ERROR_FRIENDS_CODE);
    }

    g_debug("db_get_profile: END");
    return db_resp;
}

/**
 * @brief save binary data in db and transit 
 * @drv_response - response from drivers to db
 * @return db_response - response from db to ui
**/
xmlDocPtr DB_SAVE_BINARY_DATA(xmlDocPtr drv_response) 
{

	xmlDocPtr db_resp;
	xmlDocPtr struct_response = xmlParseDoc(BAD_CAST db_response);
	xmlNodePtr node_bin;
	xmlNodePtr node;

	gchar* xpath_req = g_strconcat(RESPONSE_PARAMS_TAG_XPATH, "/img", NULL);
	xmlXPathObject *Obj = db_xpath(xpath_req, drv_response);
	if (Obj->nodesetval->nodeNr != 0) {	
		node = Obj->nodesetval->nodeTab[0];

	} else {
		g_debug("db generate message start");
		db_resp = xmlParseDoc(BAD_CAST db_message); 
		db_resp = db_generate_message(db_resp, ERROR_MESSAGE, 
									  DB_GET_ERROR_BINARY_TEXT, 
									  DB_GET_ERROR_BINARY_CODE);
		return db_resp;
	}

	/** save binary data in db **/
	gchar* binary_id;
	gchar* binary_content;
	for (node_bin = node; node_bin != NULL; node_bin = node_bin->next) {

		g_debug("db: try save in bd %s",xmlGetProp(node_bin,BAD_CAST "uri"));

		/** if uri empty **/	
		binary_id = (gchar*)xmlGetProp(node_bin,BAD_CAST "uri");
		if (strlen(binary_id) > 0) {
		        binary_content = (gchar*)xmlNodeGetContent(node_bin);
			if ( set_binary_data_content(binary_id, binary_content) != TRUE ) {

				db_resp = xmlParseDoc(BAD_CAST db_message); 
				db_resp = db_generate_message(db_resp, ERROR_MESSAGE, 
                                                              DB_SAVE_ERROR_BINARY_TEXT, 
							      DB_SAVE_ERROR_BINARY_CODE);
				g_free(binary_id);
				g_free(binary_content);
				return db_resp;
	
			} // set binary
			g_free(binary_id);
			g_free(binary_content);
		} // if uri
	} // for

	/** get params node **/
	node = node->parent;

	xmlUnlinkNode(node);	

	db_resp = db_generate_response(struct_response, node, CLASS_BINARY_DATA, GET_BINARY_DATA);	

    xmlXPathFreeObject(Obj);
    xmlFreeDoc(drv_response);
	return db_resp;

}

/**
 * @brief save message in db 
 * @drv_response - response from drivers to db
 * @return db_response - response from db to ui
**/
xmlDocPtr DB_SAVE_INBOX_MESSAGES(xmlDocPtr drv_response)
{
    gint (*save_func) (xmlDocPtr, gchar*) = NULL;    
    gchar* text_ok = NULL;
    gchar* code_ok = NULL;
    gchar* text_error = NULL;
    gchar* code_error = NULL;

    /** get function **/
	xmlXPathObject *Obj = db_xpath(RESPONSE_TAG_XPATH, drv_response);
    if (Obj->nodesetval->nodeNr != 0 ) {
        xmlNodePtr func_node = Obj->nodesetval->nodeTab[0];
        
        xmlChar* func = xmlGetProp(func_node, BAD_CAST "function");
        if (strcmp((char*)func, UPDATE_INBOX_MESSAGES) == 0){
                save_func = create_or_update_message; 
                text_ok = DB_UPDATE_MESSAGE_INBOX_MESSAGES_TEXT;
                code_ok = DB_UPDATE_MESSAGE_INBOX_MESSAGES_CODE;
                text_error = DB_GET_ERROR_INBOX_MESSAGES_TEXT;
                code_error = DB_GET_ERROR_INBOX_MESSAGES_CODE;
        }

        if (strcmp((char*)func, UPDATE_OUTBOX_MESSAGES) == 0){
                save_func = create_or_update_owner_message; 
                text_ok = DB_UPDATE_MESSAGE_OUTBOX_MESSAGES_TEXT;
                code_ok = DB_UPDATE_MESSAGE_OUTBOX_MESSAGES_CODE;
                text_error = DB_GET_ERROR_OUTBOX_MESSAGES_TEXT;
                code_error = DB_GET_ERROR_OUTBOX_MESSAGES_CODE;
        }    
        xmlFree(func);
    
    }
    xmlXPathFreeObject(Obj);

	gchar* xpath_req = g_strconcat(RESPONSE_PARAMS_TAG_XPATH, "/array/struct", NULL);
	Obj = db_xpath(xpath_req, drv_response);
	g_free(xpath_req);

	xmlNodePtr node;
	xmlDocPtr db_resp;
	gint i;

	gchar* message_id;

    gboolean save_error = FALSE;

	if (Obj->nodesetval->nodeNr != 0) {	
		node = Obj->nodesetval->nodeTab[0];

	} else {
		g_debug("db generate message start");
		db_resp = xmlParseDoc(BAD_CAST db_message); 
		db_resp = db_generate_message(db_resp, ERROR_MESSAGE, 
                                               text_error, 
                                               code_error);
        xmlFreeDoc(drv_response);
        xmlXPathFreeObject(Obj);
		return db_resp;
	}

	
	for(i = 0; i<Obj->nodesetval->nodeNr; i++) {

        node = Obj->nodesetval->nodeTab[i];

        if (node == NULL) {
            save_error = TRUE;
            continue;        
        }

		db_resp = xmlNewDoc(BAD_CAST XML_VERSION);
		message_id = (gchar*)xmlGetProp(node,BAD_CAST "id");
	    xmlUnlinkNode(node);
		xmlDocSetRootElement(db_resp, node);
				
		if (save_func(db_resp, message_id) != 0) {
            save_error = TRUE;
		}
		g_free(message_id);  
        xmlFreeDoc(db_resp);
	}

    xmlXPathFreeObject(Obj);
    xmlFreeDoc(drv_response);
    
    if(save_error == TRUE || i == 0) {
            db_resp = xmlParseDoc(BAD_CAST db_message); 
			db_resp = db_generate_message(db_resp, ERROR_MESSAGE, 
                                                   text_error, 
                                                   code_error);
    } else {
            db_resp = xmlParseDoc(BAD_CAST db_message); 
            db_resp = db_generate_message(db_resp, INFO_MESSAGE, text_ok, code_ok);
    }    
   
    return db_resp;		
	
}


// NEW
/**
 * @brief get message in db 
 * @drv_response - response from db to ui
 * @return db_response - response from db to ui
**/
xmlDocPtr DB_GET_INBOX_MESSAGES(xmlDocPtr ui_request)
{
    /** get function **/    
    GSList* response_list = NULL;
    xmlNodePtr node_root = xmlNewNode(NULL, BAD_CAST "Params");
    xmlNodePtr node_array = xmlNewNode(NULL, BAD_CAST "array");
    xmlAddChild(node_root, node_array);

    xmlDocPtr struct_resp = xmlParseDoc(BAD_CAST db_response);

    gint messages_from = 0;
    gint messages_to = 0;
    gchar* messages_from_str = NULL;
    gchar* messages_to_str = NULL;

   /** get parametrs in Params node**/
   xmlXPathObjectPtr Obj = db_xpath(REQUEST_PARAMS_TAG_XPATH, ui_request);
   if (Obj->nodesetval->nodeNr != 0) {
       xmlNodePtr node_set = Obj->nodesetval->nodeTab[0];

       /** get From **/			
       gchar* str = (gchar*)xmlGetProp(node_set, BAD_CAST "From");
      
       if (str != NULL) {
             messages_from = atoi(str);
       } else {
             messages_from = 0;			
       }
       g_free(str);        
       
       messages_from_str =  g_strdup_printf("%d", messages_from);
       xmlSetProp(node_root, BAD_CAST "From", BAD_CAST messages_from_str);

	    /** get TO **/ 
	    str = (gchar*)xmlGetProp(node_set,BAD_CAST "To");
	    if (str != NULL) {
		    messages_to = atoi(str);
	    } else {
		    messages_to = 10;			
	    } 
	
        messages_to_str = g_strdup_printf("%d", messages_to);
        xmlSetProp(node_root, BAD_CAST "To", BAD_CAST messages_to_str);
            g_free(str);
   }

    guint messages_count = 0;
    xmlDocPtr messages = NULL;
    
        gchar* text_error = NULL;
    gchar* code_error = NULL;
    gchar* cur_function = NULL;
    
    Obj = db_xpath(REQUEST_TAG_XPATH, ui_request);
    if (Obj->nodesetval->nodeNr != 0 ) {
        xmlNodePtr func_node = Obj->nodesetval->nodeTab[0];
        
        xmlChar* func_name = xmlGetProp(func_node,BAD_CAST "function");
        if (strcmp((char*)func_name, GET_INBOX_MESSAGES) == 0){
//            response_list = get_messages();
            messages = get_messages_with_filter(messages_from_str, messages_to_str, &messages_count);
            text_error = DB_GET_ERROR_INBOX_MESSAGES_TEXT;
            code_error = DB_GET_ERROR_INBOX_MESSAGES_CODE;
            cur_function = GET_INBOX_MESSAGES;    
        }

        if (strcmp((char*)func_name, GET_OUTBOX_MESSAGES) == 0){
    //        return DB_GET_INBOX_MESSAGES_(ui_request);
            messages = get_owner_messages_with_filter(messages_from_str, messages_to_str, &messages_count);
            text_error = DB_GET_ERROR_OUTBOX_MESSAGES_TEXT;
            code_error = DB_GET_ERROR_OUTBOX_MESSAGES_CODE;      
            cur_function = GET_OUTBOX_MESSAGES;        
        }    
        xmlFree(func_name);        
    
    }
    xmlXPathFreeObject(Obj);
    
    
    
    
    g_free(messages_from_str);
    g_free(messages_to_str);

    /** error **/
    if (messages == NULL ) {

        xmlDocPtr db_resp = xmlParseDoc(BAD_CAST db_message); 
        db_resp = db_generate_message(db_resp, ERROR_MESSAGE, 
                                               DB_GET_ERROR_OUTBOX_MESSAGES_TEXT,
		                                       DB_GET_ERROR_OUTBOX_MESSAGES_CODE);
      printf("\nEND\n");
        return db_resp;
   }

   gchar* text_count = g_strdup_printf("%d", messages_count);		
    xmlSetProp(node_root, BAD_CAST "messagesQuantity", BAD_CAST text_count);
    g_free(text_count);

    xmlNodePtr messages_root = xmlDocGetRootElement(messages);

    xmlNodePtr messages_nodes = messages_root->children;
    messages_root->children = NULL;
    
    while (messages_nodes != NULL) {
        xmlNodePtr tmpNode = messages_nodes->next;
        xmlUnlinkNode(messages_nodes);            
        xmlAddChild(node_array, messages_nodes);
        messages_nodes = tmpNode;
    }

    xmlFreeDoc(messages);



    xmlDocPtr db_resp = db_generate_response(struct_resp, node_root,
				   CLASS_BOX_MESSAGE, GET_OUTBOX_MESSAGES);//cur_function);
    
    xmlFreeDoc(ui_request);
    //xmlXPathFreeObject(Obj);

    return db_resp;
}



/**
 * @brief get message in db 
 * @drv_response - response from db to ui
 * @return db_response - response from db to ui
**/
xmlDocPtr DB_GET_INBOX_MESSAGES_OLD(xmlDocPtr ui_request)
{
    /** get function **/    
    GSList* response_list = NULL;
    xmlNodePtr node_root = xmlNewNode(NULL, BAD_CAST "Params");
    xmlNodePtr node_array = xmlNewNode(NULL, BAD_CAST "array");
    xmlAddChild(node_root, node_array);
		
    xmlNodePtr node_set;
    xmlDocPtr db_resp;
    gint i;	

    gchar* text_error = NULL;
    gchar* code_error = NULL;
    gchar* cur_function = NULL;

    xmlXPathObject *Obj = db_xpath(REQUEST_TAG_XPATH, ui_request);
    if (Obj->nodesetval->nodeNr != 0 ) {
        xmlNodePtr func_node = Obj->nodesetval->nodeTab[0];
        
        xmlChar* func_name = xmlGetProp(func_node,BAD_CAST "function");
        if (strcmp((char*)func_name, GET_INBOX_MESSAGES) == 0){
            response_list = get_messages();
            text_error = DB_GET_ERROR_INBOX_MESSAGES_TEXT;
            code_error = DB_GET_ERROR_INBOX_MESSAGES_CODE;
            cur_function = GET_INBOX_MESSAGES;    
        }

        if (strcmp((char*)func_name, GET_OUTBOX_MESSAGES) == 0){
            response_list = get_owner_messages(); 
            text_error = DB_GET_ERROR_OUTBOX_MESSAGES_TEXT;
            code_error = DB_GET_ERROR_OUTBOX_MESSAGES_CODE;      
            cur_function = GET_OUTBOX_MESSAGES;        
        }    
        xmlFree(func_name);        
    
    }
    xmlXPathFreeObject(Obj);
    
    guint messages_count = g_slist_length(response_list);
    gchar* text_count = g_strdup_printf("%d", messages_count);		
    xmlSetProp(node_root, BAD_CAST "messagesQuantity", BAD_CAST text_count);
    g_free(text_count);
    
    /** error **/
    if (response_list == NULL ) {

        db_resp = xmlParseDoc(BAD_CAST db_message); 
        db_resp = db_generate_message(db_resp, ERROR_MESSAGE, 
                                               text_error,
		                                       code_error);
        return db_resp;
   }
   response_list = g_slist_sort(response_list, (GCompareFunc)db_sort_time);
   xmlDocPtr struct_resp = xmlParseDoc(BAD_CAST db_response);

   gint messages_from = 0;
   gint messages_to = 0;

   gchar* str = NULL;

   /** get parametrs in Params node**/
   Obj = db_xpath(REQUEST_PARAMS_TAG_XPATH, ui_request);
   if (Obj->nodesetval->nodeNr != 0) {
       node_set = Obj->nodesetval->nodeTab[0];

       /** get From **/			
       str = (gchar*)xmlGetProp(node_set, BAD_CAST "From");
       if (str != NULL) {
             messages_from = atoi(str);
       } else {
             messages_from = 0;			
       }
       g_free(str);        
       
       char * node_prop =  g_strdup_printf("%d", messages_from);
       xmlSetProp(node_root, BAD_CAST "From", BAD_CAST node_prop);
       free(node_prop);		

	    /** get TO **/ 
	    str = (gchar*)xmlGetProp(node_set,BAD_CAST "To");
	    if (str != NULL) {
		    messages_to = atoi(str);
	    } else {
		    messages_to = 10;			
	    } 
	
        char *message_to_prop = g_strdup_printf("%d", messages_to);
        xmlSetProp(node_root, BAD_CAST "To", BAD_CAST message_to_prop);
        g_free(message_to_prop);
   }

	GSList* list = g_slist_nth(response_list, messages_from);

    for (i = messages_from; i < messages_to; i++) {

            node_set = xmlDocGetRootElement((xmlDocPtr)list->data);
            xmlUnlinkNode(node_set);
            xmlAddChild(node_array, node_set);

            if (list->next == NULL) {
                    break;
            }

            xmlDocSetRootElement((xmlDocPtr)list->data, NULL);

            list = list->next;

    }
    
    /* free memory in list elements */
    while(response_list != NULL)
    {
	    xmlDocPtr cur_node = (xmlDocPtr)response_list->data;
	    response_list = g_slist_remove(response_list, cur_node);
	    xmlFreeDoc(cur_node);
    }

    db_resp = db_generate_response(struct_resp, node_root,
				   CLASS_BOX_MESSAGE, cur_function);
    
    xmlFreeDoc(ui_request);
    //xmlXPathFreeObject(Obj);
    g_free(str);
    return db_resp;
}

/**
 * @brief save new message in BD and generate infoMessage 
 *
 * @brief save message in db 
 * @drv_response - response from db to ui
 * @return db_response - response from db to ui
**/
xmlDocPtr GET_NEW_MESSAGES_FUNC(xmlDocPtr drv_response)
{
    xmlXPathObject *Obj = db_xpath("//TransitData/Content/Response/Params/array/struct", drv_response);

    xmlNodePtr node;
    xmlDocPtr db_resp;
    xmlChar* message_id;
    
    int i;
    int message_count = Obj->nodesetval->nodeNr;

    //g_debug("message count !!!! %d", message_count);
    //xmlDocDump(stdout, drv_response);    
    //sleep(10);

	for(i = 0; i<message_count; i++) {

        node = Obj->nodesetval->nodeTab[i];
	    db_resp = xmlNewDoc(BAD_CAST XML_VERSION);
	    message_id = xmlGetProp(node, BAD_CAST "id");

	    xmlUnlinkNode(node);
	    xmlDocSetRootElement(db_resp, node);
			
	    if (create_or_update_message(db_resp, (gchar*)message_id) != 0)
	    {
		    db_resp = xmlParseDoc(BAD_CAST db_message); 
		    db_resp = db_generate_message(db_resp, ERROR_MESSAGE, 
                                                   DB_GET_ERROR_INBOX_MESSAGES_TEXT, 
                                                   DB_GET_ERROR_INBOX_MESSAGES_CODE);
		    return db_resp;
	    }
    }

    /** if response have any new message **/
    if (message_count != 0) {

        db_resp = xmlParseDoc(BAD_CAST db_message); 
        db_resp = db_generate_message(db_resp, INFO_MESSAGE, 
                                               g_strdup_printf("%s %d!", DB_NEW_INBOX_MESSAGES_TEXT, message_count),
                                               DB_NEW_INBOX_MESSAGES_CODE);
         xmlFreeDoc(drv_response);
         return db_resp;   
     }

     xmlXPathFreeObject(Obj);
     xmlFreeDoc(drv_response);
     return NULL;
}

/**
 * @brief create response 
 *
 * @param response - defined response struct
 * @param resp_params - Params node from external response  
 * @param class_name - class response name
 * @param function_name - func response name
 */
static xmlDocPtr db_generate_response(xmlDocPtr response, xmlNodePtr resp_params, 
							char* class_name, char* function_name)
{
	g_debug("db_generate_response: START");

	xmlNodePtr node;
	xmlXPathObject *Obj;

    if (response == NULL) {
        g_debug("response is null!");
        return 0;
    }

	/** get request node and set class and function**/	
    Obj = db_xpath(RESPONSE_TAG_XPATH, response);
    if (Obj == NULL) {
        g_debug("obj is null!");
        return 0;
    }
        
	if (Obj->nodesetval->nodeNr != 0) {

		node = Obj->nodesetval->nodeTab[0];
	
		g_debug("obj is not null %s", node->name);		
	
		xmlSetProp(node, BAD_CAST "class", BAD_CAST class_name);	
		xmlSetProp(node, BAD_CAST "function", BAD_CAST function_name);
		
		/** add resp_params node with all information, in response struct **/
		xmlAddChild(node, resp_params);

//        xmlChar *content = xmlNodeGetContent(node);
//        xmlFree(content);

//        free(class_name);
//        free(function_name);    
    }

    xmlXPathFreeObject(Obj);
	g_debug("db_generate_response: END");
    return response;
}


/**
 * @brief create db message 
 *
 * @param response - define response 
 * @param function_name - func response name
 * @param text - text message
 * @param code - code message
 */
static xmlDocPtr db_generate_message(xmlDocPtr response, char* function_name,  
	     					  char* text, char* code)
{
	g_debug("db_generate_message: START");

	xmlNodePtr node;
	xmlXPathObject *Obj;

	/** get request node and set function**/
	Obj = db_xpath(RESPONSE_TAG_XPATH, response);
	if (Obj->nodesetval->nodeNr != 0) {
		node = Obj->nodesetval->nodeTab[0];
		xmlSetProp(node, BAD_CAST "function", BAD_CAST function_name);		
	} else {
		return NULL;	
	}

	xmlXPathFreeObject(Obj);

	/** set text **/
	char* xpath_req = g_strconcat(RESPONSE_TAG_XPATH,
							 	  "/Params/string[@name='text']", NULL);

	Obj = db_xpath(xpath_req, response);
    g_free(xpath_req);
	if (Obj->nodesetval->nodeNr != 0) {
		node = Obj->nodesetval->nodeTab[0];
		xmlNodeSetContent(node, BAD_CAST (text));
	} 

	xmlXPathFreeObject(Obj);

	/** set code **/
	xpath_req = g_strconcat(RESPONSE_TAG_XPATH,
							"/Params/string[@name='code']", NULL);

	Obj = db_xpath(xpath_req, response);
	g_free(xpath_req);
	if (Obj->nodesetval->nodeNr != 0) {
		node = Obj->nodesetval->nodeTab[0];
		xmlNodeSetContent(node, BAD_CAST code);
	}

	xmlXPathFreeObject(Obj);

	return response;

}

/**
 * @brief get id from Params node.
 *
 * @param class - request class.
 * @param req_type - type of request 
 * @param request - xmlDocPtr.
 * @return id or NULL if error
 */
static char* db_get_request_id(char* class, db_type req_type, xmlDocPtr request)
{
	g_debug("db_get_request_id: START");
	xmlXPathObject *Obj = NULL;
	xmlNodePtr node;
	
	if (strcmp(class, CLASS_PROFILE) == 0) {

		if (req_type == TYPE_REQUEST) {
			Obj = db_xpath(REQUEST_PARAMS_TAG_XPATH, request);
		}

		if (req_type == TYPE_RESPONSE) {
			Obj = db_xpath(RESPONSE_PARAMS_TAG_XPATH, request);
		}
			
		if (Obj->nodesetval->nodeNr != 0) {
			node = Obj->nodesetval->nodeTab[0];
		    gchar *id = (char*) xmlGetProp(node, BAD_CAST "id");
			g_debug("db_get_request_id: END %s", id);
            xmlXPathFreeObject(Obj);
			return id;

		}

	}

    xmlXPathFreeObject(Obj);
	g_debug("db_get_request_id: END with error");
	return NULL;
}

/**
 * @brief sort gslist asc.
 */
static gint db_sort_asc(gpointer a, gpointer b)
{
	xmlDocPtr doc_a = (xmlDocPtr)a;
	xmlDocPtr doc_b = (xmlDocPtr)b;
	
	xmlNodePtr node_a = NULL;
	xmlNodePtr node_b = NULL;	

	xmlXPathObject *Obj = db_xpath("//struct/string[@name='FriendName']", doc_a);

	if (Obj->nodesetval->nodeNr > 0) {
		node_a = Obj->nodesetval->nodeTab[0];
	}
    xmlXPathFreeObject(Obj);

	Obj = db_xpath("//struct/string[@name='FriendName']", doc_b);

	if (Obj->nodesetval->nodeNr > 0) {
		node_b = Obj->nodesetval->nodeTab[0];
	}

    xmlChar* cont_a = xmlNodeGetContent(node_a);
    xmlChar* cont_b = xmlNodeGetContent(node_b);
    
    gint icont_count = strcmp((char*)cont_b, (char*)cont_a);

    xmlFree(cont_a);
    xmlFree(cont_b);
    xmlXPathFreeObject(Obj);

	return icont_count;
}

/**
 * @brief sort gslist asc.
 */
static gint db_sort_time(gpointer a, gpointer b)
{

    xmlDocPtr doc_a = (xmlDocPtr)a;
    xmlDocPtr doc_b = (xmlDocPtr)b;

    xmlNodePtr node_a = NULL;
    xmlNodePtr node_b = NULL;


    xmlXPathObject *Obj = db_xpath("//struct/date[@name='Time']", doc_a);

    if (Obj != NULL && Obj->nodesetval!= NULL && Obj->nodesetval->nodeNr > 0) {
            node_a = Obj->nodesetval->nodeTab[0];
    }

    xmlXPathFreeObject(Obj);

    Obj = db_xpath("//struct/date[@name='Time']", doc_b);

    if (Obj!= NULL && Obj->nodesetval != NULL && Obj->nodesetval->nodeNr > 0) {
            node_b = Obj->nodesetval->nodeTab[0];
    }

    xmlChar* cont_a = xmlNodeGetContent(node_a);
    xmlChar* cont_b = xmlNodeGetContent(node_b);
    
    gint icont_count = g_strcmp0((gchar*)cont_b, (gchar*)cont_a);

    xmlFree(cont_a);
    xmlFree(cont_b);

    xmlXPathFreeObject(Obj);
    return icont_count;

} 

/**
 * @brief sort gslist desc.
 */
static gint db_sort_desc(gpointer a, gpointer b)
{
	xmlDocPtr doc_a = (xmlDocPtr)b;
	xmlDocPtr doc_b = (xmlDocPtr)a;
	
	xmlNodePtr node_a = NULL;
	xmlNodePtr node_b = NULL;	

	xmlXPathObject *Obj = db_xpath("//struct/string[@name='FriendName']", doc_a);

	if (Obj->nodesetval->nodeNr > 0) {
		node_a = Obj->nodesetval->nodeTab[0];
	}
    xmlXPathFreeObject(Obj);

	Obj = db_xpath("//struct/string[@name='FriendName']", doc_b);

	if (Obj->nodesetval->nodeNr > 0) {
		node_b = Obj->nodesetval->nodeTab[0];
	}

    xmlChar* cont_a = xmlNodeGetContent(node_a);
    xmlChar* cont_b = xmlNodeGetContent(node_b);
    
    gint icont_count = strcmp((char*)cont_b, (char*)cont_a);

    xmlFree(cont_a);
    xmlFree(cont_b);

    xmlXPathFreeObject(Obj);
	return icont_count;
	
} 


/**
 * @brief search friend.
 */
static gint db_friends_find(gpointer _data, gpointer _filter)
{
	xmlChar* filter = (xmlChar*)_filter;
	xmlChar* name;
	xmlDocPtr doc_data = (xmlDocPtr)_data;

	xmlXPathObject *Obj = db_xpath("//struct/string[@name='FriendName']", doc_data);
	
	if (Obj->nodesetval->nodeNr > 0) {
		xmlNodePtr node_data = Obj->nodesetval->nodeTab[0];
		name = (xmlChar*)g_utf8_strdown((gchar*)xmlNodeGetContent(node_data), -1);
		filter = (xmlChar*)g_utf8_strdown((gchar*)filter, -1);
		if (xmlStrcasestr(name, filter) != NULL) {
                xmlXPathFreeObject(Obj);
			return 0;		
		}	
	}
	
    xmlXPathFreeObject(Obj);	
	return 1;	
}

/**
 * @brief search online friends
 */
static gint db_friends_online(gpointer _data, gpointer _filter)
{
	xmlChar* filter = (xmlChar*)_filter;
	xmlChar* name;
	xmlDocPtr doc_data = (xmlDocPtr)_data;

	xmlXPathObject *Obj = db_xpath("//struct/string[@name='FriendStatus']", doc_data);
	
	if (Obj->nodesetval->nodeNr > 0) {
		xmlNodePtr node_data = Obj->nodesetval->nodeTab[0];
		name = (xmlChar*)g_utf8_strdown((gchar*)xmlNodeGetContent(node_data), -1);
		filter = (xmlChar*)g_utf8_strdown((gchar*)filter, -1);
		if (xmlStrcasestr(name, filter) != NULL) {
		    xmlFree(name);
		    xmlXPathFreeObject(Obj);
			return 0;		
		}	
	}
	
    xmlXPathFreeObject(Obj);	
	return 1;	

}


