/*
 * kernel.c - kernel of MSA program.
 * This file is part of MSA program.
 *
 * Copyright (C) 2009 - Alexander A. Lomov
 *
 * MSA program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MSA program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MSA program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA  02110-1301  USA
 */
 

//#define TEST_KERNEL

#include "kernel.h"
#include "repository.h"
#include "handler.h"


#ifdef TEST_KERNEL
#include "test.h"
#endif

#define INCLUDE_TEST_FUNC 1

/* Max count of attemps for sending data to driver */
#define MAX_KERNEL_SEND_ATTEMPT_COUNT 20

/* Wake signal settings */
#define WAKE_KERNEL_START_TIME "20"
#define WAKE_KERNEL_REPEAT_TIME "10"
#define WAKE_KERNEL_SIGNAL_ID "kernel_1"
 


/* Indicates that a kernel work on transit data */
static gboolean is_process_proceed = FALSE;
G_LOCK_DEFINE_STATIC(is_process_proceed);

/* Main thread of kernel packet processing */
static GThread* process_thread = NULL;

/* Signal for Handler */
static xmlChar* wake_kernel_signal = NULL;

/* Status for transit data structure, sets for each target */
typedef enum process_status {
    NOT_SEND = 0,       // Message not send yet.
    IN_PROCESS = 2,     // Message is processed by "target". 
    PROCESSED = 4,      // Message processed by "target".
    FORCE_REMOVE = 8,   // Remove "target" from send message process. 
    IGNORE = 16         // Ignore send message for "target'.
} process_status;

/* Describe target for transit data */ 
typedef struct targets {
    xmlChar* name;          // Target name.
    gint order_number;      // Order number in query.
    process_status status;  // Target status.
    guint attempts_count;   // Count of send message. 
} target_data;


/* Init and set wake signal function */
static void set_wake_signal(const gchar* signal_id, const gchar* action,  
                            const gchar* start_time, const gchar* repeat_time);
static gint register_wake_signal();


/* Function for work with xml data */
static xmlChar* get_source(const xmlDocPtr doc);
static GList* get_targets(xmlDocPtr doc);
static target_data* create_target(const xmlNodePtr target_node);
static xmlXPathObjectPtr get_nodeset(const xmlDocPtr doc, 
                                     const xmlChar* xpath);
                                

/* Function convert xml data to inner format */
static transit_data* convert_to_kernel_format(const xmlDocPtr doc);
static gint compare_target_order(const target_data* a, const target_data* b);

/* Function for work with transit data structures, and drivers */
static void processing();                                   
static void free_targets_list(GList* targets);
static void free_transit_data(transit_data* tr_data);
static gint send_data_to_driver(const target_data* target, 
                                const xmlDocPtr data);
static gboolean check_transit_data_status(const transit_data* tr_data, 
                                          const process_status status);

/* Other functions */
static void free_reposityry_data();
static void shutdown_drivers();



#ifdef INCLUDE_TEST_FUNC
#include "../../test/kernel_test/kernel_test_func.c"
#endif


/**************************************************/
/*************** External functions ***************/

/**
 * 	@brief Initializes kernel. First start this function, then use other.
 *
 */
gint kernel_initialization()
{
    
    KERNEL_DEBUG(1) ("KERNEL: kernel_initialization START");

    if (g_thread_supported() == FALSE) {
	    g_thread_init(NULL);
	}
    
    if (msa_disp_init() == FAILURE) {
        KERNEL_DEBUG(3) ("KERNEL: msa_disp_init error!");
        return FAILURE;
    }
    
    if (handler_initialization() == FAILURE) {
        KERNEL_DEBUG(3) ("KERNEL: handler_initialization error!");
        return FAILURE; 
    }
 
    set_wake_signal(WAKE_KERNEL_SIGNAL_ID, SET_SIGNAL_TEXT, 
                    WAKE_KERNEL_START_TIME, WAKE_KERNEL_REPEAT_TIME);

   //gint register_result = 0;
   // if ((register_result = register_wake_signal()) != SUCCESS) {
   //     return register_result;
   // }
    KERNEL_DEBUG(1) ("KERNEL: kernel_initialization END");
    //KERNEL_DEBUG(1, "KERNEL: kernel_initialization END");
	  
    return FAILURE;
    
}


/**
 * @brief Takes transit data processing.
 *
 * @param data xml document describing transit data.
 *
 * @return 0 on success or not 0 otherwise.
 */
gint kernel_put_data(xmlDocPtr data)
{
    transit_data* tr_data = convert_to_kernel_format(data); 
 
    KERNEL_DEBUG(5) ("\n\n\nKERNEL: data in kernel:"); \
//    xmlDocDump(stdout, data);

    if (tr_data == NULL) {
        KERNEL_DEBUG(4) ("KERNEL: kernel_put_data; tr_data = NULL");
        return FAILURE;
    } 

    if(repository_add_data(tr_data) != SUCCESS) {
        free_transit_data(tr_data);
        return FAILURE;
    }
//Debug
//   tr_data = repository_get_transit_data_by_index(0, TRUE);
//  free_transit_data(tr_data);       
//return 0;
    G_LOCK(is_process_proceed);
    if (is_process_proceed != TRUE) {
        KERNEL_DEBUG(4) ("KERNEL: Start thread of processing");
        is_process_proceed = TRUE;
        if (process_thread != NULL) {
             g_thread_join(process_thread);
             process_thread = NULL;
        }
        process_thread = g_thread_create((GThreadFunc)processing, 
                                         NULL, TRUE, NULL);
    }
    G_UNLOCK(is_process_proceed);    

    return SUCCESS;
}


/**
 * @brief Takes transit data processing.
 *
 * @param data text describing xml format if transit data.
 *
 * @return 0 on success or not 0 otherwise.
 */
gint kernel_put_text_data(xmlChar* data)
{
    KERNEL_DEBUG(4) ("KERNEL: kernel_put_text_data START");
    if (data == NULL) {
        return FAILURE;
    } 

    xmlDocPtr doc = xmlParseDoc(data);

    xmlFree(data);

    data = NULL;

    if (doc == NULL) {
        return ERROR_CANT_PARSE_DATA;
    } 
    
    KERNEL_DEBUG(4) ("KERNEL: kernel_put_text_data END");
    return kernel_put_data(doc);
}

//FIXME: need for equality between other plugins and modules
gint kernel_ext_put_text_data(xmlDocPtr request, xmlDocPtr* response)
{
    return kernel_put_data(request);
}

/**
 * 	@brief Shutdown kernel.
 *
 */
void kernel_shutdown()
{
    KERNEL_DEBUG(1) ("KERNEL: shutdown START");
    G_LOCK(is_process_proceed);
    is_process_proceed = FALSE;
    G_UNLOCK(is_process_proceed);
   
    if (process_thread != NULL) {
        KERNEL_MESSAGE(5) ("KERNEL: join to process");
        g_thread_join(process_thread);
        KERNEL_MESSAGE(5) ("KERNEL: shutdown join to process end");
        process_thread = NULL;
    }
    free_reposityry_data();
  
    xmlFree(wake_kernel_signal);
    wake_kernel_signal  = NULL;;   

    KERNEL_DEBUG(1) ("KERNEL: shutdown END");
}


/**
 * @brief Free all data in repository and reposityry.
 *
 */
static void free_reposityry_data()
{
    transit_data* tr_data = NULL;
    while ((tr_data = repository_get_oldest_transit_data(TRUE)) != NULL) {
        free_transit_data(tr_data);     
    }

    repository_free();
}


/**************************************************/
/***************** Static functions ****************/
/**
 * @brief Fills wake signal content by given data.
 *
 * @param signal_id identifier of signal.
 * @param action action for handler.
 * @param start_time start time for first call.
 * @param repeat_time time for repeat calls.
 */
static void set_wake_signal(const gchar* signal_id, const gchar* action,  
                            const gchar* start_time, const gchar* repeat_time)
{
    xmlFree(wake_kernel_signal);
        
    wake_kernel_signal = BAD_CAST g_strconcat("<?xml version=\"1.0\"?>\
<TransitData type=\"signal\">\
<SourceID>", KERNEL_ID, "</SourceID>\
<TargetID>", HANDLER_ID, "</TargetID>\
<Content>\
<Signal id=\"", signal_id,"\">\
<StartTime>", start_time, "</StartTime>\
<Action>", action,"</Action>\
<RepeatTime>", repeat_time,"</RepeatTime>\
<Content>\
<TransitData id=\"0\" type=\"data\">\
<SourceID>kernel</SourceID>\
<TargetID>kernel</TargetID>\
<Content></Content>\
</TransitData>\
</Content>\
</Signal>\
</Content>\
</TransitData>", NULL);

}


/**
 * @brief Register wake kernel signal for kernel.
 *
 * @return 0 on success registration or not 0 otherwise
 */
static gint register_wake_signal()
{
    if (handler_put_data(xmlStrdup(wake_kernel_signal)) != 0) {
        return ERROR_CANT_SEND_DATA;
    }
 
    return SUCCESS;
}


/**
 * @brief Gets a source of transin data.
 *
 * @param doc xml document describing transit data.
 *
 * @return content of source node, NULL otherwise
 */
static xmlChar* get_source(const xmlDocPtr doc)
{
    xmlXPathObjectPtr result = get_nodeset(doc, BAD_CAST SOURCE_TAG_XPATH);
   
    if (result == NULL) {
        return NULL;
    }
     
    xmlNodeSetPtr nodeset = result->nodesetval;
   
    xmlChar* source_id = NULL;
    if (nodeset->nodeNr > 0) {
        source_id = xmlNodeGetContent(nodeset->nodeTab[0]);
        KERNEL_DEBUG(5) ("KERNEL: get_source; Source id = %s", source_id);
    }
   
    xmlXPathFreeObject(result);
    return source_id;
}


/**
 * @brief Gets the targets from given transit data.
 *
 * @param doc xml document describing transit data.
 *
 * @return list of targets or NULL otherwise.
 */
static GList* get_targets(xmlDocPtr doc)
{
    KERNEL_DEBUG(5) ("KERNEL: get_targets START");
    xmlXPathObjectPtr result = get_nodeset(doc, BAD_CAST TARGET_TAG_XPATH);

    if (result == NULL) {
        return NULL;
    }
     
    xmlNodeSetPtr nodeset = result->nodesetval;
   
    GList* targets = NULL;

    for (gint i = 0; i < nodeset->nodeNr; ++i) {
        target_data* target = create_target(nodeset->nodeTab[i]);
        
        targets = g_list_insert_sorted(targets, (gpointer)target, 
                                       (GCompareFunc)compare_target_order);
    }
    
    for (gint i = 0; i < nodeset->nodeNr; ++i) {
        xmlUnlinkNode(nodeset->nodeTab[i]);
        xmlFreeNode(nodeset->nodeTab[i]);
        nodeset->nodeTab[i] = NULL;
    }
    
    // Kulakov: valgrind says that is already freed.
    // Lomov: Valgrind is lier. Now all is well.
    xmlXPathFreeObject(result);
    
    KERNEL_DEBUG(5) ("KERNEL: get_targets END");
    return targets;
}


/**
 * @brief Create target data from xml node.
 *
 * @param xml node describing target.
 *
 * @return new target data or NULL otherwise.
 */
static target_data* create_target(const xmlNodePtr target_node)
{
    target_data* target = g_try_new(target_data, 1);
    
    if (target == NULL ) {
        return NULL;
    }

    target->name = xmlNodeGetContent(target_node);

    xmlChar* order = xmlGetProp(target_node, BAD_CAST TARGET_TAG_ORDER_ATTR);

    target->order_number = (order != NULL) 
                            ? (gint)strtod((gchar*)order, NULL)
                            : -1;

    target->status = NOT_SEND;
    target->attempts_count = 0;
    
    xmlFree(order);
    
    return target;
}


/**
 * @brief Gets nodes from xml document using XPath expression.
 *
 * @param doc xml document.
 * @param xpath XPath expression.
 *
 * @return find nodes or NULL otherwise.
 */
static xmlXPathObjectPtr get_nodeset(const xmlDocPtr doc, const xmlChar* xpath)
{
    if (xmlDocGetRootElement(doc) == NULL || xpath == NULL) {
        KERNEL_DEBUG(3) ("KERNEL: get_nodeset; root = NULL or xpath: %s\n", xpath);
        return NULL;
    }
	xmlXPathContextPtr context = xmlXPathNewContext(doc);
   
	if (context == NULL) {
		KERNEL_DEBUG (4) ("Error in xmlXPathNewContext\n");
		return NULL;
	}
	
	xmlXPathObjectPtr result = xmlXPathEvalExpression(xpath, context);
	xmlXPathFreeContext(context);
	
	if (result == NULL) {
		KERNEL_DEBUG (4) ("Error in xmlXPathEvalExpression\n");
		return NULL;
	}
	
	if (xmlXPathNodeSetIsEmpty(result->nodesetval)) {
	    KERNEL_MESSAGE(4)("KERNEL: get_nodeset; EMPTY XPATH\n");
		xmlXPathFreeObject(result);
		return NULL;
	}
	
	return result;
}


/**
 * @brief Add target to transit data.
 *
 * @param doc xml document describing transit data.
 * @param target_id identifier of target.
 * @param order_number order number.
 */
void add_target(const xmlDocPtr doc, const xmlChar* target_id, 
                       const gint order_number)
{
    xmlNodePtr root_node = xmlDocGetRootElement(doc);
    if (root_node == NULL || target_id == NULL) {
        return;
    }
    
    if (xmlStrlen(target_id) == 0) {
        return;
    }
    
    xmlNodePtr target = xmlNewNode(NULL, BAD_CAST TARGET_TAG);
    xmlNodeSetContent(target, target_id);
    
    if (order_number > 0) {
        gchar buffer[25];
        gint char_count = sprintf(buffer, "%i", order_number);

        if (char_count > 0) {
            xmlSetProp(target, BAD_CAST TARGET_TAG_ORDER_ATTR, BAD_CAST buffer);
        }    
    }
    
    xmlAddChild(root_node, target);
}

/**
 * @brief Convert xml document into kernel transit data structure.
 *
 * @param doc xml document describing transit data.
 *
 * @return new transit data structure or NULL otherwise.
 */
static transit_data* convert_to_kernel_format(const xmlDocPtr doc)
{
    KERNEL_DEBUG(3) ("KERNEL: convert_to_kernel_format START");
    GList* targets = get_targets(doc);
    xmlChar* source_id = get_source(doc);
    transit_data* tr_data = g_try_new(transit_data, 1);


    if(tr_data == NULL || targets == NULL || source_id == NULL) {
        free_targets_list(targets);
        xmlFree(source_id);
        g_free(tr_data);

        return NULL;
    } 
    
    tr_data->id = NULL;
    tr_data->source_id = source_id;
    tr_data->targets = targets;  
    tr_data->content = doc;
    
    KERNEL_DEBUG(3) ("KERNEL: convert_to_kernel_format END");
    return tr_data;
} 


/**
 * @brief Compare order of target.
 *
 * @param a first target.
 * @param b second target.
 *
 * @return if both greater than zero: -1 - order of b greater than a, 
 *         otherwise 1
 *         if one is less or equal zero: 1 - order of b greater or equal a, 
 *         otherwise -1 
 */
static gint compare_target_order(const target_data* a, const target_data* b)
{
    if (a->order_number <= 0 || b->order_number <= 0) {
        return (a->order_number <= b->order_number) ? 1 : -1;
    }
    
    return (a->order_number < b->order_number) ? -1 : 1;
}

// FIXME: make function simply!
/**
 * @brief Gets transit data from repository, and try send it to targets.
 *
 * @return TRUE if status of transit data equal given, otherwise FALSE.
 */
static void processing()
{
    KERNEL_DEBUG(1) ("KERNEL: Processing START");
    guint index = 0;

    while (index < repository_get_length() && is_process_proceed == TRUE) {
        KERNEL_MESSAGE(3) ("KERNEL: Target Index = %i, all targets = %i", index, repository_get_length());
        transit_data* tr_data = repository_get_transit_data_by_index(index, FALSE); 	    

	    if (tr_data == NULL) {   
	        repository_get_transit_data_by_index(index, TRUE);
	        continue;
	    }    
        
        //GList* targets = tr_data->targets;
        GList* targets_walker = tr_data->targets;//targets;	

    	while (targets_walker != NULL && is_process_proceed == TRUE) {
    	     target_data* target = (target_data*)targets_walker->data;
    	     
    	     ++target->attempts_count;
    	     KERNEL_MESSAGE(3) ("KERNEL: Target name %s; attempts = %i", target->name, \
    	     target->attempts_count);

     	     if(target->status != NOT_SEND) {
                targets_walker = g_list_next(targets_walker);
    	        continue;
    	     }
    	     
    	     if(send_data_to_driver(target, tr_data->content) == SUCCESS
    	          || target->attempts_count > MAX_KERNEL_SEND_ATTEMPT_COUNT ) {
    	        target->status = PROCESSED;
    	     }
    	     
    	     targets_walker = g_list_next(targets_walker);
    	}
    	
        if (check_transit_data_status(tr_data, PROCESSED) == TRUE) {
            tr_data = repository_get_transit_data_by_index(index, TRUE);
            free_transit_data(tr_data);       
            continue;     
        }
        ++index;
	}

    G_LOCK(is_process_proceed);
    is_process_proceed = FALSE;
    G_UNLOCK(is_process_proceed);

    KERNEL_DEBUG(1) ("KERNEL: Processing END");
}


/**
 * @brief Free resources allocated for targets list.
 *
 * @param targets list of targets.
 */
static void free_targets_list(GList* targets)
{
    KERNEL_DEBUG(3) ("KERNEL: free_targets_list START");
    GList* targets_walker = targets;
    
    while (targets_walker != NULL) {
        target_data* target = (target_data*)targets_walker->data;
    	     
        xmlFree(target->name);     
        target->name = NULL;
   	    g_free(target);
   	    target = NULL; 
   	     
        targets_walker = g_list_next(targets_walker);
    }
    
    g_list_free(targets);
    KERNEL_DEBUG(3) ("KERNEL: free_targets_list END");
}


/**
 * @brief Free resources allocated for transit data.
 *
 * @param doc transit data structure.
 */
static void free_transit_data(transit_data* tr_data)
{   
    if (tr_data == NULL) {
        return;
    }
    
    KERNEL_DEBUG(3) ("KERNEL: free_transit_data FREE START");
    
    xmlFree(tr_data->source_id);
    xmlFree(tr_data->id);
    xmlFreeDoc(tr_data->content);

    tr_data->content = NULL;
    tr_data->source_id = NULL;
    tr_data->id = NULL;

    free_targets_list(tr_data->targets);
    tr_data->targets = NULL;

    g_free(tr_data);
    tr_data = NULL;

    KERNEL_DEBUG(3) ("KERNEL: free_transit_data FREE END");    
}


/**
 * Send xml document to driver with given id.
 *
 * @param driver_id identifier of driver.
 * @param data xml document.
 *
 * @return 0 on success or not 0 otherwise.
 */
static gint send_data_to_driver(const target_data* target, const xmlDocPtr data)
{
    if (xmlStrcmp(target->name, BAD_CAST KERNEL_ID) == 0) {
          return SUCCESS;
    }

    xmlDocPtr data_with_target = xmlCopyDoc(data, 1);
    add_target(data_with_target, target->name, target->order_number);
    
    KERNEL_DEBUG(3) ("\n\nKERNEL: Send data to driver (%s): ", target->name); \
   // xmlDocDump(stdout, data_with_target);
    
    //xmlChar* raw_data = NULL;
    //gint size = 0;
    //xmlDocDumpMemory(data_with_target, &raw_data, &size);
    //xmlFreeDoc(data_with_target);

        
    if (msa_disp_send_request(target->name, data_with_target) != SUCCESS) {
	xmlFreeDoc(data_with_target);
        KERNEL_MESSAGE(2) ("KERNEL: Can't send data to driver (%s)", target->name);
        return ERROR_CANT_SEND_DATA;
    }
    
    return SUCCESS;
}


static xmlChar* create_message(const target_data* target, const xmlDocPtr data)
{
    if (target == NULL || data == NULL ) {
        return NULL;
    }

    add_target(data, target->name, target->order_number);
    
    xmlChar* raw_data = NULL;
    gint size = 0;
    //xmlDocDumpMemory(data, &raw_data, &size);

    return raw_data;
}
    

static gint send_message(const target_data* target, xmlChar* message)
{
    gint (*func)(xmlChar*) = NULL; //get_execution_function((gchar*)target->name);
    
    if (func == NULL) {
        return ERROR_NO_DRIVER_FUNCTION;
    }

    if (func(message) != 0) {
        xmlFree(message);
        return ERROR_CANT_SEND_DATA;
    }
    
    return SUCCESS;
}

/**
 * @brief Check status of transit data. 
 *
 * If all targets has given status, 
 * transit data has this status too.
 *
 * @param tr_data transit data.
 * @param status status for check.
 *
 * @return TRUE if status of transit data equal given, otherwise FALSE.
 */
static gboolean check_transit_data_status(const transit_data* tr_data, 
                                          const process_status status)
{
    GList* targets_walker = tr_data->targets;
    
    while (targets_walker != NULL) {
        target_data* target = (target_data*)targets_walker->data;
    	     
    	if (status != target->status) {
            return FALSE;
        }     
    	     
        targets_walker = g_list_next(targets_walker);
    }
    
    return TRUE ;
}

