#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QDebug>
#include <datatypes/message.h>
#include <widgets/qstatwidget.h>

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);    

    init();

    this->sm = new ServiceMgr(this);
    this->sm->canUpdateAlbums = true;
    this->sm->canUpdateFeeds = true;
    this->sm->canUpdateFriends = true;
    this->sm->canUpdateMessages = true;

    imagesPanel = new ImageBrowserPanel(this->usedFont, this->sm, this);
    profilesPanel = new ProfileBrowserPanel(this->usedFont, this->sm, this);
    messagesPanel = new MessageBrowserPanel(this->usedFont, this->sm, this);
    msgComposer = new MessageComposer(this->sm, this);
    newsPanel = new NewsBrowserPanel(this->sm, this);

    QMap<int, QList<int> > panelTypeMap;
    int friendIndex = ui->stackedWidget->addWidget(profilesPanel);
    int albumsIndex = ui->stackedWidget->addWidget(imagesPanel);
    int messagesIndex = ui->stackedWidget->addWidget(messagesPanel);
    ui->stackedWidget->addWidget(newsPanel);
    ui->stackedWidget->addWidget(msgComposer);

    panelTypeMap[friendIndex] << qMetaTypeId<Friend>();
    panelTypeMap[albumsIndex] << qMetaTypeId<Album>() << qMetaTypeId<Photo>() << qMetaTypeId<Friend>();
    panelTypeMap[messagesIndex] << qMetaTypeId<Message>();

    this->sm->setPanelToTypeMap(panelTypeMap);

    connect(this->sm, SIGNAL(errorOccured(QString, QTransport::Action, bool)),
            this, SLOT(gotErrorMsg(QString, QTransport::Action, bool)));

    connect(imagesPanel, SIGNAL(openImage(Photo)), this, SLOT(openImage(Photo)));
    connect(newsPanel, SIGNAL(openImage(Photo)), this, SLOT(openImage(Photo)));

    connect(profilesPanel, SIGNAL(showFriendAlbums(Friend)), this,
            SLOT(showFriendAlbums(Friend)));
    connect(profilesPanel, SIGNAL(showFriendMessages(Friend)), this,
            SLOT(showFriendMessages(Friend)));
    connect(profilesPanel, SIGNAL(sendMessage(Friend)), this,
            SLOT(sendMessage(Friend)));
    connect(messagesPanel, SIGNAL(newMessage()), this, SLOT(createNewMessage()));
    connect(messagesPanel, SIGNAL(replyMessage(Message)), this,
            SLOT(replyMessage(Message)));
    connect(messagesPanel, SIGNAL(deleteMessage(Message)), this, SLOT(messageDeleted(Message)));
    connect(msgComposer, SIGNAL(saveMessage(Message)), this, SLOT(saveMessage(Message)));
    connect(msgComposer, SIGNAL(messageSent()), this, SLOT(messageSent()));
    connect(messagesPanel, SIGNAL(editDraftMessage(Message)), this,
            SLOT(editMessage(Message)));
    connect(ui->stackedWidget, SIGNAL(currentChanged(int)), this->sm, SLOT(changeActivePanel(int)));

    // device rotation
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));

    // check that one account must be setup
    if (this->sm->getAccounts().length() == 0) {
        UIAccountSettings *settings = new UIAccountSettings(this->sm, this);
        settings->setModal(true);
        settings->addNewAccount();
    }

    friendAction->trigger();
}

MainWindow::~MainWindow()
{
    delete sm;
    delete ui;
}

void MainWindow::init()
{
#ifdef Q_WS_MAEMO_5
    this->usedFont = QFont(this->font().family(), 14);
    this->setAttribute(Qt::WA_Maemo5StackedWindow);
    this->setAttribute(Qt::WA_Maemo5AutoOrientation, true);
    this->setWindowFlags(this->windowFlags() | Qt::Window);
    this->fullScreenButton = new FullScreenExitButton(this);
    this->fullScreenButton->hide();
#else
    this->usedFont = this->font();
#endif

    actionGroup = new QActionGroup(this);

    this->ui->mainToolBar->setIconSize(QSize(_MAINICONSIZE_,_MAINICONSIZE_));

    createActions();
    createMenus();

    ui->mainToolBar->layout()->setSpacing(0);
    ui->mainToolBar->layout()->setMargin(0);
    ui->mainToolBar->addAction(friendAction);
    ui->mainToolBar->addAction(msgAction);
    ui->mainToolBar->addAction(albumAction);
    ui->mainToolBar->addAction(newsAction);
    ui->mainToolBar->addAction(moreAction);
}

void MainWindow::createActions()
{
    bool isShowLabels = true;

#if defined(Q_WS_MAEMO_5) || defined(MEEGO_EDITION_HARMATTAN)
    isShowLabels = false;
#endif

    // toolbar actions
    friendAction = new QAction(isShowLabels ? tr("&Friends") : "", this);
    friendAction->setIcon(QIcon(":/res/friends.png"));
    friendAction->setShortcut(QKeySequence(Qt::ALT + Qt::Key_1));
    friendAction->setCheckable(true);
    connect(friendAction, SIGNAL(triggered()), SLOT(onFriendsTriggered()));
    actionGroup->addAction(friendAction);

    msgAction = new QAction(isShowLabels ? tr("&Messages") : "", this);
    msgAction->setIcon(QIcon(":/res/messages.png"));
    msgAction->setShortcut(QKeySequence(Qt::ALT + Qt::Key_2));
    msgAction->setCheckable(true);
    connect(msgAction, SIGNAL(triggered()), SLOT(onMessagesTriggered()));
    actionGroup->addAction(msgAction);

    albumAction = new QAction(isShowLabels ? tr("&Albums") : "", this);
    albumAction->setIcon(QIcon(":/res/albums.png"));
    albumAction->setShortcut(QKeySequence(Qt::ALT + Qt::Key_3));
    albumAction->setCheckable(true);
    connect(albumAction, SIGNAL(triggered()), SLOT(onAlbumsTriggered()));
    actionGroup->addAction(albumAction);

    newsAction = new QAction(isShowLabels ? tr("&News") : "", this);
    newsAction->setIcon(QIcon(":/res/general_rss.png"));
    newsAction->setCheckable(true);
    connect(newsAction, SIGNAL(triggered()), SLOT(onNewsTriggered()));
    actionGroup->addAction(newsAction);

    moreAction = new QAction(isShowLabels ? tr("&More") : "", this);
    moreAction->setIcon(QIcon(":/res/more.png"));
    moreAction->setShortcut(QKeySequence(Qt::ALT + Qt::Key_0));
    moreAction->setCheckable(true);
    connect(moreAction, SIGNAL(triggered()), SLOT(onMoreTriggered()));
    actionGroup->addAction(moreAction);

    connect(actionGroup, SIGNAL(triggered(QAction*)), SLOT(onActionTriggered(QAction*)));
}

void MainWindow::createMenus()
{
    // main menu
    QMenu *mainMenu = menuBar()->addMenu(tr("&File"));
    QAction *action = mainMenu->addAction(tr("Full screen"));
    action->setCheckable(true);
    action->setShortcut(QKeySequence(Qt::Key_F11));
    connect(action, SIGNAL(triggered()), this, SLOT(onFullScreenToggled()));

    action = mainMenu->addAction(tr("&Upload images"));
    action->setShortcut(QKeySequence(Qt::CTRL + Qt::Key_U));
    connect(action, SIGNAL(triggered()), this, SLOT(onUploadImagesToggled()));

#ifndef Q_WS_MAEMO5
    action = mainMenu->addAction(tr("&Exit"));
    action->setShortcut(QKeySequence::Close);
    connect(action, SIGNAL(triggered()), this, SLOT(close()));
#endif

   // settings menu
#ifndef Q_WS_MAEMO5
   mainMenu = menuBar()->addMenu(tr("&Settings"));
#endif

   action = mainMenu->addAction(tr("Accounts"));
   connect(action, SIGNAL(triggered()), this, SLOT(onAccountSettingsToggled()));
   action = mainMenu->addAction(tr("Network"));
   connect(action, SIGNAL(triggered()), this, SLOT(onNetworkSettingsToggled()));
   action = mainMenu->addAction(tr("Program settings"));
   connect(action, SIGNAL(triggered()), this, SLOT(onApplicationSettingsToggled()));

   // help menu
#ifndef Q_WS_MAEMO5
   mainMenu = menuBar()->addMenu(tr("&Help"));
#endif

   action = mainMenu->addAction(tr("Help"));
   action->setShortcut(QKeySequence::HelpContents);
   connect(action, SIGNAL(triggered()), this, SLOT(help()));
   action = mainMenu->addAction(tr("&About"));
   connect(action, SIGNAL(triggered()), this, SLOT(about()));
   action = mainMenu->addAction(tr("Statistics"));
   connect(action, SIGNAL(triggered()), SLOT(statistics()));
}

// handler for toolbar actions
void MainWindow::onActionTriggered(QAction *action)
{
    QList<QAction*> actions = actionGroup->actions();
    foreach (QAction *a, actions) {
        if (!(a->objectName() == action->objectName()))
            a->setChecked(false);
    }
}

// toggles fullscreen mode
void MainWindow::onFullScreenToggled()
{
    bool isFullScreen = windowState() & Qt::WindowFullScreen;
    if (isFullScreen) {
#ifdef Q_WS_MAEMO_5
        fullScreenButton->hide();
#endif
        showNormal();
    } else {
        showFullScreen();
#ifdef Q_WS_MAEMO_5
        fullScreenButton->show();
#endif
    }
}

// shows account settings
void MainWindow::onAccountSettingsToggled()
{
    UIAccountSettings *settings = new UIAccountSettings(this->sm, this);
    settings->setModal(true);
    settings->show();
}

// shows network settings
void MainWindow::onNetworkSettingsToggled()
{
    UINetworkSettings *settings = new UINetworkSettings(this->sm, this);
    settings->setModal(true);
    settings->show();
}

// shows application settings
void MainWindow::onApplicationSettingsToggled()
{
    UIApplicationSettings *settings = new UIApplicationSettings(this->sm, this);
    connect(settings, SIGNAL(updateSettings()), this, SLOT(settingsUpdated()));
    settings->setModal(true);
    settings->show();
}

void MainWindow::onUploadImagesToggled()
{
    QPhotoUploader *uploader = new QPhotoUploader(this->sm, this);
    uploader->show();
}

// Friends
void MainWindow::onFriendsTriggered()
{
    ui->stackedWidget->setCurrentIndex(FRIENDS_PAGE);
}

// Messages
void MainWindow::onMessagesTriggered()
{
    ui->stackedWidget->setCurrentIndex(MESSAGES_PAGE);
}

// Albums
void MainWindow::onAlbumsTriggered()
{
    ui->stackedWidget->setCurrentIndex(IMAGES_PAGE);
}

void MainWindow::onNewsTriggered()
{
    ui->stackedWidget->setCurrentIndex(NEWS_PAGE);
}

// More
void MainWindow::onMoreTriggered()
{
//    if (this->testAttribute(Qt::WA_Maemo5PortraitOrientation))
//        this->setAttribute(Qt::WA_Maemo5PortraitOrientation, false);
//    else
//        this->setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
}

void MainWindow::openImage(Photo photo)
{
    QImageViewer *viewer = new QImageViewer(this->font(), this);

#ifdef Q_WS_MAEMO_5
    viewer->setAttribute(Qt::WA_Maemo5StackedWindow);
#endif
    viewer->loadImage(this->sm, photo);
}

void MainWindow::gotErrorMsg(QString errorMsg, QTransport::Action acc, bool isMajor)
{
    if (!isMajor)
	return;

#ifdef Q_WS_MAEMO_5
//hildon_banner_show_information(0, "", errorMsg.toUtf8().data());
    QDBusMessage msg = QDBusMessage::createMethodCall("org.freedesktop.Notifications", "/org/freedesktop/Notifications",
                                                     "org.freedesktop.Notifications", "SystemNoteDialog");

    QList<QVariant> args;
    args.append(errorMsg);
    args.append(static_cast<quint32>(0));
    args.append("ok");

    msg.setArguments(args);

    QDBusConnection::systemBus().call(msg);
#else
#ifdef ENABLE_NOTIFY
    NotifyNotification *notification;

#if defined(NOTIFY_CHECK_VERSION)
#    if NOTIFY_CHECK_VERSION(0, 7, 0)
#        define NOTIFY_VERSION_MORE_7
#    endif
#endif
    /* Create notification */
    notification = notify_notification_new(APP_NAME,
        errorMsg.toUtf8().data(), ICONPATH
#ifndef NOTIFY_VERSION_MORE_7
        ,NULL
#endif
        );
    
    if (notification) {
        /* Set timeout */
        notify_notification_set_timeout(notification, 3000);
        /* Schedule notification for showing */
        if (!notify_notification_show(notification, NULL))
            qDebug("Failed to send notification");

        /* Clean up the memory */
        g_object_unref(notification);
    } else {
        qDebug("Failed to create notification");
        QMessageBox::warning(this, tr("Error occured"), errorMsg);
    }
#else /* if notify not supported */
    qDebug("Failed to create notification");
    QMessageBox::warning(this, tr("Error occured"), errorMsg);
#endif
#endif
}

void MainWindow::showNotification(QString message)
{
#ifdef Q_WS_MAEMO_5
    QDBusMessage msg = QDBusMessage::createMethodCall("org.freedesktop.Notifications", "/org/freedesktop/Notifications",
                                                     "org.freedesktop.Notifications", "SystemNoteDialog");

    QList<QVariant> args;
    args.append(message);
    args.append(static_cast<quint32>(0));
    args.append("ok");

    msg.setArguments(args);

    QDBusConnection::systemBus().call(msg);
#else
#ifdef ENABLE_NOTIFY
    NotifyNotification *notification;

#if defined(NOTIFY_CHECK_VERSION)
#    if NOTIFY_CHECK_VERSION(0, 7, 0)
#        define NOTIFY_VERSION_MORE_7
#    endif
#endif
    /* Create notification */
    notification = notify_notification_new(APP_NAME,
        message.toUtf8().data(), ICONPATH
#ifndef NOTIFY_VERSION_MORE_7
        ,NULL
#endif
        );

    if (notification) {
        /* Set timeout */
        notify_notification_set_timeout(notification, 3000);
        /* Schedule notification for showing */
        if (!notify_notification_show(notification, NULL))
            qDebug("Failed to send notification");

        /* Clean up the memory */
        g_object_unref(notification);
    } else {
        qDebug("Failed to create notification");
        QMessageBox::information(this, tr("Notification"), message);
    }
#else /* if notify not supported */
    qDebug("Failed to create notification");
    QMessageBox::information(this, tr("Notification"), message);
#endif
#endif
}

void MainWindow::showFriendInfo(Friend fd)
{
    this->profilesPanel->showFriendInfo(fd);
    ui->stackedWidget->setCurrentIndex(FRIENDS_PAGE);
    friendAction->setChecked(TRUE);
}

void MainWindow::showFriendAlbums(Friend fd)
{
    this->imagesPanel->showFriendAlbums(fd);
    ui->stackedWidget->setCurrentIndex(IMAGES_PAGE);
    albumAction->setChecked(TRUE);
}

void MainWindow::showFriendMessages(Friend fd)
{
    this->messagesPanel->showFriendMessages(fd);
    ui->stackedWidget->setCurrentIndex(MESSAGES_PAGE);
    msgAction->setChecked(TRUE);
}

void MainWindow::sendMessage(Friend fd)
{
    this->msgComposer->resetMessage();
    this->ui->stackedWidget->setCurrentIndex(MSG_COMPOSER_PAGE);
    this->msgComposer->setRecepient(fd);
}

void MainWindow::createNewMessage()
{
    this->msgComposer->resetMessage();
    this->ui->stackedWidget->setCurrentIndex(MSG_COMPOSER_PAGE);
}

void MainWindow::replyMessage(Message msg)
{
    this->msgComposer->resetMessage();
    this->msgComposer->replyMessage(msg);
    this->ui->stackedWidget->setCurrentIndex(MSG_COMPOSER_PAGE);
}

void MainWindow::editMessage(Message msg)
{
    this->msgComposer->resetMessage();
    this->msgComposer->setRecepient(msg.recepientList().first().id);
    this->msgComposer->setTitle(msg.title());
    this->msgComposer->setMessageBody(msg.text());
    this->ui->stackedWidget->setCurrentIndex(MSG_COMPOSER_PAGE);
}

void MainWindow::saveMessage(Message)
{
    this->showNotification(tr("Your message was saved in Drafts."));
    this->ui->stackedWidget->setCurrentIndex(MESSAGES_PAGE);
}

void MainWindow::messageSent()
{
    this->showNotification(tr("Your message was successfully sent."));
    this->ui->stackedWidget->setCurrentIndex(MESSAGES_PAGE);
}

void MainWindow::messageDeleted(Message)
{
    this->showNotification(tr("Message was deleted."));
}

void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void MainWindow::closeEvent(QCloseEvent *)
{
    emit closeDialogs();
}

void MainWindow::help()
{
    QString path;

#ifdef DOCDIR
    path = DOCDIR;
#else
    path = Utils::getDataDir() + "/doc/" + APP_NAME;
#endif

    uihelp *windowHelp = new uihelp(path + tr("/help.html"),
                                    tr("Help - %1").arg(APP_NAME), this->usedFont, this);
#ifdef Q_WS_MAEMO_5
    windowHelp->setAttribute(Qt::WA_Maemo5StackedWindow);
#endif
    windowHelp->show();
}

void MainWindow::about()
{
    QString path;

#ifdef DOCDIR
    path = DOCDIR;
#else
    path = Utils::getDataDir() + "/doc/" + APP_NAME;
#endif

    uihelp *windowHelp = new uihelp(path + tr("/about.html"),
                                    tr("About - %1").arg(APP_NAME), this->usedFont,this);
#ifdef Q_WS_MAEMO_5
    windowHelp->setAttribute(Qt::WA_Maemo5StackedWindow);
#endif
    windowHelp->show();
}

void MainWindow::orientationChanged()
{
    QRect geom = QApplication::desktop()->screenGeometry();
    if (geom.width() > geom.height())
    {
        // landscape
        this->addToolBar(Qt::LeftToolBarArea, ui->mainToolBar);
    }
    else
    {
        // portrait
        this->addToolBar(Qt::BottomToolBarArea, ui->mainToolBar);
    }
}

void MainWindow::statistics()
{
    QStatWidget *statWindow = new QStatWidget(this->usedFont, this->sm, this);
    connect(this->sm, SIGNAL(statsUpdated()), statWindow, SLOT(updateStats()));
    statWindow->show();
}
