#include "album.h"
#include "utils/utils.h"

class AlbumData : public QSharedData
{
public:
    AlbumData();
    AlbumData(const AlbumData& other);
    ~AlbumData();

    QString albumId;     /*!< album ID from service */
    QString ownerId;     /*!< ID of album owner */
    QString title;       /*!< album title (optional) */
    QString description; /*!< album description (optional) */
    QString time_create; /*!< Unix timestamp when the album was created. */
    QString time_update; /*!< Unix timestamp when the album was modifed. */
    int size;            /*!< Number of images in this album */
    QString icon;        /*!< Full path in system to the file with album's icon */
    QString icon_url;    /*!< Permalink on the file with album's icon */
    QString accountId;   /*!< account id which belongs to the album (not stored in cache) */
};

AlbumData::AlbumData()
    : QSharedData()
{
}

AlbumData::AlbumData(const AlbumData& other)
    : QSharedData(other)
{
    this->accountId = other.accountId;
    this->albumId = other.albumId;
    this->description = other.description;
    this->icon = other.icon;
    this->icon_url = other.icon_url;
    this->ownerId = other.ownerId;
    this->size = other.size;
    this->time_create = other.time_create;
    this->time_update = other.time_update;
    this->title = other.title;
}



AlbumData::~AlbumData()
{

}


Album::Album()
{
    d = new AlbumData;
}

Album::Album(const Album &src)
    : d(src.d)
{
}

Album::~Album()
{
}

QDomElement Album::toQDomElement(QDomDocument& doc) const
{
    QDomElement ret = doc.createElement(NODE_ALBUM_ROOT);

    // ID
    QDomElement node = doc.createElement(NODE_ALBUM_ID);
    QDomText t = doc.createTextNode(this->albumId());
    ret.appendChild(node);
    node.appendChild(t);

    // Name
    if (!this->title().isEmpty())
    {
        node = doc.createElement(NODE_ALBUM_TITLE);
        t = doc.createTextNode(this->title());
        ret.appendChild(node);
        node.appendChild(t);
    }

    // description
    if (!this->description().isEmpty())
    {
        node = doc.createElement(NODE_ALBUM_DESCRIPTION);
        t = doc.createTextNode(this->description());
        ret.appendChild(node);
        node.appendChild(t);
    }

    // size
    node = doc.createElement(NODE_ALBUM_SIZE);
    t = doc.createTextNode(QString("%1").arg(this->size()));
    ret.appendChild(node);
    node.appendChild(t);

    //time create
    if (!this->time_create().isEmpty())
    {
        node = doc.createElement(NODE_ALBUM_CREATE);
        t = doc.createTextNode(this->time_create());
        ret.appendChild(node);
        node.appendChild(t);
    }

    //time update
    if (!this->time_update().isEmpty())
    {
        node = doc.createElement(NODE_ALBUM_UPDATE);
        t = doc.createTextNode(this->time_update());
        ret.appendChild(node);
        node.appendChild(t);
    }

    //icon
    if (!this->icon().isEmpty())
    {
        node = doc.createElement(NODE_ALBUM_ICON);
        t = doc.createTextNode(this->icon());
        ret.appendChild(node);
        node.appendChild(t);
    }

    //icon url
    if (!this->icon_url().isEmpty())
    {
        node = doc.createElement(NODE_ALBUM_ICON_URL);
        t = doc.createTextNode(this->icon_url());
        ret.appendChild(node);
        node.appendChild(t);
    }

    return ret;
}

Album Album::fromQDomElement(const QDomElement& fr, const QString& accountId, const QString& friendId)
{
    Album ret = Album();

    ret.setAlbumId(fr.elementsByTagName(NODE_ALBUM_ID).at(0).firstChild().toText().data());
    if (fr.elementsByTagName(NODE_ALBUM_TITLE).count() > 0)
        ret.setTitle(fr.elementsByTagName(NODE_ALBUM_TITLE).at(0).firstChild().toText().data());
    if (fr.elementsByTagName(NODE_ALBUM_ICON).count() > 0)
        ret.setIcon(fr.elementsByTagName(NODE_ALBUM_ICON).at(0).firstChild().toText().data());
    if (fr.elementsByTagName(NODE_ALBUM_ICON_URL).count() > 0)
        ret.setIcon_url(fr.elementsByTagName(NODE_ALBUM_ICON_URL).at(0).firstChild().toText().data());
    if (fr.elementsByTagName(NODE_ALBUM_DESCRIPTION).count() > 0)
        ret.setDescription(fr.elementsByTagName(NODE_ALBUM_DESCRIPTION).at(0).firstChild().toText().data());
    if (fr.elementsByTagName(NODE_ALBUM_CREATE).count() > 0)
        ret.setTime_create(fr.elementsByTagName(NODE_ALBUM_CREATE).at(0).firstChild().toText().data());
    if (fr.elementsByTagName(NODE_ALBUM_UPDATE).count() > 0)
        ret.setTime_update(fr.elementsByTagName(NODE_ALBUM_UPDATE).at(0).firstChild().toText().data());
    if (fr.elementsByTagName(NODE_ALBUM_SIZE).count() > 0)
        ret.setSize(fr.elementsByTagName(NODE_ALBUM_SIZE).at(0).firstChild().toText().data().toInt());

    ret.setAccountId(accountId);
    ret.setOwnerId(friendId);

    return ret;
}

// load album list from cache file.
PhotoList Album::getPhotoList() const
{
    return Photo::loadPhotoList(this->accountId(), this->ownerId(), this->albumId());
}

void Album::setPhotoList(PhotoList list, QString accountId, QString friendId, QString albumId)
{
    // store photo lists into cache
    QDomDocument out("MyDoc");
    QDomElement profile = out.createElement(NODE_PHOTOS_ROOT);
    profile.setAttribute(ATTR_REFRESH, list.timeCreate().toString(DATE_TIME_FORMAT));
    out.appendChild(profile);

    for (int i = 0; i < list.length(); i++) {
        Photo curPh = list.at(i);
        //qDebug() << "Image " << curPh.photoId << " stored at " << curPh.photo;
        profile.appendChild(Photo::toQDomElement(out, curPh));
    }

    QDir dir;

    dir.setPath(dir.cleanPath(Utils::getAccountDir(accountId)));
    if (!dir.exists())
        dir.mkpath(dir.absolutePath());

    QFile file(dir.absolutePath() + "/" + QString(FILE_PHOTOS_DATA).arg(friendId, albumId));
    qDebug() << file.fileName();
    if (file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate) == false) {
        qDebug() << "can't open file: " + file.errorString();
    }
    //file.write(output.toUtf8());
    file.write(out.toByteArray(1));
    file.close();
}

Album& Album::operator=(const Album& other)
{
    d = other.d;
    return *this;
}

// Getters
const QString Album::albumId() const { return d->albumId; }
const QString Album::ownerId() const { return d->ownerId; }
const QString Album::title() const { return d->title; }
const QString Album::description() const { return d->description; }
const QString Album::time_create() const { return d->time_create; }
const QString Album::time_update() const { return d->time_update; }
int Album::size() const { return d->size; }
const QString Album::icon() const { return d->icon; }
const QString Album::icon_url() const { return d->icon_url; }
const QString Album::accountId() const { return d->accountId; }

// Setters
void Album::setAlbumId(const QString& v) { d->albumId = v; }
void Album::setOwnerId(const QString& v) { d->ownerId = v; }
void Album::setTitle(const QString& v) { d->title = v; }
void Album::setDescription(const QString& v) { d->description = v; }
void Album::setTime_create(const QString& v) { d->time_create = v; }
void Album::setTime_update(const QString& v) { d->time_update = v; }
void Album::setSize(int v) { d->size = v; }
void Album::setIcon(const QString& v) { d->icon = v; }
void Album::setIcon_url(const QString& v) { d->icon_url = v; }
void Album::setAccountId(const QString& v) { d->accountId = v; }

