#include "qtransport.h"
#include <QDebug>

typedef  gint (*driver_init)(msa_module *);

QTransport::QTransport(QString libraryname, QString accountName, QObject *parent) :
    QObject(parent), driverModule(0), isLibraryInit(false)
{
    this->activeRequests = 0;
    this->needShutdown = false;
    this->timer.start();

    if (libraryname.contains(Utils::getDriversDir()))
        libraryname.remove(Utils::getDriversDir(), Qt::CaseInsensitive);
    this->libName = libraryname;
    // Add dynamic library (libvkontakte.so)
    if ((this->driver = new QLibrary(Utils::getDriversDir() + libraryname)) == 0)
        qFatal("Problems with initiation QLibrary object");

    driver->setLoadHints(QLibrary::ResolveAllSymbolsHint); // analog RTLD_LAZY
    // Check driver existance
    if (!driver->isLoaded())
        if (!driver->load()) {
            qCritical() << "Can't load driver library: %s" << driver->errorString();
            return;
        }

    if ((driverModule = new struct msa_module) == 0)
        qFatal("can't init msa_module");

    // fill important properties
    driverModule->proxy = 0;
    driverModule->port = 0;

    if (accountName.isNull() || accountName.isEmpty())
        driverModule->id = NULL;
    else
        driverModule->id = g_strdup((gchar*) accountName.toUtf8().data());

    driver_init dr_init = (driver_init)driver->resolve(MODULE_INIT);
    dr_init(driverModule);

    if (driverModule == 0) {
        qCritical() << "can't resolve funtion of  library: " << driver->errorString();
        return;
    }
    this->accId = QTextCodec::codecForName("utf8")->toUnicode(driverModule->id);

    this->getMethods();
    isLibraryInit = true;

    connect(this, SIGNAL(errorOccurred(QString,QErrorMsg,QTransport::Action)), SLOT(gotError()));
    this->stats.setAccountId(this->accId);
}

QTransport::~QTransport()
{
    if (driver != 0) {
//        if (!driver->unload()) {
//            qWarning() << "can't unload  library: " << driver->errorString();
//        }
        delete driver;
        if (driverModule != NULL)
            delete driverModule;
        qDebug() <<__FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << ":" << this->accId << "(" << this->timer.elapsed()<< ")" << "qtransport deleted";
    }

}

QString QTransport::serviceName() const
{
    return QString::fromUtf8(this->driverModule->driverName);
}

QString QTransport::libraryName() const
{
    return this->libName;
}

QString QTransport::accountId() const
{
    return this->accId;
}

bool QTransport::ready()
{
    return isLibraryInit;

}

DriverInfo *QTransport::getDriverInfo()
{
    DriverInfo *ret = new DriverInfo(this->parent());

    // name
    ret->name = QString::fromUtf8(this->driverModule->name);
    // library
    ret->library = this->libName;

    ret->serviceName = this->serviceName();
    // icon
    QImage image;
    if (!image.loadFromData(QByteArray::fromBase64(this->driverModule->pic)))
        qDebug() <<__FILE__ <<":" << __LINE__ << ":" << __FUNCTION__ << ":"<< " Error loading image for " << this->accId << ": " << this->driverModule->pic;
    QPixmap pix = QPixmap::fromImage(image);
    ret->icon = QIcon(pix);

    return ret;
}

DriverInfo *QTransport::getDriverInfo(QString libraryPath)
{
    QLibrary *lib;
    // Add dynamic library (libvkontakte.so)
    if ((lib = new QLibrary(libraryPath)) == 0) {
        qWarning("Problems with initiation QLibrary object");
        return NULL;
    }

    lib->setLoadHints(QLibrary::ResolveAllSymbolsHint); // analog RTLD_LAZY
    // Check driver existance
    if (!lib->isLoaded())
        if (!lib->load()) {
            qWarning() << QString("can't load driver library: ") + lib->errorString();
            delete lib;
            return NULL;
        }

    struct msa_module * dm;

    if ((dm = new struct msa_module) == 0) {
        qWarning() << "can't init msa_module";
        lib->unload();
        delete lib;
        return NULL;
    }

    driver_init dr_init = (driver_init)lib->resolve(MODULE_GET_INFO);
    if (dr_init == NULL) {
        qWarning() << QString("Cann't find function ") + QString(MODULE_GET_INFO) +
                QString(". Probably old library type");
        delete dm;
        lib->unload();
        delete lib;
        return NULL;
    }

    dr_init(dm);

    if (dm == 0) {
        qWarning() << QString("can't resolve funtion of  library: ") + lib->errorString();
        lib->unload();
        delete lib;
        return NULL;
    }
    DriverInfo *ret = new DriverInfo();
    ret->name = QString::fromUtf8(dm->name);
    // library
    ret->library = libraryPath;
    // icon
    QImage image;
    if (!image.loadFromData(QByteArray::fromBase64(dm->pic)))
        qDebug() << "Error loading image: " << dm->pic;
    QPixmap pix = QPixmap::fromImage(image);
    ret->icon = QIcon(pix);
    ret->serviceName = QString::fromUtf8(dm->driverName);

    delete dm;
    lib->unload();
    delete lib;

    return ret;
}

/** init()
  * @param strProxyHost
  * @param strProxyPort
  */
void QTransport::init(QString proxyHost, uint proxyPort)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ": accountId=" <<this->accId << "(" << this->timer.elapsed() << ")";

    if (proxyHost.isNull()) {
        driverModule->proxy = (gchar*) NULL;
    } else {
        char *host = new char[proxyHost.toLocal8Bit().size()+1];
        strcpy(host,proxyHost.toLocal8Bit().data());
        driverModule->proxy = (gchar*) host;
    }
    driverModule->port = (gint) proxyPort;
}

void QTransport::close()
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ": accountId=" <<this->accId << "(" << this->timer.elapsed() << ")";
    this->needShutdown = true;
    this->stats.store();
    this->stats.clean();
    while (this->activeRequests != 0) {
        sleep(1);
    }

    this->stats.store();

    if (this->activeRequests == 0)
        if (driverModule != NULL && this->ready()) {
            qDebug() <<__FILE__<<":"<<__LINE__<<":"<<__FUNCTION__ << "Shutdown..."<< endl;
            driverModule->shutdown(driverModule);
            qDebug() <<__FILE__<<":"<<__LINE__<<":"<<__FUNCTION__ << "Shutdown released"<< endl;
        }
}

QString QTransport::generateFileName(const QString& path, const QString& url)
{
    QString icon = NULL;
    if (url != NULL) {
        QFileInfo fi(url);
        QString ext = fi.suffix();
        icon = path + QString(url.toUtf8().toBase64()) + (ext.isEmpty() ? QString() : QString(".") + ext);
    }
    return icon;
}

QDomDocument QTransport::createRequest(const QString& usedClass, const QString& usedFunction, const bool noAuthorize) const
{
    return this->createRequest(usedClass, usedFunction, noAuthorize, "<Params/>");
}

QDomDocument QTransport::createRequest(const QString& usedClass, const QString& usedFunction,
                                       const bool noAuthorize, const QString& content) const
{
    QDomDocument ret;

    QDomElement req = ret.createElement("Request");
    req.setAttribute("class", usedClass);
    req.setAttribute("function", usedFunction);
    req.setAttribute("noAuthorize", noAuthorize ? "true" : "false");
    ret.appendChild(req);

    // parse content
    QDomDocument c;
    c.setContent(content);
    QDomNode contNode = ret.importNode(c.documentElement(), true);
    req.appendChild(contNode);

    return ret;
}

QDomDocument QTransport::sendRequest(const QDomDocument& doc, Action acc)
{
    QDomDocument ret;
    if (this->needShutdown) {
        return ret;
    } else
        this->activeRequests++;

    xmlChar* request = xmlCharStrdup(doc.toByteArray(0).constData());
    xmlDocPtr reqXml = xmlParseDoc(request);
    xmlDocPtr respXml = NULL;

    xmlFree(request);
    // Process request
    this->stats.addRequest();
    if (this->driverModule != NULL) {
        qDebug() << __FILE__ << ":" << __LINE__  << "(" << this->timer.elapsed() << ")" << ": ========== REQUEST ==============\n" <<
                doc.toString() << "============ END REQUEST =============";
        try {
            driverModule->send(reqXml, &respXml, driverModule);
        } catch (...) {
            qDebug() << "Caught an exception.";
            QErrorMsg msg(APP_LIBRARY_EXCEPTION_CODE, tr("Caught an exception."));
            emit errorOccurred(this->accId, msg, acc);
            return ret;
        }
    } else {
        qDebug() << "driverModule == NULL" << endl;
    }
    activeRequests--;

    xmlChar *resp = NULL;
    int respLen = 0;
    xmlDocDumpMemory(respXml, &resp, &respLen);

    ret.setContent(QString((char *)resp));
    xmlFree(resp);
    xmlFreeDoc(respXml);

    qDebug() << __FILE__ << ":" << __LINE__  << "(" << this->timer.elapsed() << ")"<< ": ========== RESPONSE ==============\n" <<
            ret.toString() << "============ END RESPONSE =============";

    //add statistics to collection
    QDomNodeList msg = ret.elementsByTagName("Response");
    for (int i = 0; i < msg.count(); i++) {
        this->stats.addUpload(msg.at(i).toElement().attribute("upload").toLong());
        this->stats.addDownload(msg.at(i).toElement().attribute("download").toLong());
    }
    emit this->statsUpdated(this->accId);

    return ret;
}

bool QTransport::checkGoodResponse(const QDomDocument& resp, Action acc)
{
    if (!this->checkBadResponse(resp, acc))
        return false;

    qDebug() << __FILE__ <<":"<< __LINE__ << "(" << this->timer.elapsed() << ")"<<": response not error!";

    QDomNodeList msg = resp.elementsByTagName("Response");
    for (int i = 0; i < msg.count(); i++) {
        if (msg.at(i).toElement().attribute("class").compare(CLASS_SYSTEM_MESSAGES) == 0 &&
            msg.at(i).toElement().attribute("function").compare(INFO_MESSAGE) == 0) {
            return true;
        }
    }
    qDebug() << __FILE__ <<":"<< __LINE__ << "(" << this->timer.elapsed() << ")"<<": Response code not found!";

    QErrorMsg err(APP_LIBRARY_WRONG_RESPONSE, tr("Bad response from driver"));
    emit this->errorOccurred(this->accId, err, acc);
    return false;
}

bool QTransport::checkBadResponse(const QDomDocument& resp, Action acc)
{
    bool ret = true;

    QDomNodeList msg = resp.elementsByTagName("Response");
    for (int i = 0; i < msg.count(); i++) {
        QDomElement curNode = msg.at(i).toElement();

        // check authorization flag
        if (curNode.attribute("authorized").compare("true") == 0)
            this->getSettings();

        // check error message
        if (curNode.attribute("class").compare(CLASS_SYSTEM_MESSAGES) == 0) {
            if (curNode.attribute("function").compare(ERROR_MESSAGE) == 0) {
                QErrorMsg msg;
                QDomNodeList params = curNode.elementsByTagName("Params").at(0).toElement().childNodes();
                for (int j = 0; j < params.count(); j++) {
                    QDomElement curNode = params.at(j).toElement();
                    if (curNode.nodeName().compare("string") == 0) {
                        if (curNode.attribute("name").compare("code") == 0)
                            msg.code = curNode.firstChild().nodeValue().trimmed().toAscii();
                        if (curNode.attribute("name").compare("text") == 0)
                            msg.text = tr(curNode.firstChild().nodeValue().trimmed().toAscii());
                        if (curNode.attribute("name").compare("comment") == 0)
                            msg.comment = tr(curNode.firstChild().nodeValue().trimmed().toAscii());
                    }
                }
                qDebug() << __FILE__ << ":" << __LINE__  << "(" << this->timer.elapsed() << ")"<< ": code=" << msg.code;
                emit errorOccurred(this->accId, msg, acc);
                ret = false;
            }
        }
    }
    return ret;
}

bool QTransport::checkFunction(const QString &usedClass, const QString &usedFunction) const
{
    for (int i = 0; i < this->registerdFunctions.length(); i++) {
        if (this->registerdFunctions.at(i).className.compare(usedClass) == 0 &&
            this->registerdFunctions.at(i).functionName.compare(usedFunction) == 0)
            return true;
    }

    return false;
}

bool QTransport::checkFunction(const QString& usedClass, const QString& usedFunction,
                               QTransport::Action acc) const
{
    bool ret = this->checkFunction(usedClass, usedFunction);

    if (!ret)
    {
        QErrorMsg msg;
        msg.code = DRV_UNKNOWN_METHOD_CALL;
        msg.text = tr("Unsupported method was called");
        msg.comment = tr("This method didn't supported by selected driver");

        emit errorOccurred(this->accId, msg, acc);
    }

    return ret;
}

void QTransport::getMethods()
{
    QDomDocument req = this->createRequest(CLASS_SETTINGS, GET_LIST_METHODS, true);
    QDomDocument res = this->sendRequest(req, this->getListMethodsAction);

    if (!this->checkBadResponse(res, this->getListMethodsAction))
        return;

    // parse response
    if (res.elementsByTagName("Response").count() == 0 ||
        res.elementsByTagName("Response").at(0).toElement().attribute("function").compare(GET_LIST_METHODS) != 0)
        return;

    QDomNodeList methods = res.elementsByTagName("Response").at(0).toElement().
                           elementsByTagName("Params").at(0).toElement().
                           elementsByTagName("string");

    this->registerdFunctions.clear();

    for (int i = 0; i < methods.count(); i++) {
        MethodElement elem;
        elem.className = methods.at(i).toElement().attribute("class");
        elem.functionName = methods.at(i).toElement().attribute("function");

        this->registerdFunctions.append(elem);
    }

}

void QTransport::getFriends(bool alsoLoadFriendImages)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ": accountId=" <<this->accId << "(" << this->timer.elapsed() << ")";
    if (!this->checkFunction(CLASS_FRIENDS, GET_LIST_FRIENDS, this->getListFriendsAction))
        return;

    int page = 1;
    int pageSize = 100;

    FriendList ret;
    ret.setTimeCreate(QDateTime::currentDateTime());

    while(true)
    {
        QDomDocument req = this->createRequest(CLASS_FRIENDS, GET_LIST_FRIENDS);
        QDomNodeList params = req.elementsByTagName("Params");

        QDomElement pageElem = req.createElement("number");
        pageElem.setAttribute("name", "page");
        pageElem.appendChild(req.createTextNode(QString::number(page)));
        params.at(0).toElement().appendChild(pageElem);

        QDomElement pageSElem = req.createElement("number");
        pageSElem.setAttribute("name", "pageSize");
        pageSElem.appendChild(req.createTextNode(QString::number(pageSize)));
        params.at(0).toElement().appendChild(pageSElem);

        QDomDocument res = this->sendRequest(req, this->getListFriendsAction);

        if (!this->checkBadResponse(res, this->getListFriendsAction))
            return;

        // parse response
        if (res.elementsByTagName("Response").count() == 0 ||
                res.elementsByTagName("Response").at(0).toElement().attribute("function").compare(GET_LIST_FRIENDS) != 0)
            return;

        QDomNodeList friends = res.elementsByTagName("Response").at(0).toElement().
                elementsByTagName("Params").at(0).toElement().
                elementsByTagName("array").at(0).toElement().
                elementsByTagName("struct");

        if (friends.count() == 0)
            break;

        for (int i = 0; i < friends.count(); i++) {
            Friend fr;
            QDomElement curFr = friends.at(i).toElement();
            fr.setAccountId(this->accId);
            fr.setServiceId(this->serviceName());
            fr.setOwnerId(curFr.attribute("id"));

            for (uint j = 0; j < curFr.childNodes().length(); j++) {
                QDomElement curElem = curFr.childNodes().at(j).toElement();
                const QString curElemName = curElem.attribute("name");
                if (curElem.nodeName().compare("string") == 0) {
                    if (curElemName.compare("FirstName") == 0)
                        fr.setFirstName(curElem.firstChild().nodeValue().trimmed());
                    else
                        if (curElemName.compare("NickName") == 0)
                            fr.setNickName(curElem.firstChild().nodeValue().trimmed());
                        else
                            if (curElemName.compare("LastName") == 0)
                                fr.setLastName(curElem.firstChild().nodeValue().trimmed());
                            else
                                if (curElemName.compare("FriendStatus") == 0)
                                    fr.setOnline(curElem.firstChild().nodeValue().trimmed() == "0" ? false : true);
                }
                else
                    if (curElem.nodeName().compare("img") == 0) {
                        if (curElemName.compare("Img") == 0) {
                            // Icon
                            fr.setIcon_url(curElem.firstChild().nodeValue().trimmed());

                            QDir dir;
                            if (!dir.exists(Utils::getFriendsIconDir(this->serviceName())))
                                dir.mkpath(Utils::getFriendsIconDir(this->serviceName()));
                            QString icon = NULL;
                            if (!fr.icon_url().isEmpty()){
                                icon = generateFileName(Utils::getFriendsIconDir(this->serviceName()), fr.icon_url());
                                QFile file(icon);
                                if(file.exists())
                                    fr.setIcon(icon);
                            }
                        }
                    }
            }
            ret.append(fr);
        }
        page++;
    }
    qDebug() <<__FILE__ << __LINE__ << __FUNCTION__  << "(" << this->timer.elapsed() << ")" << "Total friends:" + ret.length();

    if (alsoLoadFriendImages)
    {
        emit friendsReceived(this->accId, ret, false);

        // === update friend icons ===
        int count = 0;

        for (int i = 0; i < ret.length(); i++) {
            if (ret[i].icon_url().isEmpty())
                continue;

            QString icon = generateFileName(Utils::getFriendsIconDir(this->serviceName()),
                                            ret[i].icon_url());
            QFile file(icon);
            if (!file.exists()) {
                count++;
                if (downloadPhoto(ret[i].icon_url(), icon))
                    ret[i].setIcon(icon);
                if (count > 0 && count%10 == 0) {
                    qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "(" << this->timer.elapsed() << ")" << "emit friendsReceived signal" << endl;
                    emit friendsReceived(this->accId, ret, false);
                }
            } else
                ret[i].setIcon(icon);
        }
    }

    emit friendsReceived(this->accId, ret, true);
}

bool QTransport::deleteFriend(Friend owner)
{
    return this->deleteFriend(owner.ownerId());
}

bool QTransport::deleteFriend(QString ownerId)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ": accountId=" <<this->accId << "(" << this->timer.elapsed() << ")";
    if (!this->checkFunction(CLASS_FRIENDS, DELETE_FRIEND, this->deleteFriendAction))
        return false;

    QDomDocument req = this->createRequest(CLASS_FRIENDS, DELETE_FRIEND);
    if (!ownerId.isEmpty()) {
        QDomNodeList params = req.elementsByTagName("Params");
        params.at(0).toElement().setAttribute("id", ownerId);
    }
    QDomDocument res = this->sendRequest(req, this->deleteFriendAction);

    // parse response
    if (this->checkGoodResponse(res, this->deleteFriendAction)) {
        emit this->friendDeleted(this->accId, ownerId);
        return true;
        qDebug() << __FILE__ <<":"<<__LINE__ << "Response is correct";
    }

    qDebug() << __FILE__ <<":"<<__LINE__ << "Response incorrect!!!";
    return false;
}

void QTransport::getSettings()
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ": accountId=" <<this->accId << "(" << this->timer.elapsed() << ")";
    if (!this->checkFunction(CLASS_SETTINGS, GET_SETTINGS, this->getSettingsAction))
        return;

    QDomDocument req = this->createRequest(CLASS_SETTINGS, GET_SETTINGS);
    QDomDocument res = this->sendRequest(req, this->getSettingsAction);

    if (this->checkBadResponse(res, this->getSettingsAction)) {
        QString settings;
        QTextStream str(&settings);
        res.elementsByTagName("Response").at(0).toElement().elementsByTagName("Params").at(0).save(str, 0);
        emit this->settingsReceived(this->accId, settings);
    }
}

void QTransport::getProfile()
{
    this->getProfile(QString::null);
}

void QTransport::getProfile(const QString& ownerId)
{
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << ": accountId=" << this->accId << "; ownerId=" << ownerId;
    if (!this->checkFunction(CLASS_PROFILE, GET_PROFILE, this->getProfileAction))
        return;

    QDomDocument req = this->createRequest(CLASS_PROFILE, GET_PROFILE);
    if (!ownerId.isEmpty()) {
        QDomNodeList params = req.elementsByTagName("Params");
        params.at(0).toElement().setAttribute("id", ownerId);
    }
    QDomDocument res = this->sendRequest(req, this->getProfileAction);

    if (!this->checkBadResponse(res, this->getProfileAction))
        return;

    // parse response
    if (res.elementsByTagName("Response").count() == 0 ||
        res.elementsByTagName("Response").at(0).toElement().attribute("function").compare(GET_PROFILE) != 0)
        return;

    QDomElement params = res.elementsByTagName("Response").at(0).toElement().
                         elementsByTagName("Params").at(0).toElement();
    QDomNodeList paramNodes = params.childNodes();

    Friend profile;
    profile.setAccountId(this->accId);
    profile.setServiceId(this->serviceName());
    QString icon;

    profile.setOwnerId(params.attribute("id"));

    for (int i = 0; i < paramNodes.count(); i++) {
        QDomElement curElem = paramNodes.at(i).toElement();
        const QString curElemName = curElem.attribute("name");
        if (curElem.nodeName().compare("string") == 0) {
            if (curElemName.compare("FirstName") == 0)
                profile.setFirstName(curElem.firstChild().nodeValue().trimmed());
            else
            if (curElemName.compare("NickName") == 0)
                profile.setNickName(curElem.firstChild().nodeValue().trimmed());
            else
            if (curElemName.compare("LastName") == 0)
                profile.setLastName(curElem.firstChild().nodeValue().trimmed());
            else
            if (curElemName.compare("Gender") == 0)
                profile.setGender(curElem.firstChild().nodeValue());
            else
            if (curElemName.compare("Birthday") == 0)
                profile.setBirthday(curElem.firstChild().nodeValue());
            else
            if (curElemName.compare("MobilePhone") == 0)
                profile.setMobilePhone(curElem.firstChild().nodeValue());
            else
            if (curElemName.compare("HomePhome") == 0)
                profile.setHomePhone(curElem.firstChild().nodeValue());
            else
            if (curElemName.compare("CityName") == 0)
                profile.setCity(curElem.firstChild().nodeValue());
            else
            if (curElemName.compare("CountryName") == 0)
                profile.setCountry(curElem.firstChild().nodeValue());
        }
        else
        if (curElem.nodeName().compare("img") == 0) {
            if (curElemName.compare("Img") == 0) {
                // Icon
                profile.setIcon_url(curElem.firstChild().nodeValue().trimmed());
                QDir dir;
                if (!dir.exists(Utils::getFriendsIconDir(this->serviceName())))
                    dir.mkpath(Utils::getFriendsIconDir(this->serviceName()));
                if (!profile.icon_url().isEmpty()){
                    icon = generateFileName(Utils::getFriendsIconDir(this->serviceName()), profile.icon_url());
                    QFile file(icon);
                    if(file.exists())
                        profile.setIcon(icon);
                }
            }
        }

    }

    if (!icon.isEmpty() && !profile.icon_url().isEmpty()) {
        QFile file(icon);
        if (!file.exists()) {
            if (downloadPhoto(profile.icon_url(), icon)) {
                profile.setIcon(icon);
            }
        }
    }
    emit profileReceived(this->accId, ownerId, profile);
}

void QTransport::setSettings(QString settings)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ": accountId=" <<this->accId;

    if (!this->checkFunction(CLASS_SETTINGS, SET_SETTINGS, this->setSettingsAction))
        return;

    QDomDocument req = this->createRequest(CLASS_SETTINGS, SET_SETTINGS, true, settings);
    QDomDocument res = this->sendRequest(req, this->setSettingsAction);

    // parse response
    if (this->checkGoodResponse(res, this->setSettingsAction))
        qDebug() << __FILE__ <<":"<<__LINE__ << "Response is correct";
    else
        qDebug() << __FILE__ <<":"<<__LINE__ << "Response incorrect!!!";
}

void QTransport::getAlbums(QString ownerId, bool alsoLoadIcons)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ": accountId=" <<this->accId << "; ownerId=" << ownerId;

    if (!this->checkFunction(CLASS_PHOTOS, GET_LIST_ALBUMS, this->getListAlbumsAction))
        return;

    QDomDocument req = this->createRequest(CLASS_PHOTOS, GET_LIST_ALBUMS);

    QDomNodeList params = req.elementsByTagName("Params");
    params.at(0).toElement().setAttribute("id", ownerId);

    QDomDocument res = this->sendRequest(req, this->getListAlbumsAction);

    if (!this->checkBadResponse(res, this->getListAlbumsAction))
        return;

    QDomElement array = res.elementsByTagName("Response").at(0).toElement().
                        elementsByTagName("Params").at(0).toElement().
                        elementsByTagName("array").at(0).toElement();

    AlbumList ret;
    ret.setTimeCreate(QDateTime::currentDateTime());

    if (array.attribute("ownerId").compare(ownerId) != 0) {
        emit albumsReceived(this->accId, ownerId, ret, true);
        return;
    }

    QDomNodeList albums = array.elementsByTagName("struct");

    for (int i = 0; i < albums.count(); i++) {
        QDomElement curAlbum = albums.at(i).toElement();
        Album al;
        al.setAccountId(this->accId);
        al.setOwnerId(ownerId);
        al.setAlbumId(curAlbum.attribute("id"));

        for (int j = 0; j < curAlbum.childNodes().count(); j++) {
            QDomElement curItem = curAlbum.childNodes().at(j).toElement();
            const QString curItemName = curItem.attribute("name");
            if (curItem.nodeName().compare("string") == 0) {
                if (curItemName.compare("title") == 0)
                    al.setTitle(curItem.firstChild().nodeValue().trimmed());

                if (curItemName.compare("description") == 0)
                    al.setDescription(curItem.firstChild().nodeValue().trimmed());

                if (curItemName.compare("ownerId") == 0)
                    al.setOwnerId(curItem.firstChild().nodeValue().trimmed());
            }
            else
            if (curItem.nodeName().compare("img") == 0) {
                if (curItemName.compare("Img") == 0) {
                    al.setIcon_url(curItem.firstChild().nodeValue().trimmed());

                    QDir dir;
                    if (!dir.exists(Utils::getAlbumsIconDir(this->serviceName())))
                        dir.mkpath(Utils::getAlbumsIconDir(this->serviceName()));
                    QString icon = NULL;
                    if (!al.icon_url().isEmpty()){
                        icon = generateFileName(Utils::getAlbumsIconDir(this->serviceName()), al.icon_url());
                        QFile file(icon);
                        if(file.exists())
                            al.setIcon(icon);
                    }
                }
            }
            else
            if (curItem.nodeName().compare("number") == 0) {
                if (curItemName.compare("created") == 0) {
                    QDateTime mdf;
                    mdf.setTime_t(curItem.firstChild().nodeValue().toInt());
                    al.setTime_create(mdf.toString(DATE_TIME_FORMAT));
                }

                if (curItemName.compare("updated") == 0) {
                    QDateTime mdf;
                    mdf.setTime_t(curItem.firstChild().nodeValue().toInt());
                    al.setTime_update(mdf.toString(DATE_TIME_FORMAT));
                }

                if (curItemName.compare("size") == 0)
                    al.setSize(curItem.firstChild().nodeValue().toInt());
            }
        }
        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << ": albumId=" << al.albumId() << "; time_update=" << al.time_update() << "; time_ceate=" << al.time_create();
        if (al.time_update().isEmpty() && !al.time_create().isEmpty())
            al.setTime_update(al.time_create());
        ret.append(al);
    }

    if (alsoLoadIcons)
    {
        emit albumsReceived(this->accId, ownerId, ret, ret.length() == 0 ? true : false);

        if (ret.length() == 0)
            return;

        // update album icons
        int count = 0;

        QDir dir;
        if (!dir.exists(Utils::getAlbumsIconDir(this->serviceName())))
            dir.mkpath(Utils::getAlbumsIconDir(this->serviceName()));

        qDebug()<< "Length AlbumList = "<<ret.length()<<endl;
        for (int i = 0; i < ret.length(); i++) {
            if (ret.at(i).icon_url().isEmpty())
                continue;

            QString icon = NULL;
            icon = generateFileName(Utils::getAlbumsIconDir(this->serviceName()), ret.at(i).icon_url());
            QFile file(icon);
            if (!file.exists()) {
                if (downloadPhoto(ret.at(i).icon_url(), icon))
                    ret[i].setIcon(icon);
                count++;
                if (count > 0 && count%10 == 0) {
                    qDebug() << "emit AlbumsReceived signal" << endl;
                    emit albumsReceived(this->accId, ownerId, ret, false);
                }

            } else
                ret[i].setIcon(icon);
        }
    }

    emit albumsReceived(this->accId, ownerId, ret, true);
}

void QTransport::getPhotos(QString ownerId, QString albumId, bool alsoLoadIcons)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ": accountId=" <<this->accId;
    if (!this->checkFunction(CLASS_PHOTOS, GET_LIST_PHOTOS, this->getListPhotosAction))
        return;

    int page = 1;
    int pageSize = 100;
    PhotoList ret;
    ret.setTimeCreate(QDateTime::currentDateTime());

    while(true)
    {
        QDomDocument req = this->createRequest(CLASS_PHOTOS, GET_LIST_PHOTOS);

        QDomNodeList params = req.elementsByTagName("Params");
        params.at(0).toElement().setAttribute("id", ownerId);

        QDomElement albElem = req.createElement("string");
        albElem.setAttribute("name", "albumId");
        albElem.appendChild(req.createTextNode(albumId));
        params.at(0).toElement().appendChild(albElem);

        if (page > 1)
        {
            QDomElement pageElem = req.createElement("number");
            pageElem.setAttribute("name", "page");
            pageElem.appendChild(req.createTextNode(QString::number(page)));
            params.at(0).toElement().appendChild(pageElem);

            QDomElement pageSElem = req.createElement("number");
            pageSElem.setAttribute("name", "pageSize");
            pageSElem.appendChild(req.createTextNode(QString::number(pageSize)));
            params.at(0).toElement().appendChild(pageSElem);
        }

        QDomDocument res = this->sendRequest(req, this->getListPhotosAction);

        if (!this->checkBadResponse(res, this->getListPhotosAction))
            return;

        QDomElement array = res.elementsByTagName("Response").at(0).toElement().
                elementsByTagName("Params").at(0).toElement().
                elementsByTagName("array").at(0).toElement();

        if (array.attribute("ownerId").compare(ownerId) != 0 ||
                array.attribute("albumId").compare(albumId) != 0)
        {
            qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "() wrong Id's: ownerId="
                     << array.attribute("ownerId") << "; albumId=" << array.attribute("albumId") << "; need:"
                     << ownerId << " and " << albumId;
            QErrorMsg msg;
            msg.code = APP_LIBRARY_WRONG_RESPONSE;
            msg.comment = tr("The friend or album identificators are not equal");
            msg.text = tr("Incorrect identificators");
            emit errorOccurred(this->accId, msg, this->getListPhotosAction);
            return;
        }

        QDomNodeList photos = array.elementsByTagName("struct");

        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "() page=" << page << " pageSize=" << pageSize;

        // not more images, exit
        if (photos.count() == 0)
            break;

        for (int i = 0; i < photos.count(); i++) {
            QDomElement curPhoto = photos.at(i).toElement();
            Photo ph;
            ph.setPhotoId(curPhoto.attribute("id"));
            ph.setAlbumId(albumId);
            ph.setOwnerId(ownerId);
            ph.setAccountId(this->accId);

            for (int j = 0; j < curPhoto.childNodes().count(); j++) {
                QDomElement curItem = curPhoto.childNodes().at(j).toElement();
                const QString curItemName = curItem.attribute("name");
                if (curItem.nodeName().compare("string") == 0) {
                    if (curItemName.compare("albumId") == 0)
                        ph.setAlbumId(curItem.firstChild().nodeValue().trimmed());
                    else
                        if (curItemName.compare("ownerId") == 0)
                            ph.setOwnerId(curItem.firstChild().nodeValue().trimmed());
                        else
                            if (curItemName.compare("description") == 0)
                                ph.setDescription(curItem.firstChild().nodeValue().trimmed());
                            else
                                if (curItemName.compare("urlSmall") == 0) {
                                    ph.setIcon_url(curItem.firstChild().nodeValue().trimmed());

                                    QDir dir;
                                    if (!dir.exists(Utils::getAlbumsIconDir(this->serviceName())))
                                        dir.mkpath(Utils::getAlbumsIconDir(this->serviceName()));
                                    QString icon = NULL;
                                    if (!ph.icon_url().isEmpty()) {
                                        icon = generateFileName(Utils::getAlbumsIconDir(this->serviceName()),
                                                                ph.icon_url());
                                        QFile file(icon);
                                        if(file.exists())
                                            ph.setIcon(icon);
                                    }
                                }
                                else
                                    if (curItemName.compare("urlBig") == 0) {
                                        ph.setPhoto_url(curItem.firstChild().nodeValue().trimmed());

                                        QDir dir;
                                        if (!dir.exists(Utils::getPhotoDir(this->serviceName())))
                                            dir.mkpath(Utils::getPhotoDir(this->serviceName()));

                                        QString img = NULL;
                                        if (!ph.photo_url().isEmpty()) {
                                            img = generateFileName(Utils::getPhotoDir(this->serviceName()),
                                                                   ph.photo_url());
                                            QFile file(img);
                                            if (file.exists())
                                                ph.setPhoto(img);
                                        }
                                    }
                }
                else
                    if (curItem.nodeName().compare("number") == 0) {
                        if (curItemName.compare("created") == 0) {
                            QDateTime mdf;
                            mdf.setTime_t(curItem.firstChild().nodeValue().toInt());
                            ph.setTime_create(mdf.toString(DATE_TIME_FORMAT));
                        }
                    }
            }
            ret.append(ph);
        }

        page++;
        if (page == 2)
            pageSize = photos.count();
    }

    if (alsoLoadIcons)
    {
        emit this->photosReceived(this->accId, ownerId, albumId, ret, ret.length() > 0 ? false : true);

        if (ret.length() == 0)
            return;

        int count = 0;

        QDir dir;
        if (!dir.exists(Utils::getAlbumsIconDir(this->serviceName())))
            dir.mkpath(Utils::getAlbumsIconDir(this->serviceName()));
        qDebug()<< "Length AlbumList = "<<ret.length()<<endl;

        for (int i = 0; i < ret.length(); i++) {
            if (ret.at(i).icon_url().isEmpty())
                continue;

            QString icon = NULL;
            icon = generateFileName(Utils::getAlbumsIconDir(this->serviceName()),
                                    ret.at(i).icon_url());
            QFile file(icon);
            if (!file.exists()) {
                count++;
                if (downloadPhoto(ret.at(i).icon_url(), icon))
                    ret[i].setIcon(icon);
                if (count > 0 && count%10 == 0) {
                    qDebug() << "emit PhotosReceived signal";
                    emit this->photosReceived(this->accId, ownerId, albumId, ret, false);
                }
            } else
                ret[i].setIcon(icon);

        }
    }

    emit this->photosReceived(this->accId, ownerId, albumId, ret, true);
}

void QTransport::getPhotoComments(Photo photo)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ": accountId=" <<this->accId;

    if (!this->checkFunction(CLASS_PHOTOS, GET_LIST_PHOTO_COMMENTS, this->getCommentsAction))
        return;

    QDomDocument req = this->createRequest(CLASS_PHOTOS, GET_LIST_PHOTO_COMMENTS);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    QDomElement pr = req.createElement("string");
    pr.setAttribute("name", "ownerId");
    pr.appendChild(req.createTextNode(photo.ownerId()));
    params.appendChild(pr);

    pr = req.createElement("string");
    pr.setAttribute("name", "albumId");
    pr.appendChild(req.createTextNode(photo.albumId()));
    params.appendChild(pr);

    pr = req.createElement("string");
    pr.setAttribute("name", "photoId");
    pr.appendChild(req.createTextNode(photo.photoId()));
    params.appendChild(pr);

    QDomDocument res = this->sendRequest(req, this->getCommentsAction);

    if (!this->checkBadResponse(res, this->getCommentsAction))
        return;


    QDomElement array = res.elementsByTagName("Response").at(0).toElement().
                        elementsByTagName("Params").at(0).toElement().
                        elementsByTagName("array").at(0).toElement();

    PhotoCommentList ret;
    ret.setTimeCreate(QDateTime::currentDateTime());

    if (array.attribute("ownerId").compare(photo.ownerId()) != 0 ||
        array.attribute("albumId").compare(photo.albumId()) != 0 ||
        array.attribute("photoId").compare(photo.photoId()) != 0) {
        emit this->commentsReceived(this->accId, photo.ownerId(), photo.albumId(), photo.photoId(), ret);
        return;
    }

    QDomNodeList comments = array.elementsByTagName("struct");

    for (int i = 0; i < comments.count(); i++) {
        QDomElement curComment = comments.at(i).toElement();
        PhotoComment ch;
        ch.setCommentId(curComment.attribute("id"));
        ch.setAccountId(this->accId);
        ch.setOwnerId(photo.ownerId());
        ch.setAlbumId(photo.albumId());
        ch.setPhotoId(photo.photoId());

        for (int j = 0; j < curComment.childNodes().count(); j++) {
            QDomElement curItem = curComment.childNodes().at(j).toElement();
            const QString curItemName = curItem.attribute("name");
            if (curItem.nodeName().compare("string") == 0) {
                if (curItemName.compare("SenderId") == 0)
                    ch.setSenderId(curItem.firstChild().nodeValue().trimmed());
                else
                if (curItemName.compare("SenderName") == 0)
                    ch.setSenderName(curItem.firstChild().nodeValue().trimmed());
                else
                if (curItemName.compare("Time") == 0) {
                    QDateTime mdf;
                    mdf.setTime_t(curItem.firstChild().nodeValue().toInt());
                    ch.setTime(mdf.toString(DATE_TIME_FORMAT));
                }
                else
                if (curItemName.compare("Text") == 0)
                    ch.setText(curItem.firstChild().nodeValue().trimmed());
            }
        }
        ret.append(ch);
    }
    emit this->commentsReceived(this->accId, photo.ownerId(), photo.albumId(), photo.photoId(), ret);
}

bool QTransport::sendPhotoComment(Photo photo, QString comment)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ": accountId=" <<this->accId;

    if (!this->checkFunction(CLASS_PHOTOS, SEND_PHOTO_COMMENT, this->sendCommentAction))
        return false;

    QDomDocument req = this->createRequest(CLASS_PHOTOS, SEND_PHOTO_COMMENT);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    QDomElement node = req.createElement("string");
    node.setAttribute("name", "ownerId");
    node.appendChild(req.createTextNode(photo.ownerId()));
    params.appendChild(node);

    node = req.createElement("string");
    node.setAttribute("name", "albumId");
    node.appendChild(req.createTextNode(photo.albumId()));
    params.appendChild(node);

    node = req.createElement("string");
    node.setAttribute("name", "photoId");
    node.appendChild(req.createTextNode(photo.photoId()));
    params.appendChild(node);

    node = req.createElement("string");
    node.setAttribute("name", "text");
    node.appendChild(req.createTextNode(comment));
    params.appendChild(node);

    QDomDocument res = this->sendRequest(req, this->sendCommentAction);

    if (this->checkGoodResponse(res, this->sendCommentAction))
        return true;
    else
        return false;
}

bool QTransport::uploadPhoto(QString accountId, QString albumId, QString file_name, QString description="")
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ": accountId=" <<this->accId;

    if (!this->checkFunction(CLASS_PHOTOS, UPLOAD_PHOTO, this->uploadPhotoAction))
        return false;

    QFileInfo finfo(file_name);
    if (!finfo.isFile() || !finfo.isReadable()) {
        QErrorMsg msg(APP_FILE_READ_ERROR, tr("Can't read file to upload"));
        emit this->errorOccurred(accountId, msg,
                                 QTransport::uploadPhotoAction);
        qWarning() << "Can't read file to upload: " << file_name;
        return false;
    }

    QDomDocument req = this->createRequest(CLASS_PHOTOS, UPLOAD_PHOTO);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    QDomElement node = req.createElement("string");
    node.setAttribute("name", "albumId");
    node.appendChild(req.createTextNode(albumId));
    params.appendChild(node);

    node = req.createElement("string");
    node.setAttribute("name", "file");
    node.appendChild(req.createTextNode(finfo.absoluteFilePath()));
    params.appendChild(node);

    node = req.createElement("string");
    node.setAttribute("name", "fileName");
    node.appendChild(req.createTextNode(finfo.fileName()));
    params.appendChild(node);

    if (!description.isEmpty()) {
        node = req.createElement("string");
        node.setAttribute("name", "description");
        node.appendChild(req.createTextNode(description));
        params.appendChild(node);
    }

    QDomDocument res = this->sendRequest(req, this->uploadPhotoAction);

    QDomNodeList respNode = res.elementsByTagName("Response");
    if (respNode.count() != 0 &&
        respNode.at(0).toElement().attribute("function").compare(UPLOAD_PHOTO) == 0) {
        respNode=respNode.at(0).toElement().elementsByTagName("Params");
        if (respNode.count()==0) {
            QErrorMsg err(APP_LIBRARY_WRONG_RESPONSE, tr("Bad response from driver"));
            emit this->errorOccurred(this->accId, err, QTransport::uploadPhotoAction);
            return false;
        }
        respNode = respNode.at(0).toElement().childNodes();
        QDomElement curElem;
        QString curElemName;
        QString photoId;
        for (int i = 0; i < respNode.count(); i++) {
            curElem = respNode.at(i).toElement();
            if (curElem.nodeName().compare("string") != 0)
                continue;
            curElemName = curElem.attribute("name");
            if (curElemName.compare("albumId") == 0)
                albumId = curElem.firstChild().nodeValue().trimmed();
            else
            if (curElemName.compare("photoId") == 0)
                photoId = curElem.firstChild().nodeValue().trimmed();
        }
        emit this->photoUploaded(accountId, albumId, file_name, photoId);
        return true;
    } else
    if (this->checkGoodResponse(res, this->uploadPhotoAction)) {
        emit this->photoUploaded(accountId, albumId, file_name, "");
        return true;
    } else {
        return false;
    }
}

bool QTransport::uploadPhoto(Album al, QString file_name, QString description)
{
    return uploadPhoto(al.accountId(), al.albumId(), file_name, description);
}

bool QTransport::downloadPhoto(QString url, QString file_name)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ": accountId=" <<this->accId;

    if (!this->checkFunction(CLASS_PHOTOS, GET_PHOTO, this->getPhotoAction))
        return false;

    QDomDocument req = this->createRequest(CLASS_PHOTOS, GET_PHOTO);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    QDomElement urlNode = req.createElement("string");
    urlNode.setAttribute("name", "url");
    urlNode.appendChild(req.createTextNode(url));
    params.appendChild(urlNode);

    QDomElement fileNode = req.createElement("string");
    fileNode.setAttribute("name", "path");
    fileNode.appendChild(req.createTextNode(file_name));
    params.appendChild(fileNode);

    QDomDocument res = this->sendRequest(req, this->getPhotoAction);

    if (this->checkGoodResponse(res, this->getPhotoAction))
        return true;
    else
        return false;
}

void QTransport::downloadPhotoList(PhotoList list, Photo curPhoto, int nearest)
{
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "() Album " << curPhoto.albumId() << " has size " << list.length();
    if (nearest < 0) {
        qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << "(): nearest < 0";
        QErrorMsg msg;
        msg.code = APP_FUNCTION_WRONG_ARGS;
        msg.comment = tr("QTransport::downloadPhotoLost() with nearest < 0");
        msg.text = tr("Wrong arguments of function");
        emit this->errorOccurred(this->accId, msg, this->getPhotoAction);
        return;
    }

    QDir dir;
    if (!dir.exists(Utils::getPhotoDir(this->serviceName())))
        dir.mkpath(Utils::getPhotoDir(this->serviceName()));

    int len = list.length();
    int startItem = -1;
    for (int i = 0; i < len; i++) {
        if (list.at(i).photoId().compare(curPhoto.photoId()) == 0) {
            startItem = i;
            break;
        }
    }

    if (len == 0 || startItem < 0)
    {
        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "() Empty photo list for album " << curPhoto.albumId();
        QString photo_path = generateFileName(Utils::getPhotoDir(this->serviceName()), curPhoto.photo_url());
        QFile file(photo_path);
        if (!file.exists()) {
            if (!downloadPhoto(curPhoto.photo_url(), file.fileName()))
                return;
            curPhoto.setPhoto(photo_path);
        } else {
            curPhoto.setPhoto(photo_path);
        }
        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "() Finish download image for album " << curPhoto.albumId();
        PhotoList lst;
        lst.append(curPhoto);
        emit photosReceived(curPhoto.accountId(), curPhoto.ownerId(), curPhoto.albumId(), lst, true);
        return;
    } else {
        QSet<QString> photoSet;
        for (int i = 0; i <= nearest; i++) {
            for (int j = -1; j < 2; j += 2) {
                if (this->needShutdown)
                    return;

                if (i == 0 && j > 0)
                    continue;

                int cp = startItem + i * j;
                qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "() download image: start=" << startItem << " i=" << i << " cp=" << cp;
                while (cp < 0)
                    cp += len;

                // FIXME
                while (cp >= list.size())
                    cp -= len;

                if (photoSet.contains(list.at(cp).photoId())) {
                    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "() Duplicate of photo was found";
                    continue;
                }
                photoSet.insert(list.at(cp).photoId());
                qDebug()<< "photo_url = "<< list.at(cp).photo_url() << endl;

                QString photo_path = generateFileName(Utils::getPhotoDir(this->serviceName()), list.at(cp).photo_url());
                QFile file(photo_path);
                if (!file.exists()) {
                    if (!downloadPhoto(list.at(cp).photo_url(), file.fileName()))
                        return;
                    list[cp].setPhoto(photo_path);
                    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "() emit PhotosDownload signal" << endl;
                    emit photosReceived(curPhoto.accountId(), curPhoto.ownerId(), curPhoto.albumId(), list, false);
                } else {
                    list[cp].setPhoto(photo_path);
                }
            }
        }
    }

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "() Finish download image for album " << curPhoto.albumId();
    emit photosReceived(curPhoto.accountId(), curPhoto.ownerId(), curPhoto.albumId(), list, true);
}

MessageList QTransport::getInbox()
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ":" <<this->accId << "(" << this->timer.elapsed() << ")";

    if (!this->checkFunction(CLASS_MESSAGES, GET_LIST_INBOX_MESSAGES, this->getMessagesAction))
        return MessageList();

    MessageList ret;
    ret.setTimeCreate(QDateTime::currentDateTime());

    int pageNum = 1;
    int pageSize = 100;

    while(true)
    {
        qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ":" <<this->accId << "(" << this->timer.elapsed() << ")" << "pageNum=" << pageNum;
        QDomDocument req = this->createRequest(CLASS_MESSAGES, GET_LIST_INBOX_MESSAGES);

        QDomElement params = req.elementsByTagName("Params").at(0).toElement();

        QDomElement pr = req.createElement("number");
        pr.setAttribute("name", "page");
        pr.appendChild(req.createTextNode(QString::number(pageNum)));
        params.appendChild(pr);

        pr = req.createElement("number");
        pr.setAttribute("name", "pageSize");
        pr.appendChild(req.createTextNode(QString::number(pageNum + pageSize)));
        params.appendChild(pr);

        QDomDocument res = this->sendRequest(req, this->getMessagesAction);

        if (!this->checkBadResponse(res, this->getMessagesAction)) {
            qDebug() << __FILE__ << ":" << __LINE__ << ":" << this->accId  << "(" << this->timer.elapsed() << ")" << ": getInbox check response error";
            return ret;
        }

        QDomElement array = res.elementsByTagName("Response").at(0).toElement().
                elementsByTagName("Params").at(0).toElement().
                elementsByTagName("array").at(0).toElement();

        QDomNodeList messages = array.childNodes();

        if (messages.count() == 0)
            break;

        for (int i = 0; i < messages.count(); i++) {
            QDomElement curMessage = messages.at(i).toElement();

            Message msg;
            msg.setMessageId(curMessage.attribute("id"));
            msg.setAccountId(this->accId);
            msg.setReceived(true);

            for (int j = 0; j < curMessage.childNodes().count(); j++) {
                QDomNode curNode = curMessage.childNodes().at(j);

                // parse recepient list
                if (curNode.nodeName().compare("array") == 0) {
                    QDomElement recElement = curNode.toElement();
                    int recQuantity = recElement.attribute("quantity").toInt();

                    for (int i = 0; i < recQuantity; i++) {
                        QDomNode recNode = curNode.childNodes().at(i);

                        Recepient rec;
                        for (int k = 0; k < recNode.childNodes().count(); k++) {
                            QDomElement recItem = recNode.childNodes().at(k).toElement();

                            if (recItem.nodeName().compare("string") == 0) {
                                if (recItem.attribute("name").compare("RecipientId") == 0)
                                    rec.id = recItem.firstChild().nodeValue();

                                if (recItem.attribute("name").compare("RecipientName") == 0)
                                    rec.name = recItem.firstChild().nodeValue();
                            }
                        }
                        if (rec.id.compare(this->accId) != 0)
                            msg.recepientList().append(rec);
                    }
                }

                QDomElement curItem = curMessage.childNodes().at(j).toElement();
                const QString curItemName = curItem.attribute("name");
                if (curItem.nodeName().compare("string") == 0) {
                    if (curItemName.compare("SenderId") == 0)
                    {
                        msg.setSenderId(curItem.firstChild().nodeValue());
                        if (msg.senderId().compare(this->accId) == 0)
                            msg.setSended(true);
                    } else
                        if (curItemName.compare("SenderName") == 0)
                            msg.setSenderName(curItem.firstChild().nodeValue());
                        else
                            if (curItemName.compare("Time") == 0) {
                                QDateTime mdf;
                                mdf.setTime_t(curItem.firstChild().nodeValue().toInt());
                                msg.setTime(mdf.toString(DATE_TIME_FORMAT));
                            }
                            else
                                if (curItemName.compare("Text") == 0)
                                    msg.setText(curItem.firstChild().nodeValue());
                                else
                                    if (curItemName.compare("Title") == 0)
                                        msg.setTitle(curItem.firstChild().nodeValue());
                                    else
                                        if (curItemName.compare("Status") == 0)
                                            msg.setReaded(curItem.firstChild().nodeValue().startsWith('1'));
                }
            }
            if (msg.senderId().isEmpty())
                msg.setSended(true);
            ret.append(msg);
        }
        pageNum++;
    }
    return ret;
}

MessageList QTransport::getOutbox()
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ": accountId=" <<this->accId << "(" << this->timer.elapsed() << ")";

    if (!this->checkFunction(CLASS_MESSAGES, GET_LIST_OUTBOX_MESSAGES, this->getMessagesAction))
        return MessageList();

    MessageList ret;
    ret.setTimeCreate(QDateTime::currentDateTime());

    int pageNum = 1;
    int pageSize = 100;

    while(true)
    {
        qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ":" <<this->accId << "(" << this->timer.elapsed() << ")" << "pageNum=" << pageNum;
        QDomDocument req = this->createRequest(CLASS_MESSAGES, GET_LIST_OUTBOX_MESSAGES);

        QDomElement params = req.elementsByTagName("Params").at(0).toElement();

        QDomElement pr = req.createElement("number");
        pr.setAttribute("name", "page");
        pr.appendChild(req.createTextNode(QString::number(pageNum)));
        params.appendChild(pr);

        pr = req.createElement("number");
        pr.setAttribute("name", "pageSize");
        pr.appendChild(req.createTextNode(QString::number(pageSize)));
        params.appendChild(pr);

        QDomDocument res = this->sendRequest(req, this->getMessagesAction);

        if (!this->checkBadResponse(res, this->getMessagesAction)) {
            qDebug() << __FILE__ << ":" << __LINE__ << ":" << this->accId << ": getOutbox check response error";
            return ret;
        }

        QDomElement array = res.elementsByTagName("Response").at(0).toElement().
                elementsByTagName("Params").at(0).toElement().
                elementsByTagName("array").at(0).toElement();

        QDomNodeList messages = array.childNodes();

        if (messages.count() == 0)
            break;

        for (int i = 0; i < messages.count(); i++) {
            QDomElement curMessage = messages.at(i).toElement();

            Message msg;
            msg.setMessageId(curMessage.attribute("id"));
            msg.setAccountId(this->accId);
            msg.setSended(true);

            for (int j = 0; j < curMessage.childNodes().count(); j++) {
                QDomNode curNode = curMessage.childNodes().at(j);

                // parse recepient list
                if (curNode.nodeName().compare("array") == 0) {
                    QDomElement recElement = curNode.toElement();
                    int recQuantity = recElement.attribute("quantity").toInt();

                    for (int i = 0; i < recQuantity; i++) {
                        QDomNode recNode = curNode.childNodes().at(i);

                        Recepient rec;
                        for (int k = 0; k < recNode.childNodes().count(); k++) {
                            QDomElement recItem = recNode.childNodes().at(k).toElement();

                            if (recItem.nodeName().compare("string") == 0) {
                                if (recItem.attribute("name").compare("RecipientId") == 0)
                                    rec.id = recItem.firstChild().nodeValue();

                                if (recItem.attribute("name").compare("RecipientName") == 0)
                                    rec.name = recItem.firstChild().nodeValue();
                            }
                        }
                        if (rec.id.compare(this->accId) == 0)
                            msg.setReceived(true);
                        else
                            msg.recepientList().append(rec);
                    }
                }

                QDomElement curItem = curMessage.childNodes().at(j).toElement();
                const QString curItemName = curItem.attribute("name");

                if (curItem.nodeName().compare("string") == 0) {
                    if (curItemName.compare("SenderId") == 0)
                        msg.setSenderId(curItem.firstChild().nodeValue());
                    else
                        if (curItemName.compare("SenderName") == 0)
                            msg.setSenderName(curItem.firstChild().nodeValue());
                        else
                            if (curItemName.compare("Time") == 0) {
                                QDateTime mdf;
                                mdf.setTime_t(curItem.firstChild().nodeValue().toInt());
                                msg.setTime(mdf.toString(DATE_TIME_FORMAT));
                            }
                            else
                                if (curItemName.compare("Text") == 0)
                                    msg.setText(curItem.firstChild().nodeValue());
                                else
                                    if (curItemName.compare("Title") == 0)
                                        msg.setTitle(curItem.firstChild().nodeValue());
                                    else
                                        if (curItemName.compare("Status") == 0)
                                            msg.setReaded(curItem.firstChild().nodeValue().startsWith('1'));
                }
            }
            if (msg.recepientList().isEmpty())
                msg.setReceived(true);
            ret.append(msg);
        }

        pageNum++;
    }

    return ret;
}

MessageList QTransport::getThreadList()
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << ": accountId=" <<this->accId;

    if (!this->checkFunction(CLASS_MESSAGES, GET_LIST_THREADS, this->getMessagesAction))
        return MessageList();

    // thie list of thread IDs.
    QList<QString> threadIds;

    // STEP 1. get thread IDs.

    QDomDocument req = this->createRequest(CLASS_MESSAGES, GET_LIST_THREADS);

//    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

//    QDomElement pr = req.createElement("number");
//    pr.setAttribute("name", "from");
//    pr.appendChild(req.createTextNode(QString::number(0)));
//    params.appendChild(pr);
//
//    pr = req.createElement("number");
//    pr.setAttribute("name", "to");
//    pr.appendChild(req.createTextNode(QString::number(100)));
//    params.appendChild(pr);

    QDomDocument res = this->sendRequest(req, this->getMessagesAction);

    if (!this->checkBadResponse(res, this->getMessagesAction)) {
        return MessageList();
    }

    QDomElement array = res.elementsByTagName("Response").at(0).toElement().
                        elementsByTagName("Params").at(0).toElement().
                        elementsByTagName("array").at(0).toElement();

    QDomNodeList messages = array.childNodes();

    for (int i = 0; i < messages.count(); i++) {
        QDomElement curMessage = messages.at(i).toElement();

        threadIds.append(curMessage.attribute("id"));
    }

    MessageList ret;

    // STEP 2. get messages from thread.

    for (int thi = 0; thi < threadIds.length(); thi++)
    {
        req = this->createRequest(CLASS_MESSAGES, GET_LIST_THREAD_MESSAGES);

        QDomElement params = req.elementsByTagName("Params").at(0).toElement();

        QDomElement thID = req.createElement("string");
        thID.setAttribute("name", "threadId");
        thID.appendChild(req.createTextNode(threadIds.at(thi)));
        params.appendChild(thID);

        res = this->sendRequest(req, this->getMessagesAction);

        qDebug() << "RESPONSE: " << res.toString();

        if (!this->checkBadResponse(res, this->getMessagesAction)) {
            continue;
        }

        array = res.elementsByTagName("Response").at(0).toElement().
                            elementsByTagName("Params").at(0).toElement().
                            elementsByTagName("array").at(0).toElement();

        messages = array.childNodes();

        for (int i = 0; i < messages.count(); i++) {
            QDomElement curMessage = messages.at(i).toElement();

            Message msg;
            msg.setMessageId(curMessage.attribute("id"));
            msg.setAccountId(this->accId);
            //msg.setThreadId(params.attribute("threadId"));
            msg.setThreadId(threadIds.at(thi));

            for (int j = 0; j < curMessage.childNodes().count(); j++) {
                QDomNode curNode = curMessage.childNodes().at(j);

                // parse recepient list
                if (curNode.nodeName().compare("array") == 0) {
                    QDomElement recElement = curNode.toElement();
                    int recQuantity = recElement.attribute("quantity").toInt();

                    for (int i = 0; i < recQuantity; i++) {
                        QDomNode recNode = curNode.childNodes().at(i);

                        Recepient rec;
                        for (int k = 0; k < recNode.childNodes().count(); k++) {
                            QDomElement recItem = recNode.childNodes().at(k).toElement();

                            if (recItem.nodeName().compare("string") == 0) {
                                if (recItem.attribute("name").compare("RecipientId") == 0)
                                    rec.id = recItem.firstChild().nodeValue();

                                if (recItem.attribute("name").compare("RecipientName") == 0)
                                    rec.name = recItem.firstChild().nodeValue();
                            }
                        }
                        if (rec.id.compare(this->accId) == 0)
                            msg.setReceived(true);
                        else
                            msg.recepientList().append(rec);
                    }
                }

                QDomElement curItem = curMessage.childNodes().at(j).toElement();
                const QString curItemName = curItem.attribute("name");

                if (curItem.nodeName().compare("string") == 0) {
                    Recepient rec;
                    if (curItemName.compare("SenderId") == 0)
                    {
                        msg.setSenderId(curItem.firstChild().nodeValue());
                        if (msg.senderId().compare(this->accId) == 0)
                            msg.setSended(true);
                    }
                    else
                    if (curItemName.compare("SenderName") == 0)
                        msg.setSenderName(curItem.firstChild().nodeValue());
                    else
                    if (curItemName.compare("RecepientId") == 0)
                    {
                        rec.id = curItem.firstChild().nodeValue();
                        if (rec.id.compare(this->accId) == 0)
                        {
                            msg.setReceived(true);
                        }
                    }
                    else
                    if (curItemName.compare("RecepientName") == 0)
                        rec.name = curItem.firstChild().nodeValue();
                    else
                    if (curItemName.compare("Time") == 0) {
                        QDateTime mdf;
                        mdf.setTime_t(curItem.firstChild().nodeValue().toInt());
                        msg.setTime(mdf.toString(DATE_TIME_FORMAT));
                    }
                    else
                    if (curItemName.compare("Text") == 0)
                        msg.setText(curItem.firstChild().nodeValue());
                    else
                    if (curItemName.compare("Title") == 0)
                        msg.setTitle(curItem.firstChild().nodeValue());
                    else
                    if (curItemName.compare("Status") == 0)
                        msg.setReaded(curItem.firstChild().nodeValue().startsWith('1'));
                }
            }

            if (msg.recepientList().isEmpty())
                msg.setReceived(true);

            if (msg.senderId().isEmpty())
                msg.setSended(true);

            if (msg.messageId().isEmpty())
                msg.setMessageId(msg.threadId() + "_" + msg.time());

            ret.append(msg);
        }
    }

    return ret;
}

void QTransport::getMessages()
{
    MessageList ret;

    if (this->checkFunction(CLASS_MESSAGES, GET_LIST_THREADS))
        ret = this->getThreadList();
    else if (this->checkFunction(CLASS_MESSAGES, GET_LIST_INBOX_MESSAGES))
    {
        ret = this->getInbox();
        ret.append(this->getOutbox());
    }

    emit this->messagesReceived(this->accId, ret, true);
}

void QTransport::readMessage(QString messageId)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << "()";

    if (!this->checkFunction(CLASS_MESSAGES, READ_MESSAGE, this->readMessageAction))
        return;

    QDomDocument req = this->createRequest(CLASS_MESSAGES, READ_MESSAGE);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    QDomElement node = req.createElement("string");
    node.setAttribute("name", "messageId");
    node.appendChild(req.createTextNode(messageId));
    params.appendChild(node);

    QDomDocument res = this->sendRequest(req, this->readMessageAction);

    if (this->checkGoodResponse(res, this->readMessageAction)) {
        emit this->messageReaded(this->accId, messageId);
    }
}

void QTransport::deleteMessage(QString messageId)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << "()";

    if (!this->checkFunction(CLASS_MESSAGES, DELETE_MESSAGE, this->deleteMessageAction))
        return;

    QDomDocument req = this->createRequest(CLASS_MESSAGES, DELETE_MESSAGE);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    QDomElement node = req.createElement("string");
    node.setAttribute("name", "messageId");
    node.appendChild(req.createTextNode(messageId));
    params.appendChild(node);

    QDomDocument res = this->sendRequest(req, this->deleteMessageAction);

    if (this->checkGoodResponse(res, this->deleteMessageAction)) {
        emit this->messageDeleted(this->accId, messageId);
    }
}

bool QTransport::sendMessage(Friend owner, QString title, QString message)
{
    return this->sendMessage(owner.ownerId(), title, message);
}

bool QTransport::sendMessage(QString ownerId, QString title, QString message)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << "()";

    if (!this->checkFunction(CLASS_MESSAGES, SEND_MESSAGE, this->sendMessageAction))
        return false;

    QDomDocument req = this->createRequest(CLASS_MESSAGES, SEND_MESSAGE);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    params.setAttribute("id", ownerId);

    QDomElement node = req.createElement("string");
    node.setAttribute("name", "text");
    node.appendChild(req.createTextNode(message));
    params.appendChild(node);

    node = req.createElement("string");
    node.setAttribute("name", "title");
    node.appendChild(req.createTextNode(title));
    params.appendChild(node);

    QDomDocument res = this->sendRequest(req, this->sendMessageAction);

    if (this->checkGoodResponse(res, this->sendMessageAction)) {
        emit this->messageSended(this->accId, ownerId, title, message);
        return true;
    } else {
        return false;
    }
}

void QTransport::getFeed(QEventFeed::FeedType type)
{
    if (!this->checkFunction(CLASS_NEWS, GET_LIST_NEWS, this->getFeedAction))
        return;

    QDomDocument req = this->createRequest(CLASS_NEWS, GET_LIST_NEWS);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    QDomElement nodeType = req.createElement("string");
    nodeType.setAttribute("name", "newsType");
    switch(type)
    {
    case QEventFeed::audioFeed:
        nodeType.appendChild(req.createTextNode("audio"));
        break;
    case QEventFeed::commentFeed:
        nodeType.appendChild(req.createTextNode("comment"));
        break;
    case QEventFeed::friendFeed:
        nodeType.appendChild(req.createTextNode("friend"));
        break;
    case QEventFeed::likeFeed:
        nodeType.appendChild(req.createTextNode("like"));
        break;
    case QEventFeed::messageFeed:
        nodeType.appendChild(req.createTextNode("message"));
        break;
    case QEventFeed::photoFeed:
        nodeType.appendChild(req.createTextNode("photo"));
        break;
    case QEventFeed::tagFeed:
        nodeType.appendChild(req.createTextNode("tag"));
        break;
    case QEventFeed::videoFeed:
        nodeType.appendChild(req.createTextNode("video"));
        break;
    default:
        qWarning() << __FILE__ << __LINE__ << "unknown feed type";
    }
    params.appendChild(nodeType);

    QDomDocument res = this->sendRequest(req, this->getFeedAction);

    if (!this->checkBadResponse(res, this->getFeedAction))
        return;

    QDomElement array = res.elementsByTagName("Response").at(0).toElement().
                        elementsByTagName("Params").at(0).toElement().
                        elementsByTagName("array").at(0).toElement();

   //qDebug() << res.toString(1);

    QEventFeedList ret;
    ret.setTimeCreate(QDateTime::currentDateTime());

    for (int i = 0; i < array.childNodes().count(); i++) {
        if (!array.childNodes().at(i).isElement())
            continue;
        QDomElement curEvent = array.childNodes().at(i).toElement();
        QEventFeed ef;
        ef.setAccountId(this->accId);

        for (int j = 0; j < curEvent.childNodes().count(); j++) {
            QDomElement curItem = curEvent.childNodes().at(j).toElement();
            const QString curItemName = curItem.attribute("name");
            if (curItem.nodeName().compare("string") == 0) {
                if (curItemName.compare("authorId") == 0)
                    ef.setOwnerId(curItem.firstChild().nodeValue().trimmed());
                else
                if (curItemName.compare("authorName") == 0)
                    ef.setOwnerName(curItem.firstChild().nodeValue().trimmed());
                else
                if (curItemName.compare("text") == 0)
                    ef.setText(curItem.firstChild().nodeValue().trimmed());
            }
            else
            if (curItem.nodeName().compare("number") == 0) {
                if (curItemName.compare("created") == 0) {
                    QDateTime oldCreated = ef.created();
                    oldCreated.setTime_t(curItem.firstChild().nodeValue().toInt());
                    ef.setCreated(oldCreated);
                }
            }
            else
                if(curItem.nodeName().compare("struct") == 0) {
                    if (curItemName.compare("attachment") == 0) {
                        ef.setAttach(this->loadAttachment(curItem));
                    }
                }
            else
            {
                qDebug() << "Unknown node: " << curItem.nodeName() << " with name=" << curItemName;
            }
        }
        //qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): ownerId=" << ef.ownerId << " ownerName=" << ef.ownerName;
        ret.append(ef);
    }

    emit this->eventFeedReceived(this->accId, ret, type, true);
}

QAttachment QTransport::loadAttachment(const QDomElement &node)
{
    QAttachment ret;

    ret.setId(node.attribute("id"));
    ret.setOwnerId(node.attribute("ownerId"));

    QString type = node.attribute("type");

    if (type.compare("image") == 0)
    {
        ret.setType(QAttachment::ImageAttach);
        for (int i = 0; i < node.childNodes().count(); i++)
        {
            QDomElement curItem = node.childNodes().at(i).toElement();
            const QString curItemName = curItem.attribute("name");
            if (curItem.nodeName().compare("string") == 0)
            {
                if (curItemName.compare("name") == 0)
                    ret.setName(curItem.firstChild().nodeValue().trimmed());
                else
                    if (curItemName.compare("albumId") == 0)
                        ret.setAlbumId(curItem.firstChild().nodeValue().trimmed());
                else
                    if (curItemName.compare("urlSmall") == 0)
                        ret.setIconUrl(curItem.firstChild().nodeValue().trimmed());
                else
                    if (curItemName.compare("urlBig") == 0)
                        ret.setObjectUrl(curItem.firstChild().nodeValue().trimmed());
            }
        } // for (int i = 0; i < node.childNodes(); i++)

    } else
        if (type.compare("video") == 0)
    {
        ret.setType(QAttachment::VideoAttach);
        for (int i = 0; i < node.childNodes().count(); i++)
        {
            QDomElement curItem = node.childNodes().at(i).toElement();
            const QString curItemName = curItem.attribute("name");
            if (curItem.nodeName().compare("string") == 0)
            {
                if (curItemName.compare("name") == 0)
                    ret.setName(curItem.firstChild().nodeValue().trimmed());
                else
                    if (curItemName.compare("url") == 0)
                        ret.setObjectUrl(curItem.firstChild().nodeValue().trimmed());
                else
                    if (curItemName.compare("urlImage") == 0)
                        ret.setIconUrl(curItem.firstChild().nodeValue().trimmed());
            } else
                if (curItem.nodeName().compare("number") == 0)
            {
                if (curItemName.compare("duration") == 0)
                    ret.setDuration(curItem.firstChild().nodeValue().trimmed());
            }
        }
    } else
        if (type.compare("audio") == 0)
    {
        ret.setType(QAttachment::AudioAttach);
        for (int i = 0; i < node.childNodes().count(); i++)
        {
            QDomElement curItem = node.childNodes().at(i).toElement();
            const QString curItemName = curItem.attribute("name");
            if (curItem.nodeName().compare("string") == 0)
            {
                if (curItemName.compare("name") == 0)
                    ret.setName(curItem.firstChild().nodeValue().trimmed());
                else
                    if (curItemName.compare("url") == 0)
                        ret.setObjectUrl(curItem.firstChild().nodeValue().trimmed());
            } else
                if (curItem.nodeName().compare("number") == 0)
            {
                if (curItemName.compare("duration") == 0)
                    ret.setDuration(curItem.firstChild().nodeValue().trimmed());
            }
        }
    } else
        if (type.compare("link") == 0)
    {
        ret.setType(QAttachment::LinkAttach);
        for (int i = 0; i < node.childNodes().count(); i++)
        {
            QDomElement curItem = node.childNodes().at(i).toElement();
            const QString curItemName = curItem.attribute("name");
            if (curItem.nodeName().compare("string") == 0)
            {
                if (curItemName.compare("name") == 0)
                    ret.setName(curItem.firstChild().nodeValue().trimmed());
                else
                    if (curItemName.compare("url") == 0)
                        ret.setObjectUrl(curItem.firstChild().nodeValue().trimmed());
                else
                    if (curItemName.compare("urlImage") == 0)
                        ret.setIconUrl(curItem.firstChild().nodeValue().trimmed());
            }
        }
    } else
        if (type.compare("note") == 0)
    {
        for (int i = 0; i < node.childNodes().count(); i++)
        {
            QDomElement curItem = node.childNodes().at(i).toElement();
            const QString curItemName = curItem.attribute("name");
            if (curItem.nodeName().compare("string") == 0)
            {
                if (curItemName.compare("name") == 0)
                    ret.setName(curItem.firstChild().nodeValue().trimmed());
                else
                    if (curItemName.compare("url") == 0)
                        ret.setObjectUrl(curItem.firstChild().nodeValue().trimmed());
            }
        }
    }

    QDir dir;
    if (!dir.exists(Utils::getAlbumsIconDir(this->serviceName())))
        dir.mkpath(Utils::getAlbumsIconDir(this->serviceName()));

    if (!ret.iconUrl().isEmpty())
    {
        QString icon = NULL;
        icon = generateFileName(Utils::getAlbumsIconDir(this->serviceName()), ret.iconUrl());
        QFile file(icon);
        if (!file.exists()) {
            if (downloadPhoto(ret.iconUrl(), icon))
                ret.setIcon(icon);
        } else
            ret.setIcon(icon);
    }

    return ret;
}

void QTransport::gotError()
{
    this->stats.addError();
    emit this->statsUpdated(this->accId);
}

QStatInfo QTransport::getStats() const
{
    return this->stats;
}
