#include "photo.h"
#include "utils/utils.h"

class PhotoData : public QSharedData
{
public:
    PhotoData();
    PhotoData(const PhotoData& other);
    ~PhotoData();

    QString photoId;     //<! photo ID from service
    QString albumId;     //<! album ID from service
    QString ownerId;     //<! user ID from service
    QString time_create; //<! creation time
    QString icon_url;    //<! path to thumbnail (icon)
    QString icon;        //<! path to downloaded icon
    QString photo_url;   //<! path to large image
    QString photo;       //<! path to downloaded large image
    QString description; //<! photo description
    QString accountId; //<! the name of account of this friend (not stored in cache)
};

PhotoData::PhotoData()
    : QSharedData()
{
}

PhotoData::PhotoData(const PhotoData& other)
    : QSharedData(other),
      photoId(other.photoId),
      albumId(other.albumId),
      ownerId(other.ownerId),
      time_create(other.time_create),
      icon_url(other.icon_url),
      icon(other.icon),
      photo_url(other.photo_url),
      photo(other.photo),
      description(other.description),
      accountId(other.accountId)
{
}

PhotoData::~PhotoData()
{
}


Photo::Photo()
{
    d = new PhotoData;
}

Photo::Photo(const Photo &src)
    : d(src.d)
{
}

Photo::~Photo()
{

}

QDomElement Photo::toQDomElement(QDomDocument& doc) const
{
    return Photo::toQDomElement(doc, *this);
}

QDomElement Photo::toQDomElement(QDomDocument& doc, const Photo& src)
{
    QDomElement ret = doc.createElement(NODE_PHOTO_ROOT);

    // ID
    QDomElement node = doc.createElement(NODE_PHOTO_ID);
    QDomText t = doc.createTextNode(src.photoId());
    ret.appendChild(node);
    node.appendChild(t);

    //time create
    node = doc.createElement(NODE_PHOTO_CREATE);
    t = doc.createTextNode(src.time_create());
    ret.appendChild(node);
    node.appendChild(t);

    //icon url
    node = doc.createElement(NODE_PHOTO_ICON_URL);
    t = doc.createTextNode(src.icon_url());
    ret.appendChild(node);
    node.appendChild(t);

    //icon
    node = doc.createElement(NODE_PHOTO_ICON);
    t = doc.createTextNode(src.icon());
    ret.appendChild(node);
    node.appendChild(t);

    //image url
    node = doc.createElement(NODE_PHOTO_IMAGE_URL);
    t = doc.createTextNode(src.photo_url());
    ret.appendChild(node);
    node.appendChild(t);

    //image
    node = doc.createElement(NODE_PHOTO_IMAGE);
    t = doc.createTextNode(src.photo());
    ret.appendChild(node);
    node.appendChild(t);

    //description
    node = doc.createElement(NODE_PHOTO_DESCRIPTION);
    t = doc.createTextNode(src.description());
    ret.appendChild(node);
    node.appendChild(t);

    return ret;
}

Photo Photo::fromQDomElement(const QDomElement& fr, const QString& accountId, const QString& friendId, const QString& albumId)
{
    Photo ret = Photo();

    ret.setPhotoId(fr.elementsByTagName(NODE_PHOTO_ID).at(0).firstChild().toText().data());
    ret.setIcon_url(fr.elementsByTagName(NODE_PHOTO_ICON_URL).at(0).firstChild().toText().data());
    ret.setIcon(fr.elementsByTagName(NODE_PHOTO_ICON).at(0).firstChild().toText().data());
    ret.setTime_create(fr.elementsByTagName(NODE_PHOTO_CREATE).at(0).firstChild().toText().data());
    ret.setPhoto_url(fr.elementsByTagName(NODE_PHOTO_IMAGE_URL).at(0).firstChild().toText().data());
    ret.setPhoto(fr.elementsByTagName(NODE_PHOTO_IMAGE).at(0).firstChild().toText().data());
    QDomNodeList description = fr.elementsByTagName(NODE_PHOTO_DESCRIPTION);
    if (!description.isEmpty())
        ret.setDescription(description.at(0).firstChild().toText().data());
    else
        ret.setDescription("");
    ret.setAccountId(accountId);
    ret.setOwnerId(friendId);
    ret.setAlbumId(albumId);

    return ret;
}

Photo& Photo::operator =(Photo const& src)
{
    d = src.d;
    return *this;
}

PhotoCommentList Photo::getPhotoComments() const
{
    PhotoCommentList ret;
    QFile file(Utils::getAccountDir(this->accountId()) + QString(FILE_PHOTO_COMMENTS_DATA).arg(this->ownerId(), this->albumId(), this->photoId()));
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): " << file.fileName();
    if (file.open(QIODevice::ReadOnly) == false)
        return ret;

    QDomDocument doc;
    doc.setContent(&file);
    file.close();

    // get list updated time
    QDomElement root = doc.elementsByTagName(NODE_PHOTO_COMMENTS_ROOT).at(0).toElement();
    QDateTime time_create = QDateTime::fromString(root.attribute(ATTR_REFRESH, ""), DATE_TIME_FORMAT);
    ret.setTimeCreate(time_create);

    QDomNodeList lst = doc.elementsByTagName(NODE_PHOTO_COMMENT_ROOT);

    qDebug() << "Start load of " << lst.length() << " comments for " << this->photoId();
    for (uint i = 0; i < lst.length(); i++) {
        ret.append(PhotoComment::fromQDomElement(lst.at(i).toElement(), this->accountId(), this->ownerId(), this->albumId(), this->photoId()));
    }

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): loaded " << ret.length() << " comments";
    return ret;
}

void Photo::setPhotoComments(const PhotoCommentList& list, const QString& accountId, const QString& friendId, const QString& albumId, const QString& photoId)
{
    // store photo lists into cache
    QDomDocument out("MyDoc");
    QDomElement root = out.createElement(NODE_PHOTO_COMMENTS_ROOT);
    root.setAttribute(ATTR_REFRESH, list.timeCreate().toString(DATE_TIME_FORMAT));
    out.appendChild(root);

    for (int i = 0; i < list.length(); i++) {
        PhotoComment curPh = list.at(i);
        root.appendChild(PhotoComment::toQDomElement(out, curPh));
    }

    QDir dir;

    dir.setPath(dir.cleanPath(Utils::getAccountDir(accountId)));
    if (!dir.exists())
        dir.mkpath(dir.absolutePath());

    QFile file(dir.absolutePath() + "/" + QString(FILE_PHOTO_COMMENTS_DATA).arg(friendId, albumId, photoId));
    qDebug() << __FILE__ << ":" << __LINE__ << ": " << file.fileName();
    if (file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate) == false) {
        qDebug() << "can't open file: " + file.errorString();
    }
    //file.write(output.toUtf8());
    file.write(out.toByteArray(1));
    file.close();
}

PhotoList Photo::loadPhotoList(const QString& accountId, const QString& ownerId, const QString& albumId)
{
    PhotoList ret;
    QFile file(Utils::getAccountDir(accountId) + QString(FILE_PHOTOS_DATA).arg(ownerId, albumId));
    qDebug() << file.fileName();
    if (file.open(QIODevice::ReadOnly) == false)
        return ret;

    QDomDocument doc;
    doc.setContent(&file);
    file.close();

    // get list updated time
    QDomElement root = doc.elementsByTagName(NODE_PHOTOS_ROOT).at(0).toElement();
    QDateTime time_create = QDateTime::fromString(root.attribute(ATTR_REFRESH, ""), DATE_TIME_FORMAT);
    ret.setTimeCreate(time_create);

    QDomNodeList lst = doc.elementsByTagName(NODE_PHOTO_ROOT);

    qDebug() << "Start load of " << lst.length() << " photos for " << albumId;
    for (uint i = 0; i < lst.length(); i++) {
        ret.append(Photo::fromQDomElement(lst.at(i).toElement(), accountId, ownerId, albumId));
    }

    return ret;
}

QString Photo::getKey(const PhotoList& photos)
{
    if (photos.length() > 0)
        return QString("photos_%1_%2").arg(photos.at(0).accountId()).arg(photos.at(0).albumId());
    else
        return QString("photos");
}

QString Photo::getKey(const Album& album)
{
    return QString("photos_%1_%2").arg(album.accountId()).arg(album.albumId());
}

QString Photo::getKey(const Photo& val)
{
    return QString("photos_%1_%2").arg(val.accountId()).arg(val.albumId());
}

// Getters
const QString Photo::photoId() const { return d->photoId; }
const QString Photo::albumId() const { return d->albumId; }
const QString Photo::ownerId() const { return d->ownerId; }
const QString Photo::time_create() const { return d->time_create; }
const QString Photo::icon_url() const { return d->icon_url; }
const QString Photo::icon() const { return d->icon; }
const QString Photo::photo_url() const { return d->photo_url; }
const QString Photo::photo() const { return d->photo; }
const QString Photo::description() const { return d->description; }
const QString Photo::accountId() const { return d->accountId; }

// Setters
void Photo::setPhotoId(const QString& v) { d->photoId = v; }
void Photo::setAlbumId(const QString& v) { d->albumId = v; }
void Photo::setOwnerId(const QString& v) { d->ownerId = v; }
void Photo::setTime_create(const QString& v) { d->time_create = v; }
void Photo::setIcon_url(const QString& v) { d->icon_url = v; }
void Photo::setIcon(const QString& v) { d->icon = v; }
void Photo::setPhoto_url(const QString& v) { d->photo_url = v; }
void Photo::setPhoto(const QString& v) { d->photo = v; }
void Photo::setDescription(const QString& v) { d->description = v; }
void Photo::setAccountId(const QString& v) { d->accountId = v; }
