#include "qphotouploader.h"
#include "ui_qphotouploader.h"
#include <QPalette>
#include <QInputDialog>
#include <QTimer>
#include <QUrl>

Q_DECLARE_METATYPE(QPhotoUploader::FileInfo)
QPhotoUploader::QPhotoUploader(ServiceMgr *sm, QWidget *parent) :
    QDialog(parent),
    ui(new Ui::QPhotoUploader),
    totalSize(0),
    refreshStarted(false)
{
    setAttribute(Qt::WA_DeleteOnClose);
    connect(parent,SIGNAL(closeDialogs()),this, SLOT(close()));
    this->setParent(0);

    ui->setupUi(this);

    ui->addNewAlbumButton->hide();

#ifndef ALWAYS_UPDATE
    ui->refreshAlbumsButton->hide();
#endif
    setRefreshIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));
    ui->refreshAlbumsButton->setFocusPolicy(Qt::NoFocus);
    connect(ui->refreshAlbumsButton, SIGNAL(clicked()), this, SIGNAL(refreshPressed()));

    ui->cancelButton->hide();
    ui->listFiles->setContextMenuPolicy(Qt::ActionsContextMenu);
    actionEdit = new QAction(tr("Add description"),ui->listFiles);
    actionDelete = new QAction(tr("Delete"),ui->listFiles);
    ui->listFiles->addAction(actionEdit);
    ui->listFiles->addAction(actionDelete);
    connect(actionDelete,SIGNAL(triggered()),this,SLOT(deleteItem()));
    connect(actionEdit,SIGNAL(triggered()),this,SLOT(editItem()));
    this->sm = sm;
    const QString bgColor(this->palette().window().color().name());
    this->setStyleSheet(QString::fromLatin1("QListWidget { background-color: %1; border-color: %2; gridline-color: %3; selection-color: %4; border: 0px solid gray;}").arg(bgColor,bgColor,bgColor,bgColor));
    AccountList acc = this->sm->getAccounts();
    this->ui->listAccounts->setIconSize(QSize(48,48));
    this->ui->listAccounts->setMinimumHeight(70);

    connect(this->sm, SIGNAL(updateAccounts(AccountList)), this, SLOT(updateAccounts(AccountList)));
    this->updateAccounts(acc);
    connect(this->ui->listAccounts, SIGNAL(currentIndexChanged(int)), this, SLOT(selectAccount(int)));
    this->ui->listAccounts->setInsertPolicy(QComboBox::InsertAlphabetically);

    if (this->ui->listAccounts->count() == 0) {
        this->ui->listAccounts->addItem(tr("No active accounts"));
        this->ui->photoOpen->setDisabled(true);
        this->ui->listAccounts->setDisabled(true);
        this->ui->totalSizeLabel->hide();
        this->ui->humanSizeLabel->hide();
        this->ui->listAlbums->hide();
    }
    updateTotalSize();
    connect(this,SIGNAL(startUploadSignal()),this,SLOT(startUpload()),Qt::QueuedConnection);
}

QPhotoUploader::~QPhotoUploader()
{
    delete ui;
}

void QPhotoUploader::setRefreshIcon(QIcon icon)
{
    ui->refreshAlbumsButton->setIcon(icon);
}

void QPhotoUploader::updateAccounts(AccountList acc)
{
    DriverInfo *info;
    QString title;
    QIcon curIcon;
    QVariant v;
    this->ui->listAccounts->clear();
    for (int i = 0; i < acc.length(); i++) {
        if (!acc.at(i)->isNetworkEnabled)
            continue;

        info = acc.at(i)->transport->getDriverInfo();
        title = acc.at(i)->transport->accountId + " (" + info->name + ")\n" +
                        acc.at(i)->getProfile(false).name();
        curIcon = QIcon(acc.at(i)->getProfile(false).icon());
        v.setValue(acc.at(i));

        if (curIcon.isNull())
            this->ui->listAccounts->addItem(info->icon,title,v);
        else
            this->ui->listAccounts->addItem(curIcon,title,v);
    }

    selectAccount(this->ui->listAccounts->currentIndex());
}

void QPhotoUploader::selectAccount(int index=-1)
{
    if (index < 0)
        index = this->ui->listAccounts->currentIndex();

    QVariant v = this->ui->listAccounts->itemData(index);
    if (v == QVariant::Invalid)
        return;
    this->ui->listAlbums->clear();
    Account* acc = v.value<Account*>();
    //qDebug() << "accountId: " << acc->accountId();

    foreach(Album album, acc->getAlbumList()) {
        v.setValue(QStringList() << album.accountId << album.albumId);
        this->ui->listAlbums->addItem(
                Utils::decodeString(album.title) + " - "
                + tr("%1 photo(s)", "", album.size).arg(album.size),v);
    }
    if (this->ui->listAlbums->count() == 0) {
        this->ui->listAlbums->addItem(tr("No avalible albums"));
        this->ui->listAccounts->setMinimumWidth(this->ui->listAccounts->width());
#ifndef ALWAYS_UPDATE
        this->ui->refreshAlbumsButton->show();
#endif
        this->ui->photoOpen->setDisabled(true);
        this->ui->listAlbums->setDisabled(true);
        if (ui->listFiles->count()==0) {
            this->ui->totalSizeLabel->hide();
            this->ui->humanSizeLabel->hide();
        }
        this->ui->uploadButton->setDisabled(true);
    } else {
#ifndef ALWAYS_UPDATE
        this->ui->refreshAlbumsButton->hide();
        this->ui->listAccounts->setMinimumWidth(0);
        this->ui->listAccounts->setMaximumWidth(65536);
#endif
        this->ui->photoOpen->setEnabled(true);
        this->ui->listAlbums->setEnabled(true);
        this->ui->totalSizeLabel->show();
        this->ui->humanSizeLabel->show();
        ui->uploadButton->setEnabled(ui->listFiles->count()>0);
    }
    if (refreshStarted)
        on_refreshAlbumsButton_clicked();
}

QString QPhotoUploader::getShotName(QString name)
{
    if (name.length()>40)
        return name.left(30)+"..."+name.right(6);
    return name;
}

QString QPhotoUploader::getHumanSize(qint64 size)
{
    if (size < 1024)
        return QString::number(size)+" "+tr("bytes");
    if (size < 1024*1024)
        return QString::number(size/1024.0,'f',2)+" "+tr("Kbytes");
    return QString::number(size/(1024*1024.0),'f',3)+" "+tr("Mbytes");
}

void QPhotoUploader::on_photoOpen_clicked()
{
    QFileDialog dialog(this);
    dialog.setFileMode(QFileDialog::ExistingFiles);
    dialog.setNameFilter(tr("Images")+" (*.png *.gif *.jpg *.jpeg *.bmp)");
    dialog.setLabelText(QFileDialog::LookIn,tr("Look in:"));
    dialog.setLabelText(QFileDialog::FileName,tr("File name:"));
    dialog.setLabelText(QFileDialog::FileType,tr("File types:"));
    dialog.setLabelText(QFileDialog::Reject,tr("Cancel"));
    dialog.setLabelText(QFileDialog::Accept,tr("Add"));
    dialog.setDirectory(QDesktopServices::storageLocation(QDesktopServices::PicturesLocation));
    QList<QUrl> urls;
    urls << QUrl::fromLocalFile(QDesktopServices::storageLocation(QDesktopServices::HomeLocation))
         << QUrl::fromLocalFile(QDesktopServices::storageLocation(QDesktopServices::DesktopLocation))
         << QUrl::fromLocalFile(QDesktopServices::storageLocation(QDesktopServices::DocumentsLocation))
         << QUrl::fromLocalFile(QDesktopServices::storageLocation(QDesktopServices::PicturesLocation));
    dialog.setSidebarUrls(urls);
    QListWidgetItem *item;
    QFileInfo *fi;
    if(dialog.exec()) {
        foreach(QString name, dialog.selectedFiles()) {
            fi = new QFileInfo(name);
            if (fi->exists() && fi->isReadable()) {
                FileInfo mfi;
                mfi.size=fi->size();
                totalSize+=mfi.size;
                mfi.path=name;
                mfi.shotName=getShotName(fi->fileName());
                item = new QListWidgetItem(mfi.shotName+" - "+getHumanSize(mfi.size),ui->listFiles);
                QVariant v;
                v.setValue(mfi);
                item->setData(Qt::UserRole,v);
                QCoreApplication::processEvents();
            }
            delete fi;
        }
    }
    ui->uploadButton->setEnabled(ui->listFiles->count()>0);
    updateTotalSize();
    /*
    QString fileName = QFileDialog::getOpenFileName(this,
         tr("Open Image"), QDir::homePath(), tr("Image Files (*.png *.jpg *.jpeg *.gif *.bmp)"));
    this->ui->photoPath->setText(fileName);
    QPixmap photo(fileName);
    this->ui->photoIcon->setPixmap(photo.scaledToHeight(64));
    this->ui->photoIcon->setMinimumHeight(64);
    */
}

void QPhotoUploader::on_uploadButton_clicked()
{
    cancel = false;
    if (ui->listFiles->count()==0)
        return;
    ui->listAccounts->setDisabled(true);
    ui->listAlbums->setDisabled(true);
    ui->photoOpen->setDisabled(true);
    ui->uploadButton->hide();
    ui->cancelButton->show();
    this->repaint();
    QCoreApplication::processEvents();

    QVariant v;
    int index;
    QStringList al;

    index = this->ui->listAlbums->currentIndex();
    v = this->ui->listAlbums->itemData(index);
    if (v.canConvert<QStringList>())
        al = v.value<QStringList>();
    //qDebug() << "accountId: " << al[0] << " albumId: " << al[1];
    disconnect(this, SLOT(uploaded(QString, QString, QString,QString)));
    connect(sm, SIGNAL(photoUploaded(QString, QString, QString,QString)),
            this, SLOT(uploaded(QString, QString, QString,QString)), Qt::QueuedConnection);
    disconnect(this, SLOT(errorOccured(QString,QTransport::Action)));
    connect(sm, SIGNAL(errorOccured(QString,QTransport::Action)),
            this, SLOT(errorOccured(QString,QTransport::Action)), Qt::QueuedConnection);
    accountId = al[0];
    albumId = al[1];
    emit startUploadSignal();
    //for(int i=0; i<ui->listFiles->count(); i++) {
    //    sm->account(al[0])->transport->uploadPhoto(al[0],al[1],ui->listFiles->item(i)->text());
    //}

    //v = ui->listFiles->item(0)->data(Qt::UserRole);
    //if (v.canConvert<QString>())
    //   sm->account(al[0])->transport->uploadPhoto(al[0],al[1],v.value<QString>());
}

void QPhotoUploader::uploaded(QString accountId, QString albumId, QString file_name="", QString photoId="")
{
    Q_UNUSED(photoId)
    if (accountId != this->accountId)
        return;
    QVariant v;
    if (!file_name.isEmpty())
    {
        for(int i=0; i<ui->listFiles->count(); i++)
        {
            v = ui->listFiles->item(i)->data(Qt::UserRole);
            FileInfo fi;
            if (v.canConvert<FileInfo>())
                fi = v.value<FileInfo>();
            if (fi.path == file_name) {
                delete ui->listFiles->takeItem(i);
                break;
            }
        }
    }
    if (cancel)
        return;
    if (ui->listFiles->count() <= 0) {
        disconnect(this, SLOT(uploaded(QString, QString, QString, QString)));
        this->close();
        return;
    }
    v = ui->listFiles->item(0)->data(Qt::UserRole);
    if (v.canConvert<FileInfo>()) {
       FileInfo fi = v.value<FileInfo>();
       //sm->account(accountId)->transport->uploadPhoto(accountId,albumId,fi.path);
       sm->uploadPhoto(accountId,albumId,fi.path,fi.description);
    }
}

void QPhotoUploader::errorOccured(QString,QTransport::Action action)
{
    if (action==QTransport::uploadPhotoAction)
        on_cancelButton_clicked();
}

void QPhotoUploader::startUpload()
{
    uploaded(accountId, albumId);
}

void QPhotoUploader::enableUpload()
{
    ui->uploadButton->setEnabled(ui->listFiles->count()>0);
}

void QPhotoUploader::on_cancelButton_clicked()
{
    ui->listAccounts->setDisabled(false);
    ui->listAlbums->setDisabled(false);
    ui->photoOpen->setDisabled(false);
    ui->uploadButton->show();
    ui->cancelButton->hide();
    cancel=true;
}

void QPhotoUploader::editItem() {
    QListWidgetItem *item = ui->listFiles->item(ui->listFiles->currentRow());

    QVariant v = item->data(Qt::UserRole);
    FileInfo fi;
    if (v.canConvert<FileInfo>())
        fi = v.value<FileInfo>();

    bool ok;
    QString text = QInputDialog::getText(this, fi.shotName,
                                         tr("Enter descripton:"), QLineEdit::Normal,
                                         fi.description, &ok);
    if (ok) {
        qDebug() << "Sending descriptions not yet implemented";
        fi.description=text;
        v.setValue(fi);
        item->setData(Qt::UserRole,v);
    }
}

void QPhotoUploader::deleteItem() {
    QListWidgetItem *item = ui->listFiles->takeItem(ui->listFiles->currentRow());
    QVariant v = item->data(Qt::UserRole);
    FileInfo fi;
    if (v.canConvert<FileInfo>())
        fi = v.value<FileInfo>();
    totalSize-=fi.size;
    updateTotalSize();
    delete item;
    ui->uploadButton->setEnabled(ui->listFiles->count()>0);
}

void QPhotoUploader::updateTotalSize() {
    ui->humanSizeLabel->setText(getHumanSize(totalSize));
}

void QPhotoUploader::on_refreshAlbumsButton_clicked()
{
    if (refreshStarted) {
        refreshStarted = false;
        setRefreshIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));
        ui->refreshAlbumsButton->setEnabled(true);
    } else {
        refreshStarted = true;
        setRefreshIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
        ui->refreshAlbumsButton->setEnabled(false);
        ui->listAlbums->setEnabled(false);
        updateAlbums();
    }
}

void QPhotoUploader::updateAlbumList(QString,AlbumList,bool)
{
    selectAccount();
}


void QPhotoUploader::updateAlbums()
{
    disconnect(this, SLOT(updateAlbumList(QString,AlbumList,bool)));
    connect(this->sm, SIGNAL(updateAlbumList(QString,AlbumList,bool)),
            this, SLOT(updateAlbumList(QString,AlbumList,bool)));

    QVariant v = this->ui->listAccounts->itemData(this->ui->listAccounts->currentIndex());
    if (v == QVariant::Invalid)
        return;
    Account* acc = v.value<Account*>();
    if (acc) {
        accountId=acc->accountId();
        this->sm->getAlbums(acc->getProfile(false), true, true);
    }
}

#ifndef ALWAYS_UPDATE
void QPhotoUploader::resizeEvent(QResizeEvent * event)
{
    QDialog::resizeEvent(event);
    this->ui->listAccounts->setMinimumWidth(0);
    //if (this->ui->refreshAlbumsButton->isVisible())
    //    this->ui->listAccounts->setMinimumWidth(this->ui->listAccounts->width());
}
#endif

void QPhotoUploader::on_addNewAlbumButton_clicked()
{
    bool ok;
    QString name = QInputDialog::getText(this, tr("Creating album"),
                                         tr("Enter name of the album:"), QLineEdit::Normal,
                                         "", &ok);
    if (!ok)
        return;
/*
    QString description = QInputDialog::getText(this, tr("Creating album"),
                                         tr("Enter description of the album:"), QLineEdit::Normal,
                                         "", &ok);
    if (!ok)
        return;
*/
    this->ui->photoOpen->setEnabled(true);
    this->ui->listAlbums->setEnabled(true);
    this->ui->totalSizeLabel->show();
    this->ui->humanSizeLabel->show();
    ui->uploadButton->setEnabled(ui->listFiles->count()>0);
    QVariant v;
    v.setValue(QStringList() << "" << name);
    this->ui->listAlbums->addItem( Utils::decodeString(name)
                                   + " - "+ tr("%1 photo(s)", "", 0).arg(0),v);
    this->ui->listAlbums->setCurrentIndex(this->ui->listAlbums->count()-1);
}
