#include "servicemgr.h"

#include <cassert>

#include <QApplication>
#include <QTimer>

#include "qtransport.h"

#include "filter/servicefilter.h"
#include "filter/filtermanager.h"

#include "filter/filterfriendonline.h"
#include "filter/filterfriendgender.h"
#include "filter/filtermessagereaded.h"
#include "filter/filtermessagetype.h"

/*
 * Datatype functions:
 *
 * Friend: getFriends; gotFriends // done
 * Albums: 3 * getAlbums; gotAlbumList // done
 * Photos: 2 * getPhotos; gotPhotoList // done
 * Inbox Messages: getInboxMessages; gotInboxMessageList // done
 * Outbox Messages: getOutboxMessages; gotOutboxMessageList // done
 * Comments: getComments; gotComments // done
 */

ServiceMgr::ServiceMgr(QObject *parent) :
    QObject(parent), friendsUpdate(0), albumsUpdate(0), photosUpdate(0),
    feedsUpdate(0), messagesUpdate(0), filteredDriver(0)
{
    this->isSkipFriendListUpdate = false;

    settings = new QSettings(Utils::getHomeDir() + QString(FILE_SETTINGS).arg(QApplication::applicationName()), QSettings::NativeFormat, this);
    if (!settings->contains(SETTINGS_PROXY))
        settings->setValue(SETTINGS_PROXY, this->SystemProxy);
    if (!settings->contains(SETTINGS_MULTIPANEL))
    {
#ifdef Q_WS_MAEMO_5
        settings->setValue(SETTINGS_MULTIPANEL, false);
#else
        settings->setValue(SETTINGS_MULTIPANEL, true);
#endif
    }

#ifdef Q_WS_MAEMO_5
    if (!settings->contains(SETTINGS_AUTOROTATE))
    {
        settings->setValue(SETTINGS_AUTOROTATE, true);
    }
#endif

    loadAccounts();

    qDebug()<< "loaded " << this->accounts.length() << " accounts";

    updateDriverSettings();

    qRegisterMetaType<FriendList>("FriendList");
    qRegisterMetaType<AlbumList>("AlbumList");
    qRegisterMetaType<PhotoList>("PhotoList");
    qRegisterMetaType<MessageList>("MessageList");
    qRegisterMetaType<Friend>("Friend");
    qRegisterMetaType<QErrorMessage>("QErrorMessage");
    qRegisterMetaType<PhotoCommentList>("PhotoCommentList");
    qRegisterMetaType<QTransport::Action>("QTransport::Action");
    qRegisterMetaType<QEventFeedList>("QEventFeedList");
    qRegisterMetaType<QEventFeed::FeedType>("QEventFeed::FeedType");

    for(int i = 0; i < this->accounts.length(); i++) {
        this->connectToTransport(this->accounts.at(i));
        // load profiles for each account
        this->accounts[i]->getProfile(false);
    }

    connect(this, SIGNAL(updateAccounts(QString,AccountList)), this, SLOT(storeAccounts(QString,AccountList)));

    createFilter();
}

void ServiceMgr::connectToTransport(Account *account)
{
    connect(account->transport, SIGNAL(friendsReceived(QString, FriendList, bool)),
            SLOT(gotFriends(QString, FriendList, bool)));
    connect(account->transport, SIGNAL(messagesReceived(QString,MessageList, bool)),
            SLOT(gotMessageList(QString,MessageList,bool)));
    connect(account->transport, SIGNAL(albumsReceived(QString, QString, AlbumList, bool)),
            SLOT(gotAlbumList(QString, QString, AlbumList, bool)));
    connect(account->transport, SIGNAL(photosReceived(QString, QString, QString, PhotoList, bool)),
            SLOT(gotPhotoList(QString, QString, QString, PhotoList, bool)));
    connect(account->transport, SIGNAL(profileReceived(QString,QString,Friend)),
            SLOT(gotProfile(QString,QString,Friend)));
    connect(account->transport, SIGNAL(errorOccurred(QString,QErrorMessage,QTransport::Action)),
            SLOT(gotErrorMsg(QString,QErrorMessage,QTransport::Action)));
    connect(account->transport, SIGNAL(commentsReceived(QString,QString,QString,QString,PhotoCommentList)),
            SLOT(gotComments(QString,QString,QString,QString,PhotoCommentList)));
    connect(account->transport, SIGNAL(photoUploaded(QString,QString,QString,QString)),
            SIGNAL(photoUploaded(QString,QString,QString,QString)));
    connect(account->transport, SIGNAL(eventFeedReceived(QString,QEventFeedList,QEventFeed::FeedType,bool)),
            SLOT(gotEventFeed(QString,QEventFeedList,QEventFeed::FeedType,bool)));
}

ServiceMgr::ServiceMgr(const ServiceMgr &src) : QObject(src.parent())
{
    this->isSkipFriendListUpdate = src.isSkipFriendListUpdate;
    this->accounts = src.accounts;
    this->threads = src.threads;
    this->settings = src.settings;
    this->friendsUpdate = src.friendsUpdate;
    this->albumsUpdate = src.albumsUpdate;
    this->photosUpdate = src.photosUpdate;
    this->feedsUpdate = src.feedsUpdate;
}

ServiceMgr::~ServiceMgr()
{
    for (int i = 0; i < this->accounts.length(); i++)
        delete this->accounts.at(i);
}

void ServiceMgr::storeAccounts(QString accountId, AccountList)
{
    QDomDocument out("MyDoc");
    QDomElement profile = out.createElement(NODE_ACCOUNTS_ROOT);
    out.appendChild(profile);
    for (int i = 0; i < this->accounts.length(); i++) {
        Account *curAcc = this->accounts[i];
        if (curAcc->accountId().compare(accountId) == 0)
        {
            curAcc->saveAccount();
//            this->getFeed(curAcc, QEventFeed::photoFeed, true, true);
        }
        QDomElement anode = out.createElement(NODE_ACCOUNT);
        QDomText t = out.createTextNode(curAcc->accountId());
        anode.appendChild(t);
        profile.appendChild(anode);
    }

    QFile file(Utils::getHomeDir() + QString(FILE_ACCOUNTS_LIST));
    qDebug() << file.fileName();
    if (file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate) == false) {
        qDebug() << "can't open file: " + file.errorString();
    }
    file.write(out.toByteArray());
    file.close();

    //start update process
    //this->getFriends(false, true);
    //this->getAlbums(false, true);
}

void ServiceMgr::loadAccounts()
{
    QDomDocument doc;
    QFile file(Utils::getHomeDir() + QString(FILE_ACCOUNTS_LIST));
    if (file.open(QIODevice::ReadOnly) == false) {
        qDebug() << "can't open file: " + file.errorString();
        return;
    }

    doc.setContent(&file);
    file.close();

    QDomElement doc_elem = doc.documentElement();
    QDomNodeList lst = doc_elem.elementsByTagName("profile");

    qDebug() << "accounts:" << lst.count();
    for (int i = 0; i < lst.count(); i++) {
        QString accId = lst.at(i).firstChild().nodeValue();
        qDebug() << "Load account:" << accId;
        Account *prf = Account::loadAccount(accId);

        if (prf != NULL)
            this->accounts.append(prf);
    }
}

void ServiceMgr::cleanThreads()
{
    QList<QString> keys;
    QHashIterator<QString, QFuture<void> > i(this->threads);
    while (i.hasNext()) {
        i.next();
        if (i.value().isFinished())
            keys.append(i.key());
    }

    for(int i = 0; i < keys.length(); i++)
        this->threads.remove(keys[i]);
}

void ServiceMgr::updateDriverSettings()
{
    QString curProxyHost = QString("");
    int curProxyPort = 0;

    ProxyType proxy = (ProxyType)settings->value(SETTINGS_PROXY).toInt();

    switch (proxy) {
    case ServiceMgr::NoProxy:
        break;
    case ServiceMgr::UserProxy:
        curProxyHost = settings->value(SETTINGS_PROXYHOST).toString();
        curProxyPort = settings->value(SETTINGS_PROXYPORT).toInt();
        break;
    case ServiceMgr::SystemProxy: {
        QNetworkProxyQuery npq(QUrl(QLatin1String("http://maemo.org")));
        QList<QNetworkProxy> proxyList = QNetworkProxyFactory::systemProxyForQuery(npq);
        if (proxyList.length() > 0 && proxyList.at(0) != QNetworkProxy::NoProxy) {
    	    curProxyHost = proxyList.at(0).hostName();
    	    curProxyPort = proxyList.at(0).port();
        } else {
    	    QString var(getenv("http_proxy"));
            if (var.isNull())
                var = getenv("all_proxy");

    	    QRegExp regex("(http://)?(.*):(\\d*)/?");
    	    int pos = regex.indexIn(var);
    	    
    	    if(pos > -1) {
                curProxyHost = regex.cap(2);
                curProxyPort = regex.cap(3).toInt();
    	    }
        }
        if (curProxyHost.isEmpty())
            curProxyHost = QString::null;
        break;
    }
    default:
        qWarning() << "Unknown type of proxy";
    }

    if (curProxyHost.isNull())
        qDebug() << "using system proxy";
    else
        qDebug() << "using proxy: \""<< curProxyHost << "\":" << curProxyPort;

    for (int i = 0; i < this->accounts.length(); i++)
        this->accounts.at(i)->setProxy(curProxyHost, curProxyPort);

}

FriendList ServiceMgr::getFriends(const bool isNeedUpdate, const bool useSignal)
{
    // checks that all friend lists was downloaded
    if (!isNeedUpdate && this->cachedFriendList.length() > 0)
    {
        for (int i = 0; i < this->accounts.length(); i++)
        {
            if (this->accounts.at(i)->getFriendList().length() == 0)
            {
                this->cachedFriendList.clear();
                break;
            }
        }
    }

    if (isNeedUpdate || this->cachedFriendList.length() == 0)
    {
        this->cachedFriendList.clear();
        this->cleanThreads();

        for (int i = 0; i < this->accounts.length(); i++) {
            Account* acc = accounts.at(i);
            if (!isServiceFiltered(acc)) {
                FriendList curList = acc->getFriendList();
                qDebug() << "before merging" << this->cachedFriendList.length() << curList.length();
                this->cachedFriendList = Friend::mergeLists(this->cachedFriendList, curList);
                qDebug() << "after merging" << this->cachedFriendList.length();

                if (isNeedUpdate || curList.isEmpty()) {
                    QString threadId = QString("getFriends") + this->accounts[i]->transport->accountId;

                    // check that thread shouldn't already running
                    if (!this->threads.contains(threadId) && this->accounts.at(i)->isNetworkEnabled) {
                        // add thread to list
                        QFuture<void> status = QtConcurrent::run(this->accounts[i]->transport, &QTransport::getFriends, true);
                        this->threads.insert(threadId, status);
                        this->friendsUpdate++;
                    }
                }
            }
        }
        qSort(this->cachedFriendList.begin(), this->cachedFriendList.end());
    }

    filterManager_->filterList(cachedFriendList);

    if (useSignal)
        emit this->updateFriends(this->cachedFriendList, this->friendsUpdate > 0 ? false: true);

    return this->cachedFriendList;
}

void ServiceMgr::gotFriends(QString accountid, FriendList list, bool isLastUpdate)
{

    // skip update if it is not last
    if (!isLastUpdate && this->isSkipFriendListUpdate)
        return;

    this->cachedFriendList.clear();

    // create list of merged lists of friends
    for (int i = 0; i < this->accounts.length(); i++) {
        Account* acc = accounts.at(i);
        if (!isServiceFiltered(acc)) {
            if (this->accounts[i]->transport->accountId == accountid) {
                this->accounts[i]->setFriendList(list);
                this->cachedFriendList = Friend::mergeLists(this->cachedFriendList, list);
                qDebug() << "Add to friends " << list.length() << ". Total:" << this->cachedFriendList.length();
            } else {
                FriendList lst = this->accounts[i]->getFriendList();
                this->cachedFriendList = Friend::mergeLists(this->cachedFriendList, lst);
                qDebug() << "Add to friends " << lst.length() << ". Total:" << this->cachedFriendList.length();
            }
        }
    }

    if (isLastUpdate)
        this->friendsUpdate--;
    qDebug() << "islastUpdate=" << isLastUpdate << ", friendsUpdate=" << friendsUpdate;

    filterManager_->filterList(cachedFriendList);
    qSort(this->cachedFriendList.begin(), this->cachedFriendList.end());

    emit updateFriends(this->cachedFriendList, (this->friendsUpdate > 0 ? false: true));

    if (!isLastUpdate)
    {
        this->isSkipFriendListUpdate = true;
        QTimer::singleShot(1000, this, SLOT(disableSkipFriendListUpdate()));
    }
}

void ServiceMgr::disableSkipFriendListUpdate()
{
    this->isSkipFriendListUpdate = false;
}

AlbumList ServiceMgr::getAlbums(bool isNeedUpdate, bool useSignal)
{
    AlbumList ret;

    for (int i = 0; i < this->accounts.length(); i++) {
        ret.append(this->getAlbums(this->accounts.at(i)->getProfile(false), isNeedUpdate, false));
    }

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): this->albumsUpdate=" << this->albumsUpdate;

    filterManager_->filterList(ret);

    if (useSignal && this->accounts.length() > 0)
        emit this->updateAlbumList(this->accounts.at(0)->getProfile(false), ret, this->albumsUpdate > 0 ? false: true);

    return ret;
}

AlbumList ServiceMgr::getAlbums(const Friend& curFriend, bool isNeedUpdate, const bool useSignal)
{
    AlbumList ret;

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): Load album list from cache for " << curFriend.ownerId
            << " from " << curFriend.accountId << " needUpdate=" << isNeedUpdate << "; this->albumsUpdate="
            << this->albumsUpdate;

    ret = curFriend.getAlbumList();

    this->cleanThreads();

    if (isNeedUpdate || ret.isEmpty()) {
        for (int i = 0; i < this->accounts.length(); i++) {
            if ((curFriend.accountId == this->accounts[i]->transport->accountId) &&
                this->accounts.at(i)->isNetworkEnabled) {
                qDebug() << "Download album list for " << curFriend.ownerId;

                if (this->threads.contains(QString("albums") + curFriend.ownerId))
                    break;

                QFuture<void> status = QtConcurrent::run(this->accounts[i]->transport, &QTransport::getAlbums, curFriend.ownerId, true);
                this->threads.insert(QString("albums") + curFriend.ownerId, status);
                this->albumsUpdate++;
                break;
            }
        }
    }

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): this->albumsUpdate=" << this->albumsUpdate;

    filterManager_->filterList(ret);

    if (useSignal)
        emit this->updateAlbumList(curFriend, ret, this->albumsUpdate > 0 ? false: true);

    return ret;
}

AlbumList ServiceMgr::getAlbums(const Album& al, const bool isNeedUpdate, const bool useSignal)
{
    return this->getAlbums(this->getProfile(al.accountId, al.ownerId, false, false, false), isNeedUpdate, useSignal);
}

void ServiceMgr::gotAlbumList(QString accountId, QString owner, AlbumList list, bool isLastUpdate)
{

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): isLastUpdate=" << isLastUpdate;

    AlbumList ret;
    ret.append(list);

    // store album list to cache
    Friend::setAlbumList(list, accountId, owner);

    // check that it is not a my album list
    for (int i = 0; i < this->accounts.length(); i++) {
        if (this->accounts.at(i)->transport->accountId == accountId &&
            this->accounts.at(i)->getProfile(false).ownerId == owner)
        {
            // if it is my album list then store, generate new full list and send it
            this->accounts[i]->setAlbumList(list);
            for (int j = 0; j < this->accounts.length(); j++)
            {
                if (j != i)
                    ret.append(this->accounts[j]->getAlbumList());
            }

            break;
        }
    }

    if (isLastUpdate)
    {
        this->albumsUpdate--;
        if (this->albumsUpdate < 0)
            this->albumsUpdate = 0;
    }
    qDebug() << "islastUpdate=" << isLastUpdate << ", albumsUpdate=" << albumsUpdate << ", accountId=" << accountId << "owner=" << owner;

    filterManager_->filterList(ret);
    emit updateAlbumList(this->getProfile(accountId, owner, false, false, false), ret, this->albumsUpdate > 0 ? false: true);
}

PhotoList ServiceMgr::getPhotos(const Album& album, const bool isNeedUpdate, const bool useSignal, const bool loadIcons)
{
    bool nUpdate = isNeedUpdate;
    PhotoList ret;

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "Load photo list from cache for album " << album.albumId;

    ret = album.getPhotoList();
    this->cleanThreads();

    // if size of album not equal size of photo list then update oldest lists.
    if (album.size + ret.length() > 0 && album.size != ret.length())
    {
        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "() UNEQUAL!!! Album suze=" << album.size << " photos=" << ret.length();
        if (!ret.timeCreate().isValid() || QDateTime::currentDateTime().secsTo(ret.timeCreate()) > 60)
            nUpdate = true;

        AlbumList lst = this->getAlbums(album, false, false);
        if (!lst.timeCreate().isValid() || QDateTime::currentDateTime().secsTo(lst.timeCreate()) > 60)
            this->getAlbums(album, true, false);

        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "Update time: album=" << lst.timeCreate() << " photos=" << ret.timeCreate();
    }

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "nUpdate=" << nUpdate << " ret.length()=" << ret.length();
    if (nUpdate || ret.isEmpty()) {
        for (int i = 0; i < this->accounts.length(); i++) {
            if (album.accountId.compare(this->accounts[i]->transport->accountId) == 0 &&
                this->accounts.at(i)->isNetworkEnabled) {
                qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "Download photo list for " << album.albumId;
                if (!this->threads.contains(Photo::getKey(album)))
                {
                    QFuture<void> status = QtConcurrent::run(this->accounts[i]->transport, &QTransport::getPhotos, album.ownerId, album.albumId, loadIcons);
                    this->threads.insert(Photo::getKey(album), status);
                    this->photosUpdate++;
                    break;
                }
            }
        }
    }

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "() Loaded " << ret.length() << " photos for album " << album.albumId;

    filterManager_->filterList(ret);

    if (useSignal)
    {
        emit this->updatePhotoList(album.accountId, album.ownerId, album.albumId, ret, this->photosUpdate > 0? false: true);
    }

    return ret;
}

PhotoList ServiceMgr::getPhotos(const Photo &curPhoto, const bool isNeedUpdate, const bool useSignal, const bool loadIcons)
{
    PhotoList ret;

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "Load photo list from cache for " << curPhoto.albumId;

    ret = Photo::loadPhotoList(curPhoto.accountId, curPhoto.ownerId, curPhoto.albumId);
    this->cleanThreads();

    if (isNeedUpdate || ret.isEmpty()) {
        for (int i = 0; i < this->accounts.length(); i++) {
            if (curPhoto.accountId == this->accounts[i]->transport->accountId &&
                this->accounts[i]->isNetworkEnabled) {
                qDebug() << "Download photo list for " << curPhoto.albumId;
                if (this->threads.contains(Photo::getKey(curPhoto)) || !this->accounts.at(i)->isNetworkEnabled)
                    return ret;
                QFuture<void> status = QtConcurrent::run(this->accounts[i]->transport, &QTransport::getPhotos, curPhoto.ownerId, curPhoto.albumId, loadIcons);
                this->threads.insert(Photo::getKey(curPhoto), status);
                this->photosUpdate++;
            }
        }
    }

    filterManager_->filterList(ret);

    if (useSignal)
    {
        emit this->updatePhotoList(curPhoto.accountId, curPhoto.ownerId, curPhoto.albumId, ret, this->photosUpdate > 0? false: true);
    }

    return ret;
}

void ServiceMgr::gotPhotoList(QString accountId, QString friendId, QString albumId, PhotoList list, bool isLastUpdate)
{
    Album::setPhotoList(list, accountId, friendId, albumId);

    if (isLastUpdate)
        this->photosUpdate--;
    if (this->photosUpdate < 0)
        this->photosUpdate = 0;
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ <<"() islastUpdate=" << isLastUpdate << ", photosUpdate=" << photosUpdate;

    filterManager_->filterList(list);

    emit updatePhotoList(accountId, friendId, albumId, list, photosUpdate > 0 ? false : true);
}

MessageList ServiceMgr::getMessages(bool isNeedUpdate, bool useSignal)
{
    MessageList ret;

    qDebug() << __FILE__ << ":" << __FUNCTION__;

    this->cleanThreads();

    for (int i = 0; i < this->accounts.length(); i++) {
        if (isServiceFiltered(accounts[i])) continue;

        MessageList curList = this->accounts.at(i)->getMessageList();
        ret.append(curList);

        // drafts
        MessageList draftList = this->accounts.at(i)->getDraftMessages();
        ret.append(draftList);

        if (isNeedUpdate || curList.isEmpty()) {
            QString threadId = QString("getMessages") + this->accounts[i]->transport->accountId;

            // check that thread shouldn't already running
            if (!this->threads.contains(threadId) && this->accounts.at(i)->isNetworkEnabled) {
                // add thread to list
                QFuture<void> status = QtConcurrent::run(this->accounts[i]->transport, &QTransport::getMessages);
                this->threads.insert(threadId, status);
                this->messagesUpdate++;
            }
        }
    }

    filterManager_->filterList(ret);

    if (useSignal)
        emit this->updateMessageList(ret, this->messagesUpdate > 0 ? false: true);

    return ret;
}

void ServiceMgr::gotMessageList(QString accountId, MessageList list, bool isLastUpdate)
{
    // create list of merged lists of messages
    MessageList ret;

    qDebug() << __FILE__ << __FUNCTION__ << ": " << list.length();

    for (int i = 0; i < this->accounts.length(); i++) {
        if (isServiceFiltered(accounts[i])) continue;
        if (this->accounts[i]->transport->accountId == accountId) {
            this->accounts[i]->setMessageList(list);
            ret.append(list);
            qDebug() << "Add to messages " << list.length() << ". Total:" << ret.length();
        } else {
            MessageList lst = this->accounts[i]->getMessageList();
            ret.append(lst);
            qDebug() << "Add to messages " << lst.length() << ". Total:" << ret.length();
        }
    }

    if (isLastUpdate)
        this->messagesUpdate--;
    qDebug() << "islastUpdate=" << isLastUpdate << ", messagesUpdate=" << messagesUpdate;

    filterManager_->filterList(ret);

    emit updateMessageList(ret, (this->messagesUpdate > 0 ? false: true));
}

bool ServiceMgr::deleteMessage(const Message& msg)
{
    for (int i = 0; i < this->accounts.length(); i++) {
        if (msg.accountId == this->accounts[i]->transport->accountId) {
            qDebug() << "Deleting message: " << msg.messageId;

            return this->accounts[i]->transport->deleteMessage(msg);
        }
    }
    return false;
}

bool ServiceMgr::canSendMessage(const QString &accountId)
{
    for (int i = 0; i < this->accounts.length(); i++) {
        if (accountId == this->accounts.at(i)->transport->accountId)
            return this->accounts.at(i)->transport->checkFunction(CLASS_MESSAGES, SEND_MESSAGE);
    }
    return false;
}

bool ServiceMgr::sendMessage(const QString &accountId, const QString &ownerId, const QString &title, const QString &message)
{
    qDebug() << __FILE__ << __FUNCTION__ << "() to: " << ownerId;

    for (int i = 0; i < this->accounts.length(); i++) {
        if (accountId == this->accounts.at(i)->transport->accountId)
            return this->accounts.at(i)->transport->sendMessage(ownerId, title, message);
    }
    return false;
}

bool ServiceMgr::readMessage(const QString &accountId, const QString &messageId)
{
    qDebug() << __FILE__ << __FUNCTION__ << "() id: " << messageId;

    for (int i = 0; i < this->accounts.length(); i++) {
        if (accountId == this->accounts.at(i)->transport->accountId)
            return this->accounts.at(i)->transport->readMessage(messageId);
    }
    return false;
}

PhotoCommentList ServiceMgr::getComments(const Photo& curPhoto, const bool isNeedUpdate, const bool useSignal)
{
    PhotoCommentList ret;

    qDebug() <<__FILE__ << ":" << __LINE__ << ":" << __FUNCTION__<< "(): load comments for " << curPhoto.photoId;

    ret = curPhoto.getPhotoComments();
    this->cleanThreads();

    if (isNeedUpdate || ret.isEmpty()) {
        for (int i = 0; i < this->accounts.length(); i++) {
            if (curPhoto.accountId == this->accounts[i]->transport->accountId) {
                qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__<< "(): Download comments for " << curPhoto.photoId;
                if (this->threads.contains(Utils::getPhotoCommentListKey(curPhoto)) || !this->accounts.at(i)->isNetworkEnabled)
                    return ret;
                QFuture<void> status = QtConcurrent::run(this->accounts[i]->transport, &QTransport::getPhotoComments, curPhoto);
                this->threads.insert(Utils::getPhotoCommentListKey(curPhoto), status);
            }
        }
    }

    filterManager_->filterList(ret);

    if (useSignal)
        emit this->updatePhotoCommentList(curPhoto.photoId, ret);

    return ret;
}

void ServiceMgr::gotComments(QString accountId, QString ownerId, QString albumId, QString photoId, PhotoCommentList list)
{
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): photoId=" << photoId << " has comments=" << list.length();
    Photo::setPhotoComments(list, accountId, ownerId, albumId, photoId);
    filterManager_->filterList(list);
    emit this->updatePhotoCommentList(photoId, list);
}

bool ServiceMgr::sendComment(const Photo& curPhoto, const QString& comment)
{
    for (int i = 0; i < this->accounts.length(); i++) {
        if (curPhoto.accountId == this->accounts.at(i)->transport->accountId)
            return this->accounts.at(i)->transport->sendPhotoComment(curPhoto, comment);
    }
    return false;
}

void ServiceMgr::gotProfile(QString accountId, QString reqOwnerId, Friend profile)
{
    Friend ret = profile;
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): reqOwnerId=" << reqOwnerId;
    if (reqOwnerId.isEmpty()) {
        for (int i = 0; i < this->accounts.length(); i++) {
            if (this->accounts.at(i)->accountId().compare(accountId) == 0) {
                bool isNE = this->accounts.at(i)->isNetworkEnabled;
                this->accounts.at(i)->setProfile(profile);
                if (!isNE)
                    emit updateAccounts(this->accounts.at(i)->accountId(), this->accounts);
                ret = this->getMyProfile(false);
                break;
            }
        }
    }

    profile.storeData();
    emit this->updateProfile(profile);
}

AccountList ServiceMgr::getAccounts()
{
    // check that all accounts has profiles
    for (int i = 0; i < this->accounts.length(); i++)
        this->accounts.at(i)->getProfile(false);

    return this->accounts;
}

void ServiceMgr::addAccount(QString accountName, DriverInfo *driver)
{
    Account *newAcc = new Account(driver->library, accountName);

    if (newAcc != NULL && newAcc->ready()) {
        this->connectToTransport(newAcc);
        this->accounts.append(newAcc);
        this->updateDriverSettings();

        emit this->updateAccounts(newAcc->accountId(), this->accounts);
    }
}

Account *ServiceMgr::account(QString accountId)
{
    for (int i = 0; i < this->accounts.length(); i++)
        if (this->accounts.at(i)->accountId().compare(accountId) == 0)
            return this->accounts.at(i);

    return NULL;
}

void ServiceMgr::deleteAccount(Account *oldAcc)
{
    int i = -1;
    for (int j = 0; j < this->accounts.length(); j++) {
        if (oldAcc->transport->accountId == this->accounts.at(j)->transport->accountId)
            i = j;
    }

    if (i >= 0) {
        this->accounts.removeAt(i);
        QDir dir(Utils::getAccountDir(oldAcc->transport->accountId));

        delete oldAcc;
        qDebug() << "Try to remove dir " << dir.path();
        Utils::RemoveDirectory(dir);

        //execute accounts update process
        emit this->updateAccounts(QString::null, this->accounts);
    }
}

QList<DriverInfo*> ServiceMgr::getDrivers()
{
    if (!cachedDriverList.isEmpty())
        return this->cachedDriverList;

    QDir dir;
    dir.setFilter(QDir::Dirs|QDir::NoDotAndDotDot);
    dir.setPath(Utils::getDriversDir());
    QFileInfoList lst = dir.entryInfoList();

    qDebug() << "Found " << lst.length() << " entries at " << dir.path();

    for (int i = 0; i < lst.length(); i++) {
        QFileInfo dir = lst.at(i);
        if (dir.isDir() && dir.fileName() != "." && dir.fileName() != "..") {
            QDir files;
            files.setFilter(QDir::Files);
            files.setNameFilters(QStringList(QString("*.so")));
            files.setPath(dir.filePath());
            QFileInfoList lstFiles = files.entryInfoList();
            for (int j = 0; j < lstFiles.length(); j++) {
                QFileInfo file = lstFiles.at(j);
                if (file.isFile()) {
                    DriverInfo *curDr = QTransport::getDriverInfo(file.filePath());
                    if (curDr != NULL) {
                        qDebug() << "found driver " << curDr->name;
                        this->cachedDriverList.append(curDr);
                    }
                }
            }
        }
    }

    return this->cachedDriverList;
}

PhotoList ServiceMgr::getPhotosForAlbum(Photo curPhoto)
{
    for (int i = 0; i < this->accounts.length(); i++) {
        if (this->accounts.at(i)->transport->accountId == curPhoto.accountId) {
            return this->getPhotosForAlbum(this->accounts.at(i), curPhoto);
        }
    }

    PhotoList ret;
    return ret;
}

PhotoList ServiceMgr::getPhotosForAlbum(Account *curAcc, Photo curPhoto)
{
    AlbumList albums;

    // check that it can be my album
    if (curAcc->getProfile(false).ownerId == curPhoto.ownerId) {
        albums = curAcc->getAlbumList();
    } else {
        // or it is one of friend album
        FriendList friends;
        friends = curAcc->getFriendList();

        for (int i = 0; i < friends.length(); i++) {
            if (friends.at(i).ownerId == curPhoto.ownerId) {
                albums = friends[i].getAlbumList();
                break;
            }
        }
    }

    // get content of album
    for (int i = 0; i < albums.length(); i++) {
        if (albums.at(i).albumId == curPhoto.albumId) {
            return albums[i].getPhotoList();
        }
    }

    // try to load album if it not found
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): Can't found cached photos for album " << curPhoto.albumId;
    return this->getPhotos(curPhoto, false, false, false);
}

void ServiceMgr::downloadPhotos(Photo startPhoto, int nearest) {
    qDebug() << "Start to download photo images from " << startPhoto.photoId << " for " << startPhoto.albumId;

    this->cleanThreads();

    Account *acc = account(startPhoto.accountId);
    if (!acc || !acc->isNetworkEnabled)
        return;

    PhotoList photos = this->getPhotosForAlbum(acc, startPhoto);
    if (photos.isEmpty())
        qDebug() <<__FILE__ <<":" << __LINE__ << ":" << __FUNCTION__ << "() got empty photo list";

    if (!this->threads.contains(Photo::getKey(startPhoto)))
    {
        QFuture<void> ret = QtConcurrent::run(acc->transport, &QTransport::downloadPhotoList, photos, startPhoto, nearest);
        this->threads.insert(Photo::getKey(startPhoto), ret);
        this->photosUpdate++;
    }
}

QString ServiceMgr::downloadPhoto(Photo curPhoto)
{
    Account *acc = account(curPhoto.accountId);
    if (!acc || !acc->isNetworkEnabled)
        return QString::null;

    QString photo_path = acc->transport->generateFileName(Utils::getPhotoDir(acc->serviceName()), curPhoto.photo_url);
    QFile file(photo_path);
    if (!file.exists()) {
        if (!acc->transport->downloadPhoto(curPhoto.photo_url, file.fileName()))
            return QString::null;
    }
    return photo_path;
}

void ServiceMgr::uploadPhoto(QString accountId, QString albumId,
                             QString file, QString description="")
{
    qDebug() << "Start to upload photo image: " << file;

    this->cleanThreads();

    const Account *acc = account(accountId);
    if (!acc || !acc->isNetworkEnabled)
        return;

    QtConcurrent::run(acc->transport, &QTransport::uploadPhoto, accountId, albumId, file, description);
}

void ServiceMgr::uploadPhoto(const Album album, QString file, QString description) {
    uploadPhoto(album.accountId, album.albumId, file, description);
}

void ServiceMgr::gotErrorMsg(QString accountId, QErrorMessage msg, QTransport::Action acc)
{
    QString prefix(tr("During request "));
    QString comment;

    bool isMajor = true;

    // checks majority of error
    if (msg.code.compare(DRV_AUTH_CANCEL_CODE) == 0||
        msg.code.compare(DRV_UNKNOWN_METHOD_CALL) == 0 )
        isMajor = false;

    switch (acc) {
    case QTransport::getListFriendsAction:
        this->friendsUpdate--;
        if (this->friendsUpdate < 0)
            this->friendsUpdate = 0;
        prefix += tr("\"Update list of friends\"");
        break;
    case QTransport::deleteFriendAction:
        prefix += tr("\"Delete friend from list\"");
        break;
    case QTransport::getListAlbumsAction:
        this->albumsUpdate--;
        if (this->albumsUpdate < 0)
            this->albumsUpdate = 0;
        prefix += tr("\"Update list of albums\"");
        break;
    case QTransport::getListPhotosAction:
        this->photosUpdate--;
        if (this->photosUpdate < 0)
            this->photosUpdate = 0;
        prefix += tr("\"Update list of images\"");
        break;
    case QTransport::getProfileAction:
        prefix += tr("\"Update user profile\"");
        comment = tr("Account swiched to offline mode. Please check network settings and try again.");
        break;
    case QTransport::getSettingsAction:
        prefix += tr("\"Get settings\"");
        break;
    case QTransport::setSettingsAction:
        prefix += tr("\"Set settings\"");
        break;
    case QTransport::getPhotoAction:
        this->photosUpdate--;
        if (this->photosUpdate < 0)
            this->photosUpdate = 0;
        prefix += tr("\"Download image\"");
        break;
    case QTransport::uploadPhotoAction:
        prefix += tr("\"Upload image\"");
    case QTransport::getCommentsAction:
        prefix += tr("\"Get comments\"");
        break;
    case QTransport::sendCommentAction:
        prefix += tr("\"Send comment\"");
        break;
    case QTransport::getMessagesAction:
        prefix += tr("\"Get list of messages\"");
        break;
    case QTransport::sendMessageAction:
        prefix += tr("\"Send message\"");
        break;
    case QTransport::readMessageAction:
        prefix += tr("\"Mark message as read\"");
        break;
    case QTransport::deleteMessageAction:
        prefix += tr("\"Delete message\"");
        break;
    case QTransport::getListMethodsAction:
        prefix += tr("\"Get list of registered methods\"");
        break;
    case QTransport::getFeedAction:
        prefix += tr("\"Get feed\"");
        this->feedsUpdate--;
        if (this->feedsUpdate < 0)
            this->feedsUpdate = 0;
        break;
    }

    prefix += tr(" from account \"%1\" the error %2 was caused: ").arg(accountId).arg(msg.code) + msg.text + ".";
    if (!msg.comment.isEmpty())
        prefix += " " + msg.comment + ".";
    if (!comment.isEmpty())
        prefix += " " + comment;

    emit this->errorOccured(prefix, acc, isMajor);
}


void ServiceMgr::clearCache()
{
    QString path;
    bool ret;
    for (int i = 0; i < this->accounts.length(); i++) {
        DriverInfo *info = this->accounts.at(i)->transport->getDriverInfo();
        // remove friend icons
        path = Utils::getFriendsIconDir(info->name);
        QDir dir(path);
        ret = Utils::RemoveFiles(dir, "*");
        qDebug() << "Remove " << path << " returned " << ret;

        // remove album icons
        path = Utils::getAlbumsIconDir(info->name);
        dir = QDir(path);
        ret = Utils::RemoveFiles(dir, "*");
        qDebug() << "Remove " << path << " returned " << ret;

        // remove friend list
        path = Utils::getAccountDir(this->accounts.at(i)->transport->accountId);
        dir = QDir(path);
        ret = Utils::RemoveFiles(dir, FILE_FRIENDS_DATA);
        qDebug() << "Remove " << path << " + " << FILE_FRIENDS_DATA << " returned " << ret;
        ret = Utils::RemoveFiles(dir, QString(FILE_ALBUMS_DATA).arg("*"));
        qDebug() << "Remove " << path << " + " << QString(FILE_ALBUMS_DATA).arg("*") << " returned " << ret;
        ret = Utils::RemoveFiles(dir, QString(FILE_PHOTOS_DATA).arg("*", "*"));
        qDebug() << "Remove " << path << " + " << QString(FILE_PHOTOS_DATA).arg("*", "*") << " returned " << ret;

        path = Utils::getPhotoDir(info->name);
        dir = QDir(path);
        ret = Utils::RemoveFiles(dir, "*");
        qDebug() << "Remove " << path << " returned " << ret;
    }
}

Friend ServiceMgr::getProfile(const QString& accountId, const QString& ownerId, const bool isNeedUpdate, const bool useSignal, const bool isFullProfile)
{
    qDebug() << __FILE__ <<":" << __LINE__ <<":" << __FUNCTION__ << ": accountId=" << accountId << "; isNeedUpdate=" << isNeedUpdate << "; isFullProfile=" << isFullProfile;
    Account *acc = this->account(accountId);
    assert(acc);
    QString service = this->account(accountId)->serviceName();
    bool isFound = false;
    bool hasFullProfile = false;
    Friend ret;

    // step 1. check downloaded full profiles from all accounts
    for (int i = 0; i < this->accounts.length(); i++) {
        if (this->accounts.at(i)->serviceName().compare(service) != 0)
            continue;

        Friend nonFr = Friend::loadData(service, this->accounts.at(i)->accountId(), ownerId);
        if (nonFr.ownerId.compare(ownerId) == 0)
        {
            ret = nonFr;
            isFound = true;
            hasFullProfile = true;
            break;
        }
    }
    qDebug() << __FILE__ <<":" <<__LINE__ << ":" << __FUNCTION__ << "(): step1 isFound" << isFound;

    // step 2. check friends from all accounts
    if (!isFound)
    {
        for (int i = 0; i < this->accounts.length(); i++)
        {
            if (this->accounts.at(i)->serviceName().compare(service) != 0)
                continue;

            // check owner id
            if (this->accounts.at(i)->getProfile(false).ownerId == ownerId)
            {
                ret = this->accounts.at(i)->getProfile(false);
                isFound = true;
                break;
            }

            // check friends of owner
            FriendList frl = this->accounts.at(i)->getFriendList();
            for (int j = 0; j < frl.length(); j++)
            {
                if (frl.at(j).ownerId.compare(ownerId) == 0)
                {
                    ret = frl.at(j);
                    isFound = true;
                    break;
                }
            }

            if (isFound)
                break;
        }
    }

    qDebug() << __FILE__ <<":" <<__LINE__ << ":" << __FUNCTION__ << "(): step2 isFound" << isFound;

    // step 3. download full profile if it isn't found or need update
    if (!isFound || isNeedUpdate || (isFullProfile && !hasFullProfile))
    {
        this->cleanThreads();

        if (!this->threads.contains(Utils::getFriendKey(accountId, ownerId)))
        {
            QFuture<void> ret = QtConcurrent::run(this->account(accountId)->transport,
                                                  &QTransport::getProfile, ownerId);
            this->threads.insert(Utils::getFriendKey(accountId, ownerId), ret);
        }
    }

    // return cached value
    if (useSignal && isFound)
        emit this->updateProfile(ret);

    return ret;
}

Friend ServiceMgr::getProfile(const PhotoComment& comment, const bool isNeedUpdate, const bool useSignal)
{
    return this->getProfile(comment.accountId, comment.senderId, isNeedUpdate, useSignal, false);
}

Friend ServiceMgr::getMyProfile(bool isNeedUpdate)
{
    if (this->accounts.length() > 0)
    {
        Friend fd = this->accounts.at(0)->getProfile(isNeedUpdate);
        for (int i = 1; i < this->accounts.length(); i++)
            fd.addProfile(this->accounts.at(i)->getProfile(isNeedUpdate), false);
        return fd;
    }
    return Friend();
}

QEventFeedList ServiceMgr::getFeed(QEventFeed::FeedType type, bool isNeedUpdate, bool useSignal)
{
    return this->getFeed(NULL, type, isNeedUpdate, useSignal);
}

QEventFeedList ServiceMgr::getFeed(Account *acc, QEventFeed::FeedType type, const bool isNeedUpdate, const bool useSignal)
{
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): feedsUpdate=" << this->feedsUpdate;
    QEventFeedList ret;
    this->cleanThreads();

    for (int i = 0; i < this->accounts.length(); i++) {
        QEventFeedList curList = QEventFeed::loadFeed(this->accounts.at(i)->accountId(), type);
        qDebug() << "before merging" << ret.length() << curList.length();
        ret.append(curList);

        if ((acc == NULL || this->accounts.at(i)->accountId().compare(acc->accountId()) == 0) &&
                (isNeedUpdate || curList.isEmpty())) {
            QString threadId = QEventFeed::getKey(this->accounts[i]->transport->accountId, type);

            // check that thread shouldn't already running
            if (!this->threads.contains(threadId) && this->accounts.at(i)->isNetworkEnabled) {
                // add thread to list
                qDebug() << "Try to download feed for " << this->accounts.at(i)->accountId();
                QFuture<void> status = QtConcurrent::run(this->accounts[i]->transport, &QTransport::getFeed, type);
                this->threads.insert(threadId, status);
                this->feedsUpdate++;
            }
        }
    }

    if (useSignal)
        emit this->updateFeed(ret, type, this->feedsUpdate > 0 ? false : true);

    return ret;
}

void ServiceMgr::gotEventFeed(QString accountId, QEventFeedList list, QEventFeed::FeedType type, bool isLastUpdate)
{
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): accountId=" << accountId << " has news=" << list.length();
    QEventFeed::storeFeed(list, type, accountId);

    QEventFeedList ret;

    for (int i = 0; i < this->accounts.length(); i++)
    {
        if (this->accounts.at(i)->accountId().compare(accountId) == 0)
            ret.append(list);
        else
        {
            ret.append(QEventFeed::loadFeed(this->accounts.at(i)->accountId(), type));
        }
    }

    if (isLastUpdate)
        this->feedsUpdate--;
    if (this->feedsUpdate < 0)
        this->feedsUpdate = 0;

    emit this->updateFeed(ret, type, this->feedsUpdate > 0 ? false : true);
}

void ServiceMgr::changeFilteredService(DriverInfo* driverInfo)
{
    filteredDriver = driverInfo;
    this->cachedFriendList.clear();

    updateData();
}

void ServiceMgr::updateData(int type)
{
    bool all = (type == -1);

    if (all || type == qMetaTypeId<Friend>()) {
        getFriends(false, true);
    }

    if (all || type == qMetaTypeId<Message>()) {
        getMessages(false, true);
    }
}

bool ServiceMgr::isServiceFiltered(const Account* acc) const
{
    return filteredDriver && acc->serviceName() != filteredDriver->serviceName;
}

void ServiceMgr::createFilter()
{
    serviceFilter_ = new ServiceFilter(this, this);
    connect(serviceFilter_, SIGNAL(serviceSelected(DriverInfo*)), SLOT(changeFilteredService(DriverInfo*)));

    filterManager_ = new FilterManager(this);
    filterManager_->setObjectName("Main filter manager");

    connect(filterManager_, SIGNAL(stateChanged(int)), SLOT(updateData(int)));

    filterManager_->registerFilterableType<Friend>();
    filterManager_->registerFilterableType<Album>();
    filterManager_->registerFilterableType<Message>();
    filterManager_->registerFilterableType<Photo>();

    filterManager_->addFilter(new FilterFriendOnline(filterManager_));
    filterManager_->addFilter(new FilterFriendGender(filterManager_));
    filterManager_->addFilter(new FilterMessageReaded(filterManager_));
    filterManager_->addFilter(new FilterMessageType(this, filterManager_));
}

FilterManager* ServiceMgr::filterManager() const
{
    return filterManager_;
}

ServiceFilter* ServiceMgr::serviceFilter() const
{
    return serviceFilter_;
}
