#ifndef QTRANSPORT_H
#define QTRANSPORT_H

/**
  \file qtransport.h
  \brief Interface to MySocials libraries.

  This file describes Application Programming Interface
  which are used in MySocials project.

  \author MySocials team

  \date 2011-03-25
  */

#include <QObject>
#include <QLibrary>
#include <QDebug>
#include <QDomDocument>
#include <QDomElement>
#include <QDateTime>

#if QT_VERSION >= 0x040700
# include <QElapsedTimer>
#else
# include <QTime>
#endif

#include "mysocials_driver.h"

#include "datatypes/driverinfo.h"
#include "datatypes/photo.h"
#include "datatypes/album.h"
#include "datatypes/friend.h"
#include "datatypes/message.h"
#include "datatypes/qeventfeed.h"
#include "datatypes/qerrormessage.h"
#include "datatypes/qattachment.h"

/**
  \brief The item of method list.

  This struct describes properties of method in MySocials API.
  */
struct MethodElement
{
    QString className; /**< \brief The name of class methods. */
    QString functionName; /**< \brief The name of class function. */
};

/**
  \brief MySocials driver programming interface.

  This class describes MySocials driver programming interface. It based on MySocials API.
  */
class QTransport : public QObject
{
    Q_OBJECT
public:
    /**
      \brief List of supported actions.

      Each action corresponds function in MySocials API.
      */
    enum Action
    {
        getListMethodsAction, /**< \brief Class: settings. Method: getListMethods. */
        getProfileAction, /**< \brief Class: profile. Method: getProfile. */
        getListFriendsAction, /**< \brief Class: friends. Method: getListFriends. */
        deleteFriendAction, /**< \brief Class: friends. Method: deleteFriend. */
        getSettingsAction, /**< \brief Class: settings. Method: getSettings. */
        setSettingsAction, /**< \brief Class: settings. Method: setSettings. */
        getMessagesAction, /**< \brief Class: messages. Method: getListOutboxMessages. */
        sendMessageAction, /**< \brief Class: messages. Method: sendMessage. */
        readMessageAction, /**< \brief Class: messages. Method: readMessage. */
        deleteMessageAction, /**< \brief Class: messages. Method: deleteMessage. */
        getPhotoAction, /**< \brief Class: photos. Method: getPhoto. */
        uploadPhotoAction, /**< \brief Class: photos. Method: uploadPhoto. */
        getListPhotosAction, /**< \brief Class: photos. Method: getListPhotos. */
        getCommentsAction, /**< \brief Class: photos. Method: getListPhotoComments. */
        sendCommentAction, /**< \brief Class: photos. Method: sendPhotoComment. */
        getListAlbumsAction, /**< \brief Class: photos. Method: getListAlbums. */
        getFeedAction /**< \brief Class: news. Method: getListNews. */
    };

    bool checkFunction(const QString &usedClass, const QString &usedFunction) const;

private:
    // Pointer to social network driver library
    QLibrary *driver;

    // profile settings in library
    struct msa_module *driverModule;

    // count of active requests
    int activeRequests;

    // true if driver must shuutdown
    bool needShutdown;

    // true if library init well
    bool isLibraryInit;

    #if QT_VERSION >= 0x040700
    QElapsedTimer timer;
    #else
    QTime timer;
    #endif


    // list of registered functions
    QList<MethodElement> registerdFunctions;

    QDomDocument createRequest(const QString& usedClass, const QString& usedFunction, const bool noAuthorize = false) const;
    QDomDocument createRequest(const QString& usedClass, const QString& usedFunction, const bool noAuthorize,
                               const QString& content) const;
    QDomDocument sendRequest(const QDomDocument& doc, Action acc);

    bool checkFunction(const QString& usedClass, const QString& usedFunction, Action acc) const;
    bool checkBadResponse(const QDomDocument& resp, Action acc);
    bool checkGoodResponse(const QDomDocument& resp, Action acc);

    void getSettings();

    void getMethods();

    QAttachment loadAttachment(const QDomElement& node);

    /*!
        \brief load inbox messages if service support inbox/outbox model.
        Use \a getMessages instead of this function for universal access.
      */
    MessageList getInbox();

    /*!
      \brief load sent messages if service support inbox/outbox model.
      Use \a getMessages instead of this function for universal access.
      */
    MessageList getOutbox();

    /*!
      \brief load messages from threads if service support thread model.
      Use \a getMessages instead of this function for universal access.
      */
    MessageList getThreadList();

public:
    QTransport(QString libName, QString accName, QObject *parent = 0);
    ~QTransport();

    /**
    \brief Account ID. uses to identify account of service (e.g. vkontakte0)
    */
    QString accountId;

    /**
    \brief the name of library file (e.g. libvkontakte.so)
    */
    QString libraryName;

    QString serviceName() const;

    static QString generateFileName(const QString& path, const QString& url);

    DriverInfo *getDriverInfo();
    static DriverInfo *getDriverInfo(QString libraryPath);

    void init(QString proxyHost, uint proxyPort);
    void setSettings(QString settings);
    void close();
    bool ready();

    /**
      \brief Gets own profile.

      The profile returns by signal \a profileReceived().
      */
    void getProfile();

    /**
      \brief Gets profile by owner Id.

      The profile returns by signal \a profileReceived().

      \param ownerId The ID of owner.
      */
    void getProfile(const QString& ownerId);

    void getFriends(bool alsoLoadIcons);
    bool deleteFriend(QString ownerId);
    bool deleteFriend(Friend owner);

    void getAlbums(QString ownedId, bool alsoLoadIcons);
    void getPhotos(QString ownerId, QString albumId, bool alsoLoadIcons);
    void getPhotoComments(Photo photo);
    bool sendPhotoComment(Photo photo, QString comment);
    bool uploadPhoto(QString accountId, QString albumId, QString file_name, QString description);
    bool uploadPhoto(Album al, QString local_file_name, QString description);
    bool downloadPhoto(QString url, QString local_file_name);
    void downloadPhotoList(PhotoList list, Photo curPhoto, int nearest);

    /*!
      \brief loalist of messages from service.

      Uses \a getInbox(), \a getOutbox() and \a getThreads().
      */
    void getMessages();
    bool sendMessage(QString ownerId, QString title, QString message);
    bool sendMessage(Friend owner, QString title, QString message);
    bool readMessage(QString messageId);
    bool readMessage(Message msg);
    bool deleteMessage(QString messageId);
    bool deleteMessage(Message msg);

    /**
      \brief Gets feed from service.

      Feed returns by \a eventFeedReceived signal.
      */
    void getFeed(QEventFeed::FeedType type);

signals:
    void albumsReceived(QString accountid, QString friendId, AlbumList, bool isLastUpdate);
    void photosReceived(QString accountid, QString friendId, QString albumId, PhotoList, bool isLastUpdate);
    void photoUploaded(QString accountId, QString albumId, QString local_file_name, QString photoId);
    void commentsReceived(QString accountId, QString friendId, QString albumId, QString photoId, PhotoCommentList list);

    void friendsReceived(QString accountid, FriendList, bool isLastUpdate);
    void friendDeleted(QString accountId, QString friendId);

    void settingsReceived(QString accountid, QString) const;
    void profileReceived(QString accountId, QString reqOwnerId, Friend data);

    void messagesReceived(QString accountId, MessageList list, bool isLastUpdate);
    void messageSended(QString accountId, QString ownerId, QString title, QString text);
    void messageReaded(QString accountId, QString messageId);
    void messageDeleted(QString accountId, QString messageId);

    void errorOccurred(QString accountId, QErrorMessage msg, QTransport::Action acc) const;

    void eventFeedReceived(QString accountId, QEventFeedList list, QEventFeed::FeedType type, bool isLastUpdate);
};

// the next part of code defines standard messages that returns from driver.
// It's need for correct translation.
// Prease upgrade it if possible.

#if 0
class QTransport
{
#define DRV_SETT_ERROR QT_TR_NOOP("Unable to save settings")
#define DRV_AUTH_ERROR QT_TR_NOOP("Authorization problem")
#define DRV_AUTH_CANCEL QT_TR_NOOP("Authorization canceled")
#define DRV_REQ_ERROR QT_TR_NOOP("Unable to perform request")
#define DRV_SERVER_BUSY QT_TR_NOOP("Server busy")
#define DRV_REQ_PARAM QT_TR_NOOP("One of the parameters missing or invalid")
#define DRV_CAPT_ERROR QT_TR_NOOP("Captcha")
#define DRV_RESP_ERROR QT_TR_NOOP("Bad response from server")
#define DRV_NETW_ERROR QT_TR_NOOP("Network problem")
#define DRV_MESS_LEN QT_TR_NOOP("Empty message")
#define DRV_ACCESS_ERROR QT_TR_NOOP("Operation prohibited by privacy")
#define DRV_SERVER_ERROR QT_TR_NOOP("Internal Server Error")
#define DRV_FILE_SAVE_ERROR QT_TR_NOOP("Can't save file")
QT_TR_NOOP("male")
QT_TR_NOOP("female")

// from libvkontakte - errors.xml
QT_TR_NOOP("Incorrect signature")
};
#endif

#endif // QTRANSPORT_H
