#include "profilebrowserpanel.h"
#include "ui_profilebrowserpanel.h"

#include <QTimer>
#include <QToolBar>
#include <QActionGroup>

#include "filter/servicefilter.h"
#include "filter/filtermanager.h"

ProfileBrowserPanel::ProfileBrowserPanel(QFont usedFont, QWidget *parent) :
        QWidget(parent),
        ui(new Ui::ProfileBrowserPanel),
        isPortraitMode(false),
        state(WindowFriendList)
{
    this->usedFont = usedFont;
    this->sm = new ServiceMgr(this);
    ui->setupUi(this);

    init();
}

ProfileBrowserPanel::ProfileBrowserPanel(QFont usedFont, ServiceMgr *mgr, QWidget *parent) :
    QWidget(parent),
    ui(new Ui::ProfileBrowserPanel),
    isPortraitMode(false),
    state(WindowFriendList)
{
    this->usedFont = usedFont;

    if (mgr != NULL)
        this->sm = mgr;
    else
        this->sm = new ServiceMgr(this);
    ui->setupUi(this);

    init();
}

ProfileBrowserPanel::~ProfileBrowserPanel()
{
    delete ui;
}

void ProfileBrowserPanel::init()
{
    pb = new ProfileBrowser(this->sm, true, this);
    ui->layoutProfile->addWidget(pb, 0);

    buttonBack = new QPushButton(this);
    buttonBack->setIcon(QIcon::fromTheme("general_back",
                                         QIcon(":/res/general_back.png")));
    buttonBack->setFlat(TRUE);
    ui->layoutProfileTop->addWidget(buttonBack, 1);
    buttonBack->setVisible(FALSE);

    buttonAlbums = new QPushButton(tr("Albums"));
#ifdef Q_WS_MAEMO_5
    buttonAlbums->setText(QString(""));
#endif
    buttonAlbums->setIcon(QIcon(":/res/tasklaunch_photos.png"));
    buttonAlbums->setFlat(TRUE);
    buttonAlbums->setEnabled(FALSE);
    ui->layoutProfileTop->addWidget(buttonAlbums, 1);

    buttonMessages = new QPushButton(tr("Messages"));
#ifdef Q_WS_MAEMO_5
    buttonMessages->setText(QString(""));
#endif
    buttonMessages->setIcon(QIcon(":/res/messages.png"));
    buttonMessages->setFlat(TRUE);
    buttonMessages->setEnabled(FALSE);
    ui->layoutProfileTop->addWidget(buttonMessages, 1);

    buttonSendMsg = new QPushButton(tr("Send message"));
#ifdef Q_WS_MAEMO_5
    buttonSendMsg->setText(QString(""));
#endif
    buttonSendMsg->setIcon(QIcon(":/res/message.png"));
    buttonSendMsg->setFlat(TRUE);
    buttonSendMsg->setEnabled(FALSE);
    ui->layoutProfileTop->addWidget(buttonSendMsg, 1);

    ui->layoutProfileTop->addWidget(buttonRefreshProfile = new QPushButton(this));
    buttonRefreshProfile->setFlat(TRUE);
    buttonRefreshProfile->setIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));

    peopleradb = new QRadioButton(tr("People (%1)").arg("0"));
#ifdef Q_WS_MAEMO_5
    peopleradb->setText(QString("0"));
#endif
    peopleradb->setEnabled(FALSE);
    connect(peopleradb, SIGNAL(toggled(bool)), this, SLOT(showFriendPanel(bool)));
    peopleradb->setIcon(QIcon::fromTheme("general_contacts", QIcon(":/res/general_contacts.png")));
    ui->layoutFriendsTop->addWidget(peopleradb, 1);

    newsradb = new QRadioButton(tr("News (%1)").arg("0"));
#ifdef Q_WS_MAEMO_5
    newsradb->setText(QString("0"));
#endif
    newsradb->setEnabled(FALSE);
    ui->layoutFriendsTop->addWidget(newsradb, 1);
    newsradb->setIcon(QIcon::fromTheme("general_rss", QIcon(":/res/general_rss.png")));
    ui->layoutFriendsTop->setSpacing(0);

    // friends list
    ui->layoutFriendsTop->addWidget(buttonRefreshFriends = new QPushButton(this), 0);
    buttonRefreshFriends->setFlat(TRUE);
    buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));

    this->friendsPanel = new QFriendListWidget(this->usedFont, sm, this);
    this->ui->layoutFriends->addWidget(this->friendsPanel);

    // Filters
    QList<QActionGroup*> actionsList = sm->filterManager()->createActions<Friend>();
    foreach (QActionGroup* actions, actionsList) {
#ifndef Q_WS_MAEMO_5
        friendsPanel->filtersToolBar()->addSeparator();
#endif
        friendsPanel->filtersToolBar()->addActions(actions->actions());
    }

    connect(buttonBack, SIGNAL(clicked()), this, SLOT(hideProfile()));
    connect(buttonAlbums, SIGNAL(clicked()), this, SLOT(showFriendAlbums()));
    connect(buttonMessages, SIGNAL(clicked()), this, SLOT(showFriendMessages()));
    connect(buttonSendMsg, SIGNAL(clicked()), this, SLOT(sendMessageToFriend()));
    connect(buttonRefreshFriends, SIGNAL(clicked()), this,
            SLOT(refreshFriends()));
    connect(buttonRefreshProfile, SIGNAL(clicked()), this, SLOT(refreshProfile()));
    connect(this->friendsPanel, SIGNAL(ownerSelected(Friend)),
            this, SLOT(ownrSelected(Friend)));

    connect(this->sm, SIGNAL(updateAccounts(QString,AccountList)), this,
            SLOT(updateAccounts(QString, AccountList)));

    // device rotation
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));

    this->isFriendsUpdate = false;

    // connect to signals from service manager
    connect(sm, SIGNAL(updateFriends(FriendList, bool)), this,
            SLOT(updateListFriends(FriendList, bool)));
    connect(sm, SIGNAL(updateProfile(Friend)), this, SLOT(updateProfile(Friend)));

    // check that one acount mast be setup
    if (this->sm->getAccounts().length() != 0)
        this->updateListFriends(sm->getFriends(false, true), true);

    this->peopleradb->toggle();
    this->setState();
}

void ProfileBrowserPanel::updateAccounts(QString, AccountList)
{
    if (this->sm->getAccounts().length() != 0)
        this->updateListFriends(sm->getFriends(false, true), true);
}

void ProfileBrowserPanel::ownrSelected(Friend fd, bool isNeedUpdateState)
{
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << ": ownerId=" << fd.ownerId;

    if (!fd.isEmpty()) {
        updateProfile(fd);
        emit openProfile(fd);
        this->sm->getProfile(fd.accountId, fd.ownerId, false, true, !fd.hasFullProfile());
        if (this->sm->canSendMessage(fd.accountId))
            this->buttonSendMsg->setEnabled(true);
        else
            this->buttonSendMsg->setEnabled(false);
    } else {
        // load my profile
       Friend mFd = this->sm->getMyProfile(false);
       this->updateProfile(mFd);
       emit openProfile(mFd);
       this->buttonSendMsg->setEnabled(false);
    }

    if (isNeedUpdateState)
        this->setState(WindowProfile);
}

void ProfileBrowserPanel::updateProfile(Friend profile)
{
    Friend curFr = this->friendsPanel->getSelectedFriend();

    if (curFr.isEmpty())
        curFr = this->sm->getMyProfile(false);

    if (profile.ownerId == curFr.ownerId) {
        pb->setProfile(profile);
        this->buttonAlbums->setEnabled(TRUE);
        this->buttonMessages->setEnabled(TRUE);
        this->buttonSendMsg->setEnabled(TRUE);
    }
}

void ProfileBrowserPanel::ownrSelected(Friend item)
{
    this->ownrSelected(item, true);
}

void ProfileBrowserPanel::updateListFriends(FriendList list, bool isLastUpdate)
{
    if (isLastUpdate) {
        this->isFriendsUpdate = false;
        buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));
    } else {
        this->isFriendsUpdate = true;
        buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
    }

#ifndef Q_WS_MAEMO_5
    this->peopleradb->setText(tr("People (%1)").arg(list.length()));
#else
    this->peopleradb->setText(QString("(%1)").arg(list.length()));
#endif

    Friend curFr = this->friendsPanel->getSelectedFriend();

    if (curFr.isEmpty())
    {
        curFr = sm->getMyProfile(false);
    } else {
        if (curFr.hasFullProfile())
            curFr = this->sm->getProfile(curFr.accountId, curFr.ownerId, false, false, true);
    }
    this->updateProfile(curFr);
}

void ProfileBrowserPanel::refreshFriends()
{
    this->refreshFriends(true);
}

void ProfileBrowserPanel::refreshFriends(bool isNeedUpdate)
{
    if (!this->isFriendsUpdate) {
        this->isFriendsUpdate = true;
        buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
        sm->getFriends(isNeedUpdate, true);
    } else {
        // TODO: add stopping friends update
    }
}

void ProfileBrowserPanel::refreshProfile()
{
    Friend fr = this->friendsPanel->getSelectedFriend();
    if (fr.isEmpty())
        this->sm->getMyProfile(true);
    else
        this->sm->getProfile(fr.accountId, fr.ownerId, true, true, true);
}

void ProfileBrowserPanel::setState()
{
    this->setState(this->state);
}

void ProfileBrowserPanel::setState(WindowState state) {
    this->state = state;
    bool isMultiWindow = this->sm->settings->value(SETTINGS_MULTIPANEL).toBool() && !this->isPortraitMode;

    switch(this->state) {
    case WindowFriendList:
        hideAll(ui->layoutProfiles);
        setFriendsVisible(true);

        if (isMultiWindow) {
            showAll(ui->layoutProfiles);
            buttonBack->setVisible(FALSE);
        } else {
            hideAll(ui->layoutProfiles);
        }
        break;
    case WindowProfile:
        showAll(ui->layoutProfiles);

        if (isMultiWindow) {
            buttonBack->setVisible(FALSE);
        } else {
            setFriendsVisible(false);
            buttonBack->setVisible(TRUE);
        }
        break;
    default:
        QMessageBox::warning(this, tr("Error occured"), tr("Unknown window state"));
        break;
    }
}

void ProfileBrowserPanel::hideAll(QLayout *layout)
{
    for (int i = 0; i < layout->count(); i++) {
        if(layout->itemAt(i)->layout()) {
            hideAll(layout->itemAt(i)->layout());
        } else {
            layout->itemAt(i)->widget()->setVisible(FALSE);
        }
    }
}
void ProfileBrowserPanel::showAll(QLayout *layout)
{
    for (int i = 0; i < layout->count(); i++) {
        if(layout->itemAt(i)->layout()) {
            showAll(layout->itemAt(i)->layout());
        } else {
            layout->itemAt(i)->widget()->setVisible(TRUE);
        }
    }
}

void ProfileBrowserPanel::hideProfile()
{
    this->setState(WindowFriendList);
}

void ProfileBrowserPanel::showFriendInfo(Friend fd)
{
    this->friendsPanel->selectFriend(fd);
    ownrSelected(fd);
}

void ProfileBrowserPanel::showFriendAlbums()
{
    emit showFriendAlbums(this->friendsPanel->getSelectedFriend());
}

void ProfileBrowserPanel::showFriendMessages()
{
    emit showFriendMessages(this->friendsPanel->getSelectedFriend());
}

void ProfileBrowserPanel::sendMessageToFriend()
{
    Friend fd = this->friendsPanel->getSelectedFriend();
    emit sendMessage(fd);
}

void ProfileBrowserPanel::orientationChanged()
{
    QRect geom = QApplication::desktop()->screenGeometry();
    if (geom.width() > geom.height())
        this->isPortraitMode = false;
    else
        this->isPortraitMode = true;
}

void ProfileBrowserPanel::setFriendsVisible(const bool isShow)
{
    if (isShow) {
        this->showAll(this->ui->layoutFriendsTop);
        this->showFriendPanel(this->peopleradb->isChecked());
    } else {
        this->hideAll(this->ui->layoutFriends);
    }
}

void ProfileBrowserPanel::showFriendPanel(bool isShow)
{
    qDebug() << __FUNCTION__ << "(): isShow=" << isShow;
    if (isShow) {
        this->friendsPanel->show();
        this->ui->listNews->hide();
        if (this->isFriendsUpdate)
            buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
        else
            buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh",
                                                           QIcon(":/res/general_refresh.png")));
    } else {
        this->friendsPanel->hide();
        this->ui->listNews->show();
        buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh",
                                                           QIcon(":/res/general_refresh.png")));
    }
}
