#include "messagecomposer.h"
#include "ui_messagecomposer.h"

#include <QUuid>

MessageComposer::MessageComposer(QWidget *parent) :
        QWidget(parent), ui(new Ui::MessageComposer)
{
    ui->setupUi(this);
    this->sm = new ServiceMgr;
    init();
}

MessageComposer::MessageComposer(ServiceMgr *sm, QWidget *parent) :
        QWidget(parent), ui(new Ui::MessageComposer)
{
    ui->setupUi(this);
    this->sm = sm;
    init();
}

MessageComposer::~MessageComposer()
{
    delete ui;
}

void MessageComposer::init()
{
    ui->layoutMessageComposerBottom->addWidget(buttonSend = new QPushButton(tr("Send")), 0);
    buttonSend->setFlat(TRUE);
    buttonSend->setIcon(QIcon(":/res/msg_send.png"));

    ui->layoutMessageComposerBottom->addWidget(buttonSave = new QPushButton(tr("Save")), 0);
    buttonSave->setFlat(TRUE);
    buttonSave->setIcon(QIcon(":/res/msg_save.png"));

    connect(buttonSend, SIGNAL(clicked()), this, SLOT(sendMessage()));
    connect(buttonSave, SIGNAL(clicked()), this, SLOT(saveMessage()));

    connect(this->sm, SIGNAL(updateAccounts(QString, AccountList)), this,
            SLOT(updateAccounts(QString, AccountList)));
    connect(this->sm, SIGNAL(updateFriends(FriendList, bool)), this,
            SLOT(updateFriendList(FriendList, bool)));
}

void MessageComposer::fillFriendList(FriendList list)
{
    ui->listFriends->clear();

    for (int i = 0; i < list.count(); i++) {
        QVariant v;
        Friend fd = list.at(i);

        if (!this->sm->canSendMessage(fd.accountId))
            continue;

        v.setValue(fd);

        QImage img = QImage(fd.icon());

        if (!img.isNull()) {
            if (img.height() > img.width())
                img = img.copy(0, (img.height() - img.width()) / 2, img.width(), img.width());

            img = img.scaled(32, 32, Qt::KeepAspectRatio, Qt::SmoothTransformation);

            this->ui->listFriends->addItem(QIcon(QPixmap::fromImage(img)),
                    Utils::decodeString(fd.name()),v);
        } else {
            this->ui->listFriends->addItem(Utils::decodeString(fd.name()),v);
        }
    }
}

void MessageComposer::updateAccounts(QString accountId, AccountList list)
{
    Q_UNUSED(accountId);
    Q_UNUSED(list);

    this->fillFriendList(this->sm->getFriends(false, false));
}

void MessageComposer::updateFriendList(FriendList list, bool )
{
    this->fillFriendList(list);
}

void MessageComposer::sendMessage()
{
    QString title = this->ui->titleLineEdit->text();
    QString message = this->ui->textBrowser->toPlainText();

    int index = ui->listFriends->currentIndex();
    QVariant v = ui->listFriends->itemData(index);

    if (v == QVariant::Invalid)
        return;

    if (v.canConvert<Friend>()) {
        Friend fd = v.value<Friend>();

        if (this->sm->sendMessage(fd.accountId, fd.ownerId, title, message))
            emit messageSent();
    }
}

void MessageComposer::replyMessage(Message msg)
{
    for (int i = 0; i < ui->listFriends->count(); i++) {
        QVariant v = ui->listFriends->itemData(i);

        if (v.canConvert<Friend>()) {
            Friend item = v.value<Friend>();

            if (item.ownerId == msg.senderId)
                this->ui->listFriends->setCurrentIndex(i);
        }
    }

    this->ui->titleLineEdit->setText(QString(tr("Re: %1")).arg(msg.subject()));
    this->ui->textBrowser->setFocus();
}

void MessageComposer::saveMessage()
{
    Recepient rec;
    QString senderId;

    QVariant v = this->ui->listFriends->itemData(ui->listFriends->currentIndex());
    if (v.canConvert<Friend>()) {
        Friend fd = v.value<Friend>();

        rec.id = fd.ownerId;
        rec.name = fd.name();
        senderId = fd.accountId;
    }

    QString title = this->ui->titleLineEdit->text();
    QString message = this->ui->textBrowser->toPlainText();

    Message msg;
    msg.accountId = senderId;
    msg.title = title;
    msg.text = message;
    msg.recepientList.append(rec);
    msg.senderId = senderId;

    AccountList list = this->sm->getAccounts();    

    for (int i = 0; i < list.count(); i++) {
        if (list.at(i)->accountId() == senderId) {
            msg.senderName = list.at(i)->getProfile(false).name();

            MessageList list2 = list.at(i)->getDraftMessages();

            // Filter requires unique id
            int id;
            if (list2.count() > 0) {
                id = list2.last().messageId.toInt() + 1;
            } else {
                id = 0;
            }
            msg.messageId = QString("msa_draft_%1_%2").arg(QString::number(id))
                                                      .arg(QUuid::createUuid().toString());

            list2.append(msg);
            list.at(i)->setDraftMessages(list2);
        }
    }

    emit saveMessage(msg);
}

void MessageComposer::resetMessage()
{
    this->ui->titleLineEdit->clear();
    this->ui->textBrowser->clear();
}

void MessageComposer::setRecepient(Friend fd)
{
    this->setRecepient(fd.ownerId);
}

void MessageComposer::setRecepient(QString ownerId)
{
    for (int i = 0; i < ui->listFriends->count(); i++) {
        QVariant v = ui->listFriends->itemData(i);

        if (v.canConvert<Friend>()) {
            Friend item = v.value<Friend>();

            if (item.ownerId == ownerId)
                this->ui->listFriends->setCurrentIndex(i);
        }
    }
}

void MessageComposer::setTitle(QString title)
{
    this->ui->titleLineEdit->setText(title);
    this->ui->titleLineEdit->setCursorPosition(0);
}

void MessageComposer::setMessageBody(QString text)
{
    this->ui->textBrowser->setText(text);
}
