#include "messagebrowserpanel.h"
#include "ui_messagebrowserpanel.h"

#include <QToolBar>

#include "filter/servicefilter.h"
#include "filter/filtermanager.h"
#include "filter/filtermessagetype.h"

MessageBrowserPanel::MessageBrowserPanel(QFont uf, QWidget *parent) :
    QWidget(parent),
    ui(new Ui::MessageBrowserPanel),
    currentFriend(Friend()),
    isPortraitMode(false),
    state(WindowMessageList),
    msgState(MessagesInbox)
{
    this->usedFont = uf;
    this->sm = new ServiceMgr;

    ui->setupUi(this);
    init();

    this->setState();
    this->messagesradb->toggle();
}

MessageBrowserPanel::MessageBrowserPanel(QFont uf, ServiceMgr *mgr, QWidget *parent) :
        QWidget(parent),
        ui(new Ui::MessageBrowserPanel),
        currentFriend(Friend()),
        isPortraitMode(false),
        state(WindowMessageList),
        msgState(MessagesOutbox)
{
    this->usedFont = uf;

    if (mgr != NULL)
        this->sm = mgr;
    else
        this->sm = new ServiceMgr(this);

    ui->setupUi(this);
    init();

    this->setState();
    this->messagesradb->toggle();
}

MessageBrowserPanel::~MessageBrowserPanel()
{
    delete ui;
}

void MessageBrowserPanel::init()
{
    qDebug() << __FILE__ << ":" << __LINE__ << " - " << __FUNCTION__;
    mb = new MessageBrowser(this->sm, this);
    ui->layoutMessageBrowserBottom->addWidget(mb, 0);

    buttonBack = new QPushButton(this);
    buttonBack->setIcon(QIcon::fromTheme("general_back",
                                         QIcon(":/res/general_back.png")));
    buttonBack->setFlat(TRUE);
    ui->layoutMessageBrowserTop->addWidget(buttonBack, 0);
    buttonBack->setVisible(FALSE);

    ui->layoutMessageBrowserTop->addWidget(buttonNewMsg = new QPushButton(this), 0);
    buttonNewMsg->setFlat(TRUE);
    buttonNewMsg->setIcon(QIcon(":/res/msg_new.png"));
//    buttonNewMsg->setIconSize(QSize(48, 48));

    ui->layoutMessageBrowserTop->addWidget(buttonReply = new QPushButton(this), 0);
    buttonReply->setFlat(TRUE);
    buttonReply->setEnabled(FALSE);
    buttonReply->setIcon(QIcon(":/res/msg_reply.png"));
//    buttonReply->setIconSize(QSize(48, 48));

    ui->layoutMessageBrowserTop->addWidget(buttonEditDraft = new QPushButton(this), 0);
    buttonEditDraft->setFlat(TRUE);
    buttonEditDraft->setIcon(QIcon(":/res/msg_edit.png"));
    buttonEditDraft->setEnabled(FALSE);
//    buttonEditDraft->setIconSize(QSize(48, 48));

    ui->layoutMessageBrowserTop->addWidget(buttonDeleteMsg = new QPushButton(this), 0);
    buttonDeleteMsg->setFlat(TRUE);
    buttonDeleteMsg->setEnabled(FALSE);
    buttonDeleteMsg->setIcon(QIcon(":/res/msg_delete.png"));
//    buttonDeleteMsg->setIconSize(QSize(48, 48));

    messagesradb = new QRadioButton(tr("Messages (%1)").arg("0"));
#ifdef Q_WS_MAEMO_5
    messagesradb->setText(QString("0"));
#endif

    messagesradb->setIcon(QIcon(":/res/messages.png"));
    messagesradb->setEnabled(FALSE);
    connect(messagesradb, SIGNAL(toggled(bool)), this, SLOT(showMessagesPanel(bool)));
    ui->layoutMessagesTop->addWidget(messagesradb, 1);

    newsradb = new QRadioButton(tr("News (%1)").arg("0"));
#ifdef Q_WS_MAEMO_5
    newsradb->setText(QString("0"));
#endif
    newsradb->setEnabled(FALSE);
    ui->layoutMessagesTop->addWidget(newsradb, 1);
    newsradb->setIcon(QIcon::fromTheme("general_rss", QIcon(":/res/general_rss.png")));
    ui->layoutMessagesTop->setSpacing(0);

    ui->layoutMessagesTop->addWidget(buttonRefreshMessages = new QPushButton(this), 0);
    buttonRefreshMessages->setFlat(TRUE);
    buttonRefreshMessages->setIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));

    // message list
    this->messagesPanel = new QMessageListWidget(this->usedFont, sm, this);
    this->ui->layoutMessages->addWidget(this->messagesPanel);

    // Filters
    QList<QActionGroup*> actionsList = sm->filterManager()->createActions<Message>();
    foreach (QActionGroup* actions, actionsList) {
#ifndef Q_WS_MAEMO_5
        messagesPanel->filtersToolBar()->addSeparator();
#endif
        messagesPanel->filtersToolBar()->addActions(actions->actions());
    }

    connect(buttonBack, SIGNAL(clicked()), this, SLOT(hideMessage()));
    connect(buttonDeleteMsg, SIGNAL(clicked()), this, SLOT(deleteMessageClicked()));
    connect(buttonEditDraft, SIGNAL(clicked()), this, SLOT(editDraftMessage()));
    connect(buttonNewMsg, SIGNAL(clicked()), this, SLOT(newMessageClicked()));
    connect(buttonReply, SIGNAL(clicked()), this, SLOT(replyMessageClicked()));
    connect(this->messagesPanel, SIGNAL(ownerSelected(Message)), this,
            SLOT(ownrSelected(Message)));
    connect(buttonRefreshMessages, SIGNAL(clicked()), this,
            SLOT(refreshListMessages()));

    // device rotation
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));

    this->isFriendMessages = false;
    this->isMessagesUpdate = false;

    // connect to signals from Service Manager
    connect(sm, SIGNAL(updateMessageList(MessageList, bool)), this,
            SLOT(updateListMessages(MessageList, bool)));
    connect(sm, SIGNAL(updateAccounts(QString, AccountList)), this,
            SLOT(updateAccounts(QString, AccountList)));

    // check that one acount must be setup
    if (this->sm->getAccounts().length() != 0) {
        this->refreshListMessages(false);
    }

    orientationChanged();
}

void MessageBrowserPanel::updateAccounts(QString, AccountList)
{
    if (this->sm->getAccounts().length() != 0)
            refreshListMessages(false);
}

void MessageBrowserPanel::updateListMessages(MessageList list, bool isLastUpdate)
{
    Message curMsg = this->messagesPanel->getSelectedMessage();

    if (isLastUpdate) {
        this->isMessagesUpdate = false;
        buttonRefreshMessages->setIcon(QIcon::fromTheme("general_refresh",
                                                        QIcon(":/res/general_refresh.png")));
    } else {
        this->isMessagesUpdate = true;
        buttonRefreshMessages->setIcon(QIcon::fromTheme("general_stop",
                                                        QIcon(":/res/general_stop.png")));
    }

#ifndef Q_WS_MAEMO_5
    this->messagesradb->setText(tr("Messages (%1)").arg(list.count()));
#else
    this->messagesradb->setText(QString("(%1)").arg(list.count()));
#endif

    bool isMultiWindow = this->sm->settings->value(SETTINGS_MULTIPANEL).toBool() && !this->isPortraitMode;
    if (isMultiWindow)
        this->updateMessage(curMsg);
    else {
        this->setState(WindowMessageList);
    }
}

void MessageBrowserPanel::ownrSelected(Message msg, bool isNeedUpdateState)
{
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << ": msgId=" << msg.messageId;

    updateMessage(msg);
    emit openMessage(msg);

    if (this->sm->canSendMessage(msg.accountId))
        this->buttonReply->setEnabled(true);
    else
        this->buttonReply->setEnabled(false);

    if (isNeedUpdateState)
        this->setState(WindowMessage);
}

void MessageBrowserPanel::updateMessage(Message message)
{
    Message curMsg = this->messagesPanel->getSelectedMessage();

    // recepient for VKontakte service
    if (message.recepientList.count() == 0) {
        AccountList list = this->sm->getAccounts();

        for (int i = 0; i < list.count(); i++) {
            if (list.at(i)->accountId() == message.accountId) {
                Friend profile = list.at(i)->getProfile(false);

                Recepient rec;
                rec.name = profile.name();
                rec.id = profile.ownerId;
                message.recepientList.append(rec);
            }
        }
    }

    // sender for VKontakte service
    if (message.senderName.isEmpty()) {
        AccountList list = this->sm->getAccounts();

        for (int i = 0; i < list.count(); i++) {
            if (list.at(i)->accountId() == message.accountId) {
                Friend profile = list.at(i)->getProfile(false);

                message.senderId = profile.ownerId;
                message.senderName = profile.name();
            }
        }
    }


    if (message.messageId == curMsg.messageId) {
        mb->setMessage(message);
    }

    this->buttonDeleteMsg->setEnabled(TRUE);

    if (!message.time.isEmpty()) {
        this->buttonReply->setEnabled(TRUE);
        this->buttonEditDraft->setEnabled(FALSE);
    } else {
        if (!message.messageId.isEmpty()) {
            this->buttonReply->setEnabled(FALSE);
            this->buttonEditDraft->setEnabled(TRUE);
        } else {
            this->buttonReply->setEnabled(FALSE);
            this->buttonEditDraft->setEnabled(FALSE);
        }
    }

    this->setState(WindowMessage);
}

void MessageBrowserPanel::ownrSelected(Message item)
{
    this->ownrSelected(item, true);
}

void MessageBrowserPanel::refreshListMessages()
{
    this->refreshListMessages(true);
}

void MessageBrowserPanel::refreshListMessages(bool isNeedUpdate)
{
    if (!this->isMessagesUpdate) {
        this->isMessagesUpdate = true;
        buttonRefreshMessages->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
        this->mb->resetMessage();
        sm->getMessages(isNeedUpdate, true);
    } else {
        // TODO: add stopping messages update
    }
}

void MessageBrowserPanel::deleteMessageClicked()
{
    Message curMsg = this->messagesPanel->getSelectedMessage();

    // delete draft message
    if (curMsg.time.isEmpty()) {
        AccountList list = this->sm->getAccounts();

        for (int i = 0; i < list.count(); i++) {
            if (list.at(i)->accountId() == curMsg.accountId) {
                MessageList list2 = list.at(i)->getDraftMessages();

                for (int j = 0; j < list2.count(); j++) {
                    if (list2.at(j).messageId == curMsg.messageId)
                        list2.removeAt(j);
                }
                list.at(i)->setDraftMessages(list2);
                emit deleteMessage(curMsg);
                refreshListMessages();
                break;
            }
        }
    } else {
        // delete general message
        if (this->sm->deleteMessage(curMsg)) {
            emit deleteMessage(curMsg);
            refreshListMessages();
        }
    }
}

void MessageBrowserPanel::newMessageClicked()
{
    emit newMessage();
}

void MessageBrowserPanel::replyMessageClicked()
{
     Message curMsg = this->messagesPanel->getSelectedMessage();
     emit replyMessage(curMsg);
}

void MessageBrowserPanel::editDraftMessage()
{
    Message curMsg = this->messagesPanel->getSelectedMessage();
    emit editDraftMessage(curMsg);
}

void MessageBrowserPanel::setMessagesState(MessagesState state)
{
    this->msgState = state;    
}

void MessageBrowserPanel::setState()
{
    this->setState(this->state);
}

void MessageBrowserPanel::setState(WindowState state) {
    this->state = state;
    bool isMultiWindow = this->sm->settings->value(SETTINGS_MULTIPANEL).toBool() && !this->isPortraitMode;

    this->messagesPanel->setMultipanelEnabled(isMultiWindow);

    switch(this->state) {
    case WindowMessageList:
        hideAll(ui->layoutMessageBrowser);
        setMessagesVisible(true);

        if (isMultiWindow) {
            showAll(ui->layoutMessageBrowser);
            buttonBack->setVisible(FALSE);
        } else {
            hideAll(ui->layoutMessageBrowser);
        }
        break;
    case WindowMessage:
        showAll(ui->layoutMessageBrowser);

        if (isMultiWindow) {
            buttonBack->setVisible(FALSE);
        } else {
            setMessagesVisible(false);
            buttonBack->setVisible(TRUE);
        }
        break;
    default:
        QMessageBox::warning(this, tr("Error occured"), tr("Unknown window state"));
        break;
    }
}

void MessageBrowserPanel::hideAll(QLayout *layout)
{
    for (int i = 0; i < layout->count(); i++) {
        if(layout->itemAt(i)->layout()) {
            hideAll(layout->itemAt(i)->layout());
        } else {
            layout->itemAt(i)->widget()->setVisible(FALSE);
        }
    }
}
void MessageBrowserPanel::showAll(QLayout *layout)
{
    for (int i = 0; i < layout->count(); i++) {
        if(layout->itemAt(i)->layout()) {
            showAll(layout->itemAt(i)->layout());
        } else {
            layout->itemAt(i)->widget()->setVisible(TRUE);
        }
    }
}

void MessageBrowserPanel::hideMessage()
{
    this->setState(WindowMessageList);
}

void MessageBrowserPanel::showFriendMessages(Friend fd)
{
    this->mb->resetMessage();
    this->messagesPanel->showFriendMessages(fd);
}

void MessageBrowserPanel::setMessagesVisible(const bool isShow)
{
    if (isShow) {
        this->showAll(this->ui->layoutMessagesTop);
        this->showMessagesPanel(this->messagesradb->isChecked());
    } else {
        this->hideAll(this->ui->layoutMessages);
    }
}

void MessageBrowserPanel::showMessagesPanel(bool isShow)
{
    qDebug() << __FUNCTION__ << "(): isShow=" << isShow;
    if (isShow) {
        this->messagesPanel->show();
        this->ui->listNews->hide();
        buttonRefreshMessages->setIcon(QIcon::fromTheme("general_refresh",
                                                           QIcon(":/res/general_refresh.png")));
    } else {
        this->messagesPanel->hide();
        this->ui->listNews->show();
        buttonRefreshMessages->setIcon(QIcon::fromTheme("general_refresh",
                                                           QIcon(":/res/general_refresh.png")));
    }
}

void MessageBrowserPanel::orientationChanged()
{
    QRect geom = QApplication::desktop()->screenGeometry();
    if (geom.width() > geom.height())
        this->isPortraitMode = false;
    else
        this->isPortraitMode = true;

#ifdef Q_WS_MAEMO_5
    foreach (QAction* action, messagesPanel->filtersToolBar()->actions()) {
        if ((action->isSeparator())
                || (action->data().value<FilterMessageType::MessageType>() == FilterMessageType::ALL)) {
            if (!isPortraitMode)
                action->setVisible(false);
            else
                action->setVisible(true);
        }
    }
#endif

}
