#include "qeventfeed.h"

#include <QFile>

#include "utils/utils.h"

QEventFeed::QEventFeed(QObject *parent) :
    QObject(parent)
{
}

QEventFeed::QEventFeed(const QEventFeed &src):
        QObject(src.parent())
{
    this->copyData(this, &src);
}

void QEventFeed::copyData(QEventFeed *dst, const QEventFeed *src)
{
    dst->accountId = src->accountId;
    dst->attach = src->attach;
    dst->created = src->created;
    dst->ownerId = src->ownerId;
    dst->ownerName = src->ownerName;
    dst->text = src->text;
}

QEventFeed& QEventFeed::operator =(QEventFeed const& src)
{
    QEventFeed::copyData(this, &src);
    return *this;
}

QString QEventFeed::typeToString(const FeedType type)
{
    QString name;

    switch(type)
    {
    case photoFeed:
        name = "image";
        break;
    case messageFeed:
        name = "message";
        break;
    case commentFeed:
        name = "comment";
        break;
    case videoFeed:
        name = "video";
        break;
    case audioFeed:
        name = "audio";
        break;
    case likeFeed:
        name = "like";
        break;
    case tagFeed:
        name = "tag";
        break;
    case friendFeed:
        name = "friend";
    }

    return name;
}

void QEventFeed::storeFeed(const QEventFeedList &list, const FeedType type, QString accountId)
{
    QString name = QEventFeed::typeToString(type);

    // store feed into cache
    QDomDocument out("MyDoc");
    QDomElement root = out.createElement(NODE_FEED_ROOT);
    root.setAttribute(ATTR_REFRESH, list.timeCreate().toString(DATE_TIME_FORMAT));
    out.appendChild(root);

    for (int i = 0; i < list.length(); i++) {
        root.appendChild(list[i].toQDomElement(out));
    }

    QDir dir;

    dir.setPath(dir.cleanPath(Utils::getAccountDir(accountId)));
    if (!dir.exists())
        dir.mkpath(dir.absolutePath());

    QFile file(dir.absolutePath() + "/" + QString(FILE_FEED_DATA).arg(name));
    qDebug() << file.fileName();
    if (file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate) == false) {
        qDebug() << "can't open file: " + file.errorString();
    }
    //file.write(output.toUtf8());
    file.write(out.toByteArray(1));
    file.close();

}

QEventFeedList QEventFeed::loadFeed(QString accountId, FeedType type)
{
    QEventFeedList ret;

    QString name = QEventFeed::typeToString(type);

    QFile file(Utils::getAccountDir(accountId) + QString(FILE_FEED_DATA).arg(name));
    if (file.open(QIODevice::ReadOnly) == false)
        return ret;

    QDomDocument doc;
    doc.setContent(&file);
    file.close();

    // get list updated time
    QDomElement root = doc.elementsByTagName(NODE_FEED_ROOT).at(0).toElement();
    QDateTime time_create = QDateTime::fromString(root.attribute(ATTR_REFRESH, ""), DATE_TIME_FORMAT);
    ret.setTimeCreate(time_create);

    // parse list of events
    QDomNodeList lst = doc.elementsByTagName(NODE_EVENT_ROOT);

    qDebug() << "Start load of " << lst.length() << " events from " << accountId;
    for (uint i = 0; i < lst.length(); i++) {
        ret.append(QEventFeed::fromQDomElement(lst.at(i).toElement(), accountId, type));
    }

    return ret;
}

QString QEventFeed::getKey(QString accountId, FeedType type)
{
    QString name = QEventFeed::typeToString(type);
    return QString("feed_%1_%2").arg(accountId).arg(name);
}

QDomElement QEventFeed::toQDomElement(QDomDocument &doc) const
{
    QDomElement ret = doc.createElement(NODE_EVENT_ROOT);

    // owner ID
    QDomElement node = doc.createElement(NODE_EVENT_OWNERID);
    QDomText t = doc.createTextNode(this->ownerId);
    ret.appendChild(node);
    node.appendChild(t);

    // owner name
    node = doc.createElement(NODE_EVENT_OWNER_NAME);
    t = doc.createTextNode(this->ownerName);
    ret.appendChild(node);
    node.appendChild(t);

    // text
    if (!this->text.isEmpty())
    {
        node = doc.createElement(NODE_EVENT_TEXT);
        t = doc.createTextNode(this->text);
        ret.appendChild(node);
        node.appendChild(t);
    }

    // created
    node = doc.createElement(NODE_EVENT_CREATED);
    t = doc.createTextNode(this->created.toString(DATE_TIME_FORMAT));
    ret.appendChild(node);
    node.appendChild(t);

    // attachment
    ret.appendChild(this->attach.toQDomElement(doc));

    return ret;
}

QEventFeed QEventFeed::fromQDomElement(const QDomElement& fr, const QString& accountId, const FeedType tp)
{
    QEventFeed ret;

    if (fr.elementsByTagName(NODE_EVENT_OWNERID).count() > 0)
        ret.ownerId = fr.elementsByTagName(NODE_EVENT_OWNERID).at(0).firstChild().toText().data();

    if (fr.elementsByTagName(NODE_EVENT_OWNER_NAME).count() > 0)
        ret.ownerName = fr.elementsByTagName(NODE_EVENT_OWNER_NAME).at(0).firstChild().toText().data();

    if (fr.elementsByTagName(NODE_EVENT_TEXT).count() > 0)
        ret.text = fr.elementsByTagName(NODE_EVENT_TEXT).at(0).firstChild().toText().data();

    if (fr.elementsByTagName(NODE_EVENT_CREATED).count() > 0)
        ret.created = QDateTime::fromString(fr.elementsByTagName(NODE_EVENT_CREATED).at(0).firstChild().toText().data(), DATE_TIME_FORMAT);

    ret.attach = QAttachment::fromQDomElement(fr.firstChildElement(NODE_ATTACH_ROOT), accountId);

    ret.accountId = accountId;

    return ret;
}

Photo QEventFeed::toPhoto() const
{
    Photo ret;
    ret.accountId = this->accountId;
    ret.ownerId = this->ownerId;
    ret.albumId = this->attach.albumId;
    ret.icon = this->attach.icon;
    ret.icon_url = this->attach.iconUrl;
    ret.photo = this->attach.image;
    ret.photo_url = this->attach.objectUrl;
    ret.time_create = this->created.toString(DATE_TIME_FORMAT);
    ret.photoId = this->attach.id;

    return ret;
}
