#include "imagebrowserpanel.h"
#include "ui_imagebrowserpanel.h"

#include <QTimer>

ImageBrowserPanel::ImageBrowserPanel(QWidget *parent) :
        QWidget(parent),
        ui(new Ui::ImageBrowserPanel), state(WindowFriendAlbum), isPortraitMode(false)
{
    this->sm = new ServiceMgr(this);
    ui->setupUi(this);
    init();

    this->isSkipFriendUpdate = false;
    this->isFriendsUpdate = false;
    this->isAlbumsUpdate = false;
    this->isPhotosUpdate = false;
    this->isFeedUpdate = false;

    //connect to signals from service manager
    connect(sm, SIGNAL(updateFriends(FriendList, bool)),
            this, SLOT(updateListFriends(FriendList, bool)));
    connect(sm, SIGNAL(updateAlbumList(Friend,AlbumList,bool)),
            this, SLOT(updateListAlbums(Friend,AlbumList, bool)));
    connect(sm, SIGNAL(updatePhotoList(QString,QString,QString,PhotoList,bool)),
            this, SLOT(updateListPhotos(QString,QString,QString,PhotoList, bool)));
    connect(sm, SIGNAL(updateFeed(QEventFeedList,QEventFeed::FeedType, bool)),
            this, SLOT(updateFeed(QEventFeedList,QEventFeed::FeedType, bool)));
    connect(sm, SIGNAL(errorOccured(QString,QTransport::Action,bool)),
            SLOT(gotError(QString, QTransport::Action, bool)));
    connect(sm, SIGNAL(updateAccounts(QString,AccountList)), this,
            SLOT(updateAccounts(QString,AccountList)));

    // check that one acount mast be setup
//    if (this->sm->getAccounts().length() != 0) {
//        this->refreshFriends(false);
//        this->refreshAlbums(false);
//        this->refreshPhotos(false);
//        this->refreshFeed(false);
//    }

    this->setState();
    this->peopleradb->toggle();
}

ImageBrowserPanel::ImageBrowserPanel(ServiceMgr *mgr, QWidget *parent) :
    QWidget(parent),
    ui(new Ui::ImageBrowserPanel), state(WindowFriendAlbum), isPortraitMode(false)
{
    this->sm = mgr;
    ui->setupUi(this);
    init();

    this->isSkipFriendUpdate = false;
    this->isFriendsUpdate = false;
    this->isAlbumsUpdate = false;
    this->isPhotosUpdate = false;
    this->isFeedUpdate = false;

    //connect to signals from service manager
    connect(sm, SIGNAL(updateFriends(FriendList, bool)),
            this, SLOT(updateListFriends(FriendList, bool)));
    connect(sm, SIGNAL(updateAlbumList(Friend,AlbumList, bool)),
            this, SLOT(updateListAlbums(Friend,AlbumList, bool)));
    connect(sm, SIGNAL(updatePhotoList(QString,QString,QString,PhotoList,bool)),
            this, SLOT(updateListPhotos(QString,QString,QString,PhotoList, bool)));
    connect(sm, SIGNAL(updateFeed(QEventFeedList,QEventFeed::FeedType, bool)),
            this, SLOT(updateFeed(QEventFeedList,QEventFeed::FeedType, bool)));
    connect(sm, SIGNAL(errorOccured(QString,QTransport::Action,bool)),
            SLOT(gotError(QString, QTransport::Action, bool)));
    connect(sm, SIGNAL(updateAccounts(QString, AccountList)), this,
            SLOT(updateAccounts(QString, AccountList)));

    // check that one acount mast be setup
//    if (this->sm->getAccounts().length() != 0) {
//        this->refreshFriends(false);
//        this->refreshAlbums(false);
//        this->refreshPhotos(false);
//        this->refreshFeed(false);
//    }

    this->setState();
    this->peopleradb->toggle();
}

ImageBrowserPanel::~ImageBrowserPanel()
{
    delete ui;
}

void ImageBrowserPanel::init()
{
    // radio buttons
    peopleradb = new QRadioButton(tr("People (%1)").arg("0"));
    ui->layoutFriendsTop->addWidget(peopleradb, 1);
    connect(peopleradb, SIGNAL(toggled(bool)), this, SLOT(showFriendPanel(bool)));
    peopleradb->setIcon(QIcon::fromTheme("general_contacts", QIcon(":/res/general_contacts.png")));

    newsradb = new QRadioButton(tr("News (%1)").arg("0"));
    ui->layoutFriendsTop->addWidget(newsradb, 1);
    newsradb->setIcon(QIcon::fromTheme("general_rss", QIcon(":/res/general_rss.png")));

    ui->layoutFriendsTop->setSpacing(0);


    // friends list
    QIcon icon;
    ui->layoutFriendsTop->addWidget(buttonRefreshFriends = new QPushButton(this), 0);
    buttonRefreshFriends->setFlat(TRUE);
    buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh",
                                                   QIcon(":/res/general_refresh.png")));

    ui->listFriends->setStyleSheet("text-align: left; image-position: center;");
    ui->listFriends->setFont(this->usedFont);

    // friends filter
    ui->layoutFriendsBottom->addWidget(editOwnr = new HintEdit(this));
    ui->layoutFriendsBottom->addWidget(buttonEditOwnr = new QPushButton(this));

    buttonEditOwnr->setIcon(QIcon::fromTheme("general_backspace",
                                             QIcon(":/res/general_backspace.png")));
    buttonEditOwnr->setFlat(TRUE);
    editOwnr->setHintText(tr("Type a name..."));

    // albums
    this->albumsTopPanel = new QTopPanel(this);
    this->ui->layoutAlbums->insertWidget(0, this->albumsTopPanel);

    icon = QIcon::fromTheme("tasklaunch_photos", QIcon(":/res/tasklaunch_photos.png"));
    this->albumsTopPanel->setPixmap(icon.pixmap(48, QIcon::Normal, QIcon::On));

    this->albumsTopPanel->setRefreshIcon(QIcon::fromTheme("general_refresh",
                                                          QIcon(":/res/general_refresh.png")));
    this->albumsTopPanel->setBackIcon(QIcon::fromTheme("general_back",
                                                       QIcon(":/res/general_back.png")));

    connect(this->albumsTopPanel, SIGNAL(clicked()), this, SLOT(showFriendInfo()));

    this->ui->listAlbums->setFont(this->usedFont);
    this->albumsTopPanel->setFont(this->usedFont);

    // photos
    this->photosTopPanel = new QTopPanel(this);
    this->ui->layoutPhotos->insertWidget(0, this->photosTopPanel);

    icon = QIcon::fromTheme("general_image", QIcon(":/res/general_image.png"));
    this->photosTopPanel->setPixmap(icon.pixmap(64, QIcon::Normal, QIcon::On));

    this->photosTopPanel->setRefreshIcon(QIcon::fromTheme("general_refresh",
                                                          QIcon(":/res/general_refresh.png")));
    this->photosTopPanel->setBackIcon(QIcon::fromTheme("general_back",
                                                       QIcon(":/res/general_back.png")));

    ui->listPhotos->setDragDropMode(QAbstractItemView::NoDragDrop);
    this->ui->listPhotos->setFont(this->usedFont);
    this->photosTopPanel->setFont(this->usedFont);

    // signals
    connect(editOwnr, SIGNAL(textChanged(QString)),
            this, SLOT(editOwnr_textChanged(QString)));
    connect(buttonEditOwnr, SIGNAL(clicked()),
            this, SLOT(buttonResetOwnr_clicked()));
    connect(ui->listFriends, SIGNAL(itemClicked(QListWidgetItem*)),
            this, SLOT(ownrSelected(QListWidgetItem*)));
    connect(ui->listFriends, SIGNAL(itemActivated(QListWidgetItem*)),
            this, SLOT(ownrSelected(QListWidgetItem*)));
    connect(ui->listNews, SIGNAL(itemClicked(QListWidgetItem*)),
            this, SLOT(feedEventSelected(QListWidgetItem*)));
    connect(ui->listAlbums, SIGNAL(itemClicked(QListWidgetItem*)),
            this, SLOT(albmSelected(QListWidgetItem*)));
    connect(ui->listPhotos, SIGNAL(itemClicked(QListWidgetItem*)),
            this, SLOT(photoSelected(QListWidgetItem*)));
    connect(buttonRefreshFriends, SIGNAL(clicked()),
            this, SLOT(refreshFriends()));

    connect(this->albumsTopPanel, SIGNAL(refreshPressed()),
            this, SLOT(refreshAlbums()));
    connect(this->albumsTopPanel, SIGNAL(backPressed()),
            this, SLOT(hideAlbums()));
    connect(this->albumsTopPanel, SIGNAL(clicked()),
            this, SLOT(albumHeaderClicked()));

    connect(this->photosTopPanel, SIGNAL(refreshPressed()),
            this, SLOT(refreshPhotos()));
    connect(this->photosTopPanel, SIGNAL(backPressed()),
            this, SLOT(hidePhotos()));
    connect(this->photosTopPanel, SIGNAL(clicked()),
            this, SLOT(photoHeaderClicked()));

    // enable sorting
    this->ui->listAlbums->setSortingEnabled(true);

    // device rotation
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
}

void ImageBrowserPanel::updateAccounts(QString, AccountList )
{
    if (this->sm->getAccounts().length() != 0) {
        this->refreshFriends(false);
        this->refreshAlbums(false);
        this->refreshPhotos(false);
        this->refreshFeed(false);
    }
}

void ImageBrowserPanel::buttonResetOwnr_clicked()
{
    editOwnr->clear();
}

void ImageBrowserPanel::editOwnr_textChanged(QString str)
{
    int num = ui->listFriends->count();
    if (str.count() > 0) {
        for (int i = 0; i < num; i++)
            if (ui->listFriends->item(i)->text().indexOf(str, 0, Qt::CaseInsensitive) < 0)
                ui->listFriends->item(i)->setHidden(TRUE);
            else
                ui->listFriends->item(i)->setHidden(FALSE);
    } else {
        for (int i = 0; i < num; i++)
            ui->listFriends->item(i)->setHidden(FALSE);
    }
    ui->listFriends->scrollTo(ui->listFriends->currentIndex());
}


void ImageBrowserPanel::addMeToFriends()
{
    bool isEnabledAccountFound = false;
    Friend myProfile;
    AccountList accs = this->sm->getAccounts();
    for (int i = 0; i < accs.length(); i++) {
        if (accs.at(i)->isNetworkEnabled) {
            myProfile = accs.at(i)->getProfile(false);
            isEnabledAccountFound = true;
            break;
        }
    }

    if (!isEnabledAccountFound && accs.length() > 0)
        myProfile = accs.at(0)->getProfile(false);

    qDebug() << __FILE__<<":" << __LINE__ << ": profile icon:" << myProfile.icon();
    QListWidgetItem *item = this->createFriendWidget(myProfile);
    ui->listFriends->insertItem(0, item);
}

QListWidgetItem * ImageBrowserPanel::createFriendWidget(Friend fr)
{
    QListWidgetItem *item = new QListWidgetItem(fr.name());

    QImage img = QImage(fr.icon());
    if (!img.isNull()) {
        if (img.height() > img.width())
            img = img.copy(0, (img.height() - img.width()) / 2, img.width(), img.width());

        if (img.width() > img.height())
            img = img.copy((img.width() - img.height()) / 2, 0, img.height(), img.height());

        item->setIcon(QIcon(QPixmap::fromImage(img)));
    }

    if (item->icon().isNull())
        item->setIcon(QIcon::fromTheme("general_default_avatar", QIcon(":/res/general_default_avatar.png")));

    QVariant v;
    v.setValue(fr);
    item->setData(Qt::UserRole, v);

    return item;
}

void ImageBrowserPanel::ownrSelected(QListWidgetItem *item, bool isNeedUpdateState)
{
    qDebug() << item->text();
    this->setAlbumHeader();
    QVariant v = item->data(Qt::UserRole);
    if (v.canConvert<Friend>()) {
        Friend fd = v.value<Friend>();
        qDebug() << "Get albums for " << fd.ownerId;

        if (ui->listFriends->currentRow() == 0) {
            this->sm->getAlbums(false, true);
        } else {
            this->sm->getAlbums(fd, false, true);
        }
        this->setAlbumHeader(fd);

        if (isNeedUpdateState)
            this->setState(WindowAlbumFriend);
    }
}

void ImageBrowserPanel::ownrSelected(QListWidgetItem *item)
{
    this->ownrSelected(item, true);
}

void ImageBrowserPanel::feedEventSelected(QListWidgetItem *item)
{
    qDebug() << __FILE__ <<":"<< __LINE__ << ":"<< __FUNCTION__<< "()";
    QVariant v = item->data(Qt::UserRole);
    if (v.canConvert<QEventFeed>()) {
        QEventFeed fd = v.value<QEventFeed>();
        qDebug() << "Open image from album " << fd.attach.albumId << " for " << fd.ownerId;
        emit this->openImage(fd.toPhoto());
    }
}

void ImageBrowserPanel::albmSelected(QListWidgetItem *item)
{
    qDebug() << __FILE__ <<":"<< __LINE__ << ":"<< __FUNCTION__<< "(): Selected album name: " << item->text();
    QVariant v = item->data(Qt::UserRole);
    if (v.canConvert<Album>()) {
        Album fd = v.value<Album>();
        qDebug() << "Get photos for " << fd.albumId;
        this->sm->getPhotos(fd, false, true);
        this->setPhotoHeader(fd);
    } else {
        this->setPhotoHeader();
    }
    this->setState(WindowPhotoAlbum);
}

QString ImageBrowserPanel::albumItem(const Album& al)
{
    QString str = Utils::decodeString(al.title) + "\n";
    str += tr("%1 photo(s)", "", al.size).arg(al.size) + ".\n";
    if (al.size != 0 && !al.time_update.isEmpty())
        str += tr(" Updated: ") + al.time_update;

    return str;
}

QPixmap ImageBrowserPanel::albumIcon(const Album& al)
{
    QPixmap pm(al.icon);
    if (pm.isNull()) {
        QIcon icon = QIcon::fromTheme("tasklaunch_photos", QIcon(":/res/tasklaunch_photos.png"));
        pm = icon.pixmap(48, QIcon::Normal, QIcon::On);
    } else {
        pm = pm.scaledToWidth(80, Qt::SmoothTransformation);
    }
    return pm;
}

void ImageBrowserPanel::setAlbumHeader()
{
    this->albumsTopPanel->setText(tr("Album(s):"));
    this->albumsTopPanel->setPixmap(QIcon::fromTheme("tasklaunch_photos", QIcon(":/res/tasklaunch_photos.png")).pixmap(48, QIcon::Normal, QIcon::On));
}

void ImageBrowserPanel::setAlbumHeader(Friend fd)
{
    this->albumsTopPanel->setText(Utils::decodeString(fd.name()));
    QIcon friendIcon = QIcon(fd.icon());
    if (!friendIcon.isNull())
        this->albumsTopPanel->setPixmap(friendIcon.pixmap(48, QIcon::Normal, QIcon::On));
    else
        this->albumsTopPanel->setPixmap(QIcon::fromTheme("tasklaunch_photos", QIcon(":/res/tasklaunch_photos.png")).pixmap(48, QIcon::Normal, QIcon::On));
}

QPixmap ImageBrowserPanel::photoIcon(const Photo& al)
{
    QPixmap pm(al.icon);
    if (pm.isNull()) {
        QIcon icon = QIcon::fromTheme("general_image", QIcon(":/res/general_image.png"));
        pm = icon.pixmap(48, QIcon::Normal, QIcon::On);
    } else {
        if (pm.width() > pm.height()) {
            pm = pm.scaledToWidth(110, Qt::SmoothTransformation);
        } else {
            if (pm.height() > pm.width())
                pm = pm.scaledToHeight(110, Qt::SmoothTransformation);
            else
                pm = pm.scaled(QSize(110, 110), Qt::KeepAspectRatio, Qt::SmoothTransformation);
        }
    }

    // 130,130
    return pm;
}

QPixmap ImageBrowserPanel::photoIcon(const QAttachment& at)
{
    QPixmap pm(at.icon);
    if (pm.isNull()) {
        QIcon icon = QIcon::fromTheme("general_image", QIcon(":/res/general_image.png"));
        pm = icon.pixmap(48, QIcon::Normal, QIcon::On);
    } else {
        pm = pm.scaledToWidth(100, Qt::SmoothTransformation);
    }
    return pm;
}

void ImageBrowserPanel::setPhotoHeader()
{
    QIcon icon = QIcon::fromTheme("general_image", QIcon(":/res/general_image.png"));
    this->photosTopPanel->setPixmap(icon.pixmap(64, QIcon::Normal, QIcon::On));
    this->photosTopPanel->setText(tr("Photo(s):"));
}

void ImageBrowserPanel::setPhotoHeader(Album al)
{
    bool useDefaultIcon = false;

    al.title = cutHeader(al.title);
    al.description = cutHeader(al.description);

    this->photosTopPanel->setText(Utils::decodeString(al.title) +
                         (al.description.isNull() ? "": "\n" +
                          Utils::decodeString(al.description)) +
                         "\n" + tr("%1 photo(s):", "", al.size).arg(al.size));
    QPixmap *pm = new QPixmap(al.icon);
    if (pm->isNull()) {
        useDefaultIcon = true;
    } else {
        *pm = pm->scaledToWidth(80, Qt::SmoothTransformation);
        QIcon albumIcon = QIcon(*pm);
        if (!albumIcon.isNull()) {
            this->photosTopPanel->setPixmap(albumIcon.pixmap(64, QIcon::Normal, QIcon::On));
        } else {
            useDefaultIcon = true;
        }
    }

    if (useDefaultIcon) {
        QIcon icon = QIcon::fromTheme("general_image", QIcon(":/res/general_image.png"));
        this->photosTopPanel->setPixmap(icon.pixmap(64, QIcon::Normal, QIcon::On));
    }
}

void ImageBrowserPanel::photoSelected(QListWidgetItem *item)
{
    QVariant v = item->data(Qt::UserRole);
    if (v.canConvert<Photo>()) {
        Photo fd = v.value<Photo>();
        emit this->openImage(fd);
    }
}

void ImageBrowserPanel::updateListFriends(FriendList list, bool isLastUpdate)
{
    // skip non-last friends updates if it possible
    if (!isLastUpdate && this->isSkipFriendUpdate)
        return;

    qDebug() << "Start update Friend list process. isLastUpdate=" << isLastUpdate;

    if (isLastUpdate) {
        this->isFriendsUpdate = false;
        buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));
    } else {
        this->isFriendsUpdate = true;
        buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
    }

    if (list.isEmpty()) {
        this->ui->listFriends->clear();
        this->addMeToFriends();
        if (isLastUpdate) {
            new QListWidgetItem(tr("No Friends"), this->ui->listFriends);
        } else {
            new QListWidgetItem(tr("Download friend list"), this->ui->listFriends);
        }
        return;
    }

    if (this->ui->listFriends->count() == 0)
        this->addMeToFriends();

    FriendList newFriends(list);
    QList<QListWidgetItem *> dels;

    // check that all current friends is contains in new list
    for (int i = 1; i < ui->listFriends->count(); i++) {
        bool isFound = false;

        QListWidgetItem *item = ui->listFriends->item(i);

        QVariant v = item->data(Qt::UserRole);
        if (v.canConvert<Friend>()) {
            Friend oldFd = v.value<Friend>();

            int removedVal = -1;

            for (int j = 0; j < newFriends.length(); j++) {
                Friend curFd = newFriends.at(j);
                if (oldFd.ownerId == curFd.ownerId &&
                    oldFd.accountId == curFd.accountId) {
                    removedVal = j;
                    QImage *img = new QImage(curFd.icon());

                    if (img->height() > img->width())
                        *img = img->copy(0, (img->height() - img->width()) / 2, img->width(), img->width());
                    item->setIcon(QIcon(QPixmap::fromImage(*img)));

                    if (item->icon().isNull())
                        item->setIcon(QIcon::fromTheme("general_default_avatar", QIcon(":/res/general_default_avatar.png")));
                    v.setValue(curFd);
                    item->setData(Qt::UserRole, v);

                    if (curFd.online())
                    {
                        item->setText(curFd.name() + "\n" + tr("Online"));
                    }else{
                        item->setText(curFd.name());
                    }

                    isFound = true;
                    delete img;
                    break;
                }
            }

            if (isFound && removedVal >= 0) {
                newFriends.removeAt(removedVal);
            } else {
                qDebug() << "prepare to remove friend with id " << oldFd.ownerId << " from " << oldFd.accountId;
            }
        }

        if (!isFound)
            dels.append(item);
    }

    qDebug() << "Will be add " << newFriends.length() << " friends and removed " << dels.length() << " records. Total widgets: " << ui->listFriends->count();

    // del old friends
    for (int i = 0; i < dels.length(); i++) {
        if (dels.at(i)->isSelected()) {
            this->ui->listFriends->item(0)->setSelected(true);
            this->ui->listFriends->setCurrentRow(0);
            //this->ownrSelected(this->ui->listFriends->item(0), false);
        }
        delete dels.at(i);
    }

    // add new Friends with asceding sorting
    for (int i = 0; i < newFriends.length(); i++) {
        QListWidgetItem *item = createFriendWidget(newFriends.at(i));
        if (this->ui->listFriends->count() <= 1)
            this->ui->listFriends->addItem(item);

        bool flag = false;
        for (int j = 1; j < this->ui->listFriends->count(); j++) {
            if (this->ui->listFriends->item(j)->text().compare(item->text()) > 0) {
                this->ui->listFriends->insertItem(j, item);
                flag = true;
                break;
            }
        }
        if (!flag)
            this->ui->listFriends->addItem(item);
    }

    if (this->ui->listFriends->selectedItems().count() == 0) {
        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): no selected items in friend list!";
        this->ui->listFriends->setCurrentRow(0);
        //this->ui->listFriends->item(0)->setSelected(true);
        //this->ui->listFriends->selectedItems().append(this->ui->listFriends->item(0));
        //this->ownrSelected(this->ui->listFriends->item(0), false);
        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): Selected me!!!";
    }

    // update album header for friend
    if (this->ui->listFriends->selectedItems().length() > 0) {
        QVariant v = this->ui->listFriends->selectedItems().at(0)->data(Qt::UserRole);
        Friend fd = v.value<Friend>();
        this->setAlbumHeader(fd);
    }
    this->peopleradb->setText(tr("People (%1)").arg(this->ui->listFriends->count()));

    // set timeout for next updates
    if (!isLastUpdate) {
        this->isSkipFriendUpdate = true;
        QTimer::singleShot(1000, this, SLOT(disableFriendSkipUpdate()));
    }
}

void ImageBrowserPanel::disableFriendSkipUpdate()
{
    this->isSkipFriendUpdate = false;
}

void ImageBrowserPanel::updateListAlbums(Friend owner, AlbumList list, bool isLastUpdate)
{
    qDebug() << __FILE__ <<":"<<__LINE__<<":"<<__FUNCTION__<<"(): owner=" << owner.ownerId << ";list.length()="<<list.length() << "; isLastUpdate=" << isLastUpdate;
    bool isFoundSelectedFriend = false;

    if (this->ui->listFriends->selectedItems().length() > 0 &&
        !this->ui->listFriends->item(0)->isSelected()) {
        // check correctness that list was return for selected friend
        QList<QListWidgetItem *> selFriends = ui->listFriends->selectedItems();
        for (int i = 0; i < selFriends.length(); i++) {
            QVariant v = selFriends[i]->data(Qt::UserRole);
            if (v.canConvert<Friend>()) {
                Friend fd = v.value<Friend>();
                if (fd.hasProfile(owner)) {
                    isFoundSelectedFriend = true;
                    break;
                }
            }
        }
    } else {
        // it is my albums
        AccountList accs = this->sm->getAccounts();
        for (int i = 0; i < accs.length(); i++) {
            if (accs.at(i)->getProfile(false).ownerId == owner.ownerId) {
                isFoundSelectedFriend = true;
                break;
            }
        }
    }

    if (!isFoundSelectedFriend)
    {
        qDebug() << __FILE__ <<":"<<__LINE__<<":"<< __FUNCTION__ << " unused album list with owner=" << owner.ownerId;
        return;
    }
    qDebug() << __FILE__ <<":"<<__LINE__<<":"<<__FUNCTION__<<"(): check passed";
    if (isLastUpdate) {
        this->isAlbumsUpdate = false;
        this->albumsTopPanel->setRefreshIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));
    } else {
        this->isAlbumsUpdate = true;
        this->albumsTopPanel->setRefreshIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
    }

    if (list.isEmpty()) {
        qDebug() << "Empty album list for " << owner.ownerId << "!!!!!!";
        ui->listAlbums->clear();
        if (isLastUpdate)
            new QListWidgetItem(tr("No albums"), ui->listAlbums); //QListWidgetItem *item2 = new
        else
            new QListWidgetItem(tr("Download albums"), this->ui->listAlbums);
        ui->listAlbums->setDisabled(true);
        return;
    }

    ui->listAlbums->setDisabled(false);

    QList<QListWidgetItem *> delItems;
    AlbumList newAlbums;
    newAlbums.append(list);

    // search new albums in list of widgets
    for (int i = 0; i < this->ui->listAlbums->count(); i++) {
        QListWidgetItem *item = this->ui->listAlbums->item(i);
        QVariant v = item->data(Qt::UserRole);

        bool isFound = false;
        int removedVal = -1;

        if (v.canConvert<Album>()) {
            Album oldAlbum = v.value<Album>();

            for (int j = 0; j < newAlbums.length(); j++) {
                Album curAlbum = newAlbums.at(j);
                if (oldAlbum.albumId == curAlbum.albumId &&
                    oldAlbum.accountId == curAlbum.accountId) {
                    removedVal = j;
                    item->setData(Qt::DecorationRole, this->albumIcon(curAlbum));

                    item->setData(Qt::DisplayRole, this->albumItem(curAlbum));

                    qDebug() << __FILE__ << ":" << __LINE__ << ":" << "title=" << this->albumItem(curAlbum);

                    v.setValue(curAlbum);
                    item->setData(Qt::UserRole, v);

                    isFound = true;
                    break;
                }
            }

            if (isFound && removedVal >= 0) {
                newAlbums.removeAt(removedVal);
            } else {
                qDebug() << "prepare to remove album with id " << oldAlbum.albumId << " from " << oldAlbum.accountId;
            }
        }

        if (!isFound)
            delItems.append(item);
    }

    qDebug() << "Will be add " << newAlbums.length() << " albums and removed " << delItems.length() << " records. Total widgets: " << ui->listAlbums->count();

    // add new albums
    for (int i = 0; i < newAlbums.length(); i++) {
        //QListWidgetItem *item = new QListWidgetItem(str, ui->listAlbums);
        //item->setIcon(QIcon(list[i]->icon));
        //if (item->icon().isNull())
        //    item->setIcon(QPixmap(":/res/ms.png"));
        QListWidgetItem *albumitem = new QListWidgetItem(ui->listAlbums);
        albumitem->setData(Qt::DecorationRole, this->albumIcon(newAlbums.at(i)));

        albumitem->setData(Qt::DisplayRole, this->albumItem(newAlbums.at(i)));

        QVariant v;
        v.setValue(newAlbums[i]);
        albumitem->setData(Qt::UserRole, v);
    }

    // del old albums
    for (int i = 0; i < delItems.length(); i++) {
        if (delItems.at(i)->isSelected())
        {
            qDebug() << __FILE__<< __LINE__ << __FUNCTION__ << "(): delete items for owner" << owner.ownerId << "; new length=" << list.length();
            this->hidePhotos();
        }
        delete delItems.at(i);
    }

    // update photo header
    if (this->ui->listAlbums->selectedItems().length() > 0) {
        QVariant v = this->ui->listAlbums->selectedItems().at(0)->data(Qt::UserRole);
        Album al = v.value<Album>();
        this->setPhotoHeader(al);
    }
}

void ImageBrowserPanel::updateListPhotos(QString accountId, QString ownerId, QString albumId, PhotoList list, bool isLastUpdate)
{
    // check correctness that list was return for selected photo list
    bool isFoundSelectedAlbum = false;
    QList<QListWidgetItem *> selItems = ui->listAlbums->selectedItems();
    for (int i = 0; i < selItems.length(); i++) {
        QVariant v = selItems[i]->data(Qt::UserRole);
        if (v.canConvert<Album>()) {
            Album fd = v.value<Album>();
            if (albumId == fd.albumId) {
                isFoundSelectedAlbum = true;
                break;
            }
        }
    }
    if (!isFoundSelectedAlbum)
        return;

    if (isLastUpdate) {
        this->isPhotosUpdate = false;
        this->photosTopPanel->setRefreshIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));
    } else {
        this->isPhotosUpdate = true;
        this->photosTopPanel->setRefreshIcon(QIcon::fromTheme("general_stop",
                                                              QIcon(":/res/general_stop.png")));
    }

    if (list.isEmpty()) {
        this->ui->listPhotos->clear();
        if (isLastUpdate)
            new QListWidgetItem(tr("No images"), ui->listPhotos);
        else
            new QListWidgetItem(tr("Download images"), this->ui->listPhotos);
        ui->listPhotos->setDisabled(true);
        return;
    }

    ui->listPhotos->setDisabled(false);

    QList<QListWidgetItem *> delItems;
    PhotoList newPhoto;
    newPhoto.append(list);

    // search new images in list of widgets
    for (int i = 0; i < this->ui->listPhotos->count(); i++) {
        QListWidgetItem *item = this->ui->listPhotos->item(i);
        QVariant v = item->data(Qt::UserRole);

        bool isFound = false;
        int removedVal = -1;

        if (v.canConvert<Photo>()) {
            Photo oldPhoto = v.value<Photo>();

            for (int j = 0; j < newPhoto.length(); j++) {
                Photo curPhoto = newPhoto.at(j);
                if (oldPhoto.photoId == curPhoto.photoId &&
                    oldPhoto.accountId == curPhoto.accountId) {
                    removedVal = j;

                    QPixmap icon = this->photoIcon(curPhoto);
                    item->setData(Qt::DecorationRole, icon);
                    ((QAbstractItemView*)this->ui->listPhotos)->setIconSize(icon.size());

                    v.setValue(curPhoto);
                    item->setData(Qt::UserRole, v);

                    isFound = true;
                    break;
                }
            }

            if (isFound && removedVal >= 0) {
                newPhoto.removeAt(removedVal);
            } else {
                qDebug() << "prepare to remove photo with id " << oldPhoto.photoId << " from " << oldPhoto.accountId;
            }
        }

        if (!isFound)
            delItems.append(item);
    }
    this->ui->listPhotos->setUpdatesEnabled(false);

    // add new photos
    for (int i = 0; i < newPhoto.length(); i++) {
        qDebug() << "Show photo_thumb: " << newPhoto.at(i).photoId;
        QListWidgetItem *item = new QListWidgetItem(NULL, ui->listPhotos);
        QPixmap icon = this->photoIcon(newPhoto.at(i));
        item->setData(Qt::DecorationRole, icon);
        ((QAbstractItemView*)this->ui->listPhotos)->setIconSize(icon.size());
        QVariant v;
        v.setValue(newPhoto.at(i));
        item->setData(Qt::UserRole, v);
    }

    // del old photos
    for (int i = 0; i < delItems.length(); i++)
        delete delItems.at(i);

    this->ui->listPhotos->setUpdatesEnabled(true);
    this->ui->listPhotos->repaint();
}

void ImageBrowserPanel::albumHeaderClicked()
{

}

void ImageBrowserPanel::photoHeaderClicked()
{

}

void ImageBrowserPanel::refreshFriends()
{
    this->refreshFriends(true);
}

void ImageBrowserPanel::refreshFriends(bool isNeedUpdate)
{
    qDebug() << __FILE__ <<":"<< __LINE__ << ":" << __FUNCTION__<<"()";
    if (!newsradb->isChecked())
    {
        // friends refresh
        if (!this->isFriendsUpdate) {
            buttonResetOwnr_clicked();
            this->isFriendsUpdate = true;
            buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
            sm->getFriends(isNeedUpdate, true);
        } else {
            //TODO: add stopping friends update
        }
    } else {
        // news refresh
        if (!this->isFeedUpdate) {
            this->isFeedUpdate = true;
            buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
            sm->getFeed(QEventFeed::photoFeed, isNeedUpdate, true);
        } else {
            //TODO: add stopping friends update
        }
    }
}

void ImageBrowserPanel::refreshAlbums()
{
    this->refreshAlbums(true);
}

void ImageBrowserPanel::refreshAlbums(bool isNeedUpdate)
{
    qDebug() << __FILE__ <<":"<<__LINE__<<":"<<__FUNCTION__<<"(): isNeedUpdate=" << isNeedUpdate << "; albumUpdate=" << this->isAlbumsUpdate;

    if (!this->isAlbumsUpdate) {
        if (this->ui->listFriends->selectedItems().length() > 0
            && !this->ui->listFriends->item(0)->isSelected()) {
            QListWidgetItem *item = this->ui->listFriends->selectedItems().at(0);
            if (item != NULL) {
                qDebug() << "Update albums for " << item->text();
                QVariant v = item->data(Qt::UserRole);
                if (v.canConvert<Friend>()) {
                    Friend fd = v.value<Friend>();
                    qDebug() << "Get albums for " << fd.ownerId;
                    this->sm->getAlbums(fd, isNeedUpdate, true);
                }
            }
        } else {
            qDebug("Try to load my albums");
            this->sm->getAlbums(isNeedUpdate, true);
        }
    } else {
        //TODO: add stopping albums update
    }
}

void ImageBrowserPanel::refreshPhotos()
{
    this->refreshPhotos(true);
}

void ImageBrowserPanel::refreshPhotos(bool isNeedUpdate)
{
    qDebug() << __FILE__ <<":"<< __LINE__ << ":" << __FUNCTION__<<"()";
    QListWidgetItem *item = this->ui->listAlbums->currentItem();
    if (item == NULL) {
        qWarning() << __FILE__ <<":"<< __LINE__ << ":" << "unknown selected item";
        return;
    }

    if (!this->isPhotosUpdate) {
        this->isPhotosUpdate = true;
        this->photosTopPanel->setRefreshIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
        qDebug() << "Update photos for " << item->text();
        QVariant v = item->data(Qt::UserRole);
        if (v.canConvert<Album>()) {
            Album fd = v.value<Album>();
            qDebug() << "Get photos for " << fd.albumId;
            this->sm->getPhotos(fd, isNeedUpdate, true);
        }
    }
}

void ImageBrowserPanel::hidePhotos()
{
    this->setState(WindowAlbumPhoto);
}

void ImageBrowserPanel::hideAlbums()
{
    this->setState(WindowFriendAlbum);
}

void ImageBrowserPanel::hideAll(QLayout *layout)
{
    for (int i = 0; i < layout->count(); i++) {
        if(layout->itemAt(i)->layout()) {
            hideAll(layout->itemAt(i)->layout());
        } else {
            layout->itemAt(i)->widget()->setVisible(FALSE);
        }
    }
}
void ImageBrowserPanel::showAll(QLayout *layout)
{
    for (int i = 0; i < layout->count(); i++) {
        if(layout->itemAt(i)->layout()) {
            showAll(layout->itemAt(i)->layout());
        } else {
            layout->itemAt(i)->widget()->setVisible(TRUE);
        }
    }
}

void ImageBrowserPanel::setAlbumsVisible(const bool isShow)
{
    if (isShow)
        this->showAll(this->ui->layoutAlbums);
    else
        this->hideAll(this->ui->layoutAlbums);
}

void ImageBrowserPanel::setImagesVisible(const bool isShow)
{
    if (isShow)
        this->showAll(this->ui->layoutPhotos);
    else
        this->hideAll(this->ui->layoutPhotos);
}

void ImageBrowserPanel::setFriendsVisible(const bool isShow)
{
    if (isShow)
    {
        this->showAll(this->ui->layoutFriendsTop);
        this->showFriendPanel(this->peopleradb->isChecked());
    } else {
        this->hideAll(this->ui->layoutFriends);
    }
}

void ImageBrowserPanel::setState()
{
    this->setState(this->state);
}

void ImageBrowserPanel::setState(WindowState state) {
    this->state = state;
    bool isMultiWindow = this->sm->settings->value(SETTINGS_MULTIPANEL).toBool() && !this->isPortraitMode;

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): multiwindow=" << isMultiWindow;

    switch(this->state) {
    case WindowAlbumFriend:
        this->setAlbumsVisible(true);
        this->setImagesVisible(false);

        if (isMultiWindow) {
            this->setFriendsVisible(true);
        } else {
            this->setFriendsVisible(false);
        }
        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): show albums from friends";
        break;
    case WindowAlbumPhoto: // from photo to album
        this->setAlbumsVisible(true);
        this->setImagesVisible(false);

        if (isMultiWindow) {
            this->setFriendsVisible(true);
        } else {
            this->setFriendsVisible(false);
        }
        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): show albums from photos";
        break;
    case WindowFriendAlbum:
        this->setFriendsVisible(true);
        this->setImagesVisible(false);

        if (isMultiWindow) {
            this->setAlbumsVisible(true);
        } else {
            this->setAlbumsVisible(false);
        }
        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): show friends";
        break;
    case WindowPhotoAlbum:
        this->setFriendsVisible(false);
        this->setImagesVisible(true);

        if (isMultiWindow) {
            this->setAlbumsVisible(true);
        } else {
            this->setAlbumsVisible(false);
        }
        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): show photos";
        break;
    default:
        QMessageBox::warning(this, tr("Error occured"), tr("Unknown window state"));
        break;
    }
}

void ImageBrowserPanel::orientationChanged()
{
    QRect geom = QApplication::desktop()->screenGeometry();
    if (geom.width() > geom.height())
        this->isPortraitMode = false;
    else
        this->isPortraitMode = true;
    this->setState();
}

void ImageBrowserPanel::showFriendAlbums(Friend fd)
{
    for (int i = 0; i < ui->listFriends->count(); i++) {
        QListWidgetItem *item = ui->listFriends->item(i);

        QVariant v = item->data(Qt::UserRole);
        if (v.canConvert<Friend>()) {
            Friend oldFd = v.value<Friend>();

            if (oldFd.ownerId == fd.ownerId) {
                ui->listFriends->setCurrentRow(i);
                ownrSelected(item);
                break;
            }
        }
    }
}

void ImageBrowserPanel::showFriendInfo()
{
    QListWidgetItem *item = ui->listFriends->item(ui->listFriends->currentRow());
    if (item != NULL) {
        QVariant v = item->data(Qt::UserRole);
        if (v.canConvert<Friend>()) {
            Friend fd = v.value<Friend>();
            qDebug() << __FILE__ << "emit showFriendInfo";
            emit showFriendInfo(fd);
        }
    }
}

QString ImageBrowserPanel::cutHeader(QString text)
{
    if (text.isEmpty()) {
        return text;
    } else if (text.length() <= 40) {
        return text;
    } else if (text.length() > 40) {
        text.truncate(30);
        text.truncate(text.lastIndexOf(QRegExp("\\s")));
        text = text.trimmed();
        text.append(QString("..."));
        return text;
    }

    return QString::null;
}

void ImageBrowserPanel::updateFeed(QEventFeedList list, QEventFeed::FeedType type, bool isLastUpdate)
{
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): isLastUpdate=" << isLastUpdate;
    if (newsradb->isChecked())
    {
        if (isLastUpdate) {
            this->isFeedUpdate = false;
            buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));
        } else {
            this->isFeedUpdate = true;
            buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
        }
    }
    ui->listNews->clear();
   for (int i = 0; i < list.length(); i++)
   {
       QListWidgetItem *item = new QListWidgetItem(NULL, ui->listNews);
       QPixmap icon = this->photoIcon(list.at(i).attach);
       item->setData(Qt::DecorationRole, icon);
       ((QAbstractItemView*)this->ui->listNews)->setIconSize(icon.size());
       QVariant v;
       v.setValue(list.at(i));
       item->setData(Qt::UserRole, v);
   }
   this->newsradb->setText(tr("News (%1)").arg(this->ui->listNews->count()));
   this->ui->listNews->repaint();
}

void ImageBrowserPanel::showFriendPanel(bool isShow)
{
    qDebug() << __FUNCTION__ << "(): isShow=" << isShow;
    if (isShow)
    {
        this->ui->listFriends->show();
        this->showAll(this->ui->layoutFriendsBottom);
        this->ui->listNews->hide();
        if (this->isFriendsUpdate)
            buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
        else
            buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh",
                                                           QIcon(":/res/general_refresh.png")));
    } else {
        this->ui->listFriends->hide();
        this->hideAll(this->ui->layoutFriendsBottom);
        this->ui->listNews->show();
        if (this->isFeedUpdate)
            buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
        else
            buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh",
                                                           QIcon(":/res/general_refresh.png")));
    }
}

void ImageBrowserPanel::refreshFeed(bool isNeedUpdate)
{
    // news refresh
    if (newsradb->isChecked())
    {
        if (!this->isFeedUpdate) {
            this->isFeedUpdate = true;
            buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
            sm->getFeed(QEventFeed::photoFeed, isNeedUpdate, true);
        } else {
            //TODO: add stopping friends update
        }
    }
}

void ImageBrowserPanel::gotError(QString, QTransport::Action action, bool)
{
    switch(action)
    {
    case QTransport::getListFriendsAction:
        if (!newsradb->isChecked())
        {
            if (this->sm->getNumThreadsFriendUpdate() == 0)
                buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh",
                                                               QIcon(":/res/general_refresh.png")));
            else
                buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
        }
        break;
    case QTransport::getFeedAction:
        if (newsradb->isChecked())
        {
            if (this->sm->getNumThreadsFeedUpdate() == 0)
                buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh",
                                                               QIcon(":/res/general_refresh.png")));
            else
                buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
        }
        break;
    }
}
