#include "qimageviewer.h"

QImageViewer::QImageViewer(QFont textFont, QWidget *parent) :
    QMainWindow(parent)
{

    createMenu();

    scene = new QGraphicsScene(this);

    view = new QMouseTrackingView(scene, this);
    //FIXME
    view->setMinimumSize(50, 50);
    view->setGeometry(QApplication::desktop()->geometry());
    view->setFocusPolicy(Qt::NoFocus);

    // image
    pixmapItem = new QGraphicsPixmapItem();
    pixmapItem->setZValue(1);
    pixmapItem->setCacheMode(QGraphicsPixmapItem::DeviceCoordinateCache);
    scene->addItem(pixmapItem);

    // processing message
    processItem = new QGraphicsTextItem();
    //processItem->setDefaultTextColor(Qt::white);
    processItem->setFont(textFont);
    processItem->setZValue(0);
    scene->addItem(processItem);

    // comments
    comments = new QWidget();
    comments->setFont(textFont);
    QVBoxLayout *layout = new QVBoxLayout(comments);
    comments->setLayout(layout);
    commentsWidget = scene->addWidget(comments);
    this->commentsWidget->show();

    this->setCentralWidget(view);
    windowSize = this->centralWidget()->size();
    commentsWidget->setMinimumWidth(windowSize.width() - 20);
    comments->setMinimumWidth(windowSize.width() - 20);

    //FIXME
#ifdef Q_WS_MAEMO_5
    this->setMinimumSize(QSize(800, 480));
#else
    this->setMinimumSize(QSize(QApplication::desktop()->geometry().width() * 0.8,
                               QApplication::desktop()->geometry().height() * 0.8));
#endif

    connect(view, SIGNAL(mouseDragged(Qt::ArrowType)), this, SLOT(moveImage(Qt::ArrowType)));
    connect(view, SIGNAL(mouseDoubleClick()), this, SLOT(on_actionFull_screen_toggled()));

    // device rotation
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));

#ifdef Q_WS_MAEMO_5
    this->setAttribute(Qt::WA_Maemo5StackedWindow);
    this->setWindowFlags(this->windowFlags() | Qt::Window);
    this->fullScreenButton = new FullScreenExitButton(this);
    this->fullScreenButton->hide();
#endif
    this->setFocus();
}

QImageViewer::~QImageViewer()
{
    this->releaseMouse();
    delete pixmapItem;
    delete processItem;
    delete view;
    delete scene;
}

void QImageViewer::createMenu()
{
    // comments
    QMenu *mmenu = menuBar()->addMenu(tr("Comments"));
    QAction *action = mmenu->addAction(tr("Refresh comments"));
    connect(action, SIGNAL(triggered()), this, SLOT(action_refresh_comments_toggled()));
    action = mmenu->addAction(tr("Add comment"));
    connect(action, SIGNAL(triggered()), this, SLOT(action_add_comment_toggled()));
}

void QImageViewer::loadImage(ServiceMgr *sm, Photo image) {
    this->sm = sm;
    this->currentPhoto = image;
    this->currentImageIndex = -1;

    connect(this->sm, SIGNAL(updatePhoto(QString,QString,QString,PhotoList,bool)),
            this, SLOT(imageListUpdate(QString,QString,QString,PhotoList,bool)));
    connect(this->sm, SIGNAL(updatePhotoCommentList(QString,PhotoCommentList)),
            this, SLOT(gotComments(QString,PhotoCommentList)));
    connect(this->sm, SIGNAL(updateProfile(Friend)),
            this, SLOT(gotProfile(Friend)));

    setWindowTitle(tr(WINDOW_TITLE));
    this->showMaximized();

    this->list = this->sm->getPhotosForAlbum(this->currentPhoto);
    for (int i = 0; i < this->list.length(); i++)
    {
        if (list.at(i).photoId == this->currentPhoto.photoId)
        {
            this->currentImageIndex = i;
            break;
        }
    }

    this->sm->downloadPhotos(image, 1);
    this->sm->getComments(image, false, true);
    this->showImage(false);
}

void QImageViewer::imageListUpdate(QString accountId, QString friendId, QString albumId, PhotoList list, bool)
{
    qDebug() <<__FILE__ <<":" << __LINE__ << ":" << __FUNCTION__ << "() got new images for album " << albumId;
    if (this->currentPhoto.accountId == accountId &&
        this->currentPhoto.ownerId == friendId &&
        this->currentPhoto.albumId == albumId)
        this->list = list;
    else
        return;

    this->showImage(true);
}

void QImageViewer::showImage(bool isNeedCompare)
{
    if (this->currentImageIndex >= 0 && this->currentImageIndex < this->list.length())
    {
        this->setWindowTitle(tr(WINDOW_TITLE) + " (" +
                             QString::number(this->currentImageIndex + 1) + "/" +
                             QString::number(this->list.length()) + ")");
    } else {
        return;
    }
    Photo img = this->list.at(this->currentImageIndex);

    qDebug() << "Try to show image " << img.photoId; // << " from " << img.photo;

    bool isNeedUpdate = false;
    if (isNeedCompare)
    {        
        if (this->currentPhoto.photoId == img.photoId &&
            this->currentPhoto.photo == img.photo &&
            this->currentPhoto.icon == img.icon)
            isNeedUpdate = false;
        else
            isNeedUpdate = true;
    } else {
        isNeedUpdate = true;
    }

    qDebug() << "isNeedUpdate=" << isNeedUpdate << " isNeedCompare=" << isNeedCompare;

    QSize imageSize = this->pixmapItem->pixmap().size();
    if (isNeedUpdate)
    {
        this->view->verticalScrollBar()->setValue(0);
        this->view->horizontalScrollBar()->setValue(0);

        //try to open image from file
        if (!img.photo.isEmpty())
        {
            processItem->hide();
            pixmapItem->show();
            QImage image(img.photo);
            if (image.isNull()) {
                QMessageBox::information(this, tr("Image Viewer"),
                                         tr("Cannot load %1.").arg(img.photo));
            } else {
                // Set/scale image size into screen
                imageSize = image.size();                

                if (imageSize.height() > windowSize.height() ||
                    imageSize.width() > windowSize.width())
                {
                    imageSize.scale(windowSize, Qt::KeepAspectRatio);
                    image = image.scaled(imageSize);
                }

                imageSize = image.size();

                // Make QPixmap
                QPixmap p = QPixmap::fromImage(image);
                this->pixmapItem->setPixmap(p);
#ifdef Q_WS_MAEMO_5
                //            sharing_dialog_with_file (osso,
                //                                      NULL,
                //                                      img.photo);
#endif
            }
        } else {
            // show image preview
            QImage image(img.icon);
            if (image.isNull())
            {
                pixmapItem->hide();
            } else {
                this->pixmapItem->setPixmap(QPixmap::fromImage(image));
                imageSize = image.size();
            }
            this->sm->downloadPhotos(this->list.at(this->currentImageIndex), 0);

            processItem->show();
            processItem->setPlainText(tr("Downloading image..."));
        }
    }

    QPoint loadedArea(0,0);
    QFontMetrics fm(processItem->font());

    // set image position
    if (pixmapItem->isVisible())
    {
        QPoint center;
        center.setX(0);
        center.setY(processItem->isVisible() ? fm.height() * 2 : 0);
        pixmapItem->setPos(center);
        qDebug() << "Image size: " << imageSize << " windowSize: " << windowSize << " ImagePos: " << center;
        loadedArea.setX(imageSize.width());
        loadedArea.setY(loadedArea.y() + imageSize.height() + center.y());
    }

    // set message position
    if (processItem->isVisible())
    {
        QPoint textCenter;
        textCenter.setX(0);
        textCenter.setY(0);
        processItem->setPos(textCenter);
        if (processItem->textWidth() > loadedArea.x())
            loadedArea.setX(processItem->textWidth());
    }

    // set comments position
    QRectF commGeom = this->commentsWidget->geometry();
    commGeom.setLeft(0);
    commGeom.setTop(loadedArea.y());
    if (commGeom.height() < comments->geometry().height())
        commGeom.setHeight(comments->geometry().height());
    this->commentsWidget->setGeometry(commGeom);
    this->comments->setGeometry(commGeom.toRect());

    if (commGeom.width() > loadedArea.x())
        loadedArea.setX(commGeom.width());
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): commentsWidget geometry: " << this->commentsWidget->geometry() <<
            "; comments geometry=" << this->comments->geometry();

    //center objects horizontally
    if (pixmapItem->isVisible() && imageSize.width() < loadedArea.x())
    {
        QPointF pos = pixmapItem->pos();
        pos.setX((loadedArea.x() - imageSize.width()) / 2);
        pixmapItem->setPos(pos);
    }

    if (processItem->isVisible() && processItem->textWidth() < loadedArea.x())
    {
        QPointF pos = processItem->pos();
        pos.setX((loadedArea.x() - processItem->textWidth()) / 2);
        processItem->setPos(pos);
    }

    this->scene->setSceneRect(this->scene->itemsBoundingRect());
    this->view->updateGeometry();
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "() scene=" << this->scene->sceneRect() << "; view=" << this->view->geometry();
}

void QImageViewer::close()
{
    QMainWindow::close();
    this->destroy();
}

void QImageViewer::resizeEvent(QResizeEvent * event)
{
    QMainWindow::resizeEvent(event);
    windowSize = this->centralWidget()->size();

    view->setGeometry(QApplication::desktop()->screenGeometry());

    // Update scene and view sizes
    //scene->setSceneRect(view->rect());

    this->showImage(true);
}

void QImageViewer::moveImage(Qt::ArrowType direction)
{
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "()";
    if (this->list.isEmpty())
        return;

    bool needUpdate = false;
    switch (direction)
    {
    case Qt::LeftArrow:
        this->currentImageIndex--;       
        if (this->currentImageIndex < 0)
            this->currentImageIndex = this->list.length() - 1;
        needUpdate = true;
        break;
    case Qt::RightArrow:
        this->currentImageIndex++;
        if (this->currentImageIndex >= this->list.length())
            this->currentImageIndex = 0;
        needUpdate = true;
        break;
    default:
        break;
    }
    if (needUpdate)
    {
        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "() move image!!!";
        this->currentPhoto = this->list.at(currentImageIndex);
        this->showImage(false);
        this->sm->downloadPhotos(this->currentPhoto, 1);
        this->sm->getComments(this->currentPhoto, false, true);
    }
}

void QImageViewer::on_actionFull_screen_toggled()
{
    bool isFullScreen = windowState() & Qt::WindowFullScreen;
    if (isFullScreen) {
#ifdef Q_WS_MAEMO_5
        fullScreenButton->hide();
#endif
        showMaximized();
    } else {
        showFullScreen();
#ifdef Q_WS_MAEMO_5
        fullScreenButton->show();
#endif
    }
}

void QImageViewer::orientationChanged()
{
    windowSize = this->size();

    view->setGeometry(QApplication::desktop()->screenGeometry());

    // Update scene and view sizes
    //scene->setSceneRect(view->rect());

    this->showImage(true);
}

void QImageViewer::gotComments(QString photoId, PhotoCommentList list)
{
    if (this->currentPhoto.photoId != photoId)
        return;

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): photoId=" << photoId;

    while(this->comments->layout()->count() > 0)
    {
        QLayoutItem *delItem = this->comments->layout()->itemAt(0);
        delItem->widget()->hide();
        this->comments->layout()->removeItem(delItem);
        delete delItem;
    }

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): items=" << this->comments->layout()->count();
    int commentsHeight = 0;
    for (int i = 0; i < list.length(); i++)
    {
        QCommentWidget *comm = new QCommentWidget(comments->font(), comments);
        comm->setPhotoComment(list.at(i));
        comm->setSenderProfile(this->sm->getFriend(list.at(i)));
        comm->setWidth(comments->width() - 20);
        //comm->setWidth(this->comments->geometry().width());

//        QGraphicsProxyWidget *widg = new QGraphicsProxyWidget(commentsWidget);
//        widg->setData(Qt::DisplayRole, comm->dateTime());
//        widg->setWidget(comm);

        int position = -1;
        for(int j = 0; j < comments->layout()->count(); j++)
        {
            if (comments->layout()->itemAt(j)->widget() != NULL)
            {
                //qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): compare date " << comm->dateTime().toString(DATE_TIME_FORMAT) << " before " << ((QGraphicsProxyWidget *)comments->itemAt(j))->data(Qt::DisplayRole).toDateTime().toString(DATE_TIME_FORMAT);
                if (((QCommentWidget *)comments->layout()->itemAt(j)->widget())->dateTime() < comm->dateTime())
                {
                    //qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): set date " << comm->dateTime().toString(DATE_TIME_FORMAT) << " before " << ((QGraphicsProxyWidget *)comments->itemAt(j))->data(Qt::DisplayRole).toDateTime().toString(DATE_TIME_FORMAT);
                    position = j;
                    break;
                }
            }
        }

        if (position < 0)
        {
            //qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): position < 0, count=" << comments->count();
            position = comments->layout()->count();
        }

        comments->layout()->addWidget(comm);
        //comm->show();

        //comm->adjustSize();
        comm->updateGeometry();
        commentsHeight += comm->heightForWidth(comm->size().width());
    }

    QRect geom = this->comments->geometry();
    geom.setHeight(commentsHeight);
    this->comments->setGeometry(geom);
    this->comments->updateGeometry();
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): geometry=" <<
            this->comments->geometry() << "; count=" << comments->layout()->count();
    this->showImage(true);
}

void QImageViewer::keyPressEvent(QKeyEvent *event)
{
    if (event->key()==Qt::Key_Escape) {
        this->close();
        return;
    }
    switch(event->key()){
        case Qt::Key_Left:
            moveImage(Qt::LeftArrow);
            break;
        case Qt::Key_Right:
            moveImage(Qt::RightArrow);
            break;
    }
}

void QImageViewer::action_refresh_comments_toggled()
{
    this->sm->getComments(this->currentPhoto, true, true);
}

void QImageViewer::action_add_comment_toggled()
{
    QAddComment *dlg = new QAddComment(this->currentPhoto, this->sm, this);
    dlg->show();
}

void QImageViewer::gotProfile(Friend profile)
{
    QString service = this->sm->account(profile.accountId)->serviceName();
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): profileId=" << profile.id;
    for (int i = 0; i < this->comments->layout()->count(); i++)
    {
        PhotoComment comm = ((QCommentWidget *)comments->layout()->itemAt(i)->widget())->photoComment();
        QString curService = this->sm->account(comm.accountId)->serviceName();
        if (comm.senderId.compare(profile.id) == 0 &&
            service.compare(curService) == 0)
        {
            ((QCommentWidget *)(comments->layout()->itemAt(i))->widget())->setSenderProfile(profile);
        }
    }
}
