#include "servicemgr.h"
#include "qtransport.h"

ServiceMgr::ServiceMgr(QObject *parent) :
        QObject(parent), friendsUpdate(0), albumsUpdate(0), photosUpdate(0),
        inboxMessagesUpdate(0), outboxMessagesUpdate(0)
{
    settings = new QSettings(Utils::getHomeDir() + QString(FILE_SETTINGS), QSettings::NativeFormat, this);
    if (!settings->contains(SETTINGS_PROXY))
        settings->setValue(SETTINGS_PROXY, this->SystemProxy);
    if (!settings->contains(SETTINGS_MULTIPANEL))
        settings->setValue(SETTINGS_MULTIPANEL, true);
    loadAccounts();

    qDebug()<< "loaded " << this->accounts.length() << " accounts";

    updateDriverSettings();

    qRegisterMetaType<FriendList>("FriendList");
    qRegisterMetaType<AlbumList>("AlbumList");
    qRegisterMetaType<PhotoList>("PhotoList");
    qRegisterMetaType<MessageList>("MessageList");
    qRegisterMetaType<Friend>("Friend");
    qRegisterMetaType<QErrorMessage>("QErrorMessage");
    qRegisterMetaType<PhotoCommentList>("PhotoCommentList");
    qRegisterMetaType<QTransport::Action>("QTransport::Action");

    for(int i = 0; i < this->accounts.length(); i++) {
        this->connectToTransport(this->accounts.at(i));
        // load profiles for each account
        this->accounts[i]->getProfile(false);
    }

    connect(this, SIGNAL(updateAccounts(AccountList)), this, SLOT(storeAccounts(AccountList)));

}

void ServiceMgr::connectToTransport(Account *account)
{
    connect(account->transport, SIGNAL(friendsReceived(QString, FriendList, bool)),
            this, SLOT(gotFriends(QString, FriendList, bool)));
    connect(account->transport, SIGNAL(inboxMessagesReceived(QString,MessageList, bool)),
                     this, SLOT(gotInboxMessageList(QString,MessageList,bool)));
    connect(account->transport, SIGNAL(albumsReceived(QString, QString, AlbumList, bool)),
            this, SLOT(gotAlbumList(QString, QString, AlbumList, bool)));
    connect(account->transport, SIGNAL(photosReceived(QString, QString, QString, PhotoList, bool)),
            this, SLOT(gotPhotoList(QString, QString, QString, PhotoList, bool)));
    connect(account->transport, SIGNAL(profileReceived(QString,QString,Friend)),
            this, SLOT(gotProfile(QString,QString,Friend)));
    connect(account->transport, SIGNAL(errorOccurred(QString,QErrorMessage,QTransport::Action)),
            this, SLOT(gotErrorMsg(QString,QErrorMessage,QTransport::Action)));
    connect(account->transport, SIGNAL(commentsReceived(QString,QString,QString,QString,PhotoCommentList)),
            this, SLOT(gotComments(QString,QString,QString,QString,PhotoCommentList)));
}

ServiceMgr::ServiceMgr(const ServiceMgr &src) : QObject(src.parent())
{
    this->accounts = src.accounts;
    this->threads = src.threads;
    this->settings = src.settings;
    this->friendsUpdate = src.friendsUpdate;
    this->albumsUpdate = src.albumsUpdate;
    this->photosUpdate = src.photosUpdate;
}

ServiceMgr::~ServiceMgr()
{
    for (int i = 0; i < this->accounts.length(); i++)
    {
        delete this->accounts.at(i);
    }
}

void ServiceMgr::storeAccounts(AccountList)
{
    QDomDocument out("MyDoc");
    QDomElement profile = out.createElement(NODE_ACCOUNTS_ROOT);
    out.appendChild(profile);
    for (int i = 0; i < this->accounts.length(); i++) {
        Account *curAcc = this->accounts[i];
        curAcc->saveAccount();
        QDomElement anode = out.createElement(NODE_ACCOUNT);
        QDomText t = out.createTextNode(curAcc->transport->accountId);
        anode.appendChild(t);
        profile.appendChild(anode);
    }

    QFile file(Utils::getHomeDir() + QString(FILE_ACCOUNTS_LIST));
    qDebug() << file.fileName();
    if (file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate) == false) {
        qDebug() << "can't open file: " + file.errorString();
    }
    file.write(out.toByteArray());
    file.close();
}

void ServiceMgr::loadAccounts()
{
    QDomDocument doc;
    QFile file(Utils::getHomeDir() + QString(FILE_ACCOUNTS_LIST));
    if (file.open(QIODevice::ReadOnly) == false) {
        qDebug() << "can't open file: " + file.errorString();
        return;
    }

    doc.setContent(&file);
    file.close();

    QDomElement doc_elem = doc.documentElement();
    QDomNodeList lst = doc_elem.elementsByTagName("profile");

    qDebug() << "accounts:" << lst.count();
    for (int i = 0; i < lst.count(); i++)
    {
        QString accId = lst.at(i).firstChild().nodeValue();
        qDebug() << "Load account:" << accId;
        Account *prf = Account::loadAccount(accId);

        if (prf != NULL)
            this->accounts.append(prf);
    }
}

void ServiceMgr::cleanThreads() {
    QList<QString> keys;
    QHashIterator<QString, QFuture<void> > i(this->threads);
    while (i.hasNext()) {
        i.next();
        if (i.value().isFinished())
            keys.append(i.key());
    }

    for(int i = 0; i < keys.length(); i++) {
        this->threads.remove(keys[i]);
    }
}

void ServiceMgr::updateDriverSettings() {
    QString curProxyHost = QString("");
    int curProxyPort = 0;

    ProxyType proxy = (ProxyType)settings->value(SETTINGS_PROXY).toInt();

    switch (proxy) {
    case ServiceMgr::NoProxy:
        break;
    case ServiceMgr::UserProxy:
        curProxyHost = settings->value(SETTINGS_PROXYHOST).toString();
        curProxyPort = settings->value(SETTINGS_PROXYPORT).toInt();
        break;
    case ServiceMgr::SystemProxy: {
        QNetworkProxyQuery npq(QUrl(QLatin1String("http://maemo.org")));
        QList<QNetworkProxy> proxyList = QNetworkProxyFactory::systemProxyForQuery(npq);
        if (proxyList.length() > 0 && proxyList.at(0) != QNetworkProxy::NoProxy)
        {
    	    curProxyHost = proxyList.at(0).hostName();
    	    curProxyPort = proxyList.at(0).port();
        } else {
    	    QString var(getenv("http_proxy"));
    	    if (var.isNull()) {
    		var = getenv("all_proxy");
    	    }
    	    QRegExp regex("(http://)?(.*):(\\d*)/?");
    	    int pos = regex.indexIn(var);
    	    
    	    if(pos > -1) {
    		curProxyHost = regex.cap(2);
    		curProxyPort = regex.cap(3).toInt();
    	    }
        }
        if (curProxyHost.isEmpty())
            curProxyHost = QString::null;
        break;
    }
    default:
        qWarning() << "Unknown type of proxy";
    }

    if (curProxyHost.isNull())
        qDebug() << "using system proxy";
    else
        qDebug() << "using proxy: \""<< curProxyHost << "\":" << curProxyPort;

    for (int i = 0; i < this->accounts.length(); i++) {
        this->accounts.at(i)->setProxy(curProxyHost, curProxyPort);
    }

}

FriendList ServiceMgr::getFriends(bool isNeedUpdate, bool useSignal){
    FriendList ret;

    this->cleanThreads();

    for (int i = 0; i < this->accounts.length(); i++) {
        FriendList curList = this->accounts.at(i)->getFriendList();
        ret.append(curList);
        if (isNeedUpdate || curList.isEmpty()) {
            QString threadId = QString("getFriends") + this->accounts[i]->transport->accountId;
            // check that thread shouldn't already running
            if (!this->threads.contains(threadId) && this->accounts.at(i)->isNetworkEnabled) {
                // add thread to list
                QFuture<void> status = QtConcurrent::run(this->accounts[i]->transport, &QTransport::getFriends);
                this->threads.insert(threadId, status);
                this->friendsUpdate++;
            }
        }
    }

    if (useSignal)
        emit this->updateFriends(ret, this->friendsUpdate > 0 ? false: true);

    return ret;
}

void ServiceMgr::gotFriends(QString accountid, FriendList list, bool isLastUpdate){

    // create list of merged lists of friends
    FriendList ret;
    for (int i = 0; i < this->accounts.length(); i++) {
        if (this->accounts[i]->transport->accountId == accountid) {
            this->accounts[i]->setFriendList(list);
            ret.append(list);
            qDebug() << "Add to friends " << list.length() << ". Total:" << ret.length();
        }else{
            FriendList lst = this->accounts[i]->getFriendList();
            ret.append(lst);
            qDebug() << "Add to friends " << lst.length() << ". Total:" << ret.length();
        }
    }

    if (isLastUpdate)
        this->friendsUpdate--;
    qDebug() << "islastUpdate=" << isLastUpdate << ", friendsUpdate=" << friendsUpdate;

    emit updateFriends(ret, (this->friendsUpdate > 0 ? false: true));
}

AlbumList ServiceMgr::getMyAlbums(bool isNeedUpdate, bool useSignal) {
    AlbumList ret;

    // generate list of albums from all accounts
    for (int i = 0; i < this->accounts.length(); i++) {
        if (this->accounts.at(i)->getAlbumList().isEmpty() ||
            isNeedUpdate)
            this->accounts.at(i)->setAlbumList(this->getAlbums(this->accounts.at(i)->getProfile(false), isNeedUpdate, useSignal));
        ret.append(this->accounts.at(i)->getAlbumList());
    }

    if (useSignal)
        emit this->updateAlbumList(QString(), ret, this->albumsUpdate > 0 ? false: true);

    return ret;
}

AlbumList ServiceMgr::getAlbums(Friend curFriend, bool isNeedUpdate, bool useSignal){
    AlbumList ret;

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): Load album list from cache for " << curFriend.id  << " from " << curFriend.accountId << " needUpdate=" << isNeedUpdate << "; this->albumsUpdate=" << this->albumsUpdate;

    ret = curFriend.getAlbumList();

    this->cleanThreads();

    if (isNeedUpdate || ret.isEmpty()) {
        for (int i = 0; i < this->accounts.length(); i++) {
            if ((curFriend.accountId == this->accounts[i]->transport->accountId) &&
                this->accounts.at(i)->isNetworkEnabled) {
                qDebug() << "Download album list for " << curFriend.id;

                if (this->threads.contains(QString("albums") + curFriend.id))
                    break;

                QFuture<void> status = QtConcurrent::run(this->accounts[i]->transport, &QTransport::getAlbums, curFriend.id);
                this->threads.insert(QString("albums") + curFriend.id, status);
                this->albumsUpdate++;
                break;
            }
        }
    }

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): this->albumsUpdate=" << this->albumsUpdate;
    if (useSignal)
        emit this->updateAlbumList(curFriend.id, ret, this->albumsUpdate > 0 ? false: true);

    return ret;
}

void ServiceMgr::gotAlbumList(QString accountId, QString owner, AlbumList list, bool isLastUpdate) {

    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): isLastUpdate=" << isLastUpdate;

    // store album list to cache
    Friend::setAlbumList(list, accountId, owner);

    // check that it is not a my album list
    for (int i = 0; i < this->accounts.length(); i++) {
        if (this->accounts[i]->transport->accountId == accountId &&
            this->accounts[i]->getProfile(false).id == owner) {
            // if it is my album list then store, generate new full list and send it
            this->accounts[i]->setAlbumList(list);
            AlbumList ret;
            for (int j = 0; j < this->accounts.length(); j++) {
                ret.append(this->accounts[j]->getAlbumList());
            }

            if (isLastUpdate)
                this->albumsUpdate--;
            qDebug() << "islastUpdate=" << isLastUpdate << ", albumsUpdate=" << albumsUpdate;

            emit updateAlbumList(QString(), ret, this->albumsUpdate > 0 ? false: true);
            return;
        }
    }

    if (isLastUpdate) {
        this->albumsUpdate--;
        if (this->albumsUpdate < 0)
            this->albumsUpdate = 0;
    }
    qDebug() << "islastUpdate=" << isLastUpdate << ", albumsUpdate=" << albumsUpdate;

    emit updateAlbumList(owner, list, this->albumsUpdate > 0 ? false: true);
}

PhotoList ServiceMgr::getPhotos(Album album, bool isNeedUpdate, bool useSignal) {
    PhotoList ret;

    qDebug() << "Load photo list from cache for " << album.albumId;

    ret = album.getPhotoList();

    this->cleanThreads();

    if (isNeedUpdate || ret.isEmpty()) {
        for (int i = 0; i < this->accounts.length(); i++) {
            if (album.accountId == this->accounts[i]->transport->accountId &&
                this->accounts[i]->isNetworkEnabled) {
                qDebug() << "Download photo list for " << album.albumId;
                if (this->threads.contains(Utils::getPhotoListKey(album)) || !this->accounts.at(i)->isNetworkEnabled)
                    return ret;
                QFuture<void> status = QtConcurrent::run(this->accounts[i]->transport, &QTransport::getPhotos, album.ownerId, album.albumId);
                this->threads.insert(Utils::getPhotoListKey(album), status);
                this->photosUpdate++;
            }
        }
    }

    if (useSignal)
        emit this->updatePhotoList(album.albumId, ret, this->photosUpdate > 0? false: true);

    return ret;
}

void ServiceMgr::gotPhotoList(QString accountId, QString friendId, QString albumId, PhotoList list, bool isLastUpdate)
{
    Album::setPhotoList(list, accountId, friendId, albumId);

    if (isLastUpdate)
        this->photosUpdate--;
    if (this->photosUpdate < 0)
        this->photosUpdate = 0;
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ <<"() islastUpdate=" << isLastUpdate << ", photosUpdate=" << photosUpdate;

    emit updatePhotoList(albumId, list, photosUpdate > 0 ? false : true);
    emit updatePhoto(accountId, friendId, albumId, list, isLastUpdate);
}

MessageList ServiceMgr::getInboxMessages(bool isNeedUpdate, bool useSignal) {
    MessageList ret;

    qDebug() << __FILE__ << ":" << __FUNCTION__;

    this->cleanThreads();

    for (int i = 0; i < this->accounts.length(); i++) {
        MessageList curList = this->accounts.at(i)->getInboxMessageList();
        ret.append(curList);
        if (isNeedUpdate || curList.isEmpty()) {
            QString threadId = QString("getInbox") + this->accounts[i]->transport->accountId;
            // check that thread shouldn't already running
            if (!this->threads.contains(threadId) && this->accounts.at(i)->isNetworkEnabled) {
                // add thread to list
                QFuture<void> status = QtConcurrent::run(this->accounts[i]->transport, &QTransport::getInbox);
                this->threads.insert(threadId, status);
                this->inboxMessagesUpdate++;
            }
        }
    }

    if (useSignal)
        emit this->updateInboxMessageList(ret, this->inboxMessagesUpdate > 0 ? false: true);

    return ret;
}

void ServiceMgr::gotInboxMessageList(QString accountId, MessageList list, bool isLastUpdate) {
    // create list of merged lists of messages
    MessageList ret;

    qDebug() << __FILE__ << "GET INBOX LIST: " << list.length();

    for (int i = 0; i < this->accounts.length(); i++) {
        if (this->accounts[i]->transport->accountId == accountId) {
            this->accounts[i]->setInboxMessageList(list);
            ret.append(list);
            qDebug() << "Add to messages " << list.length() << ". Total:" << ret.length();
        } else {
            MessageList lst = this->accounts[i]->getInboxMessageList();
            ret.append(lst);
            qDebug() << "Add to messages " << lst.length() << ". Total:" << ret.length();
        }
    }

    if (isLastUpdate)
        this->inboxMessagesUpdate--;
    qDebug() << "islastUpdate=" << isLastUpdate << ", friendsUpdate=" << inboxMessagesUpdate;

    emit updateInboxMessageList(ret, (this->inboxMessagesUpdate > 0 ? false: true));
}

MessageList ServiceMgr::getOutboxMessages(bool isNeedUpdate, bool useSignal) {
    MessageList ret;

    qDebug() << __FILE__ << ":" << __FUNCTION__;

    this->cleanThreads();

    for (int i = 0; i < this->accounts.length(); i++) {
        MessageList curList = this->accounts.at(i)->getOutboxMessageList();
        ret.append(curList);
        if (isNeedUpdate || curList.isEmpty()) {
            QString threadId = QString("getOutbox") + this->accounts[i]->transport->accountId;
            // check that thread shouldn't already running
            if (!this->threads.contains(threadId) && this->accounts.at(i)->isNetworkEnabled) {
                // add thread to list
                QFuture<void> status = QtConcurrent::run(this->accounts[i]->transport, &QTransport::getOutbox);
                this->threads.insert(threadId, status);
                this->outboxMessagesUpdate++;
            }
        }
    }

    if (useSignal)
        emit this->updateOutboxMessageList(ret, this->outboxMessagesUpdate > 0 ? false: true);

    return ret;
}

void ServiceMgr::gotOutboxMessageList(QString accountId, MessageList list, bool isLastUpdate) {
    // create list of merged lists of messages
    MessageList ret;

    qDebug() << __FILE__ << "GET INBOX LIST: " << list.length();

    for (int i = 0; i < this->accounts.length(); i++) {
        if (this->accounts[i]->transport->accountId == accountId) {
            this->accounts[i]->setOutboxMessageList(list);
            ret.append(list);
            qDebug() << "Add to messages " << list.length() << ". Total:" << ret.length();
        } else {
            MessageList lst = this->accounts[i]->getOutboxMessageList();
            ret.append(lst);
            qDebug() << "Add to messages " << lst.length() << ". Total:" << ret.length();
        }
    }

    if (isLastUpdate)
        this->outboxMessagesUpdate--;
    qDebug() << "islastUpdate=" << isLastUpdate << ", friendsUpdate=" << outboxMessagesUpdate;

    emit updateOutboxMessageList(ret, (this->outboxMessagesUpdate > 0 ? false: true));
}

PhotoCommentList ServiceMgr::getComments(Photo curPhoto, bool isNeedUpdate, bool useSignal)
{
    PhotoCommentList ret;

    qDebug() <<__FILE__ << ":" << __LINE__ << ":" << __FUNCTION__<< "(): load comments for " << curPhoto.photoId;

    ret = curPhoto.getPhotoComments();

    this->cleanThreads();

    if (isNeedUpdate || ret.isEmpty()) {
        for (int i = 0; i < this->accounts.length(); i++) {
            if (curPhoto.accountId == this->accounts[i]->transport->accountId) {
                qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__<< "(): Download comments for " << curPhoto.photoId;
                if (this->threads.contains(Utils::getPhotoCommentListKey(curPhoto)) || !this->accounts.at(i)->isNetworkEnabled)
                    return ret;
                QFuture<void> status = QtConcurrent::run(this->accounts[i]->transport, &QTransport::getPhotoComments, curPhoto);
                this->threads.insert(Utils::getPhotoCommentListKey(curPhoto), status);
            }
        }
    }

    if (useSignal)
        emit this->updatePhotoCommentList(curPhoto.photoId, ret);

    return ret;
}

void ServiceMgr::gotComments(QString accountId, QString ownerId, QString albumId, QString photoId, PhotoCommentList list)
{
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): photoId=" << photoId << " has comments=" << list.length();
    Photo::setPhotoComments(list, accountId, ownerId, albumId, photoId);
    emit this->updatePhotoCommentList(photoId, list);
}

bool ServiceMgr::sendComment(Photo curPhoto, QString comment)
{
    for (int i = 0; i < this->accounts.length(); i++)
    {
        if (curPhoto.accountId == this->accounts.at(i)->transport->accountId)
        {
            return this->accounts.at(i)->transport->sendPhotoComment(curPhoto, comment);
        }
    }

    return false;
}

void ServiceMgr::gotProfile(QString accountId, QString reqOwnerId, Friend profile) {
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "(): reqOwnerId=" << reqOwnerId;
    if (reqOwnerId.isEmpty())
    {
        for (int i = 0; i < this->accounts.length(); i++)
        {
            if (this->accounts.at(i)->transport->accountId.compare(accountId) == 0)
            {
                this->accounts.at(i)->setProfile(profile);
                break;
            }
        }

        emit updateAccounts(this->accounts);
        return;
    }

    profile.storeData();

    emit this->updateProfile(profile);
}

AccountList ServiceMgr::getAccounts() {
    // check that all accounts has profiles
    for (int i = 0; i < this->accounts.length(); i++) {
        this->accounts.at(i)->getProfile(false);
    }

    return this->accounts;
}

void ServiceMgr::addAccount(QString accountName, DriverInfo *driver)
{
    Account *newAcc = new Account(driver->library, accountName);

    if (newAcc != NULL && newAcc->ready()) {
        this->connectToTransport(newAcc);
        this->accounts.append(newAcc);
        this->updateDriverSettings();

        this->updateAccounts(this->accounts);
    }
}

Account *ServiceMgr::account(QString accountId)
{
    for (int i = 0; i < this->accounts.length(); i++)
        if (this->accounts.at(i)->accountId().compare(accountId) == 0)
            return this->accounts.at(i);

    return NULL;
}

void ServiceMgr::deleteAccount(Account *oldAcc)
{
    int i = -1;
    for (int j = 0; j < this->accounts.length(); j++) {
        if (oldAcc->transport->accountId == this->accounts.at(j)->transport->accountId)
            i = j;
    }

    if (i >= 0) {
        this->accounts.removeAt(i);
        QDir dir(Utils::getAccountDir(oldAcc->transport->accountId));

        delete oldAcc;
        qDebug() << "Try to remove dir " << dir.path();
        Utils::RemoveDirectory(dir);
        //execute accounts update process
        this->updateAccounts(this->accounts);
    }
}

QList<DriverInfo*> ServiceMgr::getDrivers() {
    QList<DriverInfo*> ret;

    QDir dir;
    dir.setFilter(QDir::Dirs|QDir::NoDotAndDotDot);
    dir.setPath(Utils::getDriversDir());
    QFileInfoList lst = dir.entryInfoList();

    qDebug() << "Found " << lst.length() << " entries at " << dir.path();

    for (int i = 0; i < lst.length(); i++) {
        QFileInfo dir = lst.at(i);
        if (dir.isDir() && dir.fileName() != "." && dir.fileName() != "..")
        {
            QDir files;
            files.setFilter(QDir::Files);
            files.setNameFilters(QStringList(QString("*.so")));
            files.setPath(dir.filePath());
            QFileInfoList lstFiles = files.entryInfoList();
            for (int j = 0; j < lstFiles.length(); j++) {
                QFileInfo file = lstFiles.at(j);
                if (file.isFile()) {
                    DriverInfo *curDr = QTransport::getDriverInfo(file.filePath());
                    if (curDr != NULL) {
                        qDebug() << "found driver " << curDr->name;
                        ret.append(curDr);
                    }
                }
            }
        }
    }

    return ret;
}

PhotoList ServiceMgr::getPhotosForAlbum(Photo curPhoto) {
    for (int i = 0; i < this->accounts.length(); i++) {
        if (this->accounts.at(i)->transport->accountId == curPhoto.accountId) {
            return this->getPhotosForAlbum(this->accounts.at(i), curPhoto);
        }
    }

    PhotoList ret;
    return ret;
}

PhotoList ServiceMgr::getPhotosForAlbum(Account *curAcc, Photo curPhoto) {
    PhotoList ret;

    AlbumList albums;

    // check that it can be my album
    if (curAcc->getProfile(false).id == curPhoto.ownerId) {
        albums = curAcc->getAlbumList();
    } else {
        FriendList friends;
        friends = curAcc->getFriendList();

        for (int i = 0; i < friends.length(); i++) {
            if (friends.at(i).id == curPhoto.ownerId) {
                albums = friends[i].getAlbumList();
                break;
            }
        }
    }

    for (int i = 0; i < albums.length(); i++) {
        if (albums.at(i).albumId == curPhoto.albumId) {
            return albums[i].getPhotoList();
        }
    }

    return ret;
}

void ServiceMgr::downloadPhotos(Photo startPhoto, int nearest) {
    qDebug() << "Start to download photo images from " << startPhoto.photoId << " for " << startPhoto.albumId;

    this->cleanThreads();

    for (int i = 0; i < this->accounts.length(); i++) {
        if (this->accounts.at(i)->transport->accountId == startPhoto.accountId && this->accounts.at(i)->isNetworkEnabled) {
            PhotoList photos = this->getPhotosForAlbum(this->accounts.at(i), startPhoto);
            if (!this->threads.contains(Utils::getPhotoListKey(photos)))
            {
                QFuture<void> ret = QtConcurrent::run(this->accounts.at(i)->transport, &QTransport::downloadPhotoList, photos, startPhoto, nearest);
                this->threads.insert(Utils::getPhotoListKey(photos), ret);
                this->photosUpdate++;
            }

            // download comments for photos
            this->getComments(startPhoto, false, true);
            break;
        }
    }
}

void ServiceMgr::gotErrorMsg(QString accountId, QErrorMessage msg, QTransport::Action acc) {
    QString prefix(tr("During request "));
    QString comment;

    switch (acc) {
    case QTransport::getListFriendsAction:
        this->friendsUpdate--;
        prefix += tr("\"Update list of friends\"");
        break;
    case QTransport::deleteFriendAction:
        prefix += tr("\"Delete friend from list\"");
        break;
    case QTransport::getListAlbumsAction:
        this->albumsUpdate--;
        prefix += tr("\"Update list of albums\"");
        break;
    case QTransport::getListPhotosAction:
        this->photosUpdate--;
        prefix += tr("\"Update list of images\"");
        break;
    case QTransport::getProfileAction:
        prefix += tr("\"Update user profile\"");
        comment = tr("Account swiched to offline mode. Please check network settings and try again.");
        break;
    case QTransport::getSettingsAction:
        prefix += tr("\"Get settings\"");
        break;
    case QTransport::setSettingsAction:
        prefix += tr("\"Set settings\"");
        break;
    case QTransport::getPhotoAction:
        prefix += tr("\"Download image\"");
        break;
    case QTransport::uploadPhotoAction:
        prefix += tr("\"Upload image\"");
    case QTransport::getCommentsAction:
        prefix += tr("\"Get comments\"");
        break;
    case QTransport::sendCommentAction:
        prefix += tr("\"Send comment\"");
        break;
    case QTransport::getInboxMessagesAction:
        prefix += tr("\"Get message inbox\"");
        break;
    case QTransport::getOutboxMessagesAction:
        prefix += tr("\"Get message outbox\"");
        break;
    case QTransport::sendMessageAction:
        prefix += tr("\"Send message\"");
        break;
    case QTransport::readMessageAction:
        prefix += tr("\"Mark message as read\"");
        break;
    case QTransport::deleteMessageAction:
        prefix += tr("\"Delete message\"");
        break;
    case QTransport::getListMethodsAction:
        prefix += tr("\"Get list of registered methods\"");
        break;
    }

    prefix += tr(" from account \"%1\" the error %2 was caused: ").arg(accountId).arg(QString::number(msg.code)) + msg.text + ".";
    if (!msg.comment.isEmpty())
        prefix += " " + msg.comment + ".";
    if (!comment.isEmpty())
        prefix += " " + comment;

    emit this->errorOccured(prefix, acc);
}


void ServiceMgr::clearCache()
{
    QString path;
    bool ret;
    for (int i = 0; i < this->accounts.length(); i++)
    {
        DriverInfo *info = this->accounts.at(i)->transport->getDriverInfo();
        // remove friend icons
        path = Utils::getFriendsIconDir(info->name);
        QDir dir(path);
        ret = Utils::RemoveFiles(dir, "*");
        qDebug() << "Remove " << path << " returned " << ret;

        // remove album icons
        path = Utils::getAlbumsIconDir(info->name);
        dir = QDir(path);
        ret = Utils::RemoveFiles(dir, "*");
        qDebug() << "Remove " << path << " returned " << ret;

        // remove friend list
        path = Utils::getAccountDir(this->accounts.at(i)->transport->accountId);
        dir = QDir(path);
        ret = Utils::RemoveFiles(dir, FILE_FRIENDS_DATA);
        qDebug() << "Remove " << path << " + " << FILE_FRIENDS_DATA << " returned " << ret;
        ret = Utils::RemoveFiles(dir, QString(FILE_ALBUMS_DATA).arg("*"));
        qDebug() << "Remove " << path << " + " << QString(FILE_ALBUMS_DATA).arg("*") << " returned " << ret;
        ret = Utils::RemoveFiles(dir, QString(FILE_PHOTOS_DATA).arg("*", "*"));
        qDebug() << "Remove " << path << " + " << QString(FILE_PHOTOS_DATA).arg("*", "*") << " returned " << ret;

        path = Utils::getPhotoDir(info->name);
        dir = QDir(path);
        ret = Utils::RemoveFiles(dir, "*");
        qDebug() << "Remove " << path << " returned " << ret;
    }
}

Friend ServiceMgr::getFriend(QString accountId, QString ownerId)
{
    QString service = this->account(accountId)->serviceName();

    // step 1. check friends from all accounts

    for (int i = 0; i < this->accounts.length(); i++)
    {
        if (this->accounts.at(i)->serviceName().compare(service) != 0)
            continue;

        // check user id
        if (this->accounts.at(i)->getProfile(false).id == ownerId)
            return this->accounts.at(i)->getProfile(false);

        // check loaded user friends
        FriendList curFriends = this->accounts.at(i)->getFriendList();
        for (int j = 0; j < curFriends.length(); j++)
        {
            if (curFriends.at(j).id == ownerId)
                return curFriends.at(j);
        }
    }

    // step 2. check downloaded non-friends from all accounts
    for (int i = 0; i < this->accounts.length(); i++)
    {
        if (this->accounts.at(i)->serviceName().compare(service) != 0)
            continue;

        Friend nonFr = Friend::loadData(this->accounts.at(i)->accountId(), ownerId);
        if (nonFr.id.compare(ownerId) == 0)
            return nonFr;
    }

    // step 3. download non-friend profile
    // TODO: add parallel work
    this->account(accountId)->transport->getProfile(ownerId);

    Friend ret;

    return ret;
}

Friend ServiceMgr::getFriend(PhotoComment comment)
{
    return this->getFriend(comment.accountId, comment.senderId);
}
