#ifndef QTRANSPORT_H
#define QTRANSPORT_H

#include <QObject>
#include <QLibrary>
#include <QDebug>
#include <QDomDocument>
#include <QDomElement>
#include <QDateTime>

#include "mysocials_driver.h"

#include "datatypes/driverinfo.h"
#include "datatypes/photo.h"
#include "datatypes/album.h"
#include "datatypes/friend.h"
#include "datatypes/message.h"
#include "datatypes/qerrormessage.h"

struct MethodElement
{
    QString className;
    QString functionName;
};

class QTransport : public QObject
{
    Q_OBJECT
public:
    enum Action
    {
        getListMethodsAction,
        getProfileAction,
        getListFriendsAction,
        deleteFriendAction,
        getSettingsAction,
        setSettingsAction,
        getOutboxMessagesAction,
        getInboxMessagesAction,
        sendMessageAction,
        readMessageAction,
        deleteMessageAction,
        getPhotoAction,
        uploadPhotoAction,
        getListPhotosAction,
        getCommentsAction,
        sendCommentAction,
        getListAlbumsAction
    };

private:
    // Pointer to social network driver library
    QLibrary *driver;

    // profile settings in library
    struct msa_module *driverModule;

    // count of active requests
    int activeRequests;

    // true if driver must shuutdown
    bool needShutdown;

    // true if library init well
    bool isLibraryInit;

    // list of registered functions
    QList<MethodElement> registerdFunctions;

    QDomDocument createRequest(QString usedClass, QString usedFunction, bool noAuthorize = false);
    QDomDocument createRequest(QString usedClass, QString usedFunction, bool noAuthorize, QString content);
    QDomDocument sendRequest(QDomDocument doc, Action acc);

    QString generateFileName(QString path, QString url);

    bool checkFunction(QString usedClass, QString usedFunction, Action acc);
    bool checkBadResponse(QDomDocument resp, Action acc);
    bool checkGoodResponse(QDomDocument resp, Action acc);

    void getSettings();

    void getMethods();

public:
    QTransport(QString libName, QString accName, QObject *parent = 0);
    ~QTransport();

    // Account id. uses to identify account of service (e.g. vkontakte0)
    QString accountId;

    // the name of library file (e.g. libvkontakte.so)
    QString libraryName;

    QString serviceName();

    DriverInfo *getDriverInfo();
    static DriverInfo *getDriverInfo(QString libraryPath);

    void init(QString proxyHost, uint proxyPort);
    void setSettings(QString settings);
    void close();
    bool ready();

    void getProfile();
    void getProfile(QString ownerId);

    void getFriends();
    bool deleteFriend(QString ownerId);
    bool deleteFriend(Friend owner);

    void getAlbums(QString ownedId);
    void getPhotos(QString ownerId, QString albumId);
    void getPhotoComments(Photo photo);
    bool sendPhotoComment(Photo photo, QString comment);
    bool uploadPhoto(Album al, QString local_file_name);
    bool downloadPhoto(QString url, QString local_file_name);
    void downloadPhotoList(PhotoList list, Photo curPhoto, int nearest);


    void getInbox();
    void getOutbox();
    bool sendMessage(QString ownerId, QString title, QString message);
    bool sendMessage(Friend owner, QString title, QString message);
    bool readMessage(QString messageId);
    bool readMessage(Message msg);
    bool deleteMessage(QString messageId);
    bool deleteMessage(Message msg);


signals:
    void albumsReceived(QString accountid, QString friendId, AlbumList, bool isLastUpdate);
    void photosReceived(QString accountid, QString friendId, QString albumId, PhotoList, bool isLastUpdate);
    void photoUploaded(QString accountId, QString albumId, QString local_file_name);
    void commentsReceived(QString accountId, QString friendId, QString albumId, QString photoId, PhotoCommentList list);

    void friendsReceived(QString accountid, FriendList, bool isLastUpdate);
    void friendDeleted(QString accountId, QString friendId);

    void settingsReceived(QString accountid, QString);
    void profileReceived(QString accountId, QString reqOwnerId, Friend data);

    void inboxMessagesReceived(QString accountId, MessageList list, bool isLastUpdate);
    void outboxMessagesReceived(QString accountId, MessageList list, bool isLastUpdate);
    void messageSended(QString accountId, QString ownerId, QString title, QString text);
    void messageReaded(QString accountId, QString messageId);
    void messageDeleted(QString accountId, QString messageId);

    void errorOccurred(QString accountId, QErrorMessage msg, QTransport::Action acc);
};

// the next part of code defines standard messages that returns from driver.
// It's need for correct translation.
// Prease upgrade it if possible.

#if 0
class QTransport
{
#define DRV_SETT_ERROR QT_TR_NOOP("Unable to save settings")
#define DRV_AUTH_ERROR QT_TR_NOOP("Authorization problem")
#define DRV_AUTH_CANCEL QT_TR_NOOP("Authorization canceled")
#define DRV_REQ_ERROR QT_TR_NOOP("Unable to perform request")
#define DRV_SERVER_BUSY QT_TR_NOOP("Server busy")
#define DRV_REQ_PARAM QT_TR_NOOP("One of the parameters missing or invalid")
#define DRV_CAPT_ERROR QT_TR_NOOP("Captcha")
#define DRV_RESP_ERROR QT_TR_NOOP("Bad response from server")
#define DRV_NETW_ERROR QT_TR_NOOP("Network problem")
#define DRV_MESS_LEN QT_TR_NOOP("Empty message")
#define DRV_ACCESS_ERROR QT_TR_NOOP("Operation prohibited by privacy")
#define DRV_SERVER_ERROR QT_TR_NOOP("Internal Server Error")
#define DRV_FILE_SAVE_ERROR QT_TR_NOOP("Can't save file")

// from libvkontakte - errors.xml
QT_TR_NOOP("Incorrect signature")
};
#endif

#endif // QTRANSPORT_H
