#include "qtransport.h"

typedef  gint (*driver_init)(msa_module *);

QTransport::QTransport(QString libraryname, QString accountName, QObject *parent) :
    QObject(parent), driverModule(0), isLibraryInit(false)
{
    this->activeRequests = 0;
    this->needShutdown = false;

    this->libraryName = libraryname;
    // Add dynamic library (libvkontakte.so)
    if ((this->driver = new QLibrary(libraryname)) == 0)
        qFatal("Problems with initiation QLibrary object");

    driver->setLoadHints(QLibrary::ResolveAllSymbolsHint); // analog RTLD_LAZY
    // Check driver existance
    if (!driver->isLoaded())
        if (!driver->load())
        {
            qCritical() << "Can't load driver library: %s" << driver->errorString();
            return;
        }

    if ((driverModule = new struct msa_module) == 0)
        qFatal("can't init msa_module");

    // fill important properties
    driverModule->proxy = 0;
    driverModule->port = 0;

    if (accountName.isNull() || accountName.isEmpty())
        driverModule->id = NULL;
    else
        driverModule->id = g_strdup((gchar*) accountName.toStdString().data());

    driver_init dr_init = (driver_init)driver->resolve(MODULE_INIT);
    dr_init(driverModule);

    if (driverModule == 0)
    {
        qCritical() << "can't resolve funtion of  library: " << driver->errorString();
        return;
    }
    this->accountId = QString(driverModule->id);

    this->getMethods();
    isLibraryInit = true;
}

QTransport::~QTransport()
{
    if (driver != 0) {
        if (!driver->unload()) {
            qWarning() << "can't unload  library: " << driver->errorString();
        }
        delete driver;
        if (driverModule != NULL)
            delete driverModule;
        qDebug() << "qtransport deleted";
    }

}

QString QTransport::serviceName()
{
    return QString::fromUtf8(this->driverModule->name);
}

bool QTransport::ready()
{
    return isLibraryInit;

}
DriverInfo *QTransport::getDriverInfo()
{
    DriverInfo *ret = new DriverInfo(this->parent());

    // name
    ret->name = QString::fromUtf8(this->driverModule->name);
    // library
    ret->library = this->libraryName;
    // icon
    QImage image;
    if (!image.loadFromData(QByteArray::fromBase64(this->driverModule->pic)))
        qDebug() << "Error loading image for " << this->accountId << ": " << this->driverModule->pic;
    QPixmap pix = QPixmap::fromImage(image);
    ret->icon = QIcon(pix);

    return ret;
}

DriverInfo *QTransport::getDriverInfo(QString libraryPath)
{
    QLibrary *lib;
    // Add dynamic library (libvkontakte.so)
    if ((lib = new QLibrary(libraryPath)) == 0) {
        qWarning("Problems with initiation QLibrary object");
        return NULL;
    }

    lib->setLoadHints(QLibrary::ResolveAllSymbolsHint); // analog RTLD_LAZY
    // Check driver existance
    if (!lib->isLoaded())
        if (!lib->load()) {
            qWarning() << QString("can't load driver library: ") + lib->errorString();
            delete lib;
            return NULL;
        }

    struct msa_module * dm;

    if ((dm = new struct msa_module) == 0) {
        qWarning() << "can't init msa_module";
        lib->unload();
        delete lib;
        return NULL;
    }

    driver_init dr_init = (driver_init)lib->resolve(MODULE_GET_INFO);
    if (dr_init == NULL) {
        qWarning() << QString("Cann't find function ") + QString(MODULE_GET_INFO) +
                QString(". Probably old library type");
        delete dm;
        lib->unload();
        delete lib;
        return NULL;
    }

    dr_init(dm);

    if (dm == 0) {
        qWarning() << QString("can't resolve funtion of  library: ") + lib->errorString();
        lib->unload();
        delete lib;
        return NULL;
    }
    DriverInfo *ret = new DriverInfo();
    ret->name = QString::fromUtf8(dm->name);
    // library
    ret->library = libraryPath;
    // icon
    QImage image;
    if (!image.loadFromData(QByteArray::fromBase64(dm->pic)))
        qDebug() << "Error loading image: " << dm->pic;
    QPixmap pix = QPixmap::fromImage(image);
    ret->icon = QIcon(pix);

    delete dm;
    lib->unload();
    delete lib;

    return ret;
}

/** init()
  * @param strProxyHost
  * @param strProxyPort
  */
void QTransport::init(QString proxyHost, uint proxyPort)
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << "()";

    if (proxyHost.isNull()) {
        driverModule->proxy = (gchar*) NULL;
    } else {
        char *host = new char[proxyHost.toLocal8Bit().size()+1];
        strcpy(host,proxyHost.toLocal8Bit().data());
        driverModule->proxy = (gchar*) host;
    }
    driverModule->port = (gint) proxyPort;

    // Send data to driver module
    qDebug() << "init(): OK" << endl;
}

void QTransport::close()
{
    this->needShutdown = true;
    while (this->activeRequests != 0)
    {
        sleep(1);
    }

    if (this->activeRequests == 0)
        if (driverModule != NULL && this->ready()) {
            qDebug() << "Shutdown..."<< endl;
            driverModule->shutdown(driverModule);
            qDebug() << "Shutdown released"<< endl;
        }
}

QString QTransport::generateFileName(QString path, QString url)
{
    QString icon = NULL;
    if (url != NULL) {
        QFileInfo fi(url);
        QString ext = fi.suffix();
        icon = path + QString(url.toUtf8().toBase64()) + (ext.isEmpty() ? QString() : QString(".") + ext);
    }
    return icon;
}

QDomDocument QTransport::createRequest(QString usedClass, QString usedFunction, bool noAuthorize)
{
    return this->createRequest(usedClass, usedFunction, noAuthorize, "<Params/>");
}

QDomDocument QTransport::createRequest(QString usedClass, QString usedFunction, bool noAuthorize, QString content)
{
    QDomDocument ret;

    QDomElement req = ret.createElement("Request");
    req.setAttribute("class", usedClass);
    req.setAttribute("function", usedFunction);
    req.setAttribute("noAuthorize", noAuthorize ? "true" : "false");
    ret.appendChild(req);

    // parse content
    QDomDocument c;
    c.setContent(content);
    QDomNode contNode = ret.importNode(c.documentElement(), true);
    req.appendChild(contNode);

    return ret;
}

QDomDocument QTransport::sendRequest(QDomDocument doc, Action acc)
{
    QDomDocument ret;
    if(this->needShutdown){
        return ret;
    } else
        this->activeRequests++;

    xmlChar* request = xmlCharStrdup(doc.toByteArray(0).constData());
    xmlDocPtr reqXml = xmlParseDoc(request);
    xmlDocPtr respXml = NULL;

    xmlFree(request);
    // Process request
    if (this->driverModule != NULL) {
        qDebug() << __FILE__ << ":" << __LINE__ << ": ========== REQUEST ==============\n" <<
                doc.toString() << "============ END REQUEST =============";
        try {
            driverModule->send(reqXml, &respXml, driverModule);
        } catch (...) {
            qDebug() << "Caught an exception.";
            QErrorMessage msg(-1, tr("Caught an exception."));
            emit errorOccurred(this->accountId, msg, acc);
        }
    } else {
        qDebug() << "driverModule == NULL" << endl;
    }
    activeRequests--;

    xmlChar *resp = NULL;
    int respLen = 0;
    xmlDocDumpMemory(respXml, &resp, &respLen);

    ret.setContent(QString((char *)resp));
    xmlFree(resp);
    xmlFreeDoc(respXml);

    qDebug() << __FILE__ << ":" << __LINE__ << ": ========== RESPONSE ==============\n" <<
            ret.toString() << "============ END RESPONSE =============";

    return ret;
}

bool QTransport::checkGoodResponse(QDomDocument resp, Action acc)
{
    if (!this->checkBadResponse(resp, acc))
        return false;

    qDebug() << __FILE__ <<":"<< __LINE__<<": response not error!";

    QDomNodeList msg = resp.elementsByTagName("Response");
    for (int i = 0; i < msg.count(); i++)
    {
        if (msg.at(i).toElement().attribute("class").compare(CLASS_SYSTEM_MESSAGES) == 0 &&
            msg.at(i).toElement().attribute("function").compare(INFO_MESSAGE) == 0)
        {
            return true;
        }
    }
    qDebug() << __FILE__ <<":"<< __LINE__<<": Response code not found!";
    return false;
}

bool QTransport::checkBadResponse(QDomDocument resp, Action acc)
{
    bool ret = true;

    QDomNodeList msg = resp.elementsByTagName("Response");
    for (int i = 0; i < msg.count(); i++)
    {
        QDomElement curNode = msg.at(i).toElement();
        // check authorization flag
        if (curNode.attribute("authorized").compare("true") == 0)
        {
            this->getSettings();
        }

        // check error message
        if (curNode.attribute("class").compare(CLASS_SYSTEM_MESSAGES) == 0)
        {
            if (curNode.attribute("function").compare(ERROR_MESSAGE) == 0)
            {
                QErrorMessage msg;
                QDomNodeList params = curNode.elementsByTagName("Params").at(0).toElement().childNodes();
                for (int j = 0; j < params.count(); j++)
                {
                    QDomElement curNode = params.at(j).toElement();
                    if (curNode.nodeName().compare("string") == 0)
                    {
                        if (curNode.attribute("name").compare("code") == 0)
                            msg.code = curNode.firstChild().nodeValue().toInt();
                        if (curNode.attribute("name").compare("text") == 0)
                            msg.text = tr(curNode.firstChild().nodeValue().trimmed().toAscii());
                        if (curNode.attribute("name").compare("comment") == 0)
                            msg.comment = tr(curNode.firstChild().nodeValue().trimmed().toAscii());

                    }
                }
                qDebug() << __FILE__ << ":" << __LINE__ << ": code=" << msg.code;
                emit errorOccurred(this->accountId, msg, acc);
                ret = false;
            }
        }
    }

    return ret;
}

bool QTransport::checkFunction(QString usedClass, QString usedFunction, QTransport::Action acc)
{
    for (int i = 0; i < this->registerdFunctions.length(); i++)
    {
        if (this->registerdFunctions.at(i).className.compare(usedClass) == 0 &&
            this->registerdFunctions.at(i).functionName.compare(usedFunction) == 0)
            return true;
    }

    QErrorMessage msg;
    msg.code = 0;
    msg.text = tr("Unsupported method was called");
    msg.comment = tr("This method didn't supported by selected driver");

    emit errorOccurred(this->accountId, msg, acc);
    return false;
}

void QTransport::getMethods()
{
    QDomDocument req = this->createRequest(CLASS_SETTINGS, GET_LIST_METHODS, true);
    QDomDocument res = this->sendRequest(req, this->getListMethodsAction);

    if (!this->checkBadResponse(res, this->getListMethodsAction))
        return;

    // parse response
    if (res.elementsByTagName("Response").count() == 0 ||
        res.elementsByTagName("Response").at(0).toElement().attribute("function").compare(GET_LIST_METHODS) != 0)
        return;

    QDomNodeList methods = res.elementsByTagName("Response").at(0).toElement().
                           elementsByTagName("Params").at(0).toElement().
                           elementsByTagName("string");

    this->registerdFunctions.clear();

    for (int i = 0; i < methods.count(); i++)
    {
        MethodElement elem;
        elem.className = methods.at(i).toElement().attribute("class");
        elem.functionName = methods.at(i).toElement().attribute("function");

        this->registerdFunctions.append(elem);
    }

}

void QTransport::getFriends()
{
    if (!this->checkFunction(CLASS_FRIENDS, GET_LIST_FRIENDS, this->getListFriendsAction))
        return;

    QDomDocument req = this->createRequest(CLASS_FRIENDS, GET_LIST_FRIENDS);
    QDomDocument res = this->sendRequest(req, this->getListFriendsAction);

    if (!this->checkBadResponse(res, this->getListFriendsAction))
        return;

    // parse response
    if (res.elementsByTagName("Response").count() == 0 ||
        res.elementsByTagName("Response").at(0).toElement().attribute("function").compare(GET_LIST_FRIENDS) != 0)
        return;

    QDomNodeList friends = res.elementsByTagName("Response").at(0).toElement().
                           elementsByTagName("Params").at(0).toElement().
                           elementsByTagName("array").at(0).toElement().
                           elementsByTagName("struct");
    FriendList ret;

    for (int i = 0; i < friends.count(); i++)
    {
        Friend fr;
        QDomElement curFr = friends.at(i).toElement();
        fr.accountId = this->accountId;
        fr.id = curFr.attribute("id");

        for (uint j = 0; j < curFr.childNodes().length(); j++)
        {
            QDomElement curElem = curFr.childNodes().at(j).toElement();
            if (curElem.nodeName().compare("string") == 0)
            {
                if (curElem.attribute("name").compare("FirstName") == 0)
                {
                    fr.firstName = curElem.firstChild().nodeValue().trimmed();
                }

                if (curElem.attribute("name").compare("NickName") == 0)
                {
                    fr.nickName = curElem.firstChild().nodeValue().trimmed();
                }

                if (curElem.attribute("name").compare("LastName") == 0)
                {
                    fr.lastName = curElem.firstChild().nodeValue().trimmed();
                }

                if (curElem.attribute("name").compare("FriendStatus") == 0)
                {
                    fr.online = curElem.firstChild().nodeValue().trimmed() == "0" ? false : true;
                }
            }

            if (curElem.nodeName().compare("img") == 0)
            {
                if (curElem.attribute("name").compare("Img") == 0)
                {
                    // Icon
                    fr.icon_url = curElem.firstChild().nodeValue().trimmed();

                    QDir dir;
                    if (!dir.exists(Utils::getFriendsIconDir(this->driverModule->name)))
                        dir.mkpath(Utils::getFriendsIconDir(this->driverModule->name));
                    QString icon = NULL;
                    if (!fr.icon_url.isEmpty()){
                        icon = generateFileName(Utils::getFriendsIconDir(this->driverModule->name), fr.icon_url);
                        QFile file(icon);
                        if(file.exists())
                            fr.icon = icon;
                    }
                }
            }
        }
        ret.append(fr);
    }
    qDebug() << "Total friends:" + ret.length();
    emit friendsReceived(this->accountId, ret, false);

    // === update friend icons ===
    int count = 0;

    for(int i = 0; i < ret.length(); i++){
        if (ret.at(i).icon_url.isEmpty())
            continue;

        QString icon = generateFileName(Utils::getFriendsIconDir(this->driverModule->name),
                                        ret.at(i).icon_url);
        QFile file(icon);
        if(!file.exists()){
            count++;
            if(downloadPhoto(ret.at(i).icon_url, icon))
                ret[i].icon = icon;
            if(count > 0 && count%10 == 0){
                qDebug() << "emit friendsReceived signal" << endl;
                emit friendsReceived(this->accountId, ret, false);
            }
        } else
            ret[i].icon = icon;
    }

    emit friendsReceived(this->accountId, ret, true);
}

bool QTransport::deleteFriend(Friend owner)
{
    return this->deleteFriend(owner.id);
}

bool QTransport::deleteFriend(QString ownerId)
{
    if (!this->checkFunction(CLASS_FRIENDS, DELETE_FRIEND, this->deleteFriendAction))
        return false;

    QDomDocument req = this->createRequest(CLASS_FRIENDS, DELETE_FRIEND);
    if (!ownerId.isEmpty())
    {
        QDomNodeList params = req.elementsByTagName("Params");
        params.at(0).toElement().setAttribute("id", ownerId);
    }
    QDomDocument res = this->sendRequest(req, this->deleteFriendAction);

    // parse response
    if (this->checkGoodResponse(res, this->deleteFriendAction))
    {
        emit this->friendDeleted(this->accountId, ownerId);
        return true;
        qDebug() << __FILE__ <<":"<<__LINE__ << "Response is correct";
    }

    qDebug() << __FILE__ <<":"<<__LINE__ << "Response incorrect!!!";
    return false;
}

void QTransport::getSettings()
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << "()";

    if (!this->checkFunction(CLASS_SETTINGS, GET_SETTINGS, this->getSettingsAction))
        return;

    QDomDocument req = this->createRequest(CLASS_SETTINGS, GET_SETTINGS);
    QDomDocument res = this->sendRequest(req, this->getSettingsAction);

    if (this->checkBadResponse(res, this->getSettingsAction))
    {
        QString settings;
        QTextStream str(&settings);
        res.elementsByTagName("Response").at(0).toElement().elementsByTagName("Params").at(0).save(str, 0);
        emit this->settingsReceived(this->accountId, settings);
    }
}

void QTransport::getProfile()
{
    this->getProfile(QString::null);
}

void QTransport::getProfile(QString ownerId)
{
    if (!this->checkFunction(CLASS_PROFILE, GET_PROFILE, this->getProfileAction))
        return;

    QDomDocument req = this->createRequest(CLASS_PROFILE, GET_PROFILE);
    if (!ownerId.isEmpty())
    {
        QDomNodeList params = req.elementsByTagName("Params");
        params.at(0).toElement().setAttribute("id", ownerId);
    }
    QDomDocument res = this->sendRequest(req, this->getProfileAction);

    if (!this->checkBadResponse(res, this->getProfileAction))
        return;

    // parse response
    if (res.elementsByTagName("Response").count() == 0 ||
        res.elementsByTagName("Response").at(0).toElement().attribute("function").compare(GET_PROFILE) != 0)
        return;

    QDomElement params = res.elementsByTagName("Response").at(0).toElement().
                         elementsByTagName("Params").at(0).toElement();
    QDomNodeList paramNodes = params.childNodes();

    Friend profile;
    profile.accountId = this->accountId;
    QString icon;

    profile.id = params.attribute("id");

    for (int i = 0; i < paramNodes.count(); i++)
    {
        QDomElement curElem = paramNodes.at(i).toElement();
        if (curElem.nodeName().compare("string") == 0)
        {
            if (curElem.attribute("name").compare("FirstName") == 0)
            {
                profile.firstName = curElem.firstChild().nodeValue().trimmed();
            }

            if (curElem.attribute("name").compare("NickName") == 0)
            {
                profile.nickName = curElem.firstChild().nodeValue().trimmed();
            }

            if (curElem.attribute("name").compare("LastName") == 0)
            {
                profile.lastName = curElem.firstChild().nodeValue().trimmed();
            }

            if (curElem.attribute("name").compare("Gender") == 0) {
                profile.gender = curElem.firstChild().nodeValue();
            }

            if (curElem.attribute("name").compare("Birthday") == 0) {
                profile.birthday = curElem.firstChild().nodeValue();
            }

            if (curElem.attribute("name").compare("MobilePhone") == 0) {
                profile.mobilePhone = curElem.firstChild().nodeValue();
            }

            if (curElem.attribute("name").compare("HomePhome") == 0) {
                profile.homePhone = curElem.firstChild().nodeValue();
            }

            if (curElem.attribute("name").compare("CityName") == 0) {
                profile.city = curElem.firstChild().nodeValue();
            }

            if (curElem.attribute("name").compare("CountryName") == 0) {
                profile.country = curElem.firstChild().nodeValue();
            }
        }

        if (curElem.nodeName().compare("img") == 0)
        {
            if (curElem.attribute("name").compare("Img") == 0)
            {
                // Icon
                profile.icon_url = curElem.firstChild().nodeValue().trimmed();
                QDir dir;
                if (!dir.exists(Utils::getFriendsIconDir(this->driverModule->name)))
                    dir.mkpath(Utils::getFriendsIconDir(this->driverModule->name));
                if (!profile.icon_url.isEmpty()){
                    icon = generateFileName(Utils::getFriendsIconDir(this->driverModule->name), profile.icon_url);
                    QFile file(icon);
                    if(file.exists())
                        profile.icon = icon;
                }
            }
        }

    }
    emit profileReceived(this->accountId, ownerId, profile);

    if (!icon.isEmpty() && !profile.icon_url.isEmpty()){
        QFile file(icon);
        if(!file.exists()){
            if(downloadPhoto(profile.icon_url, icon)) {
                profile.icon = icon;
                emit profileReceived(this->accountId, ownerId, profile);
            }
        }
    }
}

void QTransport::setSettings(QString settings)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << "()";

    if (!this->checkFunction(CLASS_SETTINGS, SET_SETTINGS, this->setSettingsAction))
        return;

    QDomDocument req = this->createRequest(CLASS_SETTINGS, SET_SETTINGS, true, settings);
    QDomDocument res = this->sendRequest(req, this->setSettingsAction);

    // parse response
    if (this->checkGoodResponse(res, this->setSettingsAction))
        qDebug() << __FILE__ <<":"<<__LINE__ << "Response is correct";
    else
        qDebug() << __FILE__ <<":"<<__LINE__ << "Response incorrect!!!";
}

void QTransport::getAlbums(QString ownerId)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << "(): Prepare request getAlbums for " << ownerId;

    if (!this->checkFunction(CLASS_PHOTOS, GET_LIST_ALBUMS, this->getListAlbumsAction))
        return;

    QDomDocument req = this->createRequest(CLASS_PHOTOS, GET_LIST_ALBUMS);

    QDomNodeList params = req.elementsByTagName("Params");
    params.at(0).toElement().setAttribute("id", ownerId);

    QDomDocument res = this->sendRequest(req, this->getListAlbumsAction);

    if (!this->checkBadResponse(res, this->getListAlbumsAction))
        return;

    QDomElement array = res.elementsByTagName("Response").at(0).toElement().
                        elementsByTagName("Params").at(0).toElement().
                        elementsByTagName("array").at(0).toElement();

    AlbumList ret;

    if (array.attribute("ownerId").compare(ownerId) != 0)
    {
        emit albumsReceived(this->accountId, ownerId, ret, true);
        return;
    }

    QDomNodeList albums = array.elementsByTagName("struct");

    for (int i = 0; i < albums.count(); i++)
    {
        QDomElement curAlbum = albums.at(i).toElement();
        Album al;
        al.accountId = this->accountId;
        al.ownerId = ownerId;
        al.albumId = curAlbum.attribute("id");

        for (int j = 0; j < curAlbum.childNodes().count(); j++)
        {
            QDomElement curItem = curAlbum.childNodes().at(j).toElement();
            if (curItem.nodeName().compare("string") == 0)
            {
                if (curItem.attribute("name").compare("title") == 0)
                {
                    al.title = curItem.firstChild().nodeValue().trimmed();
                }

                if (curItem.attribute("name").compare("description") == 0)
                {
                    al.description = curItem.firstChild().nodeValue().trimmed();
                }

                if (curItem.attribute("name").compare("ownerId") == 0)
                {
                    al.ownerId = curItem.firstChild().nodeValue().trimmed();
                }
            }
            if (curItem.nodeName().compare("img") == 0)
            {
                if (curItem.attribute("name").compare("Img") == 0)
                {
                    al.icon_url = curItem.firstChild().nodeValue().trimmed();

                    QDir dir;
                    if (!dir.exists(Utils::getAlbumsIconDir(this->driverModule->name)))
                        dir.mkpath(Utils::getAlbumsIconDir(this->driverModule->name));
                    QString icon = NULL;
                    if (!al.icon_url.isEmpty()){
                        icon = generateFileName(Utils::getAlbumsIconDir(this->driverModule->name), al.icon_url);
                        QFile file(icon);
                        if(file.exists())
                            al.icon = icon;
                    }
                }
            }
            if (curItem.nodeName().compare("number") == 0)
            {
                if (curItem.attribute("name").compare("created") == 0)
                {
                    QDateTime mdf;
                    mdf.setTime_t(curItem.firstChild().nodeValue().toInt());
                    al.time_create = mdf.toString(DATE_TIME_FORMAT);
                }
                if (curItem.attribute("name").compare("updated") == 0)
                {
                    QDateTime mdf;
                    mdf.setTime_t(curItem.firstChild().nodeValue().toInt());
                    al.time_update = mdf.toString(DATE_TIME_FORMAT);
                }
                if (curItem.attribute("name").compare("size") == 0)
                {
                    al.size = curItem.firstChild().nodeValue().toInt();
                }
            }
        }
        qDebug() << "time_update=" << al.time_update << "; time_ceate=" << al.time_create;
        if (al.time_update.isEmpty() && !al.time_create.isEmpty())
            al.time_update = al.time_create;
        ret.append(al);
    }

    emit albumsReceived(this->accountId, ownerId, ret, ret.length() == 0 ? true : false);

    if (ret.length() == 0)
        return;

    // update album icons
    int count = 0;

    QDir dir;
    if (!dir.exists(Utils::getAlbumsIconDir(this->driverModule->name)))
        dir.mkpath(Utils::getAlbumsIconDir(this->driverModule->name));
    qDebug()<< "Length AlbumList = "<<ret.length()<<endl;
    for(int i = 0; i < ret.length(); i++){
        if (ret.at(i).icon_url.isEmpty())
            continue;

        QString icon = NULL;
        icon = generateFileName(Utils::getAlbumsIconDir(this->driverModule->name), ret.at(i).icon_url);
        QFile file(icon);
        if(!file.exists()){
            if(downloadPhoto(ret.at(i).icon_url, icon))
                ret[i].icon = icon;
            count++;
            if(count > 0 && count%10 == 0)
            {
                qDebug() << "emit AlbumsReceived signal" << endl;
                emit albumsReceived(this->accountId, ownerId, ret, false);
            }

        } else
            ret[i].icon = icon;
    }
    emit albumsReceived(this->accountId, ownerId, ret, true);
}

void QTransport::getPhotos(QString ownerId, QString albumId)
{
    if (!this->checkFunction(CLASS_PHOTOS, GET_LIST_PHOTOS, this->getListPhotosAction))
        return;

    QDomDocument req = this->createRequest(CLASS_PHOTOS, GET_LIST_PHOTOS);

    QDomNodeList params = req.elementsByTagName("Params");
    params.at(0).toElement().setAttribute("id", ownerId);

    QDomElement albElem = req.createElement("string");
    albElem.setAttribute("name", "albumId");
    albElem.appendChild(req.createTextNode(albumId));
    params.at(0).toElement().appendChild(albElem);

    QDomDocument res = this->sendRequest(req, this->getListPhotosAction);

    if (!this->checkBadResponse(res, this->getListPhotosAction))
        return;

    QDomElement array = res.elementsByTagName("Response").at(0).toElement().
                        elementsByTagName("Params").at(0).toElement().
                        elementsByTagName("array").at(0).toElement();

    PhotoList ret;

    if (array.attribute("ownerId").compare(ownerId) != 0 ||
        array.attribute("albumId").compare(albumId) != 0)
    {
        emit this->photosReceived(this->accountId, ownerId, albumId, ret, true);
        qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << "() wrong Id's: ownerId="
                << array.attribute("ownerId") << "; albumId=" << array.attribute("albumId") << "; need:"
                << ownerId << " and " << albumId;
        return;
    }

    QDomNodeList photos = array.elementsByTagName("struct");

    for (int i = 0; i < photos.count(); i++)
    {
        QDomElement curPhoto = photos.at(i).toElement();
        Photo ph;
        ph.photoId = curPhoto.attribute("id");
        ph.albumId = albumId;
        ph.ownerId = ownerId;
        ph.accountId = this->accountId;

        for (int j = 0; j < curPhoto.childNodes().count(); j++)
        {
            QDomElement curItem = curPhoto.childNodes().at(j).toElement();
            if (curItem.nodeName().compare("string") == 0)
            {
                if (curItem.attribute("name").compare("albumId") == 0)
                {
                    ph.albumId = curItem.firstChild().nodeValue().trimmed();
                }

                if (curItem.attribute("name").compare("ownerId") == 0)
                {
                    ph.ownerId = curItem.firstChild().nodeValue().trimmed();
                }

                if (curItem.attribute("name").compare("urlSmall") == 0)
                {
                    ph.icon_url = curItem.firstChild().nodeValue().trimmed();

                    QDir dir;
                    if (!dir.exists(Utils::getAlbumsIconDir(this->driverModule->name)))
                        dir.mkpath(Utils::getAlbumsIconDir(this->driverModule->name));
                    QString icon = NULL;
                    if (!ph.icon_url.isEmpty()){
                        icon = generateFileName(Utils::getAlbumsIconDir(this->driverModule->name),
                                                ph.icon_url);
                        QFile file(icon);
                        if(file.exists())
                            ph.icon = icon;
                    }
                }

                if (curItem.attribute("name").compare("urlBig") == 0)
                {
                    ph.photo_url = curItem.firstChild().nodeValue().trimmed();

                    QDir dir;
                    if (!dir.exists(Utils::getPhotoDir(this->driverModule->name)))
                        dir.mkpath(Utils::getPhotoDir(this->driverModule->name));

                    QString img = NULL;
                    if (!ph.photo_url.isEmpty()){
                        img = generateFileName(Utils::getPhotoDir(this->driverModule->name),
                                                ph.photo_url);
                        QFile file(img);
                        if(file.exists())
                            ph.photo = img;
                    }
                }
            }
            if (curItem.nodeName().compare("number") == 0)
            {
                if (curItem.attribute("name").compare("created") == 0)
                {
                    QDateTime mdf;
                    mdf.setTime_t(curItem.firstChild().nodeValue().toInt());
                    ph.time_create = mdf.toString(DATE_TIME_FORMAT);
                }
            }
        }
        ret.append(ph);
    }
    emit this->photosReceived(this->accountId, ownerId, albumId, ret, ret.length() > 0 ? false : true);

    if (ret.length() == 0)
        return;

    int count = 0;

    QDir dir;
    if (!dir.exists(Utils::getAlbumsIconDir(this->driverModule->name)))
        dir.mkpath(Utils::getAlbumsIconDir(this->driverModule->name));
    qDebug()<< "Length AlbumList = "<<ret.length()<<endl;

    for (int i = 0; i < ret.length(); i++) {
        if (ret.at(i).icon_url.isEmpty())
            continue;

        QString icon = NULL;
        icon = generateFileName(Utils::getAlbumsIconDir(this->driverModule->name),
                                ret.at(i).icon_url);
        QFile file(icon);
        if (!file.exists()) {
            count++;
            if (downloadPhoto(ret.at(i).icon_url, icon))
                ret[i].icon = icon;
            if (count > 0 && count%10 == 0) {
                qDebug() << "emit PhotosReceived signal";
                emit this->photosReceived(this->accountId, ownerId, albumId, ret, false);
            }
        } else
            ret[i].icon = icon;

    }
    emit this->photosReceived(this->accountId, ownerId, albumId, ret, true);
}

void QTransport::getPhotoComments(Photo photo)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << "()";

    if (!this->checkFunction(CLASS_PHOTOS, GET_LIST_PHOTO_COMMENTS, this->getCommentsAction))
        return;

    QDomDocument req = this->createRequest(CLASS_PHOTOS, GET_LIST_PHOTO_COMMENTS);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    QDomElement pr = req.createElement("string");
    pr.setAttribute("name", "ownerId");
    pr.appendChild(req.createTextNode(photo.ownerId));
    params.appendChild(pr);

    pr = req.createElement("string");
    pr.setAttribute("name", "albumId");
    pr.appendChild(req.createTextNode(photo.albumId));
    params.appendChild(pr);

    pr = req.createElement("string");
    pr.setAttribute("name", "photoId");
    pr.appendChild(req.createTextNode(photo.photoId));
    params.appendChild(pr);

    QDomDocument res = this->sendRequest(req, this->getCommentsAction);

    if (!this->checkBadResponse(res, this->getCommentsAction))
        return;


    QDomElement array = res.elementsByTagName("Response").at(0).toElement().
                        elementsByTagName("Params").at(0).toElement().
                        elementsByTagName("array").at(0).toElement();

    PhotoCommentList ret;

    if (array.attribute("ownerId").compare(photo.ownerId) != 0 ||
        array.attribute("albumId").compare(photo.albumId) != 0 ||
        array.attribute("photoId").compare(photo.photoId) != 0)
    {
        emit this->commentsReceived(this->accountId, photo.ownerId, photo.albumId, photo.photoId, ret);
        return;
    }

    QDomNodeList comments = array.elementsByTagName("struct");

    for (int i = 0; i < comments.count(); i++)
    {
        QDomElement curComment = comments.at(i).toElement();
        PhotoComment ch;
        ch.commentId = curComment.attribute("id");
        ch.accountId = this->accountId;
        ch.ownerId = photo.ownerId;
        ch.albumId = photo.albumId;
        ch.photoId = photo.photoId;

        for (int j = 0; j < curComment.childNodes().count(); j++)
        {
            QDomElement curItem = curComment.childNodes().at(j).toElement();
            if (curItem.nodeName().compare("string") == 0)
            {
                if (curItem.attribute("name").compare("SenderId") == 0)
                {
                    ch.senderId = curItem.firstChild().nodeValue().trimmed();
                }

                if (curItem.attribute("name").compare("SenderName") == 0)
                {
                    ch.senderName = curItem.firstChild().nodeValue().trimmed();
                }

                if (curItem.attribute("name").compare("Time") == 0)
                {
                    QDateTime mdf;
                    mdf.setTime_t(curItem.firstChild().nodeValue().toInt());
                    ch.time = mdf.toString(DATE_TIME_FORMAT);
                }

                if (curItem.attribute("name").compare("Text") == 0)
                {
                    ch.text = curItem.firstChild().nodeValue().trimmed();
                }
            }
        }
        ret.append(ch);
    }
    emit this->commentsReceived(this->accountId, photo.ownerId, photo.albumId, photo.photoId, ret);
}

bool QTransport::sendPhotoComment(Photo photo, QString comment)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << "()";

    if (!this->checkFunction(CLASS_PHOTOS, SEND_PHOTO_COMMENT, this->sendCommentAction))
        return false;

    QDomDocument req = this->createRequest(CLASS_PHOTOS, SEND_PHOTO_COMMENT);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    QDomElement node = req.createElement("string");
    node.setAttribute("name", "ownerId");
    node.appendChild(req.createTextNode(photo.ownerId));
    params.appendChild(node);

    node = req.createElement("string");
    node.setAttribute("name", "albumId");
    node.appendChild(req.createTextNode(photo.albumId));
    params.appendChild(node);

    node = req.createElement("string");
    node.setAttribute("name", "photoId");
    node.appendChild(req.createTextNode(photo.photoId));
    params.appendChild(node);

    node = req.createElement("string");
    node.setAttribute("name", "text");
    node.appendChild(req.createTextNode(comment));
    params.appendChild(node);

    QDomDocument res = this->sendRequest(req, this->sendCommentAction);

    if (this->checkGoodResponse(res, this->sendCommentAction))
    {
        return true;
    } else {
        return false;
    }
}

bool QTransport::uploadPhoto(Album al, QString file_name)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << "()";

    if (!this->checkFunction(CLASS_PHOTOS, UPLOAD_PHOTO, this->uploadPhotoAction))
        return false;

    QDomDocument req = this->createRequest(CLASS_PHOTOS, UPLOAD_PHOTO);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    QDomElement node = req.createElement("string");
    node.setAttribute("name", "albumId");
    node.appendChild(req.createTextNode(al.albumId));
    params.appendChild(node);

    QFileInfo finfo(file_name);

    node = req.createElement("string");
    node.setAttribute("name", "file");
    node.appendChild(req.createTextNode(finfo.dir().path()));
    params.appendChild(node);

    node = req.createElement("string");
    node.setAttribute("name", "fileName");
    node.appendChild(req.createTextNode(finfo.fileName()));
    params.appendChild(node);

    QDomDocument res = this->sendRequest(req, this->uploadPhotoAction);

    if (this->checkGoodResponse(res, this->uploadPhotoAction))
    {
        emit this->photoUploaded(al.accountId, al.albumId, file_name);
        return true;
    } else {
        return false;
    }
}

bool QTransport::downloadPhoto(QString url, QString file_name)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << "()";

    if (!this->checkFunction(CLASS_PHOTOS, GET_PHOTO, this->getPhotoAction))
        return false;

    QDomDocument req = this->createRequest(CLASS_PHOTOS, GET_PHOTO);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    QDomElement urlNode = req.createElement("string");
    urlNode.setAttribute("name", "url");
    urlNode.appendChild(req.createTextNode(url));
    params.appendChild(urlNode);

    QDomElement fileNode = req.createElement("string");
    fileNode.setAttribute("name", "path");
    fileNode.appendChild(req.createTextNode(file_name));
    params.appendChild(fileNode);

    QDomDocument res = this->sendRequest(req, this->getPhotoAction);

    if (this->checkGoodResponse(res, this->getPhotoAction))
    {
        return true;
    } else {
        return false;
    }
}

void QTransport::downloadPhotoList(PhotoList list, Photo curPhoto, int nearest) {

    if (nearest < 0)
        return;

    if(list.isEmpty())
        return;

    int count = 0;

    QDir dir;
    if (!dir.exists(Utils::getPhotoDir(this->driverModule->name)))
        dir.mkpath(Utils::getPhotoDir(this->driverModule->name));

    int len = list.length();
    int startItem = -1;
    for (int i = 0; i < len; i++) {
        if (list.at(i).photoId == curPhoto.photoId) {
            startItem = i;
            break;
        }
    }

    for(int i = 0; i <= nearest; i++)
    {
        for (int j = -1; j < 2; j += 2)
        {
            if (this->needShutdown)
                return;

            if (i == 0 && j > 0)
                continue;

            int cp = startItem + i * j;
            qDebug() << "download image: start=" << startItem << " i=" << i << " cp=" << cp;
            while (cp < 0)
                cp += len;

///FIXME
            while(cp >= list.size())
                cp -= len;

            qDebug()<< "photo_url = "<< list.at(cp).photo_url << endl;

            QString photo_path = generateFileName(Utils::getPhotoDir(this->driverModule->name), list.at(cp).photo_url);
            QFile file(photo_path);
            if(!file.exists()){
                downloadPhoto(list.at(cp).photo_url, file.fileName());
                count++;
                list[cp].photo = photo_path;
                qDebug() << "emit PhotosDownload signal" << endl;
                emit photosReceived(curPhoto.accountId, curPhoto.ownerId, curPhoto.albumId, list, false);
            }else{
                list[cp].photo = photo_path;
            }
        }
    }
    qDebug() << "Finish download image for album " << curPhoto.albumId;
    emit photosReceived(curPhoto.accountId, curPhoto.ownerId, curPhoto.albumId, list, true);
}

void QTransport::getInbox() {
    qDebug() << __FILE__ << ":" << __FUNCTION__;

    if (!this->checkFunction(CLASS_MESSAGES, GET_LIST_INBOX_MESSAGES, this->getInboxMessagesAction))
        return;

    QDomDocument req = this->createRequest(CLASS_MESSAGES, GET_LIST_INBOX_MESSAGES);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    QDomElement pr = req.createElement("number");
    pr.setAttribute("name", "from");
    pr.appendChild(req.createTextNode(QString::number(0)));
    params.appendChild(pr);

    pr = req.createElement("number");
    pr.setAttribute("name", "to");
    pr.appendChild(req.createTextNode(QString::number(100)));
    params.appendChild(pr);

    qDebug() << "REQUEST: " << req.toString();

    QDomDocument res = this->sendRequest(req, this->getInboxMessagesAction);

    qDebug() << "RESPONSE: " << res.toString();

    if (!this->checkBadResponse(res, this->getInboxMessagesAction)) {
        qDebug() << __FILE__ << ":" << __LINE__ << ": getInbox check response error";
        return;
    }

    QDomElement array = res.elementsByTagName("Response").at(0).toElement().
                        elementsByTagName("Params").at(0).toElement().
                        elementsByTagName("array").at(0).toElement();

    MessageList ret;

    QDomNodeList messages = array.childNodes();

    for (int i = 0; i < messages.count(); i++)
    {
        QDomElement curMessage = messages.at(i).toElement();

        Message msg;
        msg.messageId = curMessage.attribute("id");
        msg.accountId = this->accountId;

        for (int j = 0; j < curMessage.childNodes().count(); j++)
        {
            QDomElement curItem = curMessage.childNodes().at(j).toElement();

            if (curItem.nodeName().compare("string") == 0)
            {
                if (curItem.attribute("name").compare("SenderId") == 0)
                {
                    msg.senderId = curItem.firstChild().nodeValue();
                }

                if (curItem.attribute("name").compare("SenderName") == 0)
                {
                    msg.senderName = curItem.firstChild().nodeValue();
                }

                if (curItem.attribute("name").compare("Time") == 0)
                {
                    QDateTime mdf;
                    mdf.setTime_t(curItem.firstChild().nodeValue().toInt());
                    msg.time = mdf.toString("dd.MM.yyyy hh:mm:ss");
                }

                if (curItem.attribute("name").compare("Text") == 0)
                {
                    msg.text = curItem.firstChild().nodeValue();
                }

                if (curItem.attribute("name").compare("Title") == 0)
                {
                    msg.title = curItem.firstChild().nodeValue();
                }
            }
        }
        ret.append(msg);
    }

    qDebug() << __FILE__ << __LINE__ << "emit inboxMessagesReceived";
    emit this->inboxMessagesReceived(this->accountId, ret, true);
}

void QTransport::getOutbox() {
    qDebug() << __FILE__ << ":" << __FUNCTION__;

    if (!this->checkFunction(CLASS_MESSAGES, GET_LIST_OUTBOX_MESSAGES, this->getOutboxMessagesAction))
        return;

    QDomDocument req = this->createRequest(CLASS_MESSAGES, GET_LIST_OUTBOX_MESSAGES);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    QDomElement pr = req.createElement("number");
    pr.setAttribute("name", "from");
    pr.appendChild(req.createTextNode(QString::number(0)));
    params.appendChild(pr);

    pr = req.createElement("number");
    pr.setAttribute("name", "to");
    pr.appendChild(req.createTextNode(QString::number(100)));
    params.appendChild(pr);

    qDebug() << "REQUEST: " << req.toString();

    QDomDocument res = this->sendRequest(req, this->getOutboxMessagesAction);

    qDebug() << "RESPONSE: " << res.toString();

    if (!this->checkBadResponse(res, this->getOutboxMessagesAction)) {
        qDebug() << __FILE__ << ":" << __LINE__ << ": getOutbox check response error";
        return;
    }

    QDomElement array = res.elementsByTagName("Response").at(0).toElement().
                        elementsByTagName("Params").at(0).toElement().
                        elementsByTagName("array").at(0).toElement();

    MessageList ret;

    QDomNodeList messages = array.childNodes();

    for (int i = 0; i < messages.count(); i++)
    {
        QDomElement curMessage = messages.at(i).toElement();

        Message msg;
        msg.messageId = curMessage.attribute("id");
        msg.accountId = this->accountId;

        for (int j = 0; j < curMessage.childNodes().count(); j++)
        {
            QDomElement curItem = curMessage.childNodes().at(j).toElement();

            if (curItem.nodeName().compare("string") == 0)
            {
                if (curItem.attribute("name").compare("SenderId") == 0)
                {
                    msg.senderId = curItem.firstChild().nodeValue();
                }

                if (curItem.attribute("name").compare("SenderName") == 0)
                {
                    msg.senderName = curItem.firstChild().nodeValue();
                }

                if (curItem.attribute("name").compare("Time") == 0)
                {
                    QDateTime mdf;
                    mdf.setTime_t(curItem.firstChild().nodeValue().toInt());
                    msg.time = mdf.toString("dd.MM.yyyy hh:mm:ss");
                }

                if (curItem.attribute("name").compare("Text") == 0)
                {
                    msg.text = curItem.firstChild().nodeValue();
                }

                if (curItem.attribute("name").compare("Title") == 0)
                {
                    msg.title = curItem.firstChild().nodeValue();
                }
            }
        }
        ret.append(msg);
    }

    qDebug() << __FILE__ << __LINE__ << "emit outboxMessagesReceived";
    emit this->outboxMessagesReceived(this->accountId, ret, true);
}

bool QTransport::readMessage(Message msg)
{
    return this->readMessage(msg.messageId);
}

bool QTransport::readMessage(QString messageId)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << "()";

    if (!this->checkFunction(CLASS_MESSAGES, READ_MESSAGE, this->readMessageAction))
        return false;

    QDomDocument req = this->createRequest(CLASS_MESSAGES, READ_MESSAGE);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    QDomElement node = req.createElement("string");
    node.setAttribute("name", "messageId");
    node.appendChild(req.createTextNode(messageId));
    params.appendChild(node);

    QDomDocument res = this->sendRequest(req, this->readMessageAction);

    if (this->checkGoodResponse(res, this->readMessageAction))
    {
        emit this->messageReaded(this->accountId, messageId);
        return true;
    } else {
        return false;
    }
}

bool QTransport::deleteMessage(Message msg)
{
    return this->deleteMessage(msg.messageId);
}

bool QTransport::deleteMessage(QString messageId)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << "()";

    if (!this->checkFunction(CLASS_MESSAGES, DELETE_MESSAGE, this->deleteMessageAction))
        return false;

    QDomDocument req = this->createRequest(CLASS_MESSAGES, DELETE_MESSAGE);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    QDomElement node = req.createElement("string");
    node.setAttribute("name", "messageId");
    node.appendChild(req.createTextNode(messageId));
    params.appendChild(node);

    QDomDocument res = this->sendRequest(req, this->deleteMessageAction);

    if (this->checkGoodResponse(res, this->deleteMessageAction))
    {
        emit this->messageDeleted(this->accountId, messageId);
        return true;
    } else {
        return false;
    }
}

bool QTransport::sendMessage(Friend owner, QString title, QString message)
{
    return this->sendMessage(owner.id, title, message);
}

bool QTransport::sendMessage(QString ownerId, QString title, QString message)
{
    qDebug() << __FILE__<<":"<< __LINE__<<":" << __FUNCTION__ << "()";

    if (!this->checkFunction(CLASS_MESSAGES, SEND_MESSAGE, this->sendMessageAction))
        return false;

    QDomDocument req = this->createRequest(CLASS_MESSAGES, SEND_MESSAGE);

    QDomElement params = req.elementsByTagName("Params").at(0).toElement();

    params.setAttribute("id", ownerId);

    QDomElement node = req.createElement("string");
    node.setAttribute("name", "text");
    node.appendChild(req.createTextNode(message));
    params.appendChild(node);

    node = req.createElement("string");
    node.setAttribute("name", "title");
    node.appendChild(req.createTextNode(title));
    params.appendChild(node);

    QDomDocument res = this->sendRequest(req, this->sendMessageAction);

    if (this->checkGoodResponse(res, this->sendMessageAction))
    {
        emit this->messageSended(this->accountId, ownerId, title, message);
        return true;
    } else {
        return false;
    }
}
