/*
 * This file is part of mydicto
 *
 * Copyright (C) 2010 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <gst/gst.h>
#include <hildon/hildon.h>
#include <libosso.h>

#ifdef WITH_SHARING

#ifdef WITH_SHARING_H
#include <sharingdialog/sharing-dialog.h>
#endif

#else

void sharing_dialog_with_file (osso_context_t * osso, GtkWindow * parent, const gchar * uri);

#endif

#include <dicto/dicto.h>
#include <dicto/dicto-ui.h>

#include "config.h"

HildonProgram *program;
HildonWindow *window;

osso_context_t *osso;

GtkWidget *dw;
Dicto *d;

static gint
mydicto_dbus_req_handler(const gchar *interface, const gchar *method, GArray *arguments, gpointer data, osso_rpc_t *retval)
{
return OSSO_OK;
}

static gboolean
dicto_error_cb(Dicto *d, gpointer err, gpointer data)
{
GError *e=(GError *)err;

g_debug("Dicto reported error: (%d) %s", e->code, e->message);
hildon_banner_show_information(window, "", e->message);
return FALSE;
}

static gboolean
dicto_delete_query_cb(GtkWidget *dw, const gchar *file, gpointer data)
{
return dicto_widget_delete_query_dialog(GTK_WINDOW(window), file);
}

static gboolean
dicto_stop_cb(Dicto *d, gpointer data)
{
hildon_banner_show_information(window, "", "Stopped");
return FALSE;
}

static gboolean
dicto_playing_cb(Dicto *d, gpointer data)
{
hildon_banner_show_information(window, "", "Playing");
return FALSE;
}

static gboolean
dicto_recording_cb(Dicto *d, gpointer data)
{
hildon_banner_show_information(window, "", "Recording");
return FALSE;
}

static void
mydicto_menu_button_share_cb(GtkWidget *w, gpointer data)
{
DictoWidget *dw=(DictoWidget *)data;
gchar *file;
#ifdef WITH_SHARING
g_object_get(G_OBJECT(d), "current-file", &file, NULL);
g_debug("File: %s", file);
if (file && strlen(file)>0 && g_file_test(file, G_FILE_TEST_IS_REGULAR)) {
	sharing_dialog_with_file(osso, GTK_WINDOW(window), file);
} else {
	hildon_banner_show_information(window, "", "No file selected");
}
g_free(file);
#else
hildon_banner_show_information(window, "", "Sharing disabled");
#endif
}

static HildonAppMenu *
mydicto_menu(void)
{
HildonAppMenu *menu;
GtkWidget *button;

menu=HILDON_APP_MENU(hildon_app_menu_new());

// Create a button and add it to the menu
button=gtk_button_new_with_label("Share");
g_signal_connect_after(button, "clicked", G_CALLBACK(mydicto_menu_button_share_cb), dw);
hildon_app_menu_append(menu, GTK_BUTTON(button));

gtk_widget_show_all(GTK_WIDGET(menu));
return menu;
}

static gboolean
mydicto_init(DictoFormat f, gchar *src, gchar *sink)
{
osso_return_t result;

program=HILDON_PROGRAM(hildon_program_get_instance());
window=HILDON_WINDOW(hildon_stackable_window_new());

hildon_window_set_app_menu(HILDON_WINDOW(window), mydicto_menu());

g_set_application_name("myDicto");
gtk_window_set_default_icon_name("mydicto");

gtk_window_set_title(GTK_WINDOW(window), "myDicto");
g_signal_connect(G_OBJECT(window), "destroy", G_CALLBACK(gtk_main_quit), NULL);

osso=osso_initialize("org.tal.mydicto", VERSION, FALSE, NULL);
if (!osso)
	return FALSE;

result=osso_rpc_set_cb_f(osso, "org.tal.mydicto", "/org/tal/mydicto", "org.tal.mydicto", mydicto_dbus_req_handler, NULL);
if (result!=OSSO_OK)
	return FALSE;

d=dicto_new_full(NOTES_BASEDIR, f, sink, src);
if (!d)
	return FALSE;

dw=dicto_widget_new(d);

g_signal_connect(G_OBJECT(d), "error", G_CALLBACK(dicto_error_cb), dw);
g_signal_connect(G_OBJECT(d), "playing", G_CALLBACK(dicto_playing_cb), NULL);
g_signal_connect(G_OBJECT(d), "recording", G_CALLBACK(dicto_recording_cb), NULL);
g_signal_connect(G_OBJECT(d), "stopped-play", G_CALLBACK(dicto_stop_cb), NULL);
g_signal_connect(G_OBJECT(d), "stopped-record", G_CALLBACK(dicto_stop_cb), NULL);
g_signal_connect(G_OBJECT(dw), "delete-query", G_CALLBACK(dicto_delete_query_cb), dw);
g_object_set(dw, "enable-delete", TRUE, NULL);

gtk_container_add(GTK_CONTAINER(window), dw);
gtk_widget_show_all(GTK_WIDGET(window));

return TRUE;
}

gint 
main(gint argc, gchar * argv[])
{
gchar *sink, *src;
DictoFormat f=FORMAT_DEFAULT;

hildon_gtk_init(&argc, &argv);
gst_init(&argc, &argv);

sink=AUDIO_SINK;
src=AUDIO_SRC;

/* Veeery simple argument passing, mainly for getting working sinks under sdk */
if (argc>1)
	f=atoi(argv[1]);

if (argc>2)
	src=argv[2];

if (argc>3)
	sink=argv[3];

g_debug("Using format %d, source %s and sink %s", f, src, sink);

if (argc>4) {
	g_setenv("PULSE_PROP_media.role", argv[4], TRUE);
	g_debug("Pulseaudio role set to: %s", argv[4]);
}


if (!mydicto_init(f, src, sink)) {
	g_warning("myDicto init failed");
	return 1;
}

gtk_main();

return 0;
}

