#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "settingsdialog.h"
#include "tubestatusdelegate.h"
#include "tubestatusmodel.h"

#include <QtGui>
#include <QStyle>
#include <QtNetwork>
#if 0
#ifdef Q_WS_MAEMO_5
#include <QNetworkConfigurationManager>
#endif
#endif

#include <qjson/parser.h>
#include <qjson/serializer.h>

#define PROGRAM_NAME    "mwQTubeStatus"
#define VERSION         "0.1.0"


MainWindow::MainWindow(QWidget *parent) :
#if 1
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);
    dataView = ui->dataView;
    lastUpdatedLabel = ui->labelUpdatedTime;
#else
    QMainWindow(parent)
{
    dataView = new QTableView;
    dataView->horizontalHeader()->hide();
    dataView->verticalHeader()->hide();
    dataView->setGridStyle(Qt::NoPen);
    dataView->setStyleSheet("QTableView::item"
                            "{"
                            "  margin: 5px;"
                            "  border: 5px;"
                            "  padding: 5px;"
                            "}");
    QObject::connect(dataView, SIGNAL(clicked(QModelIndex)),
                              SLOT(onTubeStatusPressed(QModelIndex)));
    lastUpdatedLabel = new QLabel;

//    QScrollArea * sa = new QScrollArea(this);
//    this->setCentralWidget(sa);
//    QVBoxLayout * vb = new QVBoxLayout(sa);
//    sa->setLayout(vb);
//    vb->addWidget(dataView);
//    vb->addWidget(lastUpdatedLabel);
    this->setCentralWidget(dataView);
#endif

    this->setWindowTitle(PROGRAM_NAME);

#ifdef Q_WS_MAEMO_5
    qDebug("Creating Maemo 5 QT Main Window");
    this->setAttribute(Qt::WA_Maemo5StackedWindow);
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
#else
    qDebug("Creating standard QT Main Window");
#endif

    timer = NULL;
    settings = NULL;

    netMan = new QNetworkAccessManager(this);
    QObject::connect(netMan, SIGNAL(finished(QNetworkReply *)),
                              SLOT(onGetTubeUpdateFinished(QNetworkReply *)));

    QList<TubeStatus> * dataList = new QList<TubeStatus>();

    TubeStatusModel * tsm = new TubeStatusModel();
    tsm->setTubeStatusList(*dataList);

    dataView->setModel(tsm);
    dataView->setItemDelegate(new TubeStatusDelegate());
    dataView->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
    //ui->dataView->verticalHeader()->setResizeMode(QHeaderView::Stretch);

    QFile file(QDir::homePath() + "/mwtube.dat");

    bool ok;
    ok = file.open(QIODevice::ReadOnly| QIODevice::Text);
    if( ok ) {
        QJson::Parser p;
        QVariant data = p.parse(&file, &ok);
        if( ok && data.canConvert<QVariantMap>() ) {
            response = data.toMap();
            updateModel();
        }
        file.close();
    }

    this->readSettings();

#ifdef Q_WS_MAEMO_5
    this->orientationChanged();
#endif

    // Get updated tube status as soon as we start the app
    updateRunning = false;
    if( this->autoUpdate ) {
        QTimer::singleShot(0, this, SLOT(on_actionUpdate_triggered()));
    }
}

MainWindow::~MainWindow()
{
#if 0
    delete ui;
#else
    delete dataView;
    delete lastUpdatedLabel;
#endif
    if( netMan )
        delete netMan;
    if( timer )
        delete timer;
    if( settings )
        delete settings;
}

void MainWindow::onTubeStatusPressed(const QModelIndex & index)
{
    qDebug("Entered onTubeStatusPressed()");
    QVariant data = index.data();
    if( ! data.canConvert<TubeStatus>() ) {
        qDebug("Cannot convert data to TubeStatus");
        return;
    }

    TubeStatus * ts = (TubeStatus *)data.data();
    if( ts->numMessages() == 0 ) {
        qDebug("TubeStatus has no messages");
        return;
    }

    QString contents = "<p><b>" + ts->getName() + " : " + ts->getStatus() + "</b></p>\n";
    foreach(QString msg, ts->getMessages()) {
        contents += "<hr width=\"80%\">\n<p>" + msg + "</p>\n";
    }
    contents += "<hr width=\"80%\">\n<p align=\"center\"><b>Status Requested: " +
                ts->getRequestedTime() + "</b></p>\n";

    QMainWindow * mw = new QMainWindow(this);
#ifdef Q_WS_MAEMO_5
    mw->setAttribute(Qt::WA_Maemo5StackedWindow);
#else
    mw->setWindowModality(Qt::ApplicationModal);
    mw->setGeometry(this->geometry());
#endif
    QScrollArea * sa = new QScrollArea(mw);
    sa->setWidgetResizable(true);
    mw->setCentralWidget(sa);

    QLabel * l = new QLabel(sa);
    sa->setWidget(l);
    l->setBackgroundRole(QPalette::Base);
    l->setForegroundRole(QPalette::Text);
    l->setWordWrap(true);
    l->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    l->setText(contents);

    qDebug("Opening tube status window");
    mw->show();
    mw->activateWindow();
}

void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        //ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void MainWindow::on_actionUpdate_triggered()
{
    qDebug("on_actionUpdate_triggered");

#if 0
    // Hmm, not in QT4.6 for Maemo5 ?
#ifdef Q_WS_MAEMO_5
    QNetworkConfigurationManager manager;
    const bool canStartIAP = (manager.capabilities() & QNetworkConfigurationManager::BearerManagement);
    QNetworkConfiguration cfg = manager.defaultConfiguration();
    if (!cfg.isValid() || !canStartIAP)
        return;
    switch(cfg.type()) {
        case QNetworkConfiguration::InternetAccessPoint:
            // no user interaction -> system starts IAP immediately
            break;
        case QNetworkConfiguration::ServiceNetwork:
            // no user interaction -> system determines best IAP in group and starts it
            break;
        case QNetworkConfiguration::UserChoice:
            // IAP resolved by asking user as part of QNetworkSession::open()
            break;
    }
    QNetworkSession* session = new QNetworkSession(cfg);
    session->open();
#endif
#endif

#if 0
    // This only works on QT4.7 :(
    if( this->netMan->networkAccessible() == QNetworkAccessManager::NotAccessible ) {
        if( autoConnect ) {
            qDebug("Not connected - auto connecting..");
            // TODO :Connect
        #if 0
            // PSEUDOCODE:
            if( !connected &&  this->autoConnect ) {
                    // TODO : try to connect
                }
            }
            if( !connected )
                return;

            // Use QBearer mobility classes?
        #endif
        } else {
            qDebug("Not connected - aborting.");
            return;
        }
        return;
    }
#endif

    // Check whether we're currently getting updates
    mutex.lock();
    if( updateRunning ) {
        return;
        mutex.unlock();
    } else {
        updateRunning = true;
        stopAutoUpdateTimer();
    }
    mutex.unlock();

    QUrl url = QUrl("http://api.tubeupdates.com/");
    url.addQueryItem("method", "get.status");
    QNetworkRequest request;
    request.setRawHeader("User-Agent", PROGRAM_NAME " " VERSION);
    request.setUrl(url);

    QNetworkReply *reply = 0;
    reply = netMan->get(request);

//    QObject::connect(reply, SIGNAL(downloadProgress(qint64,qint64)),
//                            SLOT(slotProgress(qint64,qint64)));

}

void MainWindow::onGetTubeUpdateFinished(QNetworkReply *reply)
{
    QJson::Parser parser;
    bool ok;

    // Schedule 'reply' to be deleted when control returns to the event loop
    reply->deleteLater();

    if( reply->error() != QNetworkReply::NoError ) {
        QString errStr = reply->errorString();
        qDebug("Error response : (%d)",reply->error());
        //goto finally;
    } else {
        // Reply should be a JSON string - so let's try to parse it!
        QVariantMap data = parser.parse(reply, &ok).toMap();

        if( !ok ) {
            qDebug("Error parsing response..");
            goto finally;
        }

        qDebug("Got Response!");

        if( data.contains("error") ) {
            QString errStr = data["error"].toString();
            qDebug("Error response: %s", errStr.toLocal8Bit().data() );
            goto finally;
        }
        response = data["response"].toMap();
        updateModel();
    }

finally:
    // Re-start the auto-update timer
    mutex.lock();
    updateRunning = false;
    if( autoUpdate ) {
        startAutoUpdateTimer();
    }
    mutex.unlock();
}

void MainWindow::updateModel()
{
    TubeStatusModel * tsm = (TubeStatusModel*)dataView->model();

    foreach(QVariant lineD, response["lines"].toList() ) {
        TubeStatus * ts = new TubeStatus(lineD);
        tsm->setTubeStatus(*ts);
    }
    dataView->setModel(tsm);
    //ui->dataView->resizeColumnsToContents();
    dataView->resizeRowsToContents();

    QVariantMap line = response["lines"].toList().first().toMap();
    lastUpdatedLabel->setText("Last Updated: " +
                                  line["status_requested"].toString());
}

int MainWindow::top_application()
{
    show();
    return 0;
}

void MainWindow::closeEvent(QCloseEvent * event)
{
    // TODO: Check whether data has actually changed..
    QJson::Serializer serializer;
    QString fName = QDir::homePath() + "/mwtube.dat";
    QFile file(fName);


    // Save state
    bool ok = file.open(QIODevice::WriteOnly | QIODevice::Truncate | QIODevice::Text);
    if( ok ) {
        file.write(serializer.serialize(response));
        file.close();
    }

    // Save settings
    this->writeSettings();

    event->accept();
}

void MainWindow::on_actionAbout_triggered()
{
    // TODO: Ensure About dialog can scroll if it doesn't fit on the screen.
    QMessageBox::about(this,
            tr("About mwQTubeStatus"),
            tr("<h2>"PROGRAM_NAME" v"VERSION"</h2>"
               "<p>Copyright &copy; 2010 Michael Werle</p>"
               "<p>"PROGRAM_NAME" is a small application that retrieves the "
               "current London Tube status and displays it in a tablet-"
               "friendly format.</p>"
               "<p>This application uses data retrieved from "
               "<a href=\"http://tubeupdates.com\">tubeupdates.com</a></p>"
               "<hr>"
               "<p>This Program is licensed under the "
               "<a href=\"http://www.opensource.org/licenses/bsd-license.php\">"
               "BSD Simplified License</a>.</p>"
               ));
}

void MainWindow::on_action1_Column_triggered()
{
    TubeStatusModel * tsm = (TubeStatusModel*)dataView->model();
    tsm->setColumns(1);
    dataView->resizeRowsToContents();
}

void MainWindow::on_action2_Columns_triggered()
{
    TubeStatusModel * tsm = (TubeStatusModel*)dataView->model();
    tsm->setColumns(2);
    dataView->resizeRowsToContents();
}

#ifdef Q_WS_MAEMO_5
void MainWindow::orientationChanged()
{
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    if (screenGeometry.width() < screenGeometry.height()) {
        qDebug() << "Rotated to Portrait Mode";
        on_action1_Column_triggered();
    } else {
        qDebug() << "Rotated to Landscape Mode";
        on_action2_Columns_triggered();
    }
}
#endif

void MainWindow::on_actionSettings_triggered()
{
    SettingsDialog sd(this);
    int result;

    sd.setAutoConnect(this->autoConnect);
    sd.setAutoUpdate(this->autoUpdate);
    sd.setRotation(this->rotationMode);
    result = sd.exec();

    if( QDialog::Accepted == result ) {
        // TODO: Accept new settings
        qDebug() << "Got new settings:\n  Auto-update = " << sd.getAutoUpdate()
                << "\n  Rotation-style = " << sd.getRotation();
        this->setAutoConnect(sd.getAutoConnect());
        this->setAutoUpdate(sd.getAutoUpdate());
        this->setRotationMode(sd.getRotation());
    }
}


void MainWindow::setAutoUpdate(bool autoUpdate, bool force)
{
    if( !force && this->autoUpdate == autoUpdate )
        return;

    // Start the auto-update timer
    mutex.lock();
    this->autoUpdate = autoUpdate;
    if( autoUpdate ) {
        startAutoUpdateTimer();
    } else {
        stopAutoUpdateTimer();
    }
    mutex.unlock();
}

void MainWindow::startAutoUpdateTimer()
{
    if( !timer ) {
        timer = new QTimer(this);
        if( !timer ) {
            qDebug("Out of Memory construction auto-update timer!");
            return;
        }
        timer->setInterval(5 * 60 * 1000);  // 5 minutes
        connect(timer, SIGNAL(timeout()), this, SLOT(on_actionUpdate_triggered()));
    }
    // Only actually start the timer if it's not currently running and we're
    // not in the middle of an update.
    if( !updateRunning && !timer->isActive() ) {
        timer->start();
    }
}

void MainWindow::stopAutoUpdateTimer()
{
    if( timer && timer->isActive() ) {
        timer->stop();
    }
}

void MainWindow::setAutoConnect(bool autoConnect, bool force)
{
    if( !force && this->autoConnect == autoConnect )
        return;

    this->autoConnect = autoConnect;
}

void MainWindow::setRotationMode(int rotationMode, bool force)
{
    if( !force && this->rotationMode == rotationMode )
        return;

    this->rotationMode = rotationMode;
#ifdef Q_WS_MAEMO_5
    switch(rotationMode) {
    case 0: // Landscape
        this->setAttribute(Qt::WA_Maemo5AutoOrientation, false);
        this->setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
        break;
    case 1: //
        this->setAttribute(Qt::WA_Maemo5AutoOrientation, false);
        this->setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
        break;
    case 2: // Auto-rotate
        this->setAttribute(Qt::WA_Maemo5LandscapeOrientation, false);
        this->setAttribute(Qt::WA_Maemo5PortraitOrientation, false);
        this->setAttribute(Qt::WA_Maemo5AutoOrientation, true);
        break;
    default:
        qDebug() << "ERROR - invalid rotation mode specified!";
    }
#endif
}

void MainWindow::readSettings()
{
    if( !settings ) {
        settings = new QSettings("Lemmurg", "mwQTubeStatus");
        if( !settings ) {
            qDebug("Out of Memory - could not load settings.");
            return;
        }
    }
    settings->beginGroup("GUI");
    this->setRotationMode(settings->value("Rotation", 0).toInt(), true);
    settings->endGroup();
    settings->beginGroup("Network");
    this->setAutoConnect(autoConnect = settings->value("AutoConnect", true).toBool());
    this->setAutoUpdate(autoUpdate = settings->value("AutoUpdate", true).toBool());
    settings->endGroup();
}

void MainWindow::writeSettings() const
{
    if( !settings )
        return;
    settings->beginGroup("GUI");
    settings->setValue("Rotation", rotationMode);
    settings->endGroup();
    settings->beginGroup("Network");
    settings->setValue("AutoConnect", autoConnect);
    settings->setValue("AutoUpdate", autoUpdate);
    settings->endGroup();
}
