/*
 * Copyright (C) 2015 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "pluginplaylist.h"
#include "pluginmanager.h"
#include "resources.h"

PluginPlaylist::PluginPlaylist(QObject *parent) :
    MKPlaylist(parent),
    m_request(0)
{
}

PluginPlaylist::PluginPlaylist(const QString &service, const QString &id, QObject *parent) :
    MKPlaylist(parent),
    m_request(0)
{
    loadPlaylist(service, id);
}

PluginPlaylist::PluginPlaylist(const QString &service, const QVariantMap &playlist, QObject *parent) :
    MKPlaylist(parent),
    m_request(0)
{
    loadPlaylist(service, playlist);
}

PluginPlaylist::PluginPlaylist(const PluginPlaylist *playlist, QObject *parent) :
    MKPlaylist(playlist, parent),
    m_request(0),
    m_tracksId(playlist->tracksId())
{
}

QString PluginPlaylist::tracksId() const {
    return m_tracksId;
}

void PluginPlaylist::setTracksId(const QString &i) {
    if (i != tracksId()) {
        m_tracksId = i;
        emit changed();
        emit tracksIdChanged();
    }
}

QString PluginPlaylist::errorString() const {
    return m_request ? m_request->errorString() : QString();
}

ResourcesRequest::Status PluginPlaylist::status() const {
    return m_request ? m_request->status() : ResourcesRequest::Null;
}

void PluginPlaylist::loadPlaylist(const QString &service, const QString &id) {
    setService(service);
    setId(id);
    
    if (status() == ResourcesRequest::Loading) {
        return;
    }
    
    if (ResourcesRequest *r = request()) {
        r->get(Resources::PLAYLIST, id);
        emit changed();
        emit statusChanged(status());
    }
}

void PluginPlaylist::loadPlaylist(const QString &service, const QVariantMap &playlist) {
    setService(service);
    setArtist(playlist.value("artist").toString());
    setArtistId(playlist.value("artistId").toString());
    setDate(playlist.value("date").toString());
    setDescription(playlist.value("description").toString());
    setGenre(playlist.value("genre").toString());
    setId(playlist.value("id").toString());
    setLargeThumbnailUrl(playlist.value("largeThumbnailUrl").toString());
    setThumbnailUrl(playlist.value("thumbnailUrl").toString());
    setTitle(playlist.value("title").toString());
    setTrackCount(playlist.value("trackCount").toInt());
    setTracksId(playlist.value("tracksId").toString());
    
     if (playlist.value("duration").type() == QVariant::String) {
        setDurationString(playlist.value("duration").toString());
    }
    else {
        setDuration(playlist.value("duration").toLongLong());
    }
}

void PluginPlaylist::loadPlaylist(PluginPlaylist *playlist) {
    MKPlaylist::loadPlaylist(playlist);
    setTracksId(playlist->tracksId());
}

ResourcesRequest* PluginPlaylist::request() {
    if (!m_request) {
        m_request = PluginManager::instance()->createRequestForService(service(), this);

        if (m_request) {
            connect(m_request, SIGNAL(finished()), this, SLOT(onRequestFinished()));
        }
    }

    return m_request;
}

void PluginPlaylist::onRequestFinished() {
    if (m_request->status() == ResourcesRequest::Ready) {
        loadPlaylist(service(), m_request->result().toMap());
    }
    
    emit changed();
    emit statusChanged(status());
}
